/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.store.file.utils;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.connector.base.source.reader.SourceReaderOptions;
import org.apache.flink.connector.file.src.FileSourceSplit;
import org.apache.flink.connector.file.src.reader.BulkFormat;
import org.apache.flink.connector.file.src.util.Utils;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.store.file.utils.ObjectSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);
    private static final int LIST_MAX_RETRY = 30;
    private static final Configuration DEFAULT_READER_CONFIG = new Configuration();
    public static final ForkJoinPool COMMON_IO_FORK_JOIN_POOL;

    public static <T> List<T> readListFromFile(Path path, ObjectSerializer<T> serializer, BulkFormat<RowData, FileSourceSplit> readerFactory) throws IOException {
        ArrayList result = new ArrayList();
        Utils.forEachRemaining(FileUtils.createFormatReader(readerFactory, path), row -> result.add(serializer.fromRow((RowData)row)));
        return result;
    }

    public static long getFileSize(Path path) throws IOException {
        return path.getFileSystem().getFileStatus(path).getLen();
    }

    public static String readFileUtf8(Path file) throws IOException {
        try (FSDataInputStream in = file.getFileSystem().open(file);){
            String line;
            BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)in, StandardCharsets.UTF_8));
            StringBuilder builder = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                builder.append(line);
            }
            String string = builder.toString();
            return string;
        }
    }

    public static void writeFileUtf8(Path file, String content) throws IOException {
        try (FSDataOutputStream out = file.getFileSystem().create(file, FileSystem.WriteMode.NO_OVERWRITE);){
            FileUtils.writeOutputStreamUtf8(out, content);
        }
    }

    public static void writeOutputStreamUtf8(FSDataOutputStream out, String content) throws IOException {
        OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, StandardCharsets.UTF_8);
        writer.write(content);
        writer.flush();
    }

    public static void deleteOrWarn(Path file) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ready to delete " + file.toString());
        }
        try {
            FileSystem fs = file.getFileSystem();
            if (!fs.delete(file, false) && fs.exists(file)) {
                LOG.warn("Failed to delete file " + file);
            }
        }
        catch (IOException e) {
            LOG.warn("Exception occurs when deleting file " + file, (Throwable)e);
        }
    }

    @Nullable
    public static FileStatus[] safelyListFileStatus(Path file) throws IOException {
        FileStatus[] statuses = null;
        for (int retry = 1; retry <= 30; ++retry) {
            try {
                statuses = file.getFileSystem().listStatus(file);
                break;
            }
            catch (FileNotFoundException e) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Failed to list file status, and {}", (Object)(retry < 29 ? "will retry again" : " exceeds max retry"));
                continue;
            }
        }
        return statuses;
    }

    public static Stream<Long> listVersionedFiles(Path dir, String prefix) throws IOException {
        if (!dir.getFileSystem().exists(dir)) {
            return Stream.of(new Long[0]);
        }
        FileStatus[] statuses = FileUtils.safelyListFileStatus(dir);
        if (statuses == null) {
            throw new RuntimeException(String.format("The return value is null of the listStatus for the '%s' directory.", dir));
        }
        return Arrays.stream(statuses).map(FileStatus::getPath).map(Path::getName).filter(name -> name.startsWith(prefix)).map(name -> Long.parseLong(name.substring(prefix.length())));
    }

    public static BulkFormat.Reader<RowData> createFormatReader(BulkFormat<RowData, FileSourceSplit> format, Path file) throws IOException {
        if (!file.getFileSystem().exists(file)) {
            throw new FileNotFoundException(String.format("File '%s' not found, Possible causes: 1.snapshot expires too fast, you can configure 'snapshot.time-retained' option with a larger value. 2.consumption is too slow, you can improve the performance of consumption (For example, increasing parallelism).", file));
        }
        long fileSize = FileUtils.getFileSize(file);
        FileSourceSplit split = new FileSourceSplit("ignore", file, 0L, fileSize);
        return format.createReader(DEFAULT_READER_CONFIG, split);
    }

    static {
        DEFAULT_READER_CONFIG.setInteger(SourceReaderOptions.ELEMENT_QUEUE_CAPACITY, 1);
        ForkJoinPool.ForkJoinWorkerThreadFactory factory = pool -> {
            ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
            worker.setName("file-store-common-io-" + worker.getPoolIndex());
            return worker;
        };
        COMMON_IO_FORK_JOIN_POOL = new ForkJoinPool(Runtime.getRuntime().availableProcessors(), factory, null, false);
    }
}

