/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.codehaus.groovy.transform.tailrec

import org.codehaus.groovy.control.CompilationFailedException
import org.codehaus.groovy.control.MultipleCompilationErrorsException

class TailRecursiveCompilationFailuresTest extends GroovyShellTestCase {

    void testMethodsWithoutRecursiveCallsFailCompilation() {
        try {
            evaluate("""
            import groovy.transform.TailRecursive
            class TargetClass {
            	@TailRecursive
            	void aVoidMethod() {
            		new Object()
            	}
				@TailRecursive
				static void aStaticVoidMethod() {
					new Object()
				}
				@TailRecursive
				int aFunction() {
					42
				}
				@TailRecursive
				static int aStaticFunction() {
					43
				}
            }
            new TargetClass()
        """)
            assert false, "MultipleCompilationErrorsException expected"
        } catch (MultipleCompilationErrorsException expected) {
            assert expected.errorCollector.errors.size() == 4
        }
    }

    void testFailIfNotAllRecursiveCallsCanBeTransformed() {
		shouldFail(CompilationFailedException) { evaluate("""
            import groovy.transform.TailRecursive
            class TargetClass {
            	@TailRecursive
            	int aNonTailRecursiveMethod() {
            		return 1 + aNonTailRecursiveMethod() 
            	}
            }
        """) }
	}

	void testFailIfNotAllStaticRecursiveCallsCanBeTransformed() {
		shouldFail(CompilationFailedException) { evaluate("""
            import groovy.transform.TailRecursive
            class TargetClass {
            	@TailRecursive
            	static int aNonTailRecursiveMethod() {
            		return 1 + aNonTailRecursiveMethod() 
            	}
            }
        """) }
	}

    void testFailIfRecursiveMethodCannotBeStaticallyCompiled() {
        shouldFail(CompilationFailedException) { evaluate("""
            import groovy.transform.TailRecursive
            import groovy.transform.CompileStatic

            @CompileStatic
            class TargetClass {
            	@TailRecursive
            	static int staticCountDown(zahl) {
            		if (zahl == 0)
            			return zahl
            		return staticCountDown(zahl - 1)
            	}
            }
            new TargetClass()
        """) }
    }

    void testTailRecursiveAsFirstAnnotationIsIncompatibleWithMemoized() {
        shouldFail(CompilationFailedException) {
            evaluate("""
            import groovy.transform.TailRecursive
            import groovy.transform.Memoized

            class TargetClass {
                @Memoized
            	@TailRecursive
            	static int countDown(int zahl) {
            		if (zahl == 0)
            			return zahl
            		return countDown(zahl - 1)
            	}
            }
            new TargetClass()
        """)
        }
    }

}
