/*
 * Decompiled with CFR 0.152.
 */
package io.scalecube.cluster.gossip;

import io.scalecube.cluster.ClusterMath;
import io.scalecube.cluster.Member;
import io.scalecube.cluster.gossip.Gossip;
import io.scalecube.cluster.gossip.GossipConfig;
import io.scalecube.cluster.gossip.GossipProtocol;
import io.scalecube.cluster.gossip.GossipRequest;
import io.scalecube.cluster.gossip.GossipState;
import io.scalecube.cluster.gossip.SequenceIdCollector;
import io.scalecube.cluster.membership.MembershipEvent;
import io.scalecube.cluster.transport.api.Message;
import io.scalecube.cluster.transport.api.Transport;
import io.scalecube.net.Address;
import io.scalecube.reactor.RetryNonSerializedEmitFailureHandler;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.Disposables;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.MonoSink;
import reactor.core.publisher.Sinks;
import reactor.core.scheduler.Scheduler;

public final class GossipProtocolImpl
implements GossipProtocol {
    private static final Logger LOGGER = LoggerFactory.getLogger(GossipProtocol.class);
    public static final String GOSSIP_REQ = "sc/gossip/req";
    private final Member localMember;
    private final Transport transport;
    private final GossipConfig config;
    private long currentPeriod = 0L;
    private long gossipCounter = 0L;
    private final Map<String, SequenceIdCollector> sequenceIdCollectors = new HashMap<String, SequenceIdCollector>();
    private final Map<String, GossipState> gossips = new HashMap<String, GossipState>();
    private final Map<String, MonoSink<String>> futures = new HashMap<String, MonoSink<String>>();
    private final List<Member> remoteMembers = new ArrayList<Member>();
    private int remoteMembersIndex = -1;
    private final Disposable.Composite actionsDisposables = Disposables.composite();
    private final Sinks.Many<Message> sink = Sinks.many().multicast().directBestEffort();
    private final Scheduler scheduler;

    public GossipProtocolImpl(Member localMember, Transport transport, Flux<MembershipEvent> membershipProcessor, GossipConfig config, Scheduler scheduler) {
        this.transport = Objects.requireNonNull(transport);
        this.config = Objects.requireNonNull(config);
        this.localMember = Objects.requireNonNull(localMember);
        this.scheduler = Objects.requireNonNull(scheduler);
        this.actionsDisposables.addAll(Arrays.asList(membershipProcessor.publishOn(scheduler).subscribe(this::onMembershipEvent, ex -> LOGGER.error("[{}][onMembershipEvent][error] cause:", (Object)localMember, ex)), transport.listen().publishOn(scheduler).filter(this::isGossipRequest).subscribe(this::onGossipRequest, ex -> LOGGER.error("[{}][onGossipRequest][error] cause:", (Object)localMember, ex))));
    }

    @Override
    public void start() {
        this.actionsDisposables.add(this.scheduler.schedulePeriodically(this::doSpreadGossip, this.config.gossipInterval(), this.config.gossipInterval(), TimeUnit.MILLISECONDS));
    }

    @Override
    public void stop() {
        this.actionsDisposables.dispose();
        this.sink.emitComplete((Sinks.EmitFailureHandler)RetryNonSerializedEmitFailureHandler.RETRY_NON_SERIALIZED);
    }

    @Override
    public Mono<String> spread(Message message) {
        return Mono.just((Object)message).subscribeOn(this.scheduler).flatMap(msg -> Mono.create(sink -> this.futures.put(this.createAndPutGossip((Message)msg), (MonoSink<String>)sink)));
    }

    @Override
    public Flux<Message> listen() {
        return this.sink.asFlux().onBackpressureBuffer();
    }

    private void doSpreadGossip() {
        long period = this.currentPeriod++;
        this.checkGossipSegmentation();
        if (this.gossips.isEmpty()) {
            return;
        }
        try {
            Set<String> gossipsThatSpread;
            this.selectGossipMembers().forEach(member -> this.spreadGossipsTo(period, (Member)member));
            Set<String> gossipsToRemove = this.getGossipsToRemove(period);
            if (!gossipsToRemove.isEmpty()) {
                LOGGER.debug("[{}][{}] Sweep gossips: {}", new Object[]{this.localMember, period, gossipsToRemove});
                for (String gossipId : gossipsToRemove) {
                    this.gossips.remove(gossipId);
                }
            }
            if (!(gossipsThatSpread = this.getGossipsThatMostLikelyDisseminated(period)).isEmpty()) {
                LOGGER.debug("[{}][{}] Most likely disseminated gossips: {}", new Object[]{this.localMember, period, gossipsThatSpread});
                for (String gossipId : gossipsThatSpread) {
                    MonoSink<String> sink = this.futures.remove(gossipId);
                    if (sink == null) continue;
                    sink.success((Object)gossipId);
                }
            }
        }
        catch (Exception ex) {
            LOGGER.warn("[{}][{}][doSpreadGossip] Exception occurred:", new Object[]{this.localMember, period, ex});
        }
    }

    private String createAndPutGossip(Message message) {
        long period = this.currentPeriod;
        Gossip gossip = this.createGossip(message);
        GossipState gossipState = new GossipState(gossip, period);
        this.gossips.put(gossip.gossipId(), gossipState);
        this.ensureSequence(this.localMember.id()).add(gossip.sequenceId());
        return gossip.gossipId();
    }

    private void onGossipRequest(Message message) {
        long period = this.currentPeriod;
        GossipRequest gossipRequest = (GossipRequest)message.data();
        for (Gossip gossip : gossipRequest.gossips()) {
            GossipState gossipState = this.gossips.get(gossip.gossipId());
            if (this.ensureSequence(gossip.gossiperId()).add(gossip.sequenceId()) && gossipState == null) {
                gossipState = new GossipState(gossip, period);
                this.gossips.put(gossip.gossipId(), gossipState);
                this.sink.emitNext((Object)gossip.message(), (Sinks.EmitFailureHandler)RetryNonSerializedEmitFailureHandler.RETRY_NON_SERIALIZED);
            }
            if (gossipState == null) continue;
            gossipState.addToInfected(gossipRequest.from());
        }
    }

    private void checkGossipSegmentation() {
        int intervalsThreshold = this.config.gossipSegmentationThreshold();
        for (Map.Entry<String, SequenceIdCollector> entry : this.sequenceIdCollectors.entrySet()) {
            SequenceIdCollector sequenceIdCollector = entry.getValue();
            if (sequenceIdCollector.size() <= intervalsThreshold) continue;
            LOGGER.warn("[{}][{}] Too many missed gossip messages from original gossiper: '{}', current node({}) was SUSPECTED much for a long time or connection problem", new Object[]{this.localMember, this.currentPeriod, entry.getKey(), this.localMember});
            sequenceIdCollector.clear();
        }
    }

    private void onMembershipEvent(MembershipEvent event) {
        Member member = event.member();
        if (event.isRemoved()) {
            boolean removed = this.remoteMembers.remove(member);
            this.sequenceIdCollectors.remove(member.id());
            if (removed) {
                LOGGER.debug("[{}][{}] Removed {} from remoteMembers list (size={})", new Object[]{this.localMember, this.currentPeriod, member, this.remoteMembers.size()});
            }
        }
        if (event.isAdded()) {
            this.remoteMembers.add(member);
            LOGGER.debug("[{}][{}] Added {} to remoteMembers list (size={})", new Object[]{this.localMember, this.currentPeriod, member, this.remoteMembers.size()});
        }
    }

    private boolean isGossipRequest(Message message) {
        return GOSSIP_REQ.equals(message.qualifier());
    }

    private Gossip createGossip(Message message) {
        return new Gossip(this.localMember.id(), message, this.gossipCounter++);
    }

    private SequenceIdCollector ensureSequence(String key) {
        return this.sequenceIdCollectors.computeIfAbsent(key, s -> new SequenceIdCollector());
    }

    private void spreadGossipsTo(long period, Member member) {
        List<Gossip> gossips = this.selectGossipsToSend(period, member);
        if (gossips.isEmpty()) {
            return;
        }
        Address address = member.address();
        gossips.stream().map(this::buildGossipRequestMessage).forEach(message -> this.transport.send(address, message).subscribe(null, ex -> LOGGER.debug("[{}][{}] Failed to send GossipReq({}) to {}, cause: {}", new Object[]{this.localMember, period, message, address, ex.toString()})));
    }

    private List<Gossip> selectGossipsToSend(long period, Member member) {
        int periodsToSpread = ClusterMath.gossipPeriodsToSpread(this.config.gossipRepeatMult(), this.remoteMembers.size() + 1);
        return this.gossips.values().stream().filter(gossipState -> gossipState.infectionPeriod() + (long)periodsToSpread >= period).filter(gossipState -> !gossipState.isInfected(member.id())).map(GossipState::gossip).collect(Collectors.toList());
    }

    private List<Member> selectGossipMembers() {
        int gossipFanout = this.config.gossipFanout();
        if (this.remoteMembers.size() < gossipFanout) {
            return this.remoteMembers;
        }
        if (this.remoteMembersIndex < 0 || this.remoteMembersIndex + gossipFanout > this.remoteMembers.size()) {
            Collections.shuffle(this.remoteMembers);
            this.remoteMembersIndex = 0;
        }
        List<Member> selectedMembers = gossipFanout == 1 ? Collections.singletonList(this.remoteMembers.get(this.remoteMembersIndex)) : this.remoteMembers.subList(this.remoteMembersIndex, this.remoteMembersIndex + gossipFanout);
        this.remoteMembersIndex += gossipFanout;
        return selectedMembers;
    }

    private Message buildGossipRequestMessage(Gossip gossip) {
        GossipRequest gossipRequest = new GossipRequest(gossip, this.localMember.id());
        return Message.withData((Object)gossipRequest).qualifier(GOSSIP_REQ).build();
    }

    private Set<String> getGossipsToRemove(long period) {
        int periodsToSweep = ClusterMath.gossipPeriodsToSweep(this.config.gossipRepeatMult(), this.remoteMembers.size() + 1);
        return this.gossips.values().stream().filter(gossipState -> period > gossipState.infectionPeriod() + (long)periodsToSweep).map(gossipState -> gossipState.gossip().gossipId()).collect(Collectors.toSet());
    }

    private Set<String> getGossipsThatMostLikelyDisseminated(long period) {
        int periodsToSpread = ClusterMath.gossipPeriodsToSpread(this.config.gossipRepeatMult(), this.remoteMembers.size() + 1);
        return this.gossips.values().stream().filter(gossipState -> period > gossipState.infectionPeriod() + (long)periodsToSpread).map(gossipState -> gossipState.gossip().gossipId()).collect(Collectors.toSet());
    }

    Transport getTransport() {
        return this.transport;
    }

    Member getMember() {
        return this.localMember;
    }
}

