<!--
   Licensed to the Apache Software Foundation (ASF) under one
   or more contributor license agreements.  See the NOTICE file
   distributed with this work for additional information
   regarding copyright ownership.  The ASF licenses this file
   to you under the Apache License, Version 2.0 (the
   "License"); you may not use this file except in compliance
   with the License.  You may obtain a copy of the License at
     http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing,
   software distributed under the License is distributed on an
   "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
   KIND, either express or implied.  See the License for the
   specific language governing permissions and limitations
   under the License.
-->

## @kie-tools-scripts/build-env

Helps packages inside a monorepo to define environment variables and consume them easily.

This package allows packages to consume environment variables in a cross-platform way, while maintaining a documented list of variables that are defined.

Environment variables are not directly consumed, but rather used to create properties with a more comprehensive structure.

Env definition files are plain JavaScript, allowing for better flexibility and use through the CLI or direct imports on other JavaScript files, such as Webpack configuration files.

### Usage

- `build-env {dot.separated.property}`
  - Returns the value of a property. See available properties with `build-env --print-env`.
    - e.g. `build-env root.version` prints `0.0.0`.
  - It's possible to negate boolean values with the `--not flag.`
    - e.g. If `build-env build.runTests` prints `true`, then e.g. `build-env build.runTests --not` prints `false`
- `build-env --print-vars`
  - Prints the env var names.
- `build-env --print-env-json`
  - Prints the JSON object for this env.
- `build-env --print-dotenv`
  - Prints variables assigned to values in .env format.

> You can append `:self` to any of the three commands above to see results that ignore composition with other envs.

### Configuration

- `build-env` will recursively scan for an `env` directory with an `index.js` file inside, starting from the current directory until `env/index.js` (or a file called `.build-env-root`) is found.

- `env/index.js` files should export `vars` and `env`. See the example below.

- To compose an env with another one, use the `composeEnv` function.

### Example

Given the following package structure in a monorepo:

- packages/a/env/index.js

```js
const { varsWithName, getOrDefault } = require("@kie-tools-scripts/build-env");

module.exports = {
  vars: varsWithName({
    MY_VAR: {
      default: `foo`,
      description: "My env var",
    },
  }),
  get env() {
    return {
      myProperty: getOrDefault(this.vars.MY_VAR),
    };
  },
};
```

- packages/b/env/index.js

```js
const { varsWithName, getOrDefault, composeEnv } = require("@kie-tools-scripts/build-env");

module.exports = composeEnv([require("a/env")], {
  vars: varsWithName({
    MY_OTHER_VAR: {
      default: `bar`,
      description: "My other env var",
    },
  }),
  get env() {
    return {
      myOtherProperty: getOrDefault(this.vars.MY_OTHER_VAR),
    };
  },
});
```

Executing build-env will give you these results:

```bash
# With default values:
$ cd ~/[my-repo]/packages/a
$ build-env myProperty
foo
$ build-env myOtherProperty
[build-env] Env property 'myOtherProperty' not found.
[build-env] See all env properties with 'build-env --print-env-json'

$ cd ~/[my-repo]/packages/b
$ build-env myProperty
foo
$ build-env myOtherProperty
bar

# With custom value:
$ export MY_VAR='fooz'
$ cd ~/[my-repo]/packages/a
$ build-env myProperty
fooz
$ cd ~/[my-repo]/packages/b
$ build-env myProperty
fooz
```

---

Apache KIE (incubating) is an effort undergoing incubation at The Apache Software
Foundation (ASF), sponsored by the name of Apache Incubator. Incubation is
required of all newly accepted projects until a further review indicates that
the infrastructure, communications, and decision making process have stabilized
in a manner consistent with other successful ASF projects. While incubation
status is not necessarily a reflection of the completeness or stability of the
code, it does indicate that the project has yet to be fully endorsed by the ASF.

Some of the incubating project’s releases may not be fully compliant with ASF
policy. For example, releases may have incomplete or un-reviewed licensing
conditions. What follows is a list of known issues the project is currently
aware of (note that this list, by definition, is likely to be incomplete):

- Hibernate, an LGPL project, is being used. Hibernate is in the process of relicensing to ASL v2
- Some files, particularly test files, and those not supporting comments, may be missing the ASF Licensing Header
-

- Hibernate, an LGPL project, is being used. Hibernate is in the process of
  relicensing to ASL v2
- Some files, particularly test files, and those not supporting comments, may
  be missing the ASF Licensing Header

If you are planning to incorporate this work into your product/project, please
be aware that you will need to conduct a thorough licensing review to determine
the overall implications of including this work. For the current status of this
project through the Apache Incubator visit:
https://incubator.apache.org/projects/kie.html
