/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.webadmin;

import com.google.common.collect.ImmutableList;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.james.lifecycle.api.Startable;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.util.Port;
import org.apache.james.webadmin.CORSFilter;
import org.apache.james.webadmin.PublicRoutes;
import org.apache.james.webadmin.Routes;
import org.apache.james.webadmin.TlsConfiguration;
import org.apache.james.webadmin.WebAdminConfiguration;
import org.apache.james.webadmin.authentication.AuthenticationFilter;
import org.apache.james.webadmin.mdc.LoggingRequestFilter;
import org.apache.james.webadmin.mdc.LoggingResponseFilter;
import org.apache.james.webadmin.mdc.MDCCleanupFilter;
import org.apache.james.webadmin.mdc.MDCFilter;
import org.apache.james.webadmin.metric.MetricPostFilter;
import org.apache.james.webadmin.metric.MetricPreFilter;
import org.apache.james.webadmin.routes.CORSRoute;
import org.apache.james.webadmin.utils.ErrorResponder;
import org.apache.james.webadmin.utils.JsonExtractException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import spark.Filter;
import spark.Service;

public class WebAdminServer
implements Startable {
    private static final Logger LOGGER = LoggerFactory.getLogger(WebAdminServer.class);
    public static final int DEFAULT_PORT = 8080;
    private static final int DEFAULTS = -1;
    private final WebAdminConfiguration configuration;
    private final List<Routes> privateRoutes;
    private final List<PublicRoutes> publicRoutes;
    private final Service service;
    private final AuthenticationFilter authenticationFilter;
    private final MetricFactory metricFactory;
    private final LoggingRequestFilter loggingRequestFilter;

    @Inject
    protected WebAdminServer(WebAdminConfiguration configuration, @Named(value="webAdminRoutes") List<Routes> routesList, AuthenticationFilter authenticationFilter, MetricFactory metricFactory, LoggingRequestFilter loggingRequestFilter) {
        this.configuration = configuration;
        this.privateRoutes = WebAdminServer.privateRoutes(routesList);
        this.publicRoutes = WebAdminServer.publicRoutes(routesList);
        this.authenticationFilter = authenticationFilter;
        this.metricFactory = metricFactory;
        this.loggingRequestFilter = loggingRequestFilter;
        this.service = Service.ignite();
    }

    private static List<Routes> privateRoutes(List<Routes> routes) {
        return (List)routes.stream().filter(route -> !(route instanceof PublicRoutes)).collect(ImmutableList.toImmutableList());
    }

    private static List<PublicRoutes> publicRoutes(List<Routes> routes) {
        return (List)routes.stream().filter(PublicRoutes.class::isInstance).map(PublicRoutes.class::cast).collect(ImmutableList.toImmutableList());
    }

    public WebAdminServer start() {
        this.service.initExceptionHandler(e -> {
            throw new RuntimeException((Throwable)e);
        });
        if (this.configuration.isEnabled()) {
            this.service.threadPool(this.configuration.getMaxThreadCount().orElse(-1).intValue(), this.configuration.getMinThreadCount().orElse(8080).intValue(), -1);
            this.service.ipAddress(this.listeningIP());
            this.service.port(((Port)this.configuration.getPort().get()).getValue());
            this.configureExceptionHanding();
            this.configureHTTPS();
            this.configureCORS();
            this.configureMetrics();
            this.service.before((request, response) -> response.type("application/json"));
            this.configureMDC();
            this.privateRoutes.forEach(routes -> {
                this.service.before(routes.getBasePath(), (Filter)this.authenticationFilter);
                this.service.before(routes.getBasePath() + "/*", (Filter)this.authenticationFilter);
                routes.define(this.service);
            });
            this.publicRoutes.forEach(routes -> routes.define(this.service));
            this.service.awaitInitialization();
            LOGGER.info("Web admin server started on port {}", (Object)this.service.port());
        }
        return this;
    }

    private String listeningIP() {
        try {
            if (this.configuration.getHost().equals("0.0.0.0")) {
                return this.configuration.getHost();
            }
            return InetAddress.getByName(this.configuration.getHost()).getHostAddress();
        }
        catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    private void configureMDC() {
        this.service.before((Filter)new MDCFilter());
        this.service.before((Filter)this.loggingRequestFilter);
        this.service.after((Filter)new LoggingResponseFilter());
        this.service.after((Filter)new MDCCleanupFilter());
    }

    private void configureMetrics() {
        this.service.before((Filter)new MetricPreFilter(this.metricFactory));
        this.service.after((Filter)new MetricPostFilter());
    }

    private void configureHTTPS() {
        if (this.configuration.isTlsEnabled()) {
            TlsConfiguration tlsConfiguration = this.configuration.getTlsConfiguration();
            this.service.secure(tlsConfiguration.getKeystoreFilePath(), tlsConfiguration.getKeystorePassword(), tlsConfiguration.getTruststoreFilePath(), tlsConfiguration.getTruststorePassword());
            LOGGER.info("Web admin set up to use HTTPS");
        }
    }

    private void configureCORS() {
        if (this.configuration.isEnabled()) {
            this.service.before((Filter)new CORSFilter(this.configuration.getUrlCORSOrigin()));
            new CORSRoute().define(this.service);
            LOGGER.info("Web admin set up to enable CORS from {}", (Object)this.configuration.getUrlCORSOrigin());
        }
    }

    private void configureExceptionHanding() {
        this.service.notFound((req, res) -> ErrorResponder.builder().statusCode(404).type(ErrorResponder.ErrorType.NOT_FOUND).message("%s %s can not be found", req.requestMethod(), req.pathInfo()).asString());
        this.service.internalServerError((req, res) -> ErrorResponder.builder().statusCode(500).type(ErrorResponder.ErrorType.SERVER_ERROR).message("WebAdmin encountered an unexpected internal error").asString());
        this.service.exception(JsonExtractException.class, (ex, req, res) -> {
            res.status(400);
            res.body(ErrorResponder.builder().statusCode(400).type(ErrorResponder.ErrorType.INVALID_ARGUMENT).message("JSON payload of the request is not valid").cause(ex).asString());
        });
        this.service.exception(IllegalArgumentException.class, (ex, req, res) -> {
            LOGGER.info("Invalid arguments supplied in the user request", (Throwable)ex);
            res.status(400);
            res.body(ErrorResponder.builder().statusCode(400).type(ErrorResponder.ErrorType.INVALID_ARGUMENT).message("Invalid arguments supplied in the user request").cause(ex).asString());
        });
    }

    @PreDestroy
    public void destroy() {
        if (this.configuration.isEnabled()) {
            this.service.stop();
            this.service.awaitStop();
            LOGGER.info("Web admin server stopped");
        }
    }

    public Port getPort() {
        return new Port(this.service.port());
    }
}

