/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.protocol.internal;

import com.datastax.oss.protocol.internal.Compressor;
import com.datastax.oss.protocol.internal.Frame;
import com.datastax.oss.protocol.internal.Message;
import com.datastax.oss.protocol.internal.NoopCompressor;
import com.datastax.oss.protocol.internal.PrimitiveCodec;
import com.datastax.oss.protocol.internal.PrimitiveSizes;
import com.datastax.oss.protocol.internal.ProtocolErrors;
import com.datastax.oss.protocol.internal.ProtocolV3ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV3ServerCodecs;
import com.datastax.oss.protocol.internal.ProtocolV4ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV4ServerCodecs;
import com.datastax.oss.protocol.internal.ProtocolV5ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV5ServerCodecs;
import com.datastax.oss.protocol.internal.ProtocolV6ClientCodecs;
import com.datastax.oss.protocol.internal.ProtocolV6ServerCodecs;
import com.datastax.oss.protocol.internal.util.Flags;
import com.datastax.oss.protocol.internal.util.IntIntMap;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class FrameCodec<B> {
    public static final int V3_ENCODED_HEADER_SIZE = 9;
    private final PrimitiveCodec<B> primitiveCodec;
    private final Compressor<B> compressor;
    private final IntIntMap<Message.Codec> encoders;
    private final IntIntMap<Message.Codec> decoders;

    public static <B> FrameCodec<B> defaultClient(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor) {
        return new FrameCodec<B>(primitiveCodec, compressor, new ProtocolV3ClientCodecs(), new ProtocolV4ClientCodecs(), new ProtocolV5ClientCodecs(), new ProtocolV6ClientCodecs());
    }

    public static <B> FrameCodec<B> defaultServer(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor) {
        return new FrameCodec<B>(primitiveCodec, compressor, new ProtocolV3ServerCodecs(), new ProtocolV4ServerCodecs(), new ProtocolV5ServerCodecs(), new ProtocolV6ServerCodecs());
    }

    public FrameCodec(PrimitiveCodec<B> primitiveCodec, Compressor<B> compressor, CodecGroup ... codecGroups) {
        ProtocolErrors.check(primitiveCodec != null, "primitiveCodec can't be null", new Object[0]);
        ProtocolErrors.check(compressor != null, "compressor can't be null, use Compressor.none()", new Object[0]);
        this.primitiveCodec = primitiveCodec;
        this.compressor = compressor;
        final IntIntMap.Builder encodersBuilder = IntIntMap.builder();
        final IntIntMap.Builder decodersBuilder = IntIntMap.builder();
        CodecGroup.Registry registry = new CodecGroup.Registry(){

            @Override
            public CodecGroup.Registry addCodec(Message.Codec codec) {
                this.addEncoder(codec);
                this.addDecoder(codec);
                return this;
            }

            @Override
            public CodecGroup.Registry addEncoder(Message.Codec codec) {
                encodersBuilder.put(codec.protocolVersion, codec.opcode, codec);
                return this;
            }

            @Override
            public CodecGroup.Registry addDecoder(Message.Codec codec) {
                decodersBuilder.put(codec.protocolVersion, codec.opcode, codec);
                return this;
            }
        };
        for (CodecGroup codecGroup : codecGroups) {
            codecGroup.registerCodecs(registry);
        }
        this.encoders = encodersBuilder.build();
        this.decoders = decodersBuilder.build();
    }

    public B encode(Frame frame) {
        int protocolVersion = frame.protocolVersion;
        ProtocolErrors.check(protocolVersion >= 4 || frame.customPayload.isEmpty(), "Custom payload is not supported in protocol v%d", protocolVersion);
        ProtocolErrors.check(protocolVersion >= 4 || frame.warnings.isEmpty(), "Warnings are not supported in protocol v%d", protocolVersion);
        Message.Codec messageEncoder = this.getMessageEncoder(frame);
        int headerSize = this.encodedHeaderSize(frame);
        int bodySize = this.encodedBodySize(frame);
        int flags = this.computeFlags(frame);
        if (!Flags.contains(flags, 1)) {
            B dest = this.primitiveCodec.allocate(headerSize + bodySize);
            this.encodeInto(frame, bodySize, flags, messageEncoder, dest);
            return dest;
        }
        B uncompressedBody = this.primitiveCodec.allocate(bodySize);
        this.encodeBodyInto(frame, messageEncoder, uncompressedBody);
        B compressedBody = this.compressor.compress(uncompressedBody);
        this.primitiveCodec.release(uncompressedBody);
        int compressedBodySize = this.primitiveCodec.sizeOf(compressedBody);
        B header = this.primitiveCodec.allocate(headerSize);
        this.encodeHeaderInto(frame, flags, compressedBodySize, header);
        return this.primitiveCodec.concat(header, compressedBody);
    }

    public void encodeInto(Frame frame, int bodySize, B dest) {
        int flags = this.computeFlags(frame);
        Message.Codec encoder = this.getMessageEncoder(frame);
        this.encodeInto(frame, bodySize, flags, encoder, dest);
    }

    private Message.Codec getMessageEncoder(Frame frame) {
        Message.Codec encoder = this.encoders.get(frame.protocolVersion, frame.message.opcode);
        ProtocolErrors.check(encoder != null, "Unsupported opcode %s in protocol v%d", frame.message.opcode, frame.protocolVersion);
        return encoder;
    }

    private int computeFlags(Frame frame) {
        int flags = 0;
        if (!(this.compressor instanceof NoopCompressor) && frame.message.opcode != 1 && frame.message.opcode != 5) {
            flags = Flags.add(flags, 1);
        }
        if (frame.tracing || frame.tracingId != null) {
            flags = Flags.add(flags, 2);
        }
        if (!frame.customPayload.isEmpty()) {
            flags = Flags.add(flags, 4);
        }
        if (!frame.warnings.isEmpty()) {
            flags = Flags.add(flags, 8);
        }
        if (frame.protocolVersion == 6) {
            flags = Flags.add(flags, 16);
        }
        return flags;
    }

    private void encodeInto(Frame frame, int bodySize, int flags, Message.Codec messageEncoder, B dest) {
        this.encodeHeaderInto(frame, flags, bodySize, dest);
        this.encodeBodyInto(frame, messageEncoder, dest);
    }

    private void encodeHeaderInto(Frame frame, int flags, int bodySize, B dest) {
        if (bodySize < 0) {
            bodySize = this.encodedBodySize(frame);
        }
        int versionAndDirection = frame.protocolVersion;
        if (frame.message.isResponse) {
            versionAndDirection |= 0x80;
        }
        this.primitiveCodec.writeByte((byte)versionAndDirection, dest);
        this.primitiveCodec.writeByte((byte)flags, dest);
        this.primitiveCodec.writeUnsignedShort(frame.streamId & 0xFFFF, dest);
        this.primitiveCodec.writeByte((byte)frame.message.opcode, dest);
        this.primitiveCodec.writeInt(bodySize, dest);
    }

    private void encodeBodyInto(Frame frame, Message.Codec messageEncoder, B dest) {
        this.encodeTracingId(frame.tracingId, dest);
        this.encodeCustomPayload(frame.customPayload, dest);
        this.encodeWarnings(frame.warnings, dest);
        messageEncoder.encode(dest, frame.message, this.primitiveCodec);
    }

    private void encodeTracingId(UUID tracingId, B dest) {
        if (tracingId != null) {
            this.primitiveCodec.writeUuid(tracingId, dest);
        }
    }

    private void encodeCustomPayload(Map<String, ByteBuffer> customPayload, B dest) {
        if (!customPayload.isEmpty()) {
            this.primitiveCodec.writeBytesMap(customPayload, dest);
        }
    }

    private void encodeWarnings(List<String> warnings, B dest) {
        if (!warnings.isEmpty()) {
            this.primitiveCodec.writeStringList(warnings, dest);
        }
    }

    public int encodedHeaderSize(Frame frame) {
        return 9;
    }

    public int encodedBodySize(Frame frame) {
        int size = 0;
        if (frame.tracingId != null) {
            size += 16;
        }
        if (!frame.customPayload.isEmpty()) {
            size += PrimitiveSizes.sizeOfBytesMap(frame.customPayload);
        }
        if (!frame.warnings.isEmpty()) {
            size += PrimitiveSizes.sizeOfStringList(frame.warnings);
        }
        Message.Codec encoder = this.getMessageEncoder(frame);
        return size + encoder.encodedSize(frame.message);
    }

    public int decodeBodySize(B source) {
        return this.primitiveCodec.readInt(source, 5);
    }

    public Frame decode(B source) {
        int compressedFrameSize;
        int frameSize;
        B newSource;
        byte directionAndVersion = this.primitiveCodec.readByte(source);
        boolean isResponse = (directionAndVersion & 0x80) == 128;
        int protocolVersion = directionAndVersion & 0x7F;
        byte flags = this.primitiveCodec.readByte(source);
        boolean beta = Flags.contains(flags, 16);
        int streamId = this.readStreamId(source);
        byte opcode = this.primitiveCodec.readByte(source);
        int length = this.primitiveCodec.readInt(source);
        boolean decompressed = false;
        if (Flags.contains(flags, 1) && (newSource = this.compressor.decompress(source)) != source) {
            decompressed = true;
            source = newSource;
        }
        if (decompressed) {
            frameSize = 9 + this.primitiveCodec.sizeOf(source);
            compressedFrameSize = 9 + length;
        } else {
            frameSize = 9 + length;
            compressedFrameSize = -1;
        }
        boolean isTracing = Flags.contains(flags, 2);
        UUID tracingId = isResponse && isTracing ? this.primitiveCodec.readUuid(source) : null;
        Map<String, ByteBuffer> customPayload = Flags.contains(flags, 4) ? this.primitiveCodec.readBytesMap(source) : Collections.emptyMap();
        List<String> warnings = isResponse && Flags.contains(flags, 8) ? this.primitiveCodec.readStringList(source) : Collections.emptyList();
        Message.Codec decoder = this.decoders.get(protocolVersion, opcode);
        ProtocolErrors.check(decoder != null, "Unsupported request opcode: %s in protocol %d", opcode, protocolVersion);
        Message response = decoder.decode(source, this.primitiveCodec);
        if (decompressed) {
            this.primitiveCodec.release(source);
        }
        return new Frame(protocolVersion, beta, streamId, isTracing, tracingId, frameSize, compressedFrameSize, customPayload, warnings, response);
    }

    private int readStreamId(B source) {
        int id = this.primitiveCodec.readUnsignedShort(source);
        return (short)id;
    }

    public static interface CodecGroup {
        public void registerCodecs(Registry var1);

        public static interface Registry {
            public Registry addCodec(Message.Codec var1);

            public Registry addEncoder(Message.Codec var1);

            public Registry addDecoder(Message.Codec var1);
        }
    }
}

