/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.discovery.zen.fd;

import java.io.IOException;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.BaseTransportResponseHandler;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportService;

public class NodesFaultDetection
extends AbstractComponent {
    private final ThreadPool threadPool;
    private final TransportService transportService;
    private final boolean connectOnNetworkDisconnect;
    private final TimeValue pingInterval;
    private final TimeValue pingRetryTimeout;
    private final int pingRetryCount;
    private final boolean registerConnectionListener;
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList();
    private final ConcurrentMap<DiscoveryNode, NodeFD> nodesFD = ConcurrentCollections.newConcurrentMap();
    private final FDConnectionListener connectionListener;
    private volatile DiscoveryNodes latestNodes = DiscoveryNodes.EMPTY_NODES;
    private volatile boolean running = false;

    public NodesFaultDetection(Settings settings, ThreadPool threadPool, TransportService transportService) {
        super(settings);
        this.threadPool = threadPool;
        this.transportService = transportService;
        this.connectOnNetworkDisconnect = this.componentSettings.getAsBoolean("connect_on_network_disconnect", (Boolean)true);
        this.pingInterval = this.componentSettings.getAsTime("ping_interval", TimeValue.timeValueSeconds(1L));
        this.pingRetryTimeout = this.componentSettings.getAsTime("ping_timeout", TimeValue.timeValueSeconds(30L));
        this.pingRetryCount = this.componentSettings.getAsInt("ping_retries", (Integer)3);
        this.registerConnectionListener = this.componentSettings.getAsBoolean("register_connection_listener", (Boolean)true);
        this.logger.debug("[node  ] uses ping_interval [{}], ping_timeout [{}], ping_retries [{}]", this.pingInterval, this.pingRetryTimeout, this.pingRetryCount);
        transportService.registerHandler("discovery/zen/fd/ping", new PingRequestHandler());
        this.connectionListener = new FDConnectionListener();
        if (this.registerConnectionListener) {
            transportService.addConnectionListener(this.connectionListener);
        }
    }

    public void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(Listener listener) {
        this.listeners.remove(listener);
    }

    public void updateNodes(DiscoveryNodes nodes) {
        DiscoveryNodes prevNodes = this.latestNodes;
        this.latestNodes = nodes;
        if (!this.running) {
            return;
        }
        DiscoveryNodes.Delta delta = nodes.delta(prevNodes);
        for (DiscoveryNode newNode : delta.addedNodes()) {
            if (newNode.id().equals(nodes.localNodeId()) || this.nodesFD.containsKey(newNode)) continue;
            this.nodesFD.put(newNode, new NodeFD());
            this.threadPool.schedule(this.pingInterval, "same", new SendPingRequest(newNode));
        }
        for (DiscoveryNode removedNode : delta.removedNodes()) {
            this.nodesFD.remove(removedNode);
        }
    }

    public NodesFaultDetection start() {
        if (this.running) {
            return this;
        }
        this.running = true;
        return this;
    }

    public NodesFaultDetection stop() {
        if (!this.running) {
            return this;
        }
        this.running = false;
        return this;
    }

    public void close() {
        this.stop();
        this.transportService.removeHandler("discovery/zen/fd/ping");
        this.transportService.removeConnectionListener(this.connectionListener);
    }

    private void handleTransportDisconnect(DiscoveryNode node) {
        if (!this.latestNodes.nodeExists(node.id())) {
            return;
        }
        NodeFD nodeFD = (NodeFD)this.nodesFD.remove(node);
        if (nodeFD == null) {
            return;
        }
        if (!this.running) {
            return;
        }
        nodeFD.running = false;
        if (this.connectOnNetworkDisconnect) {
            try {
                this.transportService.connectToNode(node);
                this.nodesFD.put(node, new NodeFD());
                this.threadPool.schedule(this.pingInterval, "same", new SendPingRequest(node));
            }
            catch (Exception e) {
                this.logger.trace("[node  ] [{}] transport disconnected (with verified connect)", node);
                this.notifyNodeFailure(node, "transport disconnected (with verified connect)");
            }
        } else {
            this.logger.trace("[node  ] [{}] transport disconnected", node);
            this.notifyNodeFailure(node, "transport disconnected");
        }
    }

    private void notifyNodeFailure(final DiscoveryNode node, final String reason) {
        this.threadPool.generic().execute(new Runnable(){

            @Override
            public void run() {
                for (Listener listener : NodesFaultDetection.this.listeners) {
                    listener.onNodeFailure(node, reason);
                }
            }
        });
    }

    private static class PingResponse
    extends TransportResponse {
        private PingResponse() {
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
        }
    }

    static class PingRequest
    extends TransportRequest {
        private String nodeId;

        PingRequest() {
        }

        PingRequest(String nodeId) {
            this.nodeId = nodeId;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.nodeId = in.readString();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeString(this.nodeId);
        }
    }

    class PingRequestHandler
    extends BaseTransportRequestHandler<PingRequest> {
        public static final String ACTION = "discovery/zen/fd/ping";

        PingRequestHandler() {
        }

        @Override
        public PingRequest newInstance() {
            return new PingRequest();
        }

        @Override
        public void messageReceived(PingRequest request, TransportChannel channel) throws Exception {
            if (!NodesFaultDetection.this.latestNodes.localNodeId().equals(request.nodeId)) {
                throw new ElasticsearchIllegalStateException("Got pinged as node [" + request.nodeId + "], but I am node [" + NodesFaultDetection.this.latestNodes.localNodeId() + "]");
            }
            channel.sendResponse(new PingResponse());
        }

        @Override
        public String executor() {
            return "same";
        }
    }

    private class FDConnectionListener
    implements TransportConnectionListener {
        private FDConnectionListener() {
        }

        @Override
        public void onNodeConnected(DiscoveryNode node) {
        }

        @Override
        public void onNodeDisconnected(DiscoveryNode node) {
            NodesFaultDetection.this.handleTransportDisconnect(node);
        }
    }

    static class NodeFD {
        volatile int retryCount;
        volatile boolean running = true;

        NodeFD() {
        }
    }

    private class SendPingRequest
    implements Runnable {
        private final DiscoveryNode node;

        private SendPingRequest(DiscoveryNode node) {
            this.node = node;
        }

        @Override
        public void run() {
            if (!NodesFaultDetection.this.running) {
                return;
            }
            NodesFaultDetection.this.transportService.sendRequest(this.node, "discovery/zen/fd/ping", new PingRequest(this.node.id()), TransportRequestOptions.options().withType(TransportRequestOptions.Type.PING).withTimeout(NodesFaultDetection.this.pingRetryTimeout), new BaseTransportResponseHandler<PingResponse>(){

                @Override
                public PingResponse newInstance() {
                    return new PingResponse();
                }

                @Override
                public void handleResponse(PingResponse response) {
                    if (!NodesFaultDetection.this.running) {
                        return;
                    }
                    NodeFD nodeFD = (NodeFD)NodesFaultDetection.this.nodesFD.get(SendPingRequest.this.node);
                    if (nodeFD != null) {
                        if (!nodeFD.running) {
                            return;
                        }
                        nodeFD.retryCount = 0;
                        NodesFaultDetection.this.threadPool.schedule(NodesFaultDetection.this.pingInterval, "same", SendPingRequest.this);
                    }
                }

                @Override
                public void handleException(TransportException exp) {
                    if (!NodesFaultDetection.this.running) {
                        return;
                    }
                    if (exp instanceof ConnectTransportException) {
                        return;
                    }
                    NodeFD nodeFD = (NodeFD)NodesFaultDetection.this.nodesFD.get(SendPingRequest.this.node);
                    if (nodeFD != null) {
                        if (!nodeFD.running) {
                            return;
                        }
                        int retryCount = ++nodeFD.retryCount;
                        NodesFaultDetection.this.logger.trace("[node  ] failed to ping [{}], retry [{}] out of [{}]", exp, SendPingRequest.this.node, retryCount, NodesFaultDetection.this.pingRetryCount);
                        if (retryCount >= NodesFaultDetection.this.pingRetryCount) {
                            NodesFaultDetection.this.logger.debug("[node  ] failed to ping [{}], tried [{}] times, each with  maximum [{}] timeout", SendPingRequest.this.node, NodesFaultDetection.this.pingRetryCount, NodesFaultDetection.this.pingRetryTimeout);
                            if (NodesFaultDetection.this.nodesFD.remove(SendPingRequest.this.node) != null) {
                                NodesFaultDetection.this.notifyNodeFailure(SendPingRequest.this.node, "failed to ping, tried [" + NodesFaultDetection.this.pingRetryCount + "] times, each with maximum [" + NodesFaultDetection.this.pingRetryTimeout + "] timeout");
                            }
                        } else {
                            NodesFaultDetection.this.transportService.sendRequest(SendPingRequest.this.node, "discovery/zen/fd/ping", new PingRequest(SendPingRequest.this.node.id()), TransportRequestOptions.options().withType(TransportRequestOptions.Type.PING).withTimeout(NodesFaultDetection.this.pingRetryTimeout), this);
                        }
                    }
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }
    }

    public static interface Listener {
        public void onNodeFailure(DiscoveryNode var1, String var2);
    }
}

