/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#define CORE_PRIVATE
#include "mod_ftp.h"
#include "ftp_internal.h"
#include "apr_version.h"
#include "apr_network_io.h"
#include "http_vhost.h"

/* seem to need this for LONG_MAX */
#if APR_HAVE_LIMITS_H
#include <limits.h>
#endif

/* Wish APR had one of these */
#if defined(WIN32) || defined(USE_WINSOCK)
#define FTP_APR_EADDRINUSE (APR_OS_START_SYSERR + WSAEADDRINUSE)
#else
#define FTP_APR_EADDRINUSE EADDRINUSE
#endif                          /* WIN32 */

extern ap_filter_rec_t *ftp_count_size_filter_handle;
extern ap_filter_rec_t *ftp_content_length_filter_handle;

static apr_hash_t *FTPMethodHash;
static apr_pool_t *FTPMethodPool;
static const char *FTPHelpText;
static apr_size_t  FTPHelpTextLen;
static const char *FTPFeatText;
static apr_size_t  FTPFeatTextLen;

/*
 * The FTP command structure contains useful information about the FTP
 * handler.  This information is filled out when a command is registered
 * using ftp_hook_cmd(), which also puts the handler into the global hash.
 */
typedef struct ftp_cmd_entry
{
    const char *key;             /* The key, e.g. "DELE" */
    ftp_hook_fn *pf;             /* Pointer to the handler */
    const char *alias;           /* The aliased command e.g. "CDUP" */
    int order;                   /* Handler ordering */
    int flags;                   /* Flags for this command.  See FTP_CMD_ */
    const char *help;            /* Help string for this command */
    struct ftp_cmd_entry *next;  /* Pointer to the next handler */
} ftp_cmd_entry;


FTP_DECLARE(void) ftp_hook_cmd_any(const char *key, ftp_hook_fn *pf,
                                   const char *alias, int order,
                                   int flags, const char *help)
{
    ftp_cmd_entry *cmd, *curr;

    cmd = apr_pcalloc(FTPMethodPool, sizeof(ftp_cmd_entry));

    /* Duplicate for storage into the hash */
    key = apr_pstrdup(FTPMethodPool, key);
    help = apr_pstrdup(FTPMethodPool, help);

    cmd->key = key;
    cmd->pf = pf;
    cmd->alias = alias;
    cmd->flags = flags;
    cmd->order = order;
    cmd->help = help;

    if (!FTPMethodHash) {
        /*
         * Should never get here.  If a user tries to load an extension
         * module before the core FTP module is loaded, they should get an
         * undefined symbol.
         */
        fprintf(stderr, "Could not process registration for %s.", key);
        return;
    }

    curr = apr_hash_get(FTPMethodHash, key, APR_HASH_KEY_STRING);

    if (curr) {
        if (curr->order > cmd->order) {
            cmd->next = curr;
            apr_hash_set(FTPMethodHash, key, APR_HASH_KEY_STRING, cmd);
        }
        else {
            while (curr->next && (curr->order < cmd->order)) {
                curr = curr->next;
            }
            cmd->next = curr->next;
            curr->next = cmd;
        }
    }
    else {
        apr_hash_set(FTPMethodHash, key, APR_HASH_KEY_STRING, cmd);
    }

    /*
     * Only limit commands that are implemented and access checked (not
     * aliases).  PASS is a special case, it is verified after invocation,
     * login isn't needed prior to processing.
     */
    if (pf && (flags & FTP_NEED_LOGIN)) {
        ap_method_register(FTPMethodPool, key);
    }
    ap_method_register(FTPMethodPool, "PASS");
}

static int ftp_run_handler(request_rec *r, struct ftp_cmd_entry *cmd,
                           const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    request_rec *rr;
    int res;

    /*
     * Set the authorization header and the user name for all requests.  This
     * information may not be needed for all   requests, but is required for
     * logging purposes.
     */
    ftp_set_authorization(r);

    /*
     * Run the header parser hook for mod_setenv.  For conditional logging,
     * etc
     */
    ap_run_header_parser(r);

    /*
     * Run a subrequest before the command is run.  This allows us to check
     * if a command is allowed for the user's current location
     */
    if ((res = ftp_set_uri(r, fc->cwd))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    /*
     * XXX - JKS
     * 
     * This will check authz for the *CURRENT* location.  For commands where the
     * URI can change, the command handler will need to rerun the auth/authz
     * checks to ensure the user also has permission for the new URI.
     */
    if ((rr->status == HTTP_UNAUTHORIZED &&
         (cmd->flags & FTP_NEED_LOGIN)) ||
        ((rr->status == HTTP_FORBIDDEN) &&
         (cmd->flags & FTP_NEED_LOGIN))) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOTALLOWED,
                                 ftp_escape_control_text(r->method, r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    ap_destroy_sub_req(rr);

    /* Ok, this method is allowed, run through the list */
    res = cmd->pf(r, arg);
    if (res != DECLINED) {
        return res;
    }

    if (cmd->next) {
        cmd = cmd->next;
        if (cmd->pf) {
            return ftp_run_handler(r, cmd, arg);
        }
    }
    return (cmd->flags & FTP_EXTENSIBLE)
            ? FTP_REPLY_COMMAND_NOT_IMPL_PARAM
            : FTP_REPLY_COMMAND_NOT_IMPLEMENTED;
}

const char *ftp_get_cmd_alias(const char *key)
{
    ftp_cmd_entry *cmd;

    if (!FTPMethodHash) {
        return key;
    }

    cmd = apr_hash_get(FTPMethodHash, key, APR_HASH_KEY_STRING);

    if (cmd && cmd->alias) {
        return (const char *) cmd->alias;
    }

    return key;
}

void ftp_cmd_finalize(apr_pool_t *pool, apr_pool_t *ptemp)
{
    ftp_cmd_entry *cmd, *basecmd;
    apr_hash_index_t *hi;
    void *val;
    int i;

    FTPHelpText = apr_psprintf(ptemp, "%d-%s", FTP_REPLY_HELP_MESSAGE,
                               "The following commands are recognized "
                               "(* =>'s unimplemented).");

    FTPFeatText = apr_psprintf(ptemp, "%d-%s", FTP_REPLY_SYSTEM_STATUS,
                               "Extensions supported");

    for (hi = apr_hash_first(ptemp, FTPMethodHash), i = 0; hi;
         hi = apr_hash_next(hi), i++)
    {
        apr_hash_this(hi, NULL, NULL, &val);
        cmd = (struct ftp_cmd_entry *) val;

        if (cmd->alias)
            basecmd = apr_hash_get(FTPMethodHash, cmd->alias, APR_HASH_KEY_STRING);
        else
            basecmd = cmd;

        if (!(cmd->flags & FTP_NO_HELP))
            FTPHelpText = apr_psprintf(ptemp, "%s%s   %c%-4s",
                                       FTPHelpText, (i % 8) ? "" : CRLF,
                                       (basecmd->pf) ? ' ' : '*', cmd->key);
        else
            --i;

        if (cmd->flags & FTP_NEW_FEAT)
            FTPFeatText = apr_pstrcat(ptemp, FTPFeatText, CRLF " ", 
                                      cmd->key, NULL);
    }

    FTPHelpText = apr_pstrcat(pool, FTPHelpText, CRLF, NULL);
    FTPHelpTextLen = strlen(FTPHelpText);

    FTPFeatText = apr_pstrcat(pool, FTPFeatText, CRLF, NULL);
    FTPFeatTextLen = strlen(FTPFeatText);
}

/* ftp_parse2: Parse a FTP request that is expected to have 2 arguments.
 *
 * Arguments: pool - Pool to allocate from
 *            cmd  - The complete request (e.g. GET /foo.html)
 *            a1   - The first argument
 *            a2   - The second argument
 *
 * Returns: 1 on error or 0 on success, a1 and a2 are modified to point
 *          to the correct values.
 */
static int ftp_parse2(apr_pool_t *pool, const char *cmd,
                      char **a1, char **a2, int keepws)
{
    if (keepws)
    {
        const char *save = cmd;
        while (*cmd && *cmd != ' ') ++cmd;
        *a1 = apr_pstrndup(pool, save, cmd - save);
        if (*cmd && *cmd == ' ') ++cmd;
        *a2 = apr_pstrdup(pool, cmd);
        if (!*a1 || !*a2)
            return 1;
    }
    else
    {
        char *fix;
        *a1 = ap_getword(pool, &cmd, ' ');
        *a2 = apr_pstrdup(pool, cmd);
        if (!*a1 || !*a2)
            return 1;
        fix = strchr(*a2, '\0');
        while (fix > *a2 && *(fix - 1) == ' ')
            *(--fix) = '\0';
    }
    return 0;
}

int ftp_run_cmd(request_rec *r, const char *key)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    ftp_cmd_entry *cmd;
    char *method, *arg = NULL;
    int res;

    if (!FTPMethodHash) {
        return FTP_REPLY_LOCAL_ERROR;
    }

    cmd = apr_hash_get(FTPMethodHash, key, APR_HASH_KEY_STRING);

    if (cmd) {

        /*
         * Commands must have been traslated to their real equivilants back
         * in ftp_read_request_line, by the ftp_get_cmd_alias function above.
         * 
         * Note: recursive aliases are unsupported
         */
        if (cmd->pf == NULL) {
            return (cmd->flags & FTP_EXTENSIBLE)
                    ? FTP_REPLY_COMMAND_NOT_IMPL_PARAM
                    : FTP_REPLY_COMMAND_NOT_IMPLEMENTED;
        }

        if ((cmd->flags & FTP_NEED_LOGIN) && !fc->logged_in) {
            fc->response_notes = "Please log in with USER and PASS";
            return FTP_REPLY_NOT_LOGGED_IN;
        }
        else {
            res = ftp_parse2(r->pool, r->the_request,
                             &method, &arg, cmd->flags & FTP_KEEP_WHITESPACE);
            if (res || (!(cmd->flags & FTP_TAKE0) && !*arg)
                    || (!(cmd->flags & FTP_TAKE1) && *arg))
                return FTP_REPLY_SYNTAX_ERROR;

            return ftp_run_handler(r, cmd, arg);
        }
    }

    return FTP_REPLY_COMMAND_UNRECOGNIZED;
}

int ftp_cmd_abort_data(const char *key)
{
    ftp_cmd_entry *cmd;

    cmd = apr_hash_get(FTPMethodHash, key, APR_HASH_KEY_STRING);

    /* Return true if cmd should abort an active data connection */
    return (cmd && (cmd->flags & FTP_DATA_INTR));
}

/* Begin definition of our command handlers. */
static int ftp_cmd_abor(request_rec *r, const char *arg)
{
    r->the_request = apr_pstrdup(r->pool, "ABOR");
    r->method = apr_pstrdup(r->pool, "ABOR");
    apr_table_setn(r->subprocess_env, "ftp_transfer_ok", "0");
    return FTP_REPLY_TRANSFER_ABORTED;
}

static int ftp_cmd_auth(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    if (!ftp_have_ssl() || (!fc->ssl_input_ctx || !fc->ssl_output_ctx)) {
        fc->response_notes = "AUTH mechanism not available";
        return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
    }

    /*
     * RFC 2228 states these arguments are case insensitive.
     * draft-murray-auth-ftp-ssl-06.txt defined these 4 AUTH mechanisms. TLS
     * or TLS-C  will encrypt the control connection, leaving the data
     * channel clear.  SSL or TLS-P will encrypt both the control and data
     * connections. As it evolved to publication, all but "TLS" were dropped
     * from RFC4217, as RFC 2228 previously insisted that PROT defaults to
     * 'C'lear text.
     */
    if ((strcasecmp(arg, "SSL") == 0) ||
        (strcasecmp(arg, "TLS-P") == 0)) {

        fc->prot = FTP_PROT_PRIVATE;
        fc->auth = FTP_AUTH_SSL;
    }
    else if ((strcasecmp(arg, "TLS") == 0) ||
             (strcasecmp(arg, "TLS-C") == 0)) {

        fc->prot = FTP_PROT_CLEAR;
        fc->auth = FTP_AUTH_TLS;
    }
    else {
        fc->response_notes = "AUTH mechanism not supported";
        return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
    }

    return FTP_REPLY_SECURITY_EXCHANGE_DONE;
}

/* XXX: RPM 6/11/2002
 * This needs rewriting.  Too many special cases.
 */
static int ftp_change_dir(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    ftp_dir_config *dconf;
    request_rec *rr;
    conn_rec *c = r->connection;
    int res, response;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    dconf = ftp_get_module_config(rr->per_dir_config);

    /* Handle special case where the URI is / */
    if (r->uri[0] == '/' && !r->uri[1]) {
        apr_cpystrn(fc->cwd, r->uri, APR_PATH_MAX + 1);

        if (dconf->readme) {

            /*
             * We do not inherit readme messages.  If this readme was not
             * specifically meant for us, we skip it.  The exception to the
             * rule is if FTPReadmeMessage was placed in the global server
             * configuration.
             */
            if (!dconf->path ||
                !strncmp(dconf->path, r->filename,
                         strlen(r->filename) - 1)) {

                if (dconf->readme_isfile) {
                    ftp_show_file(c->output_filters, r->pool,
                                  FTP_REPLY_COMPLETED, fc,
                                  dconf->readme);
                }
                else {
                    char outbuf[BUFSIZ];

                    ftp_message_generate(fc, dconf->readme, outbuf,
                                         sizeof(outbuf));
                    ftp_reply(fc, c->output_filters, r->pool,
                              FTP_REPLY_COMPLETED, 1, outbuf);
                }
            }
        }

        ap_destroy_sub_req(rr);
        return FTP_REPLY_COMPLETED;
    }


    /* Check access permissions for the new directory. */
    if (!((rr->status == HTTP_OK) || (rr->status == HTTP_MOVED_PERMANENTLY))) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    if (rr->finfo.filetype != 0) {
        if (rr->finfo.filetype != APR_DIR) {
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOT_A_DIR,
                                     ftp_escape_control_text(r->parsed_uri.path,
                                                             r->pool));
            response = FTP_REPLY_FILE_NOT_FOUND;
        }
        else {
            apr_cpystrn(fc->cwd, r->parsed_uri.path, APR_PATH_MAX + 1);

            if (dconf->readme) {
                /*
                 * We do not inherit readme messages.  If this readme was not
                 * specifically meant for us, we skip it.  The exception to
                 * the rule is if FTPReadmeMessage was placed in the global
                 * server configuration.
                 */
                if (!dconf->path ||
                    !strncmp(dconf->path, r->filename,
                             strlen(r->filename) - 1)) {

                    if (dconf->readme_isfile) {
                        ftp_show_file(c->output_filters, r->pool,
                                      FTP_REPLY_COMPLETED, fc,
                                      dconf->readme);
                    }
                    else {
                        char outbuf[BUFSIZ];

                        ftp_message_generate(fc, dconf->readme, outbuf,
                                             sizeof(outbuf));
                        ftp_reply(fc, c->output_filters, r->pool,
                                  FTP_REPLY_COMPLETED, 1, outbuf);
                    }
                }
            }
            response = FTP_REPLY_COMPLETED;
        }
    }
    else {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }

    ap_destroy_sub_req(rr);
    return response;
}

static int ftp_cmd_cdup(request_rec *r, const char *arg)
{
    return ftp_change_dir(r, "..");
}

static int ftp_cmd_cwd(request_rec *r, const char *arg)
{
    return ftp_change_dir(r, arg);
}

static int ftp_cmd_dele(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    request_rec *rr;
    apr_status_t rv;
    int res, response;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    /*
     * If the user does not have permission to view the file, do not let them
     * delete it.
     */
    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }
    /* We do have permission, and the file exists.  Try to remove. */
    else if (rr->finfo.filetype == APR_DIR) {

        rv = apr_dir_remove(r->filename, r->pool);

        if (rv != APR_SUCCESS) {
            char error_str[120];
            char *err = apr_strerror(rv, error_str, sizeof(error_str));
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                     ftp_escape_control_text(err, r->pool));
            response = FTP_REPLY_FILE_NOT_FOUND;
        }
        else {
            response = FTP_REPLY_COMPLETED;
        }
    }
    else if (rr->finfo.filetype == APR_REG) {

        rv = apr_file_remove(r->filename, r->pool);

        if (rv != APR_SUCCESS) {
            /* Call to apr_file_remove failed */
            char error_str[120];
            char *err = apr_strerror(rv, error_str, sizeof(error_str));
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                     ftp_escape_control_text(err, r->pool));
            response = FTP_REPLY_FILE_NOT_FOUND;
        }
        else {
            response = FTP_REPLY_COMPLETED;
        }
    }
    else {
        /* File does not exist */
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }
    ap_destroy_sub_req(rr);
    return response;
}

static int ftp_cmd_feat(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    apr_bucket_brigade *bb;
    apr_bucket *b;

    bb = apr_brigade_create(r->pool, c->bucket_alloc);
    b = apr_bucket_immortal_create(FTPFeatText, FTPFeatTextLen, c->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    fc->traffic += FTPFeatTextLen;

    b = apr_bucket_flush_create(c->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    ap_pass_brigade(c->output_filters, bb);

    fc->response_notes = "End";
    return FTP_REPLY_SYSTEM_STATUS;
}

static int ftp_cmd_help(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    apr_bucket_brigade *bb;
    apr_bucket *b;
    ftp_cmd_entry *cmd;
    char *method;

    if (*arg) {
        method = ftp_toupper(r->pool, arg);
        cmd = apr_hash_get(FTPMethodHash, method, APR_HASH_KEY_STRING);

        if (cmd) {
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_HELP_SYNTAX,
                                              arg, cmd->help);
            return FTP_REPLY_HELP_MESSAGE;
        }
        else {
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOTIMPL,
                                     ftp_escape_control_text(arg, r->pool));
            return FTP_REPLY_COMMAND_NOT_IMPLEMENTED;
        }
    }

    /* given no argument, pre-prepared HELP message */
    bb = apr_brigade_create(r->pool, c->bucket_alloc);
    b = apr_bucket_immortal_create(FTPHelpText, FTPHelpTextLen, c->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    fc->traffic += FTPHelpTextLen;

    b = apr_bucket_flush_create(c->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    ap_pass_brigade(c->output_filters, bb);

    fc->response_notes = apr_psprintf(r->pool, FTP_MSG_HELP,
                             ftp_escape_control_text(r->server->server_admin,
                                                     r->pool));
    return FTP_REPLY_HELP_MESSAGE;
}

static int common_list(request_rec *r, const char *arg, int is_list)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    conn_rec *cdata;
    request_rec *rr;
    apr_bucket_brigade *bb;
    apr_bucket *b;
    struct ftp_direntry *direntry, *dirp;
    char *word, *buf = NULL, *pattern;
    int dashl = 0;
    apr_status_t rv;
    apr_size_t nbytes;
    char *varg = apr_pstrdup(r->pool, arg);
    const char *test, *sl;
    int res;
    int decend = 0;

    while (*varg == '-')
    {
        if (ftp_parse2(r->pool, varg, &word, &varg, FTP_KEEP_WHITESPACE)) {
            varg = word;
            break;
        }
        /* More Cowbell!  TODO: expand the accepted dash patterns */
        if (ap_strchr(word, 'l')) {
            dashl = 1;
        }
        /* -- 'end of dash-opts' by convention allows patterns like '-*' */
        if (ap_strchr(word + 1, '-')) {
            break;
        }
    }

    /* Special FTPOption that maps NLST directly to LIST */
    if (is_list && (fsc->options & FTP_OPT_LISTISNLST) && !dashl) {
        is_list = 0;
    }
    else
    /* Special FTPOption that maps NLST directly to LIST */
    if (!is_list && ((fsc->options & FTP_OPT_NLSTISLIST) || dashl)) {
        is_list = 1;
    }

    arg = varg;

    if (is_list && (ap_strchr_c(arg, '*') != NULL))
    {
        /* Prevent DOS attacks, only allow one segment to have a wildcard */
        int found = 0;          /* The number of segments with a wildcard */
        const char *pos = arg;  /* Pointer for iterating over the string */

        while (*pos != '\0') {
            if (*pos == '*') {
                /*
                 * We found a wildcard in this segment.  Increment the count
                 * and move the pointer to the beginning of the next segment
                 */
                found++;
                while (*pos != '\0' && *pos != '/') {
                    pos++;
                }
            }
            else {
                /* Nothing here, move on */
                pos++;
            }
        }

        /* In the future this can be configurable */
        if (found > 1) {
            ap_log_error(APLOG_MARK, APLOG_WARNING | APLOG_NOERRNO, 0, 0,
                         "Ignoring directory listing request for %s", arg);
            fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                     ftp_escape_control_text(arg, r->pool));
            return FTP_REPLY_FILE_NOT_FOUND;
        }
    }

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    /* Need to run intermediate subrequest to check for redirect */
    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);
    if (rr->status == HTTP_MOVED_PERMANENTLY)
    {
        ap_parse_uri(r, apr_pstrcat(r->pool, rr->uri, "/", NULL));

        /* Rerun the subrequest with the new uri */
        ap_destroy_sub_req(rr);
        rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);
    }

    if (rr->status != HTTP_OK) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    ap_destroy_sub_req(rr);

    if (arg[0] == '\0') {
        pattern = apr_pstrcat(r->pool, r->filename, "/*", NULL);
    }
    else {
        pattern = r->filename;
    }

#ifdef FTP_DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, "Pattern: %s", pattern);
#endif

    /* Construct the sorted array of directory contents */
    if ((direntry = ftp_direntry_get(r, pattern)) == NULL) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(arg, r->pool));
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    fc->response_notes = FTP_MSG_OPENASCII;
    ftp_send_response(r, FTP_REPLY_FILE_STATUS_OK);

    if (!(cdata = ftp_open_dataconn(r, 1))) {
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    if (is_list)
    {
        /*
         * We have the directory tree, and its time to print it to the client.
         * At this point we could have one of three things.
         *
         * 1. One entry, a directory.  Just print the contents
         * 2. No directories.  Just print the contents.
         * 3. Mixture.  Print files first, then directories.
         */

        /* Handle case 1 */
        if (direntry->child && !direntry->next) {
            direntry = direntry->child;
        }
        else {
            /* Handle case 2 & 3 */
            for (dirp = direntry; dirp; dirp = dirp->next) {
                if (dirp->child)
                    decend = 1;
            }
        }
    }

    bb = apr_brigade_create(r->pool, c->bucket_alloc);

    /* Print the directory listing, skipping directories as needed */
    for (dirp = direntry; dirp; dirp = dirp->next)
    {
        if (dirp->modestring[0] == 'd')
        {
            if (is_list && decend) 
                continue;
            else if (!is_list && !(fsc->options & FTP_OPT_NLSTSHOWDIRS))
                continue;
        }

        for (test = dirp->name; (sl = ap_strchr_c(test, '/')); test = sl + 1)
             /* noop */ ;

        if (!strcmp(".", test)) {
            continue;
        }

        if (is_list)
            buf = apr_psprintf(r->pool,
                               "%10s%5d %-8s %-8s%9" APR_OFF_T_FMT " %s %s"
                               CRLF, dirp->modestring, dirp->nlink,
                               dirp->username, dirp->groupname,
                               dirp->size, dirp->datestring, dirp->name);
        else
            buf = apr_psprintf(r->pool, "%s" CRLF, dirp->name);

        nbytes = strlen(buf);
        rv = apr_brigade_write(bb, ap_filter_flush, cdata->output_filters,
                               buf, nbytes);
        fc->traffic += nbytes;
    }

    if (is_list && decend) {
        for (dirp = direntry; dirp; dirp = dirp->next) {
            if (dirp->modestring[0] == 'd' && dirp->child)
            {
                ftp_direntry *decend;

                /* Iterate through the sub directory */
                buf = apr_psprintf(r->pool, CRLF "%s:" CRLF, dirp->name);
                nbytes = strlen(buf);
                rv = apr_brigade_write(bb, ap_filter_flush,
                                       cdata->output_filters, buf, nbytes);
                fc->traffic += nbytes;

                buf = apr_pstrcat(r->pool, "total 0" CRLF, NULL);
                nbytes = strlen(buf);
                rv = apr_brigade_write(bb, ap_filter_flush,
                                       cdata->output_filters, buf, nbytes);
                fc->traffic += nbytes;

                for (decend = dirp->child; decend; decend = decend->next)
                {
                    for (test = dirp->name; (sl = ap_strchr_c(test, '/'));
                         test = sl + 1)
                         /* noop */ ;

                    if (!strcmp(".", test)) {
                        continue;
                    }

                    buf = apr_psprintf(r->pool, "%10s%5d %-8s %-8s%9"
                                       APR_OFF_T_FMT " %s %s" CRLF,
                                       decend->modestring, decend->nlink,
                                       decend->username, decend->groupname,
                                       decend->size, decend->datestring,
                                       decend->name);
                    nbytes = strlen(buf);
                    rv = apr_brigade_write(bb, ap_filter_flush,
                                           cdata->output_filters, buf,
                                           nbytes);
                    fc->traffic += nbytes;
                }
            }
        }
    }

    /* If the brigade is empty, just send an eos */
    if (APR_BRIGADE_EMPTY(bb)) {
        b = apr_bucket_eos_create(c->bucket_alloc);
        APR_BRIGADE_INSERT_TAIL(bb, b);
    }

    /* Flush the brigade down the filter chain */
    b = apr_bucket_flush_create(cdata->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    ap_pass_brigade(cdata->output_filters, bb);

    ap_lingering_close(cdata);
    fc->datasock = NULL;
    fc->transfers++;

    if (cdata->aborted)
        return FTP_REPLY_TRANSFER_ABORTED;
    else
        return FTP_REPLY_DATA_CLOSE;
}

static int ftp_cmd_list(request_rec *r, const char *arg)
{
    return common_list(r, arg, 1);
}

static int ftp_cmd_nlst(request_rec *r, const char *arg)
{
    return common_list(r, arg, 0);
}

static int ftp_cmd_mdtm(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    int res;
    request_rec *rr;
    apr_time_exp_t t;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    apr_time_exp_lt(&t, rr->finfo.mtime);
    fc->response_notes = apr_psprintf(r->pool,
                                      "%04d%02d%02d%02d%02d%02d",
                                      1900 + t.tm_year, t.tm_mon + 1,
                                      t.tm_mday, t.tm_hour, t.tm_min,
                                      t.tm_sec);
    ap_destroy_sub_req(rr);
    return FTP_REPLY_FILE_STATUS;
}

static int ftp_cmd_mkd(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    apr_status_t rv;
    int res;
    request_rec *rr;
    ftp_dir_config *dconf;
    apr_fileperms_t dirperms;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    dconf = ftp_get_module_config(rr->per_dir_config);
    dirperms = dconf->dirperms;
    ap_destroy_sub_req(rr);

    if (dirperms == APR_OS_DEFAULT)
        dirperms = FTP_DEFAULT_UMASK;

    /*
     * In the config phase, ->fileperms was a negative umask. for operation,
     * exchange this with a positive protections to pass to the apr_dir_make
     * protection flag.
     */
    dirperms = (APR_UREAD | APR_UWRITE | APR_UEXECUTE |
                APR_GREAD | APR_GWRITE | APR_GEXECUTE |
                APR_WREAD | APR_WWRITE | APR_WEXECUTE)
        & ~dirperms;
    rv = apr_dir_make(r->filename, dirperms, r->pool);

    if (rv != APR_SUCCESS) {
        char error_str[120];
        char *err = apr_strerror(rv, error_str, sizeof(error_str));
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED, 
                                 ftp_escape_control_text(err, r->pool));
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    else {
        apr_file_perms_set(r->filename, dirperms);
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_DIR_CREAT,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        return FTP_REPLY_PATH_CREATED;
    }
}

static int ftp_cmd_mode(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    if (*arg && !arg[1]) {
        switch (toupper(*arg)) {
        case 'S':
            fc->response_notes = "Mode set to S";
            return FTP_REPLY_COMMAND_OK;
        }
    }
    fc->response_notes = apr_psprintf(r->pool, "Mode %s not implemented", 
                             ftp_escape_control_text(arg, r->pool));
    return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
}

static int ftp_cmd_noop(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    fc->response_notes = apr_psprintf(r->pool, FTP_MSG_SUCCESS, r->method);
    return FTP_REPLY_COMMAND_OK;
}

static int ftp_cmd_pass(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    core_server_config *ftpcore = NULL;
    char *userdir = NULL;
    ftp_dir_config *dconf;
    request_rec *rr;
    char *userpass;
    server_rec *ftpserver;
    apr_status_t rv;
    char *tmppath;

    if (fc->user == ftp_unknown_username) {
        return FTP_REPLY_BAD_SEQUENCE;
    }

    apr_cpystrn(fc->cwd, "/", APR_PATH_MAX + 1);

    /*
     * By this point we have already extracted the arguments, so rewrite the
     * original request to not include the password
     */
    r->the_request = apr_pstrdup(r->pool, "PASS xxx");

    userpass = apr_psprintf(r->pool, "%s:%s", fc->user, arg);
    fc->authorization = apr_psprintf(fc->login_pool, "Basic %s",
                                     ap_pbase64encode(r->pool, userpass));

    /*
     * This is normally set in the dispatcher, but since we just read the
     * password, we need to reset the auth header and r->user
     */
    ftp_set_authorization(r);

    /* Prepare to overwrite ap_document_root */
    if (fsc->jailuser || fsc->docrootenv) {
        int i;
        ap_conf_vector_t *conf_vector;
        module *modp;

        ftpserver = apr_pcalloc(fc->login_pool, sizeof(*ftpserver));
        memcpy(ftpserver, r->server, sizeof(*ftpserver));

        /*
         * We need to count the number of modules in order to know how big an
         * array to allocate.  There is a variable in the server for this,
         * but it isn't exported, so we are stuck doing this.  I guess this
         * could be a static variable that we compute once, but it shouldn't
         * be that expensive to do it this way.
         */
        for (i = 0; ap_loaded_modules[i]; i++);
        conf_vector = apr_pcalloc(fc->login_pool, sizeof(void *) * i);

        for (modp = ap_top_module; modp; modp = modp->next) {
            /*
             * This is a hack.  Basically, to keep the user in thier own
             * directory, we are re-writing the DocumentRoot when the user
             * logs in.  To do this, we copy the entire server_rec for this
             * Vhost, and then we copy the whole module_config for that
             * server.  For the core module, we don't just copy the pointer,
             * instead we create a new core_server_config, copy the old one
             * to it, and change the docroot.  Yuck.
             */
            if (modp == &core_module) {
                core_server_config *core;

                ftpcore = apr_pcalloc(fc->login_pool, sizeof(*ftpcore));
                core = ap_get_module_config(r->server->module_config, modp);
                *ftpcore = *core;
                ap_set_module_config(conf_vector, modp, ftpcore);
            }
            else {
                ap_set_module_config(conf_vector, modp,
                      ap_get_module_config(r->server->module_config, modp));
            }
        }
        ftpserver->module_config = (ap_conf_vector_t *) conf_vector;
        fc->orig_server = r->server = ftpserver;
    }

    if (fsc->docrootenv) {
        const char *docroot;

        /*
         * Invoke a request that forces auth to occur, Then check for an
         * fsc->docrootenv that translates to a valid envvar.
         */
        rr = ap_sub_req_method_uri(r->method, "/", r, NULL);
        docroot = apr_table_get(rr->subprocess_env, fsc->docrootenv);

        if (!docroot || !*docroot) {
            ap_log_perror(APLOG_MARK, APLOG_WARNING | APLOG_NOERRNO, 0,
                          r->pool,
                  "Warning: Document Root variable from FTPDocRootEnv [%s] "
                       "was empty or not found, using default DocumentRoot",
                          fsc->docrootenv);
        }
        else if (((rv = apr_filepath_merge(&tmppath, NULL, docroot,
                                           APR_FILEPATH_TRUENAME |
                                           APR_FILEPATH_NOTRELATIVE,
                                           fc->login_pool)) != APR_SUCCESS)
                 || (rv = APR_ENOTDIR, !ap_is_directory(r->pool, tmppath))) {
            ap_log_perror(APLOG_MARK, APLOG_WARNING, rv,
                          r->pool,
                      "Warning: Document Root [%s] from FTPDocRootEnv [%s] "
                 "is invalid or does not exist, using default DocumentRoot",
                          docroot, fsc->docrootenv);
        }
        else {
            ftpcore->ap_document_root = tmppath;
        }

        ap_destroy_sub_req(rr);
    }

    /* Check for a configured home directory */
    if (fsc->homedir) {

        apr_filepath_merge(&userdir, fsc->homedir, fc->user, 0, r->pool);

        if (userdir) {

            rr = ap_sub_req_method_uri(r->method, userdir, r, NULL);

            if (rr->finfo.filetype == APR_DIR &&
                (rr->status == HTTP_OK ||
                 rr->status == HTTP_MOVED_PERMANENTLY)) {
                tmppath = apr_pstrcat(r->pool, userdir, "/", NULL);
                apr_cpystrn(fc->cwd, tmppath, APR_PATH_MAX + 1);
            }
            else if (rr->status == HTTP_OK &&
                     rr->finfo.filetype == APR_NOFILE) {

                /* See if we should create the directory automatically */
                if (fsc->options & FTP_OPT_CREATEHOMEDIRS) {

                    if (rr->finfo.filetype == APR_NOFILE && rr->filename
                        && ap_is_directory(rr->pool,
                           ap_make_dirstr_parent(rr->pool, rr->filename))) {

                        rv = apr_dir_make(rr->filename, APR_OS_DEFAULT,
                                          r->pool);

                        if (rv != APR_SUCCESS) {
                            ap_log_error(APLOG_MARK, APLOG_ERR, rv,
                                         r->server, "Couldn't "
                                         "create home directory (%s) for "
                                         "user %s",
                                         rr->filename, fc->user);
                        }
                        else {
                            /* The new directory was created. */
                            tmppath = apr_pstrcat(r->pool, userdir, "/", NULL);
                            apr_cpystrn(fc->cwd, tmppath, APR_PATH_MAX + 1);
                            ap_log_error(APLOG_MARK, APLOG_INFO | APLOG_NOERRNO,
                                         0, r->server, "Created home "
                                         "directory (%s) for user %s",
                                         rr->filename, fc->user);
                        }
                    }
                    else {
                        ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO,
                                     0, r->server,
                                     "Home directory for user %s not "
                                     "found.  Using \"/\"", fc->user);

                    }
                }
                else {
                    ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0,
                                 r->server, "Home directory for "
                                 "user %s not found.  Using \"/\"",
                                 fc->user);
                }
            }
            else {

                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0,
                             r->server, "Permission denied entering"
                             " home directory for user %s.  Using \"/\"",
                             fc->user);
            }

            ap_destroy_sub_req(rr);
        }
    }

    if (fsc->jailuser) {
        /*
         * If we didn't set cwd the Jailed user can't be allowed to log in.
         * The cwd was set to userdir if it existed or was created above, but
         * we must ignore the trailing slash.
         */
        if (!userdir || (strncmp(userdir, fc->cwd, strlen(userdir)) != 0)) {
            goto pass_try_again;
        }
        ftpcore->ap_document_root = apr_pstrcat(fc->login_pool,
                                  ftpcore->ap_document_root, userdir, NULL);
        apr_cpystrn(fc->cwd, "/", APR_PATH_MAX + 1);
    }

    /* Our document_root and cwd are now constructed for the user... */
    rr = ap_sub_req_method_uri(r->method, fc->cwd, r, NULL);

    dconf = ftp_get_module_config(rr->per_dir_config);

    if (rr->status == HTTP_OK) {
        /*
         * We now must iterate over the entire scoreboard checking to see if
         * we have another server that can handle an incoming request. If we
         * don't see another server in the ready state, that means we are the
         * last available server, and must send the client a message that the
         * server is full.
         * 
         * For those wondering, there is a race condition here that could cause
         * a client to be put in the accept queue, but we should be able to
         * recover from this once a client disconnects.
         * 
         * This has a few side effects: - We can really only service Max - 1 FTP
         * sessions concurrently.
         * 
         * - If a hybid server is heavily loaded, such that all servers are busy
         * serving HTTP traffic, it is possible to starve FTP requests, since
         * when we check the scoreboard, all servers will be busy.
         */
        ftp_loginlimit_t limrv;

        if ((fsc->options & FTP_OPT_CHECKMAXCLIENTS) &&
            ftp_check_maxclients(r)) {
            ap_destroy_sub_req(rr);
            fc->response_notes = FTP_MSG_SESSIONLIMIT;
            fc->close_connection = 1;

            return FTP_REPLY_SERVICE_NOT_AVAILABLE;
        }

        limrv = ftp_limitlogin_check(fc->user, r);
        /* I love switch */
        switch (limrv) {
        case FTP_LIMIT_HIT_PERSERVER:
            ap_destroy_sub_req(rr);
            ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_INFO, 0,
                         r->server, "FTP per server login limit hit for %s",
                         fc->user);

            fc->response_notes = FTP_MSG_SESSIONLIMIT;
            fc->close_connection = 1;
            return FTP_REPLY_SERVICE_NOT_AVAILABLE;

        case FTP_LIMIT_HIT_PERUSER:
            ap_destroy_sub_req(rr);
            ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_INFO, 0,
                         r->server, "FTP per user login limit hit for %s",
                         fc->user);

            fc->response_notes = FTP_MSG_SESSIONLIMIT;
            fc->close_connection = 1;
            return FTP_REPLY_SERVICE_NOT_AVAILABLE;

        case FTP_LIMIT_HIT_PERIP:
            ap_destroy_sub_req(rr);
            ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_INFO, 0,
                         r->server, "FTP per IP login limit hit for %s",
                         fc->user);

            fc->response_notes = FTP_MSG_SESSIONLIMIT;
            fc->close_connection = 1;
            return FTP_REPLY_SERVICE_NOT_AVAILABLE;

        case FTP_LIMIT_ERROR:
            ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_INFO, 0,
                         r->server, "Error accessing DB for user %s; no login limit in effect",
                         fc->user);
            break;

        case FTP_LIMIT_OK:
        default:
            break;
        }

        ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_INFO, 0,
                     r->server, "FTP LOGIN FROM %s as %s",
                     c->remote_ip, fc->user);

        if (dconf->readme) {
            if (dconf->readme_isfile) {
                ftp_show_file(c->output_filters, r->pool,
                              FTP_REPLY_USER_LOGGED_IN, fc,
                              dconf->readme);
            }
            else {
                char outbuf[BUFSIZ];

                ftp_message_generate(fc, dconf->readme, outbuf,
                                     sizeof(outbuf));
                ftp_reply(fc, c->output_filters, r->pool,
                          FTP_REPLY_USER_LOGGED_IN, 1, outbuf);
            }
        }

        fc->logged_in = 1;
        ap_destroy_sub_req(rr);
        return FTP_REPLY_USER_LOGGED_IN;
    }

    ap_destroy_sub_req(rr);

    /*
     * Here we return the generic FTP_REPLY_NOT_LOGGED_IN error (530), but it
     * may have been because of a varity of reasons.  If the return status is
     * 401 HTTP_UNAUTHORIZED, then the username/password combination was
     * incorrect.  If we get a HTTP_FORBIDDEN, it is likely that the access
     * checkers failed because the location was configured with 'deny from
     * all'.
     */
    if (strcmp(fc->user, "anonymous") == 0) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, 0,
                     r->server,
                     "ANONYMOUS FTP LOGIN REFUSED FROM %s",
                     c->remote_ip);
    }
    else {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, 0,
                     r->server, "FTP LOGIN REFUSED FROM %s, %s",
                     c->remote_ip, fc->user);
    }
pass_try_again:
    if (++fc->login_attempts == fsc->max_login_attempts) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, 0,
                     r->server, "repeated login failures from %s",
                     c->remote_ip);

        fc->response_notes = "Maximum login attempts reached, closing connection.";
        fc->close_connection = 1;

        return FTP_REPLY_SERVICE_NOT_AVAILABLE;
    }

    /*
     * Sleep one second for every failed login attempt.  This is a common
     * practice among FTP servers to prevent DOS attacks
     */
    apr_sleep(fc->login_attempts * APR_USEC_PER_SEC);

    fc->user = ftp_unknown_username;
    fc->authorization = NULL;
    fc->response_notes = "Please log in with USER and PASS";
    return FTP_REPLY_NOT_LOGGED_IN;
}

static int init_pasv_socket(request_rec *r, int bindfamily,
                                const char *bindaddr)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    apr_sockaddr_t *sa;
    apr_socket_t *s;
    apr_status_t rv;
    apr_port_t port;
    int tries = 0;

    rv = apr_sockaddr_info_get(&sa, bindaddr, bindfamily, 0, 0, fc->data_pool);

    if (!sa || rv) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                      "Couldn't resolve local socket address %s (%d)"
                      " (ftp, apr or socket stack bug?)",
                      bindaddr, bindfamily);
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

#if APR_MAJOR_VERSION < 1
    rv = apr_socket_create_ex(&s, sa->family, SOCK_STREAM,
                              APR_PROTO_TCP, fc->data_pool);
#else
    rv = apr_socket_create(&s, sa->family, SOCK_STREAM,
                           APR_PROTO_TCP, fc->data_pool);
#endif
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't create passive socket");
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    rv = apr_socket_opt_set(s, APR_SO_LINGER,
                            APR_MAX_SECS_TO_LINGER);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't set APR_SO_LINGER socket option");
        apr_socket_close(s);
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    rv = apr_socket_opt_set(s, APR_SO_REUSEADDR, 1);
    if (rv != APR_SUCCESS && rv != APR_ENOTIMPL) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't set APR_SO_REUSEADDR socket option");
        apr_socket_close(s);
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    /*
     * XXX: should pick an arbitrary port within the acceptable range, but
     * dodge the throttle timer on the first reloop (init tries = -1?)
     */
    port = fsc->pasv_min;
    while (1) {
        /* Magic here when port == 0, accepts an ephemeral port assignment */
        sa->port = port;
        sa->sa.sin.sin_port = htons(port);
        rv = apr_socket_bind(s, sa);
        if (rv == APR_SUCCESS) {
            break;
        }

        if (rv != FTP_APR_EADDRINUSE || tries >= FTP_MAX_TRIES) {
            ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                         "Couldn't bind to passive socket");
            apr_socket_close(s);
            return FTP_REPLY_CANNOT_OPEN_DATACONN;
        }

        /*
         * If we are using a range, increment the port, and continue.  If we
         * have reached the top of the range start over at the beginning
         */
        if (port != 0) {
            if (port < fsc->pasv_max) {
                ++port;
                continue;
            }
            port = fsc->pasv_min;
            /* Fall through into throttle */
        }

        /*
         * Under high load we may have many sockets in TIME_WAIT, causing
         * bind to fail with EADDRINUSE.  In these conditions we throttle the
         * system by sleeping before we attempt to bind again within our
         * acceptable port range
         */
        ++tries;
        ap_log_error(APLOG_MARK, APLOG_INFO, 0, r->server,
                     "Couldn't find port within range for passive "
                   "connection.  Restarting at %d (retry %d)", port, tries);
        apr_sleep(tries * APR_USEC_PER_SEC);
    }

    rv = apr_socket_listen(s, 1);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Couldn't listen on passive socket");
        apr_socket_close(s);
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    fc->csock = s;
    fc->passive_created = apr_time_now();
    return 0;
}


static int ftp_cmd_epsv(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    int res;
    apr_sockaddr_t *sa;
    const char *addr;
    int family = 0;

    if (strcasecmp(arg, "ALL") == 0) {
        /* A contract to never respond to other data connection methods */
        fc->all_epsv = 1;
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_SUCCESS, r->method);
        return FTP_REPLY_COMMAND_OK;
    }

    ftp_reset_dataconn(fc);

    /*
     * Why don't pasv_bindaddr/bindfamily appear below? These directives
     * provide an override which the client is informed of, while EPSV only
     * informs the client of the port to use, not a family, and never an
     * address. FTPEPSVIgnoreFamily should offer sufficient customization.
     */
    if (!*arg || ((arg[0] == '1' || arg[0] == '2') && !arg[1]
                  && fsc->epsv_ignore_family)) {
#if APR_HAVE_IPV6
        if (c->local_addr->family == AF_INET6 &&
            IN6_IS_ADDR_V4MAPPED((struct in6_addr *)
                                 c->local_addr->ipaddr_ptr)) {
            /* httpd assures us local_ip is in ipv4 notation for mapped addrs */
            addr = c->local_ip;
            family = APR_INET;
        }
        else
#endif
        {
            addr = c->local_ip;
            family = c->local_addr->family;
        }
    }
    else if (arg[0] == '1' && !arg[1]) {
        if (c->local_addr->family == AF_INET
#if APR_HAVE_IPV6
            || (c->local_addr->family == AF_INET6 &&
                IN6_IS_ADDR_V4MAPPED((struct in6_addr *)
                                     c->local_addr->ipaddr_ptr))
#endif
            ) {
            /* httpd assures us local_ip is in ipv4 notation for mapped addrs */
            addr = c->local_ip;
            family = APR_INET;
        }
        else {
            return FTP_REPLY_BAD_PROTOCOL;
        }
    }
#if APR_HAVE_IPV6
    else if (arg[0] == '2' && !arg[1]) {
        family = AF_INET6;
        if (c->local_addr->family == AF_INET6 &&
            IN6_IS_ADDR_V4MAPPED((struct in6_addr *)
                                 c->local_addr->ipaddr_ptr)) {
            /* httpd assures us local_ip is in ipv4 notation for mapped addrs */
            addr = c->local_ip;
            family = APR_INET;
        }
        else if (c->local_addr->family == AF_INET6) {
            addr = c->local_ip;
            family = AF_INET6;
        }
        else {
            return FTP_REPLY_BAD_PROTOCOL;
        }
    }
#endif
    else {
        return FTP_REPLY_BAD_PROTOCOL;
    }

    if ((res = init_pasv_socket(r, family, addr))) {
        return res;
    }

    apr_socket_addr_get(&sa, APR_LOCAL, fc->csock);
    fc->response_notes = apr_psprintf(r->pool, 
                                      "Entering Extended Passive Mode (|||%u|)",
                                      sa->port);

    return FTP_REPLY_EXTENDED_PASSIVE_MODE;
}


static int ftp_cmd_pasv(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    int res;
    apr_sockaddr_t *sa;
    apr_socket_t *s;
    char *report_addr, *period;
    apr_port_t port;
    short high, low;
    const char *addr;
    int family;

    ftp_reset_dataconn(fc);

    if (fc->all_epsv) {
        fc->response_notes = "Restricted by EPSV ALL";
        return FTP_REPLY_BAD_SEQUENCE;
    }

    if (fsc->pasv_bindaddr) {
        addr = fsc->pasv_bindaddr;
        family = fsc->pasv_bindfamily;
    }
#if APR_HAVE_IPV6
    else if (c->local_addr->family == AF_INET6 &&
      IN6_IS_ADDR_V4MAPPED((struct in6_addr *) c->local_addr->ipaddr_ptr)) {
        /* httpd assures us local_ip is in ipv4 notation for mapped addrs */
        addr = c->local_ip;
        family = APR_INET;
    }
#endif
    else {
        addr = c->local_ip;
        family = c->local_addr->family;
    }

    if ((res = init_pasv_socket(r, family, addr))) {
        return res;
    }
    s = fc->csock;

    apr_socket_addr_get(&sa, APR_LOCAL, s);

    /*
     * pasv_addr is an ADVERTISED address, not related to the true address.
     * It should always be evaluated first for the benefit of servers hiding
     * behind load balancers, vpn's, nat, etc.
     */
    if (fsc->pasv_addr) {
        report_addr = apr_pstrdup(r->pool, fsc->pasv_addr);
    }
    else if (fsc->pasv_bindaddr && (fsc->pasv_bindfamily == APR_INET)) {
        report_addr = apr_pstrdup(r->pool, fsc->pasv_bindaddr);
    }
    else if ((c->local_addr->family == AF_INET)
#if APR_HAVE_IPV6
             || (c->local_addr->family == AF_INET6
                 && IN6_IS_ADDR_V4MAPPED((struct in6_addr *) c->local_addr->ipaddr_ptr))
#endif
        ) {
        /* httpd assures us local_ip is in ipv4 notation for mapped addrs */
        report_addr = apr_pstrdup(r->pool, c->local_ip);
    }
    else {
        /*
         * a bogus answer, which will not be translated below, wherein
         * clients can choose to connect back to the original, same address.
         * Suggested as an early solution at http://cr.yp.to/ftp/retr.html
         */
        report_addr = "127,555,555,555";
    }

    /* Translate x.x.x.x to x,x,x,x */
    while ((period = strstr(report_addr, ".")) != NULL) {
        *period = ',';
    }

    port = sa->port;
    high = ((port & 0xff00) >> 8);
    low = (port & 0x00ff);
    fc->response_notes = apr_psprintf(r->pool,
                                      "Entering Passive Mode (%s,%u,%u)",
                                      report_addr, high, low);
    return FTP_REPLY_PASSIVE_MODE;
}

static int ftp_cmd_pbsz(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    char *endp;

    /*
     * Check if we have completed the security exchange. Note, some
     * clients still send PBSZ even with Implicit SSL. So we
     * allow this misbehavior...
     */
    if (fc->auth == FTP_AUTH_NONE) {
        return FTP_REPLY_BAD_SEQUENCE;
    }

    fc->pbsz = strtol(arg, &endp, 10);
    /*
     * Return 501 if we were unable to parse the argument or if there was a
     * possibility of an overflow
     */
    if (((*arg == '\0') || (*endp != '\0')) || fc->pbsz < 0
        || fc->pbsz == LONG_MAX) {
        fc->response_notes = "Could not parse PBSZ argument";
        return FTP_REPLY_SYNTAX_ERROR;
    }

    fc->response_notes = apr_psprintf(r->pool, "PBSZ Command OK. "
                                      "Protection buffer size set to %d",
                                      fc->pbsz);
    return FTP_REPLY_COMMAND_OK;
}

static int get_outbound_port(apr_sockaddr_t **sa_rv, request_rec *r,
                             apr_int32_t family)
{
    conn_rec *c = r->connection;
    ftp_connection *fc = ftp_get_module_config(c->conn_config);
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    apr_sockaddr_t *sa;
    apr_socket_t *s;
    apr_status_t rv;
    apr_port_t local_port;

    if (fsc->active_min == -1) {
        local_port = 0;
    }
    else if (fsc->active_max == fsc->active_min) {
        local_port = fsc->active_min;
    }
    else {
        local_port = fsc->active_min +
            (apr_port_t) (rand() % (fsc->active_max -
                                    fsc->active_min + 1));
    }

    if (c->local_addr->family == family) {
        /* Shortcut; duplicate the contents */
        sa = apr_palloc(fc->data_pool, sizeof(apr_sockaddr_t));
        memcpy(sa, c->local_addr, sizeof(apr_sockaddr_t));
        sa->next = NULL;
        if (sa->family == APR_INET)
            sa->ipaddr_ptr = &(sa->sa.sin.sin_addr);
#if APR_HAVE_IPV6
        else if (sa->family == APR_INET6)
            sa->ipaddr_ptr = &(sa->sa.sin6.sin6_addr);
#endif
        sa->sa.sin.sin_port = htons(local_port);
    }
    else {
        /* Long way around, create a fresh sa, attempt to use 
         * the original port, before falling back on the nul adapter
         * TODO: this could use a config option to specify the
         * preferred interface/local address
         */
        rv = apr_sockaddr_info_get(&sa, c->local_ip, family,
                                   local_port, 0, fc->data_pool);
        if (!sa || rv) {
            ap_log_rerror(APLOG_MARK, APLOG_DEBUG, rv, r,
                          "Couldn't resolve explicit local socket address %s "
                          "(apr or socket stack bug?)  Retrying", c->local_ip);
            rv = apr_sockaddr_info_get(&sa, NULL, APR_INET,
                                       local_port, 0, fc->data_pool);
        }

        if (!sa || rv) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                          "Couldn't resolve emphemeral local socket address "
                          "(apr or socket stack bug?)  Giving up");
            return FTP_REPLY_CANNOT_OPEN_DATACONN;
        }
    }

#ifdef HAVE_FTP_LOWPORTD
    if ((local_port > 0) && (local_port < 1024)) {
        /*
         * Here's the case of low numbered port creation; we have spun off
         * a worker to serve socket fd's through a unix domain socket via the
         * ftp_request_lowport client.
         */
        rv = ftp_request_lowport(&s, r, sa, fc->data_pool);

        if (rv != APR_SUCCESS) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                          "Request socket failed from FTP low port daemon");
            return FTP_REPLY_CANNOT_OPEN_DATACONN;
        }
    }
    else
#endif
    {
#if APR_MAJOR_VERSION < 1
        rv = apr_socket_create_ex(&s, family, SOCK_STREAM, APR_PROTO_TCP,
                                  fc->data_pool);
#else
        rv = apr_socket_create(&s, family, SOCK_STREAM, APR_PROTO_TCP,
                               fc->data_pool);
#endif

        if (rv != APR_SUCCESS) {
            ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                          "Couldn't create socket");
            return FTP_REPLY_CANNOT_OPEN_DATACONN;
        }

        apr_socket_opt_set(s, APR_SO_REUSEADDR, 1);

        rv = apr_socket_bind(s, sa);

        if (rv != APR_SUCCESS) {
#ifdef EACCES
            if (sa->port < 1024 && rv == EACCES)
                ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                              "Couldn't bind to low numbered port (<1024).  "
                              "See FTPActiveRange directive");
            else
#endif
                ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                              "Couldn't bind to socket");
            apr_socket_close(s);
            return FTP_REPLY_CANNOT_OPEN_DATACONN;
        }
    }

    *sa_rv = sa;
    fc->csock = s;
    fc->response_notes = apr_psprintf(r->pool, FTP_MSG_SUCCESS, r->method);
    return FTP_REPLY_COMMAND_OK;
}

static int ftp_cmd_eprt(request_rec *r, const char *arg)
{
    conn_rec *c = r->connection;
    ftp_connection *fc = ftp_get_module_config(c->conn_config);
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    apr_sockaddr_t *sa;
    apr_status_t rv;
    char *arg_tok, *ip_addr;
    apr_int32_t family;
    apr_port_t port;
    int res;

    ftp_reset_dataconn(fc);

    if (fc->all_epsv) {
        fc->response_notes = "Restricted by EPSV ALL";
        return FTP_REPLY_BAD_SEQUENCE;
    }

    arg_tok = apr_pstrdup(fc->data_pool, arg);
    if ((res = ftp_eprt_decode(&family, &ip_addr, &port, arg_tok))
        != FTP_REPLY_COMMAND_OK) {
        fc->response_notes = "Invalid EPRT request";
        return res;
    }

    rv = apr_sockaddr_info_get(&fc->clientsa, ip_addr, family,
                               port, 0, fc->data_pool);
    if (!fc->clientsa || rv) {
        ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                     "Unable to resolve EPRT address of |%d|%s|%d|",
#if APR_HAVE_IPV6
                     (family == APR_INET6) ? 2 : 1, ip_addr, port);
#else
                     1, ip_addr, port);
#endif
        fc->response_notes = "Invalid EPRT command, unable to resolve request";
        return FTP_REPLY_SYNTAX_ERROR;
    }

    /*
     * Open data connection only if the EPRT connection is to the client's IP
     * address. All other EPRT connection requests are denied, unless
     * disabled using:
     * 
     * FTPOptions AllowProxyPORT
     * 
     * We must canonicalize the IP address first to compare it to our own idea
     * of the client's IP address.
     */
    if (!(fsc->options & FTP_OPT_ALLOWPROXYPORT)) {
        char *test_ip = "(unknown)";
        if (apr_sockaddr_ip_get(&test_ip, fc->clientsa) != APR_SUCCESS
            || (strcasecmp(test_ip, c->remote_ip) != 0)) {
            ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server,
                         "Rejected EPRT data connection request to %s "
                         "(doesn't match the client IP %s and "
                         "not configured to AllowProxyPORT)",
                         test_ip, c->remote_ip);
            fc->response_notes = "Invalid EPRT command, proxy EPRT is"
                                 " not permitted";
            return FTP_REPLY_SYNTAX_ERROR;
        }
    }

    return get_outbound_port(&sa, r, family);
}

static int ftp_cmd_port(request_rec *r, const char *arg)
{
    conn_rec *c = r->connection;
    ftp_connection *fc = ftp_get_module_config(c->conn_config);
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    apr_sockaddr_t *sa;
    apr_status_t rv;
    apr_port_t port;
    char *ip_addr, tc;
    int res, val[6];

    ftp_reset_dataconn(fc);

    if (fc->all_epsv) {
        fc->response_notes = "Restricted by EPSV ALL";
        return FTP_REPLY_BAD_SEQUENCE;
    }

    if ((res = sscanf(arg, "%d,%d,%d,%d,%d,%d%c", &val[0], &val[1], &val[2],
                      &val[3], &val[4], &val[5], &tc)) != 6) {
        fc->response_notes = "Invalid PORT request";
        return FTP_REPLY_SYNTAX_ERROR;
    }

    ip_addr = apr_psprintf(fc->data_pool, "%d.%d.%d.%d",
                           val[0], val[1], val[2], val[3]);

    port = ((val[4] << 8) + val[5]);

    /*
     * Open data connection only if the PORT connection is to the client's IP
     * address. All other PORT connection requests are denied, unless
     * enabled using:
     * 
     * FTPOptions AllowProxyPORT
     */
    if (!(fsc->options & FTP_OPT_ALLOWPROXYPORT)) {
        if (strcasecmp(ip_addr, c->remote_ip) != 0) {
            ap_log_error(APLOG_MARK, APLOG_ERR, 0, r->server,
                         "Rejected PORT data connection request to %s "
                         "(doesn't match the client IP %s and "
                         "not configured to AllowProxyPORT)",
                         ip_addr, c->remote_ip);
            fc->response_notes = "Invalid PORT command, proxy PORT is"
                                 " not permitted";
            return FTP_REPLY_SYNTAX_ERROR;
        }
    }

    rv = apr_sockaddr_info_get(&fc->clientsa, ip_addr, APR_INET,
                               port, 0, fc->data_pool);

    if (!fc->clientsa || rv) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, rv, r,
                      "Couldn't resolve remote socket address %s"
                      " (apr or socket stack bug?)", ip_addr);
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    return get_outbound_port(&sa, r, APR_INET);
}

static int ftp_cmd_prot(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    /*
     * Return 503 if the user has not done a AUTH command yet. Although
     * RFC2228 and RFC4217 are very explicit that PBSZ must preceed PROT,
     * it's entirely worthless in the context of TLS, and not even worth
     * enforcing.
     */
    if (fc->auth == FTP_AUTH_NONE) {
        return FTP_REPLY_BAD_SEQUENCE;
    }

    switch (*arg) {
    case 'C':
        fc->response_notes = "PROT Command OK. Using clear data channel";
        fc->prot = FTP_PROT_CLEAR;
        return FTP_REPLY_COMMAND_OK;
    case 'S':
        return FTP_REPLY_PROT_NOT_SUPPORTED;
    case 'E':
        return FTP_REPLY_PROT_NOT_SUPPORTED;
    case 'P':
        fc->response_notes = "PROT Command OK. Using private data channel";
        fc->prot = FTP_PROT_PRIVATE;
        return FTP_REPLY_COMMAND_OK;
    }

    /* We don't understand */
    fc->response_notes = "PROT argument not understood.";
    return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
}

static int ftp_cmd_pwd(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    fc->response_notes = apr_psprintf(r->pool, FTP_MSG_DIR_CUR, 
                             ftp_escape_control_text(fc->cwd, r->pool));
    return FTP_REPLY_PATH_CREATED;
}

static int ftp_cmd_quit(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    fc->close_connection = 1;
    return FTP_REPLY_CONTROL_CLOSE;
}

static int ftp_cmd_rest(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    char *endp;

#if APR_MAJOR_VERSION < 1
    fc->restart_point = apr_atoi64(arg);
    endp = (char *) arg + strspn(arg, "0123456789");
    if (
#else
    apr_status_t rv = apr_strtoff(&(fc->restart_point), arg, &endp, 10);
    if ((rv != APR_SUCCESS) ||
#endif
        (*arg == '\0') || (*endp != '\0') || (fc->restart_point < 0)) {
        fc->response_notes = "REST requires a non-negative integer value";
        return FTP_REPLY_SYNTAX_ERROR;
    }

    fc->response_notes = apr_psprintf(r->pool, "Restarting at %" APR_OFF_T_FMT
                                      ". Send STORE or RETRIEVE to initiate "
                                      "transfer.", fc->restart_point);
    return FTP_REPLY_PENDING;
}

static int ftp_cmd_retr(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    request_rec *rr;
    conn_rec *c = r->connection;
    conn_rec *cdata;
    ap_filter_t *f, *rinput, *routput, *rinput_proto, *routput_proto;
    int res;

    /* Put a note in the env table for logging */
    /*
     * This envar determines whether or not the command being
     * processed is one which causes a file to be uploaded or
     * downloaded ("transferred"). This allows a logfile to be
     * created of just transfers attempts. The success or failure
     * of the transfer is determined by the ftp_transfer_ok envar
     */
    apr_table_setn(r->subprocess_env, "do_transfer_log", "1");

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    /* If anything fails in the subrequest, simply return permission denied */
    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);
    if (rr->status != HTTP_OK) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(arg, r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    ap_destroy_sub_req(rr);

    ftp_reply(fc, c->output_filters, r->pool, FTP_REPLY_FILE_STATUS_OK, 0,
              apr_pstrcat(r->pool, "Opening ",
                          (fc->type == TYPE_A) ? "ASCII" : "BINARY",
                          " mode data connection for ", 
                          ftp_escape_control_text(arg, r->pool), NULL));

    if (!(cdata = ftp_open_dataconn(r, 1))) {
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    /* Save the original filters */
    rinput = r->input_filters;
    rinput_proto = r->proto_input_filters;
    routput = r->output_filters;
    routput_proto = r->proto_output_filters;

    r->proto_input_filters = cdata->input_filters;
    r->input_filters = r->proto_input_filters;
    r->proto_output_filters = cdata->output_filters;
    r->output_filters = r->proto_output_filters;

    ap_add_input_filter_handle(ftp_input_filter_handle, NULL, r, r->connection);

    r->connection = cdata;

    /*
     * If we are sending a ASCII file, we need to run the CRLF filter.
     * Running the CRLF filter will result in a bunch of buckets, so we
     * should also add the COALESCE filter to put everything back into a
     * single bucket.
     */
    if (fc->type == TYPE_A) {
        fc->filter_mask += FTP_NEED_CRLF;
    }

    if (fc->restart_point > 0) {
        fc->filter_mask += FTP_NEED_BYTERANGE;
    }

    rr = ap_sub_req_method_uri("GET", r->uri, r, NULL);
    if (rr->status != HTTP_OK) {
        /*
         * Should never get here since we have already run this subrequest,
         * but better safe than sorry
         */
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(arg, r->pool));
        res = FTP_REPLY_FILE_NOT_FOUND;
        goto clean_up;
    }

    /* If this is a resume request, set the "Range: {start}-" header */
    if (fc->restart_point > 0) {
        apr_table_setn(rr->headers_in, "Range",
                       apr_psprintf(r->pool, "bytes=%" APR_OFF_T_FMT "-",
                                    fc->restart_point));

        /* Byterange requires that we are not assbackwards (HTTP/0.9) */
        rr->assbackwards = 0;
    }

    /*
     * Filter manipulation.  We remove the subrequest filter so that the EOS
     * buckets stay intact.  We also add the content length filter so that
     * we can record the bytes_sent
     */
    for (f = rr->output_filters; f; f = f->next) {
        if (strcasecmp(f->frec->name, "SUBREQ_CORE") == 0) {
            ap_remove_output_filter(f);
        }
    }

    /* Need content length rr->bytes_sent */
    ap_add_output_filter_handle(ftp_content_length_filter_handle, NULL,
                                rr, rr->connection);

    if ((res = ap_run_sub_req(rr)) != OK) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(arg, r->pool));
        res = FTP_REPLY_FILE_NOT_FOUND;
    }

    else {
        res = FTP_REPLY_DATA_CLOSE;
    }

    fc->restart_point = 0;
    fc->traffic += rr->bytes_sent;
    fc->bytes += rr->bytes_sent;
    fc->files += 1;
    fc->transfers += 1;

    /*
     * We use a subrequest here in an weird way, which causes some
     * information to be lost.  Here we hack around that by setting values in
     * r->main, but in the future, we may just want to do away with running
     * the subrequest, and run the main request.
     * 
     * There may be other values we need to save here.
     */
    rr->main->sent_bodyct = 1;

    /* Check to make sure the connection wasnt aborted */
    if (rr->connection->aborted || cdata->aborted) {
        rr->main->bytes_sent = 0;
        res = FTP_REPLY_TRANSFER_ABORTED;
        rr->main->connection->aborted = 0;
    }
    else {
        rr->main->bytes_sent = rr->bytes_sent;
    }

clean_up:
    ap_destroy_sub_req(rr);

    /* Replace the filters and connection */
    r->input_filters = rinput;
    r->proto_input_filters = rinput_proto;
    r->output_filters = routput;
    r->proto_output_filters = routput_proto;
    r->connection = c;

    /* Close the data connection, send confirmation, and return  */
    ap_lingering_close(cdata);
    fc->datasock = NULL;
    fc->filter_mask = 0;

    return res;
}

static int ftp_cmd_rnfr(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    request_rec *rr;
    apr_status_t res;
    int response;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }
    else if (rr->finfo.filetype != APR_NOFILE) {
        fc->response_notes = "File exists, ready for destination name";
        apr_cpystrn(fc->rename_from, r->filename, APR_PATH_MAX + 1);
        response = FTP_REPLY_PENDING;
    }
    else {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }

    ap_destroy_sub_req(rr);

    return response;
}

static int ftp_cmd_rnto(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    apr_status_t rv;
    int response, res;
    request_rec *rr;

    if ((res = ftp_set_uri(r, arg))) {
        fc->rename_from[0] = '\0';
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    ap_destroy_sub_req(rr);

    /* RNTO *must* be preceeded by RNFR */
    if (fc->rename_from[0] == '\0') {
        return FTP_REPLY_BAD_SEQUENCE;
    }

    rv = apr_file_rename(fc->rename_from, r->filename, r->pool);

    if (rv != APR_SUCCESS) {
        response = FTP_REPLY_LOCAL_ERROR;
    }
    else {
        response = FTP_REPLY_COMPLETED;
    }

    fc->rename_from[0] = '\0';
    return response;
}

static int ftp_cmd_rmd(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    apr_status_t rv, res;
    request_rec *rr;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    ap_destroy_sub_req(rr);

    rv = apr_dir_remove(r->filename, r->pool);

    if (rv != APR_SUCCESS) {
        char error_str[120];
        char *err = apr_strerror(rv, error_str, sizeof(error_str));
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(err, r->pool));
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    else {
        return FTP_REPLY_COMPLETED;
    }
}

static int ftp_cmd_size(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    int res, response;
    request_rec *rr;

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path, r->pool));
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    if (rr->finfo.filetype == 0) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOSUCHFILE,
                                 ftp_escape_control_text(arg, r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }
    else if (rr->finfo.filetype != APR_REG) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_NOTPLAIN,
                                 ftp_escape_control_text(arg, r->pool));
        response = FTP_REPLY_FILE_NOT_FOUND;
    }
    else {
        /*
         * XXX: big bug - going back to the beginning.  We must compute size
         * rather than stating the file... this could be done trivially with
         * a request to a null 'data' port which simply counts up the bytes.
         * Will be implemented as a special-case of the ftp_core_data
         * connection endpoint-filter.
         * See RFC3659 - especially with respect to TYPE A/I
         */
        fc->response_notes = apr_psprintf(r->pool, "%" APR_OFF_T_FMT,
                                          rr->finfo.size);
        response = FTP_REPLY_FILE_STATUS;
    }

    ap_destroy_sub_req(rr);
    return response;
}

static int ftp_get_client_block(conn_rec *c, ap_input_mode_t mode,
                                apr_bucket_brigade *bb,
                                char *buffer, apr_size_t bufsiz,
                                apr_size_t *len)
{
    apr_size_t total;
    apr_status_t rv;
    apr_bucket *b;
    const char *tempbuf;

    if (APR_BRIGADE_EMPTY(bb)) {
        if ((rv = ap_get_brigade(c->input_filters, bb, mode,
                                 APR_BLOCK_READ, bufsiz)) != APR_SUCCESS) {
            apr_brigade_destroy(bb);
            return rv;
        }
    }

    b = APR_BRIGADE_FIRST(bb);
    if (APR_BUCKET_IS_EOS(b)) { /* From previous invocation */
        apr_bucket_delete(b);
        return APR_EOF;
    }

    total = 0;
    while (total < bufsiz
           && b != APR_BRIGADE_SENTINEL(bb)
           && !APR_BUCKET_IS_EOS(b)) {
        apr_size_t len_read;
        apr_bucket *old;

        if ((rv = apr_bucket_read(b, &tempbuf, &len_read,
                                  APR_BLOCK_READ)) != APR_SUCCESS) {
            return rv;
        }
        if (total + len_read > bufsiz) {
            len_read = bufsiz - total;
            apr_bucket_split(b, bufsiz - total);
        }
        memcpy(buffer, tempbuf, len_read);
        buffer += len_read;
        total += len_read;

        old = b;
        b = APR_BUCKET_NEXT(b);
        apr_bucket_delete(old);
    }

    *len = total;
    return APR_SUCCESS;
}

/*
 * This handles all STOR and APPE requests
 */
static int ftp_cmd_stor(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    conn_rec *cdata;
    ftp_dir_config *dconf;
    apr_file_t *file;
    apr_status_t rv, res;
    apr_bucket_brigade *bb;
    apr_int32_t openflag;
    char *buffer;
    apr_off_t total = 0;
    apr_size_t len;
    request_rec *rr;
    int clientstatus = APR_SUCCESS;
    int status = FTP_REPLY_DATA_CLOSE;
#ifndef WIN32
    int seen_cr = 0;
#endif
    apr_fileperms_t creatperms;
#ifdef HAVE_FCHMOD
    mode_t fixmode;
    apr_finfo_t finfo;
    int fd;
#endif

    apr_table_setn(r->subprocess_env, "do_transfer_log", "1");

    if ((res = ftp_set_uri(r, arg))) {
        return res;
    }

    rr = ap_sub_req_method_uri(r->method, r->uri, r, NULL);

    if ((rr->status == HTTP_UNAUTHORIZED) || (rr->status == HTTP_FORBIDDEN)) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(r->parsed_uri.path,
                                                         r->pool));
        ap_destroy_sub_req(rr);
        return FTP_REPLY_FILE_NOT_FOUND;
    }

    dconf = ftp_get_module_config(rr->per_dir_config);
    creatperms = dconf->fileperms;
    ap_destroy_sub_req(rr);

    if (creatperms == APR_OS_DEFAULT)
        creatperms = FTP_DEFAULT_UMASK;

    /*
     * In the config phase, ->creatperms was a negative umask. for operation,
     * exchange this with a positive protections to pass to the apr_file_open
     * protection flag.
     */
    creatperms = (APR_UREAD | APR_UWRITE |
                  APR_GREAD | APR_GWRITE |
                  APR_WREAD | APR_WWRITE)
        & ~creatperms;

#ifdef HAVE_FCHMOD
    fixmode = ftp_unix_perms2mode(creatperms);
    creatperms = 0;
#endif

    /*
     * For the remainder of the operation, (openflag & APR_APPEND) reflects
     * this was an append operation and we have no need to truncate.
     * Presume, if not append and our restart point is zero, that
     * open(O_TRUNC) is supported and preferable.
     */
    openflag = APR_WRITE | APR_CREATE;
    if (strcmp(r->method, "APPE") == 0) {
        openflag |= APR_APPEND;
    }
    else if (fc->restart_point == 0) {
        openflag |= APR_TRUNCATE;
    }

    rv = apr_file_open(&file, r->filename, openflag, creatperms, r->pool);

    /* File permissions deny server write access */
    if (rv != APR_SUCCESS) {
        char error_str[120];
        char *err = apr_strerror(rv, error_str, sizeof(error_str));
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                 ftp_escape_control_text(err, r->pool));
        fc->restart_point = 0;
        return FTP_REPLY_FILE_NOT_FOUND;
    }

#ifdef HAVE_FCHMOD
    /*
     * If we opened an existing file, we have nothing to do later in the code
     * (leave fd as -1). If we created the file with perms zero, grab the fd
     * and creatperms to adjust the perms when finished.
     */
    if (apr_file_info_get(&finfo, APR_FINFO_PROT, file)
        != APR_SUCCESS) {
        fc->response_notes = apr_psprintf(r->pool, FTP_MSG_PERM_DENIED,
                                          "Unable to perform file upload; "
                                          "failed to get fileinfo");
        fc->restart_point = 0;
        return FTP_REPLY_FILE_NOT_FOUND;
    }
    if (finfo.protection) {
        fd = -1;
    }
    else {
        apr_os_file_get(&fd, file);
    }
#endif

    /*
     * Once the file is opened, non-append/non-truncate, we need to space
     * forward to the restart point.  Unset the restart pointer once we have
     * grabbed it to space forward.
     */
    if (!(openflag & (APR_APPEND | APR_TRUNCATE))) {
        apr_off_t restart = fc->restart_point;
        if (apr_file_seek(file, APR_SET, &restart) != APR_SUCCESS) {
            fc->response_notes = "Requested action not taken: invalid REST "
                                 "parameter; failed to skip to restart point";
#ifdef HAVE_FCHMOD
            if (fd != -1)
                fchmod(fd, fixmode);
#endif
            fc->restart_point = 0;
            return FTP_REPLY_INVALID_REST_PARAM;
        }
    }
    fc->restart_point = 0;

    ftp_reply(fc, c->output_filters, r->pool, FTP_REPLY_FILE_STATUS_OK, 0,
              apr_pstrcat(r->pool, "Opening ",
                          (fc->type == TYPE_A) ? "ASCII" : "BINARY",
                          " mode data connection for ", 
                          ftp_escape_control_text(arg, r->pool), NULL));

    /*
     * Open the data connection to the client
     */
    if (!(cdata = ftp_open_dataconn(r, 0))) {
#ifdef HAVE_FCHMOD
        if (fd != -1)
            fchmod(fd, fixmode);
#endif
        return FTP_REPLY_CANNOT_OPEN_DATACONN;
    }

    bb = apr_brigade_create(r->pool, c->bucket_alloc);
    buffer = apr_palloc(r->pool, AP_IOBUFSIZE);

    while ((clientstatus = ftp_get_client_block(cdata,
                              fc->type == TYPE_A ? AP_MODE_GETLINE : AP_MODE_READBYTES,
                              bb, buffer, AP_IOBUFSIZE, &len)) == APR_SUCCESS) {
#ifndef WIN32
        /*
         * If we are in ASCII mode, translate to our own internal form. This
         * means CRLF for windows and plain LF for Unicies
         */
        if ((fc->type == TYPE_A) && (len > 0)) {
            /*
             * The last read may have been incomplete if we had activity on
             * the control connection.  Watch out for trailing CR's from the
             * last write, and back up the file a byte if one had been
             * written for this now-leading LF.
             */
            if (seen_cr && (*buffer == APR_ASCII_LF)) {
                apr_off_t off = -1;
                apr_file_seek(file, APR_CUR, &off);
                --total;
            }

            /*
             * We may reach EOF without an ending newline.  Make sure we
             * don't truncate any important data
             */
            if ((len > 1) && (*(buffer + len - 2) == APR_ASCII_CR)) {
                *(buffer + len - 2) = APR_ASCII_LF;
                len--;
            }
            else {
                /* Now note any trailing CR on this write. */
                seen_cr = (*(buffer + len - 1) == APR_ASCII_CR);
            }
        }

#endif                          /* WIN32 */

        rv = apr_file_write(file, buffer, &len);
        if (rv != APR_SUCCESS) {
            char error_str[120];
            char *err = apr_strerror(rv, error_str, sizeof(error_str));
            ap_log_error(APLOG_MARK, APLOG_ERR, rv, r->server,
                         "Error writing uploaded file");
            fc->response_notes = ftp_escape_control_text(err, r->pool);
            status = FTP_REPLY_LOCAL_ERROR;
            break;
        }

        total += len;
    }

    if ((clientstatus != APR_SUCCESS && !APR_STATUS_IS_EOF(clientstatus))
        || (cdata->aborted)) {
        status = FTP_REPLY_TRANSFER_ABORTED;
        cdata->aborted = 1;
    }

    fc->traffic += total;
    fc->bytes += total;
    fc->files += 1;
    fc->transfers += 1;

    /* Keep track of bytes sent for logging purposes. */
    r->sent_bodyct = 1;
    r->bytes_sent = total;

    /*
     * Once we have finished this upload - we need to reset the file perms to
     * no longer hold a lock on the uploaded file...
     */
#ifdef HAVE_FCHMOD
    if (fd != -1)
        fchmod(fd, fixmode);
#endif

    /*
     * and truncate anything beyond the end of the most recent upload
     */
    if (!(openflag & (APR_APPEND | APR_TRUNCATE))) {
        /* use apr_file_seek to do apr_file_tell's... */
        apr_off_t totsize = 0;
        apr_off_t restart = 0;
        if ((rv = apr_file_seek(file, APR_CUR, &restart)) != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_WARNING, rv, r->server,
                         "STOR resume: failed to determine current position for truncation");
        }
        else if ((rv = apr_file_seek(file, APR_END, &totsize)) != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_WARNING, rv, r->server,
                         "STOR resume: failed to determine current file size for truncation");
        }
        else if (totsize <= restart) {
            ;                   /* noop - nothing to truncate */
        }
        else if ((rv = apr_file_trunc(file, restart)) != APR_SUCCESS) {
            ap_log_error(APLOG_MARK, APLOG_WARNING, rv, r->server,
                 "STOR resume: failed to truncate remaining file contents");
        }
    }

    rv = apr_file_close(file);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_WARNING, rv, r->server,
                     "STOR/APPE: failed to close file");
    }

    ap_lingering_close(cdata);
    fc->datasock = NULL;

    return status;
}

static int ftp_cmd_stru(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    if (*arg && !arg[1]) {
        switch (toupper(*arg)) {
        case 'F':
            fc->response_notes = "Structure set to F";
            return FTP_REPLY_COMMAND_OK;
        }
    }
    fc->response_notes = apr_psprintf(r->pool, "Structure %s not implemented",
                             ftp_escape_control_text(arg, r->pool));

    return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
}

static int ftp_cmd_syst(request_rec *r, const char *arg)
{
    return FTP_REPLY_SYSTEM_TYPE;
}

static int ftp_cmd_type(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    if (*arg && !arg[1])
    {
        switch (toupper(*arg)) {
        case 'A':
            fc->type = TYPE_A;
            fc->response_notes = "Type set to A";
            return FTP_REPLY_COMMAND_OK;
        case 'I':
            fc->type = TYPE_I;
            fc->response_notes = "Type set to I";
            return FTP_REPLY_COMMAND_OK;
        }
    }
    else if (!strcasecmp(arg, "A N")) {
        /*
         * Accept Ascii Non-Print
         */
        fc->type = TYPE_A;
        fc->response_notes = "Type set to A N";
        return FTP_REPLY_COMMAND_OK;
    }
    else if (!strcasecmp(arg, "L 8")) {
        /*
         * Treat Local 8 as indistinguishible from Image for httpd platforms
         */
        fc->type = TYPE_I;
        fc->response_notes = "Type set to L 8";
        return FTP_REPLY_COMMAND_OK;
    }

    fc->response_notes = apr_psprintf(r->pool, "Type %s not implemented",
                             ftp_escape_control_text(arg, r->pool));

    return FTP_REPLY_COMMAND_NOT_IMPL_PARAM;
}

static int ftp_cmd_user(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);
    conn_rec *c = r->connection;
    ftp_server_config *fsc = ftp_get_module_config(r->server->module_config);
    apr_time_t prev_timeout;
    apr_status_t rv;
    char *username;
    char *hostname;

    /* Implicit logout */
    if (fc->logged_in) {
        ftp_limitlogin_loggedout(c);
    }
    fc->logged_in = 0;
    r->server = fc->orig_server = c->base_server;
    r->per_dir_config = r->server->lookup_defaults;
    r->hostname = fc->host = NULL;
    apr_pool_clear(fc->login_pool);

    fc->user = username = apr_pstrdup(fc->login_pool, arg);

    /*
     * Identify virtual host (user@{hostname}) for named vhost lookup, and
     * split from user name if so configured.
     */
    if ((hostname = ap_strchr(username, '@')) != NULL) {
        /*
         * Toggle to the Host:-based vhost's timeout mode to process this
         * login request
         */
        if (fsc->options & FTP_OPT_VHOST_BY_USER) {
            r->hostname = hostname + 1;

            ap_update_vhost_from_headers(r);

            fc->host = r->hostname;
            fc->orig_server = r->server;
        }
    }

    /* we may have switched to another server */
    fsc = ftp_get_module_config(r->server->module_config);
    r->per_dir_config = r->server->lookup_defaults;

    /*
     * Now that we switched virtual hosts, it's time to determine if the
     * username fc->user's "@{hostname}" should be discarded
     */
    if ((hostname != NULL) && (fsc->options & FTP_OPT_STRIP_HOSTNAME))
        *hostname = '\0';

    /*
     * We have nominally 'logged out', and also potentially changed virtual
     * host contexts; reset to the proper timeout_login
     */
    rv = apr_socket_timeout_get(fc->cntlsock, &prev_timeout);
    if (rv != APR_SUCCESS || prev_timeout != fsc->timeout_login) {
        rv = apr_socket_timeout_set(fc->cntlsock,
                                    fsc->timeout_login * APR_USEC_PER_SEC);
        if (rv != APR_SUCCESS)
            ap_log_error(APLOG_MARK, APLOG_DEBUG, rv, r->server,
                         "Couldn't set SO_TIMEOUT socket option");
    }

    if ((fsc->options & FTP_OPT_REQUIRESSL) && !fc->is_secure) {
        r->hostname = fc->host = NULL;
        fc->user = ftp_unknown_username;
        fc->authorization = NULL;
        r->server = fc->orig_server = c->base_server;
        r->per_dir_config = r->server->lookup_defaults;
        fc->response_notes = "This server requires the use of SSL";
        return FTP_REPLY_NOT_LOGGED_IN;
    }

    /* TODO: these should really be configurable */
    if ((strcmp(fc->user, "anonymous") == 0) ||
        (strncmp(fc->user, "anonymous@", 10) == 0) ||
        (strcmp(fc->user, "ftp") == 0) ||
        (strncmp(fc->user, "ftp@", 4) == 0)) {
        fc->response_notes = "Guest login ok, type your email address"
                             " as the password";
        /* force this for limit and access control */
        fc->user = "anonymous";
    }
    else {
        fc->response_notes = apr_psprintf(r->pool, "Password required for %s",
                                 ftp_escape_control_text(fc->user, r->pool));
    }

    return FTP_REPLY_USER_OK;
}

void ftp_register_core_cmds(apr_pool_t *p)
{

    /*
     * Create the FTP command hash.  All external modules will need to make
     * sure that their register hooks callbacks are called after the FTP
     * module
     */

    FTPMethodHash = apr_hash_make(p);
    FTPMethodPool = p;          /* The config pool */

    /* Register all of our core command handlers */

    ftp_hook_cmd("ABOR", ftp_cmd_abor, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0 | FTP_DATA_INTR,
                 "(abort operation)");

    ftp_hook_cmd("ACCT", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "(specify account)");

    ftp_hook_cmd("ALLO", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "allocate storage (vacuously)");

    ftp_hook_cmd("APPE", ftp_cmd_stor, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    /* AUTH does not require a user to be logged in */
    ftp_hook_cmd("AUTH", ftp_cmd_auth, FTP_HOOK_LAST,
                 FTP_TAKE1,
                 "<sp> mechanism-name");

    /* XXX: Advertise only if configured! */
    ftp_feat_advert("AUTH TLS");

    ftp_hook_cmd("CDUP", ftp_cmd_cdup, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "change to parent directory");

    ftp_hook_cmd("CWD", ftp_cmd_cwd, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "[ <sp> directory-name ]");

    ftp_hook_cmd("DELE", ftp_cmd_dele, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("EPRT", ftp_cmd_eprt, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1 | FTP_NEW_FEAT,
                 "<sp> <d>af<d>addr<d>port<d>");

    ftp_hook_cmd("EPSV", ftp_cmd_epsv, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0 | FTP_TAKE1 | FTP_NEW_FEAT,
                 "[ <sp> af|ALL ]");

    ftp_hook_cmd("FEAT", ftp_cmd_feat, FTP_HOOK_LAST,
                 FTP_TAKE0,
                 "show server features");

    ftp_hook_cmd("HELP", ftp_cmd_help, FTP_HOOK_LAST,
                 FTP_TAKE0 | FTP_TAKE1,
                 "[ <sp> <string> ]");

    ftp_hook_cmd("LANG", NULL, FTP_HOOK_LAST,
                 FTP_TAKE1,
                 "<sp> lang-code");

    ftp_hook_cmd("LIST", ftp_cmd_list, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0 | FTP_TAKE1_PATH,
                 "[ <sp> path-name ]");

    ftp_hook_cmd("MDTM", ftp_cmd_mdtm, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH | FTP_NEW_FEAT,
                 "<sp> path-name");

    ftp_hook_cmd("MKD", ftp_cmd_mkd, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> path-name");

    ftp_hook_cmd("MODE", ftp_cmd_mode, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1,
                 "<sp> [ S | B | C ] (specify transfer mode)");

    ftp_hook_cmd("NLST", ftp_cmd_nlst, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0 | FTP_TAKE1_PATH,
                 "[ <sp> path-name ]");

    ftp_hook_cmd("NOOP", ftp_cmd_noop, FTP_HOOK_LAST,
                 FTP_TAKE0,
                 "");

    ftp_hook_cmd("OPTS", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1 | FTP_EXTENSIBLE,
                 "<sp> command [ <sp> options ]");

    /* Pass must be handled literally to ensure leading, trailing spaces 
     * or empty string are recognized
     */
    ftp_hook_cmd("PASS", ftp_cmd_pass, FTP_HOOK_LAST,
                 FTP_TAKE0 | FTP_TAKE1_PATH,
                 "<sp> password");

    ftp_hook_cmd("PASV", ftp_cmd_pasv, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "(set server in passive mode)");

    /* PBSZ does not require a user to be logged in */
    ftp_hook_cmd("PBSZ", ftp_cmd_pbsz, FTP_HOOK_LAST,
                 FTP_TAKE1 | FTP_NEW_FEAT,
                 "<sp> decimal-integer");

    ftp_hook_cmd("PORT", ftp_cmd_port, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1,
                 "<sp> b0, b1, b1, b3, b4, b5");

    /* PROT does not require a user to be logged in */
    ftp_hook_cmd("PROT", ftp_cmd_prot, FTP_HOOK_LAST,
                 FTP_TAKE1 | FTP_NEW_FEAT,
                 "<sp> prot-code");

    ftp_hook_cmd("PWD", ftp_cmd_pwd, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "(return current directory)");

    ftp_hook_cmd("QUIT", ftp_cmd_quit, FTP_HOOK_LAST,
                 FTP_TAKE0 | FTP_DATA_INTR,
                 "(terminate service)");

    ftp_hook_cmd("REIN", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0 | FTP_DATA_INTR,
                 "(reinitialize server)");

    ftp_hook_cmd("REST", ftp_cmd_rest, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1,
                 "<sp> offset (restart command)");

    ftp_feat_advert("REST STREAM");

    ftp_hook_cmd("RETR", ftp_cmd_retr, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("RMD", ftp_cmd_rmd, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> path-name");

    ftp_hook_cmd("RNFR", ftp_cmd_rnfr, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("RNTO", ftp_cmd_rnto, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("SITE", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1 | FTP_EXTENSIBLE,
                 "<sp> site-cmd [ <sp> arguments ]");

    ftp_hook_cmd("SIZE", ftp_cmd_size, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH | FTP_NEW_FEAT,
                 "<sp> path-name");

    ftp_hook_cmd("SMNT", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "(structure mount)");

    ftp_hook_cmd("STAT", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "[ <sp> path-name ]");

    ftp_hook_cmd("STOR", ftp_cmd_stor, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("STOU", NULL, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1_PATH,
                 "<sp> file-name");

    ftp_hook_cmd("STRU", ftp_cmd_stru, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1,
                 "<sp> [ F | R | P ] (specify file structure)");

    ftp_hook_cmd("SYST", ftp_cmd_syst, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE0,
                 "(get type of operating system)");

    /* RFC3659 TVFS advertising simply insists that '/' is not a filename
     * character, but a path delimiter in a tree structured filesystem.
     * That's us, advertise it;
     */
    ftp_feat_advert("TVFS");

    ftp_hook_cmd("TYPE", ftp_cmd_type, FTP_HOOK_LAST,
                 FTP_NEED_LOGIN | FTP_TAKE1,
                 "<sp> [ A [ fmt ] | E [ fmt ] | I | L size ]");

    ftp_hook_cmd("USER", ftp_cmd_user, FTP_HOOK_LAST,
                 FTP_TAKE1,
                 "<sp> username");

    /* RFC2640 offers UTF-8; it does not insist all octets are UTF-8,
     * even for arbitrary file names.  This reflects 'most unix' today.
     * Because we want to permit admins not to advertise support,
     * this is handled in post-config; see mod_ftp.c.
     */
    /* ftp_feat_advert("UTF8"); */

    /* Unadvertised, deprecated RFC775 X-flavor CDUP */
    ftp_hook_cmd_alias("XCUP", "CDUP", FTP_HOOK_LAST,
                       FTP_NO_HELP | FTP_NEED_LOGIN | FTP_TAKE0,
                       "change to parent directory");

    /* Unadvertised, deprecated RFC542 X-flavor CWD */
    ftp_hook_cmd_alias("XCWD", "CWD", FTP_HOOK_LAST,
                       FTP_NO_HELP | FTP_NEED_LOGIN | FTP_TAKE1,
                       "[ <sp> directory-name ]");

    /* Unadvertised, deprecated RFC775 X-flavor MKD */
    ftp_hook_cmd_alias("XMKD", "MKD", FTP_HOOK_LAST,
                       FTP_NO_HELP | FTP_NEED_LOGIN | FTP_TAKE1,
                       "<sp> path-name");

    /* Unadvertised, deprecated RFC775 X-flavor PWD */
    ftp_hook_cmd_alias("XPWD", "PWD", FTP_HOOK_LAST,
                       FTP_NO_HELP | FTP_NEED_LOGIN | FTP_TAKE0,
                       "(return current directory)");

    /* Unadvertised, deprecated RFC775 X-flavor RMD */
    ftp_hook_cmd_alias("XRMD", "RMD", FTP_HOOK_LAST,
                       FTP_NO_HELP | FTP_NEED_LOGIN | FTP_TAKE1,
                       "<sp> path-name");

}
