"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const __1 = require("..");
class InvalidPathException extends __1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} is invalid.`); }
}
exports.InvalidPathException = InvalidPathException;
class PathMustBeAbsoluteException extends __1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} must be absolute.`); }
}
exports.PathMustBeAbsoluteException = PathMustBeAbsoluteException;
class PathCannotBeFragmentException extends __1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} cannot be made a fragment.`); }
}
exports.PathCannotBeFragmentException = PathCannotBeFragmentException;
/**
 * The Separator for normalized path.
 * @type {Path}
 */
exports.NormalizedSep = '/';
/**
 * The root of a normalized path.
 * @type {Path}
 */
exports.NormalizedRoot = exports.NormalizedSep;
/**
 * Split a path into multiple path fragments. Each fragments except the last one will end with
 * a path separator.
 * @param {Path} path The path to split.
 * @returns {Path[]} An array of path fragments.
 */
function split(path) {
    return path.split(exports.NormalizedSep).map(x => fragment(x));
}
exports.split = split;
/**
 *
 */
function extname(path) {
    const base = basename(path);
    const i = base.lastIndexOf('.');
    if (i < 1) {
        return '';
    }
    else {
        return base.substr(i);
    }
}
exports.extname = extname;
/**
 * Return the basename of the path, as a Path. See path.basename
 */
function basename(path) {
    const i = path.lastIndexOf(exports.NormalizedSep);
    if (i == -1) {
        return fragment(path);
    }
    else {
        return fragment(path.substr(path.lastIndexOf(exports.NormalizedSep) + 1));
    }
}
exports.basename = basename;
/**
 * Return the dirname of the path, as a Path. See path.dirname
 */
function dirname(path) {
    const i = path.lastIndexOf(exports.NormalizedSep);
    if (i == -1) {
        return '';
    }
    else {
        return normalize(path.substr(0, i));
    }
}
exports.dirname = dirname;
/**
 * Join multiple paths together, and normalize the result. Accepts strings that will be
 * normalized as well (but the original must be a path).
 */
function join(p1, ...others) {
    if (others.length > 0) {
        return normalize((p1 ? p1 + exports.NormalizedSep : '') + others.join(exports.NormalizedSep));
    }
    else {
        return p1;
    }
}
exports.join = join;
/**
 * Returns true if a path is absolute.
 */
function isAbsolute(p) {
    return p.startsWith(exports.NormalizedSep);
}
exports.isAbsolute = isAbsolute;
/**
 * Returns a path such that `join(from, relative(from, to)) == to`.
 * Both paths must be absolute, otherwise it does not make much sense.
 */
function relative(from, to) {
    if (!isAbsolute(from)) {
        throw new PathMustBeAbsoluteException(from);
    }
    if (!isAbsolute(to)) {
        throw new PathMustBeAbsoluteException(to);
    }
    let p;
    if (from == to) {
        p = '';
    }
    else {
        const splitFrom = from.split(exports.NormalizedSep);
        const splitTo = to.split(exports.NormalizedSep);
        while (splitFrom.length > 0 && splitTo.length > 0 && splitFrom[0] == splitTo[0]) {
            splitFrom.shift();
            splitTo.shift();
        }
        if (splitFrom.length == 0) {
            p = splitTo.join(exports.NormalizedSep);
        }
        else {
            p = splitFrom.map(_ => '..').concat(splitTo).join(exports.NormalizedSep);
        }
    }
    return normalize(p);
}
exports.relative = relative;
/**
 * Returns a Path that is the resolution of p2, from p1. If p2 is absolute, it will return p2,
 * otherwise will join both p1 and p2.
 */
function resolve(p1, p2) {
    if (isAbsolute(p2)) {
        return p2;
    }
    else {
        return join(p1, p2);
    }
}
exports.resolve = resolve;
function fragment(path) {
    if (path.indexOf(exports.NormalizedSep) != -1) {
        throw new PathCannotBeFragmentException(path);
    }
    return path;
}
exports.fragment = fragment;
/**
 * Normalize a string into a Path. This is the only mean to get a Path type from a string that
 * represents a system path. Normalization includes:
 *   - Windows backslashes `\\` are replaced with `/`.
 *   - Windows drivers are replaced with `/X/`, where X is the drive letter.
 *   - Absolute paths starts with `/`.
 *   - Multiple `/` are replaced by a single one.
 *   - Path segments `.` are removed.
 *   - Path segments `..` are resolved.
 *   - If a path is absolute, having a `..` at the start is invalid (and will throw).
 */
function normalize(path) {
    if (path == '' || path == '.') {
        return '';
    }
    else if (path == exports.NormalizedRoot) {
        return exports.NormalizedRoot;
    }
    // Match absolute windows path.
    const original = path;
    if (path.match(/^[A-Z]:\\/)) {
        path = '\\' + path[0] + '\\' + path.substr(3);
    }
    // We convert Windows paths as well here.
    const p = path.split(/[\/\\]/g);
    let relative = false;
    let i = 1;
    // Special case the first one.
    if (p[0] != '') {
        p.unshift('.');
        relative = true;
    }
    while (i < p.length) {
        if (p[i] == '.') {
            p.splice(i, 1);
        }
        else if (p[i] == '..') {
            if (i < 2 && !relative) {
                throw new InvalidPathException(original);
            }
            else if (i >= 2 && p[i - 1] != '..') {
                p.splice(i - 1, 2);
                i--;
            }
            else {
                i++;
            }
        }
        else if (p[i] == '') {
            p.splice(i, 1);
        }
        else {
            i++;
        }
    }
    if (p.length == 1) {
        return p[0] == '' ? exports.NormalizedSep : '';
    }
    else {
        if (p[0] == '.') {
            p.shift();
        }
        return p.join(exports.NormalizedSep);
    }
}
exports.normalize = normalize;
function asWindowsPath(path) {
    const drive = path.match(/^\/(\w)\/(.*)$/);
    if (drive) {
        return `${drive[1]}:\\${drive[2].replace(/\//g, '\\')}`;
    }
    return path.replace(/\//g, '\\');
}
exports.asWindowsPath = asWindowsPath;
function asPosixPath(path) {
    return path;
}
exports.asPosixPath = asPosixPath;
//# sourceMappingURL=data:application/json;base64,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