"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ajv = require("ajv");
const http = require("http");
const Observable_1 = require("rxjs/Observable");
const fromPromise_1 = require("rxjs/observable/fromPromise");
const of_1 = require("rxjs/observable/of");
const map_1 = require("rxjs/operators/map");
function _parseJsonPointer(pointer) {
    if (pointer === '') {
        return [];
    }
    if (pointer.charAt(0) !== '/') {
        throw new Error('Invalid JSON pointer: ' + pointer);
    }
    return pointer.substring(1).split(/\//).map(str => str.replace(/~1/g, '/').replace(/~0/g, '~'));
}
function _visitJsonSchema(schema, visitor) {
    const keywords = {
        additionalItems: true,
        items: true,
        contains: true,
        additionalProperties: true,
        propertyNames: true,
        not: true,
    };
    const propsKeywords = {
        definitions: true,
        properties: true,
        patternProperties: true,
        dependencies: true,
    };
    function _traverse(schema, jsonPtr, rootSchema, parentSchema, keyIndex) {
        if (schema && typeof schema == 'object' && !Array.isArray(schema)) {
            visitor(schema, jsonPtr, parentSchema, keyIndex);
            for (const key of Object.keys(schema)) {
                const sch = schema[key];
                if (Array.isArray(sch)) {
                    if (key == 'items') {
                        for (let i = 0; i < sch.length; i++) {
                            _traverse(sch[i], jsonPtr + '/' + key + '/' + i, rootSchema, schema, '' + i);
                        }
                    }
                }
                else if (key in propsKeywords) {
                    if (sch && typeof sch == 'object') {
                        for (const prop of Object.keys(sch)) {
                            _traverse(sch[prop], jsonPtr + '/' + key + '/' + prop.replace(/~/g, '~0').replace(/\//g, '~1'), rootSchema, schema, prop);
                        }
                    }
                }
                else if (key in keywords) {
                    _traverse(sch, jsonPtr + '/' + key, rootSchema, schema, key);
                }
            }
        }
    }
    _traverse(schema, '', schema);
}
class CoreSchemaRegistry {
    constructor(formats = []) {
        /**
         * Build an AJV instance that will be used to validate schemas.
         */
        this._uriCache = new Map();
        const formatsObj = {};
        for (const format of formats) {
            formatsObj[format.name] = format.formatter;
        }
        this._ajv = ajv({
            removeAdditional: 'all',
            useDefaults: true,
            formats: formatsObj,
            loadSchema: (uri) => this._fetch(uri),
        });
        this._ajv.addMetaSchema(require('ajv/lib/refs/json-schema-draft-04.json'));
    }
    _clean(data, // tslint:disable-line:no-any
        schema, validate, parentDataCache) {
        _visitJsonSchema(schema, (currentSchema, pointer, parentSchema, index) => {
            // If we're at the root, skip.
            if (parentSchema === undefined || index === undefined) {
                return;
            }
            const parsedPointer = _parseJsonPointer(pointer);
            // Every other path fragment is either 'properties', 'items', 'allOf', ...
            const nonPropertyParsedPP = parsedPointer.filter((_, i) => !(i % 2));
            // Skip if it's part of a definitions or too complex for us to analyze.
            if (nonPropertyParsedPP.some(f => f == 'definitions' || f == 'allOf' || f == 'anyOf')) {
                return;
            }
            let maybeParentData = parentDataCache.get(parentSchema);
            if (!maybeParentData) {
                // Every other path fragment is either 'properties' or 'items' in this model.
                const parentDataPointer = parsedPointer.filter((_, i) => i % 2);
                // Find the parentData from the list.
                maybeParentData = data;
                for (const index of parentDataPointer.slice(0, -1)) {
                    if (maybeParentData[index] === undefined) {
                        // tslint:disable-next-line:no-any
                        if (parentSchema.hasOwnProperty('items') || parentSchema['type'] == 'array') {
                            maybeParentData[index] = [];
                        }
                        else {
                            maybeParentData[index] = {};
                        }
                    }
                    maybeParentData = maybeParentData[index];
                }
                parentDataCache.set(parentSchema, maybeParentData);
            }
            if (currentSchema.hasOwnProperty('$ref')) {
                const $ref = currentSchema['$ref'];
                const refHash = $ref.split('#', 2)[1];
                const refUrl = $ref.startsWith('#') ? $ref : $ref.split('#', 1);
                let refVal = validate;
                if (!$ref.startsWith('#')) {
                    // tslint:disable-next-line:no-any
                    refVal = validate.refVal[validate.refs[refUrl[0]]];
                }
                if (refHash) {
                    // tslint:disable-next-line:no-any
                    refVal = refVal.refVal[refVal.refs['#' + refHash]];
                }
                maybeParentData[index] = {};
                this._clean(maybeParentData[index], refVal.schema, refVal, parentDataCache);
                return;
            }
            else if (!maybeParentData.hasOwnProperty(index)) {
                maybeParentData[index] = undefined;
            }
        });
    }
    _fetch(uri) {
        const maybeSchema = this._uriCache.get(uri);
        if (maybeSchema) {
            return Promise.resolve(maybeSchema);
        }
        return new Promise((resolve, reject) => {
            http.get(uri, res => {
                if (!res.statusCode || res.statusCode >= 300) {
                    // Consume the rest of the data to free memory.
                    res.resume();
                    reject(`Request failed. Status Code: ${res.statusCode}`);
                }
                else {
                    res.setEncoding('utf8');
                    let data = '';
                    res.on('data', chunk => {
                        data += chunk;
                    });
                    res.on('end', () => {
                        try {
                            const json = JSON.parse(data);
                            this._uriCache.set(uri, json);
                            resolve(json);
                        }
                        catch (err) {
                            reject(err);
                        }
                    });
                }
            });
        });
    }
    compile(schema) {
        // Supports both synchronous and asynchronous compilation, by trying the synchronous
        // version first, then if refs are missing this will fails.
        // We also add any refs from external fetched schemas so that those will also be used
        // in synchronous (if available).
        let validator;
        try {
            const maybeFnValidate = this._ajv.compile(schema);
            validator = of_1.of(maybeFnValidate);
        }
        catch (e) {
            // Propagate the error.
            if (!(e instanceof ajv.MissingRefError)) {
                throw e;
            }
            validator = new Observable_1.Observable(obs => {
                this._ajv.compileAsync(schema)
                    .then(validate => {
                    obs.next(validate);
                    obs.complete();
                }, err => {
                    obs.error(err);
                });
            });
        }
        return validator
            .pipe(
        // tslint:disable-next-line:no-any
        map_1.map(validate => (data) => {
            const result = validate(data);
            const resultObs = typeof result == 'boolean'
                ? of_1.of(result)
                : fromPromise_1.fromPromise(result);
            return resultObs
                .pipe(map_1.map(result => {
                if (result) {
                    // tslint:disable-next-line:no-any
                    const schemaDataMap = new WeakMap();
                    schemaDataMap.set(schema, data);
                    this._clean(data, schema, validate, schemaDataMap);
                    return { success: true };
                }
                return {
                    success: false,
                    errors: (validate.errors || [])
                        .map((err) => `${err.dataPath} ${err.message}`),
                };
            }));
        }));
    }
    addFormat(format) {
        // tslint:disable-next-line:no-any
        const validate = (data) => {
            const result = format.formatter.validate(data);
            return result instanceof Observable_1.Observable ? result.toPromise() : result;
        };
        this._ajv.addFormat(format.name, {
            async: format.formatter.async,
            validate,
        });
    }
}
exports.CoreSchemaRegistry = CoreSchemaRegistry;
//# sourceMappingURL=data:application/json;base64,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