/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.snapshots.blobstore;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.RateLimiter;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.cluster.metadata.SnapshotId;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.ImmutableBlobContainer;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.BytesStreamInput;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.store.ThreadSafeInputStreamIndexInput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.deletionpolicy.SnapshotIndexCommit;
import org.elasticsearch.index.gateway.RecoveryStatus;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardRepository;
import org.elasticsearch.index.snapshots.IndexShardRestoreFailedException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotFailedException;
import org.elasticsearch.index.snapshots.IndexShardSnapshotStatus;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshot;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardSnapshots;
import org.elasticsearch.index.snapshots.blobstore.RateLimitingInputStream;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.repositories.RepositoryName;

public class BlobStoreIndexShardRepository
extends AbstractComponent
implements IndexShardRepository {
    private BlobStore blobStore;
    private BlobPath basePath;
    private final String repositoryName;
    private ByteSizeValue chunkSize;
    private final IndicesService indicesService;
    private RateLimiter snapshotRateLimiter;
    private RateLimiter restoreRateLimiter;
    private RateLimiterListener rateLimiterListener;
    private RateLimitingInputStream.Listener snapshotThrottleListener;
    private static final String SNAPSHOT_PREFIX = "snapshot-";

    @Inject
    BlobStoreIndexShardRepository(Settings settings, RepositoryName repositoryName, IndicesService indicesService) {
        super(settings);
        this.repositoryName = repositoryName.name();
        this.indicesService = indicesService;
    }

    public void initialize(BlobStore blobStore, BlobPath basePath, ByteSizeValue chunkSize, RateLimiter snapshotRateLimiter, RateLimiter restoreRateLimiter, final RateLimiterListener rateLimiterListener) {
        this.blobStore = blobStore;
        this.basePath = basePath;
        this.chunkSize = chunkSize;
        this.snapshotRateLimiter = snapshotRateLimiter;
        this.restoreRateLimiter = restoreRateLimiter;
        this.rateLimiterListener = rateLimiterListener;
        this.snapshotThrottleListener = new RateLimitingInputStream.Listener(){

            @Override
            public void onPause(long nanos) {
                rateLimiterListener.onSnapshotPause(nanos);
            }
        };
    }

    @Override
    public void snapshot(SnapshotId snapshotId, ShardId shardId, SnapshotIndexCommit snapshotIndexCommit, IndexShardSnapshotStatus snapshotStatus) {
        SnapshotContext snapshotContext = new SnapshotContext(snapshotId, shardId, snapshotStatus);
        snapshotStatus.startTime(System.currentTimeMillis());
        try {
            snapshotContext.snapshot(snapshotIndexCommit);
            snapshotStatus.time(System.currentTimeMillis() - snapshotStatus.startTime());
            snapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.DONE);
        }
        catch (Throwable e) {
            snapshotStatus.time(System.currentTimeMillis() - snapshotStatus.startTime());
            snapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.FAILURE);
            if (e instanceof IndexShardSnapshotFailedException) {
                throw (IndexShardSnapshotFailedException)e;
            }
            throw new IndexShardSnapshotFailedException(shardId, e.getMessage(), e);
        }
    }

    @Override
    public void restore(SnapshotId snapshotId, ShardId shardId, ShardId snapshotShardId, RecoveryStatus recoveryStatus) {
        RestoreContext snapshotContext = new RestoreContext(snapshotId, shardId, snapshotShardId, recoveryStatus);
        try {
            recoveryStatus.index().startTime(System.currentTimeMillis());
            snapshotContext.restore();
            recoveryStatus.index().time(System.currentTimeMillis() - recoveryStatus.index().startTime());
        }
        catch (Throwable e) {
            throw new IndexShardRestoreFailedException(shardId, "failed to restore snapshot [" + snapshotId.getSnapshot() + "]", e);
        }
    }

    public void delete(SnapshotId snapshotId, ShardId shardId) {
        Context context = new Context(snapshotId, shardId, shardId);
        context.delete();
    }

    public String toString() {
        return "BlobStoreIndexShardRepository[[" + this.repositoryName + "], [" + this.blobStore + ']' + ']';
    }

    private String snapshotBlobName(SnapshotId snapshotId) {
        return SNAPSHOT_PREFIX + snapshotId.getSnapshot();
    }

    public static byte[] writeSnapshot(BlobStoreIndexShardSnapshot snapshot) throws IOException {
        XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON).prettyPrint();
        BlobStoreIndexShardSnapshot.toXContent(snapshot, builder, ToXContent.EMPTY_PARAMS);
        return builder.bytes().toBytes();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static BlobStoreIndexShardSnapshot readSnapshot(byte[] data) throws IOException {
        XContentParser parser = XContentFactory.xContent(XContentType.JSON).createParser(data);
        try {
            parser.nextToken();
            BlobStoreIndexShardSnapshot blobStoreIndexShardSnapshot = BlobStoreIndexShardSnapshot.fromXContent(parser);
            return blobStoreIndexShardSnapshot;
        }
        finally {
            parser.close();
        }
    }

    public static interface RateLimiterListener {
        public void onRestorePause(long var1);

        public void onSnapshotPause(long var1);
    }

    private class RestoreContext
    extends Context {
        private final Store store;
        private final RecoveryStatus recoveryStatus;

        public RestoreContext(SnapshotId snapshotId, ShardId shardId, ShardId snapshotShardId, RecoveryStatus recoveryStatus) {
            super(snapshotId, shardId, snapshotShardId);
            this.store = BlobStoreIndexShardRepository.this.indicesService.indexServiceSafe(shardId.getIndex()).shardInjectorSafe(shardId.id()).getInstance(Store.class);
            this.recoveryStatus = recoveryStatus;
        }

        public void restore() {
            BlobStoreIndexShardSnapshot snapshot;
            BlobStoreIndexShardRepository.this.logger.debug("[{}] [{}] restoring to [{}] ...", this.snapshotId, BlobStoreIndexShardRepository.this.repositoryName, this.shardId);
            try {
                snapshot = BlobStoreIndexShardRepository.readSnapshot(this.blobContainer.readBlobFully(BlobStoreIndexShardRepository.this.snapshotBlobName(this.snapshotId)));
            }
            catch (IOException ex) {
                throw new IndexShardRestoreFailedException(this.shardId, "failed to read shard snapshot file", (Throwable)ex);
            }
            this.recoveryStatus.updateStage(RecoveryStatus.Stage.INDEX);
            int numberOfFiles = 0;
            long totalSize = 0L;
            int numberOfReusedFiles = 0;
            long reusedTotalSize = 0L;
            ArrayList<BlobStoreIndexShardSnapshot.FileInfo> filesToRecover = Lists.newArrayList();
            for (BlobStoreIndexShardSnapshot.FileInfo fileInfo : snapshot.indexFiles()) {
                String fileName = fileInfo.physicalName();
                StoreFileMetaData md = null;
                try {
                    md = this.store.metaData(fileName);
                }
                catch (IOException e) {
                    // empty catch block
                }
                ++numberOfFiles;
                if (!fileName.startsWith("segments") && md != null && fileInfo.isSame(md)) {
                    totalSize += md.length();
                    ++numberOfReusedFiles;
                    reusedTotalSize += md.length();
                    if (!BlobStoreIndexShardRepository.this.logger.isTraceEnabled()) continue;
                    BlobStoreIndexShardRepository.this.logger.trace("not_recovering [{}], exists in local store and is same", fileInfo.physicalName());
                    continue;
                }
                totalSize += fileInfo.length();
                filesToRecover.add(fileInfo);
                if (!BlobStoreIndexShardRepository.this.logger.isTraceEnabled()) continue;
                if (md == null) {
                    BlobStoreIndexShardRepository.this.logger.trace("recovering [{}], does not exists in local store", fileInfo.physicalName());
                    continue;
                }
                BlobStoreIndexShardRepository.this.logger.trace("recovering [{}], exists in local store but is different", fileInfo.physicalName());
            }
            this.recoveryStatus.index().files(numberOfFiles, totalSize, numberOfReusedFiles, reusedTotalSize);
            if (filesToRecover.isEmpty()) {
                BlobStoreIndexShardRepository.this.logger.trace("no files to recover, all exists within the local store", new Object[0]);
            }
            if (BlobStoreIndexShardRepository.this.logger.isTraceEnabled()) {
                BlobStoreIndexShardRepository.this.logger.trace("[{}] [{}] recovering_files [{}] with total_size [{}], reusing_files [{}] with reused_size [{}]", this.shardId, this.snapshotId, numberOfFiles, new ByteSizeValue(totalSize), numberOfReusedFiles, new ByteSizeValue(reusedTotalSize));
            }
            CountDownLatch latch = new CountDownLatch(filesToRecover.size());
            CopyOnWriteArrayList<Throwable> failures = new CopyOnWriteArrayList<Throwable>();
            for (BlobStoreIndexShardSnapshot.FileInfo fileToRecover : filesToRecover) {
                BlobStoreIndexShardRepository.this.logger.trace("[{}] [{}] restoring file [{}]", this.shardId, this.snapshotId, fileToRecover.name());
                this.restoreFile(fileToRecover, latch, failures);
            }
            try {
                latch.await();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            if (!failures.isEmpty()) {
                throw new IndexShardRestoreFailedException(this.shardId, "Failed to recover index", (Throwable)failures.get(0));
            }
            long version = -1L;
            try {
                if (Lucene.indexExists(this.store.directory())) {
                    version = Lucene.readSegmentInfos(this.store.directory()).getVersion();
                }
            }
            catch (IOException e) {
                throw new IndexShardRestoreFailedException(this.shardId, "Failed to fetch index version after copying it over", (Throwable)e);
            }
            this.recoveryStatus.index().updateVersion(version);
            try {
                for (String storeFile : this.store.directory().listAll()) {
                    if (snapshot.containPhysicalIndexFile(storeFile)) continue;
                    try {
                        this.store.directory().deleteFile(storeFile);
                    }
                    catch (IOException e) {
                        // empty catch block
                    }
                }
            }
            catch (IOException e) {
                // empty catch block
            }
        }

        private void restoreFile(final BlobStoreIndexShardSnapshot.FileInfo fileInfo, final CountDownLatch latch, final List<Throwable> failures) {
            IndexOutput indexOutput;
            try {
                indexOutput = this.store.createOutputRaw(fileInfo.physicalName());
            }
            catch (IOException e) {
                failures.add(e);
                latch.countDown();
                return;
            }
            String firstFileToRecover = fileInfo.partName(0L);
            final AtomicInteger partIndex = new AtomicInteger();
            this.blobContainer.readBlob(firstFileToRecover, new BlobContainer.ReadBlobListener(){

                @Override
                public synchronized void onPartial(byte[] data, int offset, int size) throws IOException {
                    RestoreContext.this.recoveryStatus.index().addCurrentFilesSize(size);
                    indexOutput.writeBytes(data, offset, size);
                    if (BlobStoreIndexShardRepository.this.restoreRateLimiter != null) {
                        BlobStoreIndexShardRepository.this.rateLimiterListener.onRestorePause(BlobStoreIndexShardRepository.this.restoreRateLimiter.pause((long)size));
                    }
                }

                @Override
                public synchronized void onCompleted() {
                    int part = partIndex.incrementAndGet();
                    if ((long)part < fileInfo.numberOfParts()) {
                        String partName = fileInfo.partName(part);
                        RestoreContext.this.blobContainer.readBlob(partName, this);
                        return;
                    }
                    try {
                        indexOutput.close();
                        if (fileInfo.checksum() != null) {
                            RestoreContext.this.store.writeChecksum(fileInfo.physicalName(), fileInfo.checksum());
                        }
                        RestoreContext.this.store.directory().sync(Collections.singleton(fileInfo.physicalName()));
                    }
                    catch (IOException e) {
                        this.onFailure(e);
                        return;
                    }
                    latch.countDown();
                }

                @Override
                public void onFailure(Throwable t) {
                    failures.add(t);
                    latch.countDown();
                }
            });
        }
    }

    private class SnapshotContext
    extends Context {
        private final Store store;
        private final IndexShardSnapshotStatus snapshotStatus;

        public SnapshotContext(SnapshotId snapshotId, ShardId shardId, IndexShardSnapshotStatus snapshotStatus) {
            super(snapshotId, shardId);
            this.store = BlobStoreIndexShardRepository.this.indicesService.indexServiceSafe(shardId.getIndex()).shardInjectorSafe(shardId.id()).getInstance(Store.class);
            this.snapshotStatus = snapshotStatus;
        }

        public void snapshot(SnapshotIndexCommit snapshotIndexCommit) {
            ImmutableMap<String, BlobMetaData> blobs;
            BlobStoreIndexShardRepository.this.logger.debug("[{}] [{}] snapshot to [{}] ...", this.shardId, this.snapshotId, BlobStoreIndexShardRepository.this.repositoryName);
            try {
                blobs = this.blobContainer.listBlobs();
            }
            catch (IOException e) {
                throw new IndexShardSnapshotFailedException(this.shardId, "failed to list blobs", (Throwable)e);
            }
            long generation = this.findLatestFileNameGeneration(blobs);
            BlobStoreIndexShardSnapshots snapshots = this.buildBlobStoreIndexShardSnapshots(blobs);
            this.snapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.STARTED);
            CountDownLatch indexLatch = new CountDownLatch(snapshotIndexCommit.getFiles().length);
            CopyOnWriteArrayList<Throwable> failures = new CopyOnWriteArrayList<Throwable>();
            ArrayList<BlobStoreIndexShardSnapshot.FileInfo> indexCommitPointFiles = Lists.newArrayList();
            int indexNumberOfFiles = 0;
            long indexTotalFilesSize = 0L;
            for (String fileName : snapshotIndexCommit.getFiles()) {
                StoreFileMetaData md;
                if (this.snapshotStatus.aborted()) {
                    BlobStoreIndexShardRepository.this.logger.debug("[{}] [{}] Aborted on the file [{}], exiting", this.shardId, this.snapshotId, fileName);
                    throw new IndexShardSnapshotFailedException(this.shardId, "Aborted");
                }
                BlobStoreIndexShardRepository.this.logger.trace("[{}] [{}] Processing [{}]", this.shardId, this.snapshotId, fileName);
                try {
                    md = this.store.metaData(fileName);
                }
                catch (IOException e) {
                    throw new IndexShardSnapshotFailedException(this.shardId, "Failed to get store file metadata", (Throwable)e);
                }
                boolean snapshotRequired = false;
                BlobStoreIndexShardSnapshot.FileInfo fileInfo = snapshots.findPhysicalIndexFile(fileName);
                if (fileInfo == null || !fileInfo.isSame(md) || !this.snapshotFileExistsInBlobs(fileInfo, blobs)) {
                    snapshotRequired = true;
                }
                if (snapshotRequired) {
                    ++indexNumberOfFiles;
                    indexTotalFilesSize += md.length();
                    try {
                        BlobStoreIndexShardSnapshot.FileInfo snapshotFileInfo = new BlobStoreIndexShardSnapshot.FileInfo(this.fileNameFromGeneration(++generation), fileName, md.length(), BlobStoreIndexShardRepository.this.chunkSize, md.checksum());
                        indexCommitPointFiles.add(snapshotFileInfo);
                        this.snapshotFile(snapshotFileInfo, indexLatch, failures);
                    }
                    catch (IOException e) {
                        failures.add(e);
                    }
                    continue;
                }
                indexCommitPointFiles.add(fileInfo);
                indexLatch.countDown();
            }
            this.snapshotStatus.files(indexNumberOfFiles, indexTotalFilesSize);
            this.snapshotStatus.indexVersion(snapshotIndexCommit.getGeneration());
            try {
                indexLatch.await();
            }
            catch (InterruptedException e) {
                failures.add(e);
                Thread.currentThread().interrupt();
            }
            if (!failures.isEmpty()) {
                throw new IndexShardSnapshotFailedException(this.shardId, "Failed to perform snapshot (index files)", (Throwable)failures.get(0));
            }
            this.snapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.FINALIZE);
            String commitPointName = BlobStoreIndexShardRepository.this.snapshotBlobName(this.snapshotId);
            BlobStoreIndexShardSnapshot snapshot = new BlobStoreIndexShardSnapshot(this.snapshotId.getSnapshot(), snapshotIndexCommit.getGeneration(), indexCommitPointFiles);
            try {
                byte[] snapshotData = BlobStoreIndexShardRepository.writeSnapshot(snapshot);
                BlobStoreIndexShardRepository.this.logger.trace("[{}] [{}] writing shard snapshot file", this.shardId, this.snapshotId);
                this.blobContainer.writeBlob(commitPointName, new BytesStreamInput(snapshotData, false), snapshotData.length);
            }
            catch (IOException e) {
                throw new IndexShardSnapshotFailedException(this.shardId, "Failed to write commit point", (Throwable)e);
            }
            ArrayList<BlobStoreIndexShardSnapshot> newSnapshotsList = Lists.newArrayList();
            newSnapshotsList.add(snapshot);
            for (BlobStoreIndexShardSnapshot point : snapshots) {
                newSnapshotsList.add(point);
            }
            this.cleanup(newSnapshotsList, blobs);
            this.snapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.DONE);
        }

        private void snapshotFile(BlobStoreIndexShardSnapshot.FileInfo fileInfo, final CountDownLatch latch, final List<Throwable> failures) throws IOException {
            final AtomicLong counter = new AtomicLong(fileInfo.numberOfParts());
            for (long i = 0L; i < fileInfo.numberOfParts(); ++i) {
                IndexInput indexInput = null;
                try {
                    indexInput = this.store.openInputRaw(fileInfo.physicalName(), IOContext.READONCE);
                    indexInput.seek(i * fileInfo.partBytes());
                    ThreadSafeInputStreamIndexInput inputStreamIndexInput = new ThreadSafeInputStreamIndexInput(indexInput, fileInfo.partBytes());
                    final IndexInput fIndexInput = indexInput;
                    long size = inputStreamIndexInput.actualSizeToRead();
                    InputStream inputStream = BlobStoreIndexShardRepository.this.snapshotRateLimiter != null ? new RateLimitingInputStream(inputStreamIndexInput, BlobStoreIndexShardRepository.this.snapshotRateLimiter, BlobStoreIndexShardRepository.this.snapshotThrottleListener) : inputStreamIndexInput;
                    this.blobContainer.writeBlob(fileInfo.partName(i), inputStream, size, new ImmutableBlobContainer.WriterListener(){

                        @Override
                        public void onCompleted() {
                            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{fIndexInput});
                            if (counter.decrementAndGet() == 0L) {
                                latch.countDown();
                            }
                        }

                        @Override
                        public void onFailure(Throwable t) {
                            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{fIndexInput});
                            failures.add(t);
                            if (counter.decrementAndGet() == 0L) {
                                latch.countDown();
                            }
                        }
                    });
                    continue;
                }
                catch (Throwable e) {
                    IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{indexInput});
                    failures.add(e);
                    latch.countDown();
                }
            }
        }

        private boolean snapshotFileExistsInBlobs(BlobStoreIndexShardSnapshot.FileInfo fileInfo, ImmutableMap<String, BlobMetaData> blobs) {
            BlobMetaData blobMetaData = blobs.get(fileInfo.name());
            if (blobMetaData != null) {
                return blobMetaData.length() == fileInfo.length();
            }
            if (blobs.containsKey(fileInfo.partName(0L))) {
                int part = 0;
                long totalSize = 0L;
                while ((blobMetaData = blobs.get(fileInfo.partName(part++))) != null) {
                    totalSize += blobMetaData.length();
                }
                return totalSize == fileInfo.length();
            }
            return false;
        }
    }

    private class Context {
        protected final SnapshotId snapshotId;
        protected final ShardId shardId;
        protected final ImmutableBlobContainer blobContainer;

        public Context(SnapshotId snapshotId, ShardId shardId) {
            this(snapshotId, shardId, shardId);
        }

        public Context(SnapshotId snapshotId, ShardId shardId, ShardId snapshotShardId) {
            this.snapshotId = snapshotId;
            this.shardId = shardId;
            this.blobContainer = BlobStoreIndexShardRepository.this.blobStore.immutableBlobContainer(BlobStoreIndexShardRepository.this.basePath.add("indices").add(snapshotShardId.getIndex()).add(Integer.toString(snapshotShardId.getId())));
        }

        public void delete() {
            ImmutableMap<String, BlobMetaData> blobs;
            try {
                blobs = this.blobContainer.listBlobs();
            }
            catch (IOException e) {
                throw new IndexShardSnapshotException(this.shardId, "Failed to list content of gateway", (Throwable)e);
            }
            BlobStoreIndexShardSnapshots snapshots = this.buildBlobStoreIndexShardSnapshots(blobs);
            String commitPointName = BlobStoreIndexShardRepository.this.snapshotBlobName(this.snapshotId);
            try {
                this.blobContainer.deleteBlob(commitPointName);
            }
            catch (IOException e) {
                BlobStoreIndexShardRepository.this.logger.debug("[{}] [{}] failed to delete shard snapshot file", this.shardId, this.snapshotId);
            }
            ArrayList<BlobStoreIndexShardSnapshot> newSnapshotsList = Lists.newArrayList();
            for (BlobStoreIndexShardSnapshot point : snapshots) {
                if (point.snapshot().equals(this.snapshotId.getSnapshot())) continue;
                newSnapshotsList.add(point);
            }
            this.cleanup(newSnapshotsList, blobs);
        }

        protected void cleanup(List<BlobStoreIndexShardSnapshot> snapshots, ImmutableMap<String, BlobMetaData> blobs) {
            BlobStoreIndexShardSnapshots newSnapshots = new BlobStoreIndexShardSnapshots(snapshots);
            for (String blobName : blobs.keySet()) {
                if (!blobName.startsWith("__") || newSnapshots.findNameFile(BlobStoreIndexShardSnapshot.FileInfo.canonicalName(blobName)) != null) continue;
                try {
                    this.blobContainer.deleteBlob(blobName);
                }
                catch (IOException e) {
                    BlobStoreIndexShardRepository.this.logger.debug("[{}] [{}] error deleting blob [{}] during cleanup", e, this.snapshotId, this.shardId, blobName);
                }
            }
        }

        protected String fileNameFromGeneration(long generation) {
            return "__" + Long.toString(generation, 36);
        }

        protected long findLatestFileNameGeneration(ImmutableMap<String, BlobMetaData> blobs) {
            long generation = -1L;
            for (String name : blobs.keySet()) {
                if (!name.startsWith("__")) continue;
                name = BlobStoreIndexShardSnapshot.FileInfo.canonicalName(name);
                try {
                    long currentGen = Long.parseLong(name.substring(2), 36);
                    if (currentGen <= generation) continue;
                    generation = currentGen;
                }
                catch (NumberFormatException e) {
                    BlobStoreIndexShardRepository.this.logger.warn("file [{}] does not conform to the '__' schema", new Object[0]);
                }
            }
            return generation;
        }

        protected BlobStoreIndexShardSnapshots buildBlobStoreIndexShardSnapshots(ImmutableMap<String, BlobMetaData> blobs) {
            ArrayList<BlobStoreIndexShardSnapshot> snapshots = Lists.newArrayList();
            for (String name : blobs.keySet()) {
                if (!name.startsWith(BlobStoreIndexShardRepository.SNAPSHOT_PREFIX)) continue;
                try {
                    snapshots.add(BlobStoreIndexShardRepository.readSnapshot(this.blobContainer.readBlobFully(name)));
                }
                catch (IOException e) {
                    BlobStoreIndexShardRepository.this.logger.warn("failed to read commit point [{}]", e, name);
                }
            }
            return new BlobStoreIndexShardSnapshots(snapshots);
        }
    }
}

