"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.XhrQueueController = void 0;
const Lang_1 = require("./Lang");
var debounce = Lang_1.ExtLang.debounce;
/**
 * A simple XHR queue controller
 * following the async op -> next pattern
 * Faces enforces for the XHR handling
 */
class XhrQueueController {
    constructor() {
        this.queue = [];
        this.taskRunning = false;
    }
    /**
     * executes or enqueues an element
     * @param runnable the runnable (request) to be enqueued
     * @param timeOut timeout if > 0 which defers the execution
     * until the debounce window for the timeout is closed.
     */
    enqueue(runnable, timeOut = 0) {
        debounce("xhrQueue", () => {
            const requestHandler = this.enrichRunnable(runnable);
            if (!this.taskRunning) {
                this.signalTaskRunning();
                requestHandler.start();
            }
            else {
                this.queue.push(requestHandler);
            }
        }, timeOut);
    }
    /**
     * trigger the next element in the queue
     * to be started!
     */
    next() {
        this.updateTaskRunning();
        const next = this.queue.shift();
        next === null || next === void 0 ? void 0 : next.start();
    }
    /**
     * clears and resets the queue
     */
    clear() {
        this.queue.length = 0;
        this.updateTaskRunning();
    }
    /**
     * true if queue is empty
     */
    get isEmpty() {
        return !this.queue.length;
    }
    /**
     * Enriches the incoming async asyncRunnable
     * with the error and next handling
     * (aka: asyncRunnable is done -> next
     *                   error -> clear queue
     * @param asyncRunnable the async runnable which needs enrichment
     * @private
     */
    enrichRunnable(asyncRunnable) {
        /**
         * we can use the Promise pattern asyncrunnable uses
         * to trigger queue control callbacks of next element
         * and clear the queue (theoretically this
         * would work with any promise)
         */
        return asyncRunnable
            .then(() => this.next())
            .catch(() => this.clear());
    }
    /**
     * alerts the queue that a task is running
     *
     * @private
     */
    signalTaskRunning() {
        this.taskRunning = true;
    }
    /**
     * updates the task running status according to the current queue
     * @private
     */
    updateTaskRunning() {
        this.taskRunning = !this.isEmpty;
    }
}
exports.XhrQueueController = XhrQueueController;
//# sourceMappingURL=XhrQueueController.js.map