/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.nifi.processors.gcp.credentials.service;

import com.google.auth.http.HttpTransportFactory;
import com.google.auth.oauth2.GoogleCredentials;
import org.apache.nifi.annotation.behavior.Restricted;
import org.apache.nifi.annotation.behavior.Restriction;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnEnabled;
import org.apache.nifi.components.ConfigVerificationResult;
import org.apache.nifi.components.ConfigVerificationResult.Outcome;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.RequiredPermission;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.controller.AbstractControllerService;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.controller.VerifiableControllerService;
import org.apache.nifi.gcp.credentials.service.GCPCredentialsService;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processors.gcp.ProxyAwareTransportFactory;
import org.apache.nifi.processors.gcp.credentials.factory.CredentialsFactory;
import org.apache.nifi.proxy.ProxyConfiguration;
import org.apache.nifi.reporting.InitializationException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static org.apache.nifi.processors.gcp.credentials.factory.CredentialPropertyDescriptors.SERVICE_ACCOUNT_JSON;
import static org.apache.nifi.processors.gcp.credentials.factory.CredentialPropertyDescriptors.SERVICE_ACCOUNT_JSON_FILE;
import static org.apache.nifi.processors.gcp.credentials.factory.CredentialPropertyDescriptors.USE_APPLICATION_DEFAULT_CREDENTIALS;
import static org.apache.nifi.processors.gcp.credentials.factory.CredentialPropertyDescriptors.USE_COMPUTE_ENGINE_CREDENTIALS;

/**
 * Implementation of GCPCredentialsService interface
 *
 * @see GCPCredentialsService
 */
@CapabilityDescription("Defines credentials for Google Cloud Platform processors. " +
        "Uses Application Default credentials without configuration. " +
        "Application Default credentials support environmental variable (GOOGLE_APPLICATION_CREDENTIALS) pointing to " +
        "a credential file, the config generated by `gcloud auth application-default login`, AppEngine/Compute Engine" +
        " service accounts, etc.")
@Tags({ "gcp", "credentials","provider" })
@Restricted(
        restrictions = {
                @Restriction(
                        requiredPermission = RequiredPermission.ACCESS_ENVIRONMENT_CREDENTIALS,
                        explanation = "The default configuration can read environment variables and system properties for credentials"
                )
        }
)
public class GCPCredentialsControllerService extends AbstractControllerService implements GCPCredentialsService, VerifiableControllerService {

    private static final List<PropertyDescriptor> properties;

    static {
        final List<PropertyDescriptor> props = new ArrayList<>();
        props.add(USE_APPLICATION_DEFAULT_CREDENTIALS);
        props.add(USE_COMPUTE_ENGINE_CREDENTIALS);
        props.add(SERVICE_ACCOUNT_JSON_FILE);
        props.add(SERVICE_ACCOUNT_JSON);
        props.add(ProxyConfiguration.createProxyConfigPropertyDescriptor(false, ProxyAwareTransportFactory.PROXY_SPECS));
        properties = Collections.unmodifiableList(props);
    }

    private volatile GoogleCredentials googleCredentials;
    protected final CredentialsFactory credentialsProviderFactory = new CredentialsFactory();

    @Override
    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return properties;
    }

    public GoogleCredentials getGoogleCredentials() throws ProcessException {
        return googleCredentials;
    }

    @Override
    protected Collection<ValidationResult> customValidate(final ValidationContext validationContext) {
        final Collection<ValidationResult> results = credentialsProviderFactory.validate(validationContext);
        ProxyConfiguration.validateProxySpec(validationContext, results, ProxyAwareTransportFactory.PROXY_SPECS);
        return results;
    }

    @Override
    public List<ConfigVerificationResult> verify(final ConfigurationContext context, final ComponentLog verificationLogger, final Map<String, String> variables) {
        ConfigVerificationResult result;
        try {
            final GoogleCredentials credentials = getGoogleCredentials(context);
            result = new ConfigVerificationResult.Builder()
                    .verificationStepName("Provide Google Credentials")
                    .outcome(Outcome.SUCCESSFUL)
                    .explanation(String.format("Successfully provided [%s] as Google Credentials", credentials.getClass().getSimpleName()))
                    .build();
        } catch (final IOException e) {
            result = new ConfigVerificationResult.Builder()
                    .verificationStepName("Provide Google Credentials")
                    .outcome(Outcome.FAILED)
                    .explanation(String.format("Failed to provide Google Credentials: " + e.getMessage()))
                    .build();
        }
        return Collections.singletonList(result);
    }

    @OnEnabled
    public void onConfigured(final ConfigurationContext context) throws InitializationException {
        try {
            googleCredentials = getGoogleCredentials(context);
        } catch (final IOException e) {
            throw new InitializationException(e);
        }
    }

    private GoogleCredentials getGoogleCredentials(final ConfigurationContext context) throws IOException {
        final ProxyConfiguration proxyConfiguration = ProxyConfiguration.getConfiguration(context);
        final HttpTransportFactory transportFactory = new ProxyAwareTransportFactory(proxyConfiguration);
        return credentialsProviderFactory.getGoogleCredentials(context.getProperties(), transportFactory);
    }

    @Override
    public String toString() {
        return "GCPCredentialsControllerService[id=" + getIdentifier() + "]";
    }
}