/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.rocketmq.eventbridge.tools.transform;

import com.google.common.collect.Lists;
import java.util.List;
import org.apache.rocketmq.eventbridge.exception.EventBridgeException;

public interface Transform {

    String DEFAULT_VALUE_NAME = "SYS_DEFAULT";

    /**
     * transform the inputData to outputData
     *
     * @param inputData
     * @return
     * @throws EventBridgeException
     */
    Data process(Data inputData) throws EventBridgeException;

    /**
     * extract variable from input data
     *
     * @param inputData
     * @param extract
     * @return
     * @throws EventBridgeException
     */
    default List<Variable> extract(Data inputData, Extract extract) throws EventBridgeException {
        if (inputData == null) {
            return Lists.newArrayListWithCapacity(0);
        }
        return extract.parse(inputData);
    }

    /**
     * Follow the template to format the variableList
     *
     * @param variableList
     * @param template
     * @return
     * @throws EventBridgeException
     */
    default Data format(List<Variable> variableList, Template template) throws EventBridgeException {
        if (template == null) {
            return new StringData();
        }
        return template.parse(variableList);
    }
}
