<?php
/**
 * @version     1.0.0

 * @date        2023-05-29
 * @copyright   2023 Tim Leibacher
 * @license     https://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPLv2
 */
defined("_JEXEC") or die("Restricted Access");

use Joomla\CMS\Uri\Uri;

/**
 * @since     1.0.0
 */
class PayageModelTaler_Leibacher extends PayageModelAccount
{
	/**
	 * Default response timeout (in seconds).
	 */
	public const DEFAULT_TIMEOUT = 10;
	/**
	 * Default connect timeout (in seconds).
	 */
	public const DEFAULT_CONNECT_TIMEOUT = 5;

	/**
	 * HTTP Methods
	 */
	public const HTTP_GET = "GET";
	public const HTTP_POST = "POST";

	var $app = null;

	var $common_data = null;

	var $specific_data = null;

	function __construct()
	{
		parent::__construct();
		$xml_array = JInstaller::parseXMLInstallFile(JPATH_ADMINISTRATOR . '/components/com_payage/payage_taler_leibacher.xml');
		$this->gw_addon_version = $xml_array['version'];
		LAPG_trace::trace("Taler: v" . $this->gw_addon_version);
	}

	// -------------------------------------------------------------------------------
	// Initialise data items specific to this gateway
	// - the account class initialises the common data items
	// 
	public function initData($gateway_info)
	{
		parent::initData($gateway_info);
		$this->specific_data = new stdClass;
		$this->specific_data->backend_url = '';
		$this->specific_data->backend_key = '';
		$this->specific_data->currency = '';
		$this->specific_data->refund_time = '';
		$this->specific_data->tid_optional = true;

	}

	// -------------------------------------------------------------------------------
	// Get the post data specific to this gateway
	// - the account class gets the common data items
	// 
	public function getPostData()
	{
		parent::getPostData();
		$this->specific_data = new stdClass;
		$jinput = JFactory::getApplication()->input;
		$this->specific_data->test_mode = $jinput->get('test_mode', '0', 'STRING');
		$this->specific_data->backend_url = $jinput->get('backend_url', '', 'STRING');
		$this->specific_data->backend_key = $jinput->get('backend_key', '', 'STRING');
		$this->specific_data->refund_time = $jinput->get('refund_time', '', 'STRING');
		$currency = $this->create_payment(self::HTTP_GET, $this->specific_data->backend_url . '/config', null)['currency'];
		$this->specific_data->tid_optional = true;

		// To allow for custom currencies which don't use 3 Letter abbreviation
		if (strlen($currency) != 3)
		{
			$this->specific_data->currency = $this->convertCurrency($currency, "currency2Abr");
		}
		else
		{
			$this->specific_data->currency = $currency;
		}

		$this->common_data->account_currency = $this->specific_data->currency;

		$languages = PayageHelper::get_site_languages();

		foreach ($languages as $tag => $name)
		{
			$this->translations[$tag]['account_language'] = $jinput->get($tag . '_account_language', '', 'string');
		}

		return $this->specific_data;
	}

	function convertCurrency($currency, $conversionType)
	{
		defined('JPATH_PAYMENT') or define('JPATH_PAYMENT', JPATH_SITE . '/administrator/components/com_payage');
		$file = JPATH_PAYMENT . '/currencies.csv';

		if (JFile::exists($file))
		{
			// Open the CSV file
			$file = fopen($file, 'r');

			while (($row = fgetcsv($file)) !== false)
			{
				if ($conversionType === 'currency2Abr' && $row[0] == $currency)
				{
					$value = $row[1];
					fclose($file);
					LAPG_trace::trace($value);

					return $value;
				}
				elseif ($conversionType === 'abr2Currency' && $row[1] == $currency)
				{
					$value = $row[0];
					fclose($file);

					return $value;
				}
			}
		}

		$errors[] = JText::_('COM_PAYAGE_INVALID') . ' ' . JText::_('COM_PAYAGE_TALER_CURRENCY_NOT_FOUND');
		$this->app->enqueueMessage(implode('<br />', $errors), 'error');

		return 'ERR';
	}

	// -------------------------------------------------------------------------------
	// Validate the account details
	// - the account class checks the common data items
	// 
	public function check_post_data()
	{
		$errors = array();
		$ok = parent::check_post_data();    // Check the common data

		if (!str_starts_with($this->specific_data->backend_url, "http"))
		{
			$errors[] = JText::_('COM_PAYAGE_INVALID') . ' ' . JText::_('COM_PAYAGE_TALER_BACKEND_URL');
		}

		if (!str_starts_with($this->specific_data->backend_key, "secret-token:"))
		{
			$errors[] = JText::_('COM_PAYAGE_INVALID') . ' ' . JText::_('COM_PAYAGE_TALER_KEY');
		}

		if (!ctype_digit(trim($this->specific_data->refund_time)))
		{
			$errors[] = JText::_('COM_PAYAGE_INVALID') . ' ' . JText::_('COM_PAYAGE_TALER_REFUND_TIME');
		}

		if (!empty($errors))
		{
			$this->app->enqueueMessage(implode('<br />', $errors), 'error');
			$ok = false;
		}

		return $ok;
	}

	// -------------------------------------------------------------------------------
	// handle an incoming request from the payment gateway
	// - we assume this is a genuine request because the front end found the account and payment records
	// our model instance already has $this->common_data and $this->specific_data
	// 
	public function Gateway_handle_request($payment_model)
	{

		$jinput = JFactory::getApplication()->input;
		$task = $jinput->get('task', '', 'STRING');
		$this->payment_model = $payment_model;
		$this->payment_data = $payment_model->data;

		switch ($task)
		{
			case 'create':                        // Someone clicked a Payage Taler payment button
				$action = $this->create();

		return $action;

			case 'return':
			case 'update':
			case 'refund':          // Used for the webhook
				return $this->handle_return($task);

			case 'cancel':
				return LAPG_CALLBACK_CANCEL;

			default:
				LAPG_trace::trace("Taler handle_request() unknown task $task");

		return LAPG_CALLBACK_BAD;            // Should never happen
		}

	}

	// -------------------------------------------------------------------------------
	// Create a payment in Taler
	// - we redirect here from Taler payment buttons
	// 
	private function create()
	{
		LAPG_trace::trace('Taler create() for Payage transaction id: ' . $this->payment_data->pg_transaction_id);

		$this->payment_data->account_id = $this->common_data->id;	// Save the account_id now in case of errors
		$this->payment_data->gw_addon_version = $this->gw_addon_version;
		$stored = $this->payment_model->store();

		if (!function_exists('curl_version'))
		{
			LAPG_trace::trace("CURL not installed - cannot use Taler");
			$this->payment_data->pg_status_code = LAPG_STATUS_FAILED;
			$this->payment_data->pg_status_text = JText::_('COM_PAYAGE_CURL_NOT_INSTALLED');
			$stored = $this->payment_model->store();

			return LAPG_CALLBACK_USER;			// Return to the calling application
		}

		// Set up the payment in the gateway

		$redirectUrl = htmlentities(JURI::root() . 'index.php?option=com_payage&task=return&aid=' . $this->common_data->id . '&tid=' . $this->payment_data->pg_transaction_id . '&tmpl=component&format=raw');
		$customer_fee = parent::calculate_gateway_fee($this->common_data, $this->payment_data->gross_amount);
		$total_amount = $this->payment_data->gross_amount + $customer_fee;
		$total_amount = number_format($total_amount, 2, '.', '');	// We currently only support currencies that use two decimal places
		$currency = $this->convertCurrency($this->common_data->account_currency, "abr2Currency");

		if ($this->specific_data->refund_time < 10)
		{
			$refund_time = 0;
		}
		else
		{
			$refund_time = intval($this->specific_data->refund_time) * 1000;
		}

		try
		{
			$data = array(
				'refund_delay' => array(
					'd_us' => (int) $refund_time
				),
				'order' => array(
					'amount' => $currency . ':' . $total_amount,
					'summary' => Uri::getInstance()->getHost() . ' - ' . $this->payment_data->item_name,
					'fulfillment_url' => $redirectUrl
				)
			);

			// Redirect the browser to the Taler gateway

			$response_data = $this->create_payment(self::HTTP_POST, $this->specific_data->backend_url . '/private/orders', json_encode($data));

			$order_id = $response_data['order_id'];
			$token = $response_data['token'];

			// Save the payment details so far

			$this->payment_data->gw_transaction_id = $order_id;
			$stored = $this->payment_model->store();

			$url = $this->specific_data->backend_url . '/orders/' . $order_id . '?token=' . $token;
			LAPG_trace::trace("Taler redirecting to: $url");
			$app = JFactory::getApplication();
			$app->redirect($url);

			return LAPG_CALLBACK_NONE;			// We are at the gateway - we will go back to the calling application later
		}
		catch (Exception $e)
		{
			$html = $e->getMessage() . '<br>' . JText::_('COM_PAYAGE_GATEWAY_TEST_RESPONSE_NOT_OK');
			$this->app->enqueueMessage($html, 'error');

			return;
		}
	}

	private function create_payment($httpMethod, $url, $httpBody)
	{
		$curl = curl_init();

		$headers = array(
			'Authorization: Bearer ' . $this->specific_data->backend_key,
			'Content-Type: application/json'
		);

		curl_setopt($curl, CURLOPT_URL, $url);

		curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, self::DEFAULT_CONNECT_TIMEOUT);
		curl_setopt($curl, CURLOPT_TIMEOUT, self::DEFAULT_TIMEOUT);

		switch ($httpMethod)
		{
			case self::HTTP_POST:
				curl_setopt($curl, CURLOPT_POST, true);
				curl_setopt($curl, CURLOPT_POSTFIELDS, $httpBody);
				break;
			case self::HTTP_GET:
				break;
			default:
				throw new InvalidArgumentException("Invalid http method: " . $httpMethod);
		}

		$response = curl_exec($curl);
		$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

		if ($response === false)
		{
			$curlErrorMessage = "Curl error: " . curl_error($curl);
			curl_close($curl);
			throw new ErrorException($curlErrorMessage);
		}

		curl_close($curl);
		$responseData = json_decode($response, true);
		$responseData['httpCode'] = $httpCode;

		return $responseData;
	}

	// -------------------------------------------------------------------------------
	// handle a return from Taler
	// - this is when Taler re-directs back to the client site after a payment
	// 
	private function handle_return($task)
	{
		LAPG_trace::trace("Taler handle_return($task) for Payage transaction id: " . $this->payment_data->pg_transaction_id);

		// Used for the Webhook to get the correct Order
		if ($task == 'refund')
		{
			if ($_SERVER['REQUEST_METHOD'] === 'POST')
			{
				if ($_SERVER['CONTENT_TYPE'] === 'application/json')
				{
					$json_data = file_get_contents('php://input');
					$body = json_decode($json_data, true);

					if ($body !== null)
					{
						$orderId = $body['order_id'];

						LAPG_trace::trace('Order with Id: ' . $orderId . ' is being refunded');

						$this->payment_data = $this->payment_model->getOne($orderId, 'gw_transaction_id');
					}
					else
					{
						LAPG_trace::trace("Missing Order ID in JSON body");

						return LAPG_STATUS_FAILED;
					}
				}
				else
				{
					LAPG_trace::trace("refunds triggered but without a JSON header");

					return LAPG_STATUS_FAILED;
				}
			}
			else
			{
				LAPG_trace::trace("refund triggered but wihtout a POST request");

				return LAPG_STATUS_FAILED;
			}
		}

		// Get the payment status from the gateway
		$response_data = $this->create_payment(self::HTTP_GET, $this->specific_data->backend_url . '/private/orders/' . $this->payment_data->gw_transaction_id, null);
		$status = $response_data['order_status'];

		switch ($status)
		{
			case 'paid':
				$new_status_code = LAPG_STATUS_SUCCESS;
				break;
			case 'expired':
			case 'canceled':
				$new_status_code = LAPG_STATUS_CANCELLED;
				break;
			case 'failed':
				$new_status_code = LAPG_STATUS_FAILED;
				break;
			case 'pending':
			case 'unpaid':
				$new_status_code = LAPG_STATUS_PENDING;
				break;
			default:
				$new_status_code = LAPG_STATUS_FAILED;
		}

		// Check for refunds

		$refund_description = '';

		if ($response_data['refunded'])
		{
			$amount_refunded = trim(explode(':', $response_data['refund_amount'])[1]);
			$amount_total = trim(explode(':', $response_data['contract_terms']['amount'])[1]);
			$refund_details = $response_data['refund_details'];
			$refund_description = 'Taler: ' . JText::_('COM_PAYAGE_REFUNDED') . ' ' . $amount_refunded . ' details: ' . end($response_data['refund_details'])['reason'];
			LAPG_trace::trace("Taler: $refund_description");
			$amount_remaining = (float) $amount_total - (float) $amount_refunded;

			if ($amount_remaining == 0.0)
			{
				$new_status_code = LAPG_STATUS_REFUNDED;
				$status = 'refunded';
			}
		}

		// Multiple calls

		$result = $this->payment_model->ladb_lockTable('#__payage_payments');

		if ($result === true)
		{
			LAPG_trace::trace('Locked the payment table ok');
		}
		else
		{
			LAPG_trace::trace('Failed to lock the payment table: ' . $this->payment_model->ladb_error_text);
		}

		$this->payment_data = $this->payment_model->getOne($this->payment_data->id);
		$this->payment_data->account_id = $this->common_data->id;
		$this->payment_data->gw_addon_version = $this->gw_addon_version;
		$this->payment_data->pg_status_code = $new_status_code;
		$this->payment_data->pg_status_text = $status;

		// Update the history

		if ($refund_description != '')
		{
			$this->payment_model->add_history_entry($refund_description);
		}
		else
		{
			$status_description = PayageHelper::getPaymentDescription($this->payment_data->pg_status_code);
			$this->payment_model->add_history_entry("Taler: $status_description");
		}

		// Update the payment record
		// we store some details in the root of gw_transaction_details so they are easily visible
		// we also store each update separately in case something goes wrong and we need to see the full history

		$stored = $this->payment_model->store();
		$this->payment_model->ladb_unlock();

		// For a 'return' task, we redirect to the calling application

		if ($task == 'return')
		{
			LAPG_trace::trace("Taler $task returning LAPG_CALLBACK_USER");

			return LAPG_CALLBACK_USER;
		}

		// It's an 'update'
		// if this was a refund, we must update the application

		if ($new_status_code == LAPG_STATUS_REFUNDED)
		{
			LAPG_trace::trace("Taler $task [refund], returning LAPG_CALLBACK_UPDATE");

			return LAPG_CALLBACK_UPDATE;
		}

		LAPG_trace::trace("Taler $task, returning LAPG_CALLBACK_NONE");

		return LAPG_CALLBACK_NONE;
	}


	// -------------------------------------------------------------------------------
	// Verify the amount, currency and recipient of a payment
	// - set the payment status code and text accordingly
	// 
	private function check_payment($customer_fee, $gross_received, $currency_received, $receiver_email)
	{
		$expected_gross = $this->payment_data->gross_amount;
		$expected_total = $expected_gross + $customer_fee;
		$str_expected_total = number_format($expected_total, 2);
		$str_actual_total = number_format($gross_received, 2);
		LAPG_trace::trace("check_payment() gross_amount = $expected_gross, customer_fee = $customer_fee, expected_payment_amount = $expected_total, gross_received = $gross_received");

		if (!isset($this->specific_data->auto_tax))
		{
								// New parameter may not have been saved yet
			$this->specific_data->auto_tax = 0;                                    // default to no
		}

		if (($this->specific_data->auto_tax == 0) && ($str_expected_total != $str_actual_total))
		{
			$this->payment_data->pg_status_code = LAPG_STATUS_FAILED;
			$this->payment_data->pg_status_text = JText::sprintf("COM_PAYAGE_MISMATCH_AMOUNT", $str_expected_total, $str_actual_total);
			LAPG_trace::trace($this->payment_data->pg_status_text);
		}

		if (($this->specific_data->auto_tax == 1) && ($str_actual_total < $str_expected_total))
		{
			$this->payment_data->pg_status_code = LAPG_STATUS_FAILED;
			$this->payment_data->pg_status_text = JText::sprintf("COM_PAYAGE_MISMATCH_AMOUNT", $str_expected_total, $str_actual_total);
			LAPG_trace::trace($this->payment_data->pg_status_text);
		}

		if ($this->common_data->account_currency != $currency_received)
		{
			$this->payment_data->pg_status_code = LAPG_STATUS_FAILED;
			$this->payment_data->pg_status_text = JText::sprintf("COM_PAYAGE_MISMATCH_CURRENCY", $this->common_data->account_currency, $currency_received);
			LAPG_trace::trace($this->payment_data->pg_status_text);
		}

		if (strcasecmp($this->common_data->account_email, $receiver_email) == 0)
		{
			return;
		}

		if (isset($this->specific_data->account_primary_email) && strcasecmp($this->specific_data->account_primary_email, $receiver_email) == 0)
		{
			return;
		}

		$this->payment_data->pg_status_code = LAPG_STATUS_FAILED;

		if (isset($this->specific_data->account_primary_email))
		{
			$email_address = $this->common_data->account_email . ' OR ' . $this->specific_data->account_primary_email;
		}
		else
		{
			$email_address = $this->common_data->account_email;
		}

		$this->payment_data->pg_status_text = JText::sprintf("COM_PAYAGE_MISMATCH_RECIPIENT", $email_address, $receiver_email);
		LAPG_trace::trace($this->payment_data->pg_status_text);
	}


	// -------------------------------------------------------------------------------
	// Build a Buy Now button
	// 
	public function Gateway_make_button($payment_data, $call_array, $app_fee)
	{

		$process_url = JURI::root() . 'index.php?option=com_payage&task=create&aid=' . $this->common_data->id . '&tid=' . $payment_data->pg_transaction_id . '&tmpl=component&format=raw';
		$button_url = JURI::base(true) . '/' . $this->common_data->button_image;
		$html = '<form class="pb-form pb-taler" action="' . $process_url . '" method="post" >';
		$html .= '<input type="image" src="' . $button_url . '" alt="Taler" title="' . $this->common_data->button_title . '" ' . $call_array['button_extra'] . '>';
		$html .= "</form>";

		return $html;
	}

	// -------------------------------------------------------------------------------
	// Test a Taler API call
	// 
	public function Gateway_test()
	{
		LAPG_trace::trace("Taler gateway test with key " . $this->specific_data->backend_key);
		$curl_info = curl_version();
		$curl_version = $curl_info['version'];
		LAPG_trace::trace("PHP version: " . PHP_VERSION);
		LAPG_trace::trace("CURL version: $curl_version");
		LAPG_trace::trace("Openssl version: " . OPENSSL_VERSION_TEXT);

		if (!function_exists('curl_version'))
		{
			$this->app->enqueueMessage("FAIL: CURL is not installed", 'error');

			return;
		}

		try                                                    // Try to connect to Taler
		{
			$response_data = $this->create_payment(self::HTTP_GET, $this->specific_data->backend_url . '/private', json_encode($data));
			$httpCode = $response_data['httpCode'];

			switch ($httpCode)
			{
				case 200:
				case 201:
					$msg = JText::_('COM_PAYAGE_GATEWAY_TEST_PASSED');
					$this->app->enqueueMessage($msg, 'message');
					break;
				case 401:
					$msg = JText::_('COM_PAYAGE_GATEWAY_TEST_BAD_KEY');
					$this->app->enqueueMessage($msg, 'error');
					break;
                case 404:
                    $msg = JText::_('COM_PAYAGE_GATEWAY_TEST_BAD_URL');
                    $this->app->enqueueMessage($msg, 'error');
                    break;
                case str_starts_with($httpCode, '5'):
                    $msg = JText::_('COM_PAYAGE_GATEWAY_TEST_SERVER_ERROR');
                    $this->app->enqueueMessage($msg, 'error');
                    break;
			}
		}
		catch (Exception $e)
		{
			$html = $e->getMessage() . '<br>' . JText::_('COM_PAYAGE_GATEWAY_TEST_RESPONSE_NOT_OK');
			$this->app->enqueueMessage($html, 'error');

			return;
		}

		LAPG_trace::trace("Taler Gateway_test done");
	}
}

