// DO NOT EDIT. Generated by code.google.com/p/rsc/cmd/bundle
// bundle -p main -x x86_ rsc.io/x86/x86asm

/* decode.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Table-driven decoding of x86 instructions.

package main

import (
	"bytes"
	"encoding/binary"
	"errors"
	"fmt"
	"runtime"
	"strings"
)

// Set trace to true to cause the decoder to print the PC sequence
// of the executed instruction codes. This is typically only useful
// when you are running a test of a single input case.
const x86_trace = false

// A decodeOp is a single instruction in the decoder bytecode program.
//
// The decodeOps correspond to consuming and conditionally branching
// on input bytes, consuming additional fields, and then interpreting
// consumed data as instruction arguments. The names of the xRead and xArg
// operations are taken from the Intel manual conventions, for example
// Volume 2, Section 3.1.1, page 487 of
// http://www.intel.com/content/dam/www/public/us/en/documents/manuals/64-ia-32-architectures-software-developer-manual-325462.pdf
//
// The actual decoding program is generated by ../x86map.
//
// TODO(rsc): We may be able to merge various of the memory operands
// since we don't care about, say, the distinction between m80dec and m80bcd.
// Similarly, mm and mm1 have identical meaning, as do xmm and xmm1.

type x86_decodeOp uint16

const (
	x86_xFail  x86_decodeOp = iota // invalid instruction (return)
	x86_xMatch                     // completed match
	x86_xJump                      // jump to pc

	x86_xCondByte     // switch on instruction byte value
	x86_xCondSlashR   // read and switch on instruction /r value
	x86_xCondPrefix   // switch on presence of instruction prefix
	x86_xCondIs64     // switch on 64-bit processor mode
	x86_xCondDataSize // switch on operand size
	x86_xCondAddrSize // switch on address size
	x86_xCondIsMem    // switch on memory vs register argument

	x86_xSetOp // set instruction opcode

	x86_xReadSlashR // read /r
	x86_xReadIb     // read ib
	x86_xReadIw     // read iw
	x86_xReadId     // read id
	x86_xReadIo     // read io
	x86_xReadCb     // read cb
	x86_xReadCw     // read cw
	x86_xReadCd     // read cd
	x86_xReadCp     // read cp
	x86_xReadCm     // read cm

	x86_xArg1            // arg 1
	x86_xArg3            // arg 3
	x86_xArgAL           // arg AL
	x86_xArgAX           // arg AX
	x86_xArgCL           // arg CL
	x86_xArgCR0dashCR7   // arg CR0-CR7
	x86_xArgCS           // arg CS
	x86_xArgDR0dashDR7   // arg DR0-DR7
	x86_xArgDS           // arg DS
	x86_xArgDX           // arg DX
	x86_xArgEAX          // arg EAX
	x86_xArgEDX          // arg EDX
	x86_xArgES           // arg ES
	x86_xArgFS           // arg FS
	x86_xArgGS           // arg GS
	x86_xArgImm16        // arg imm16
	x86_xArgImm32        // arg imm32
	x86_xArgImm64        // arg imm64
	x86_xArgImm8         // arg imm8
	x86_xArgImm8u        // arg imm8 but record as unsigned
	x86_xArgImm16u       // arg imm8 but record as unsigned
	x86_xArgM            // arg m
	x86_xArgM128         // arg m128
	x86_xArgM1428byte    // arg m14/28byte
	x86_xArgM16          // arg m16
	x86_xArgM16and16     // arg m16&16
	x86_xArgM16and32     // arg m16&32
	x86_xArgM16and64     // arg m16&64
	x86_xArgM16colon16   // arg m16:16
	x86_xArgM16colon32   // arg m16:32
	x86_xArgM16colon64   // arg m16:64
	x86_xArgM16int       // arg m16int
	x86_xArgM2byte       // arg m2byte
	x86_xArgM32          // arg m32
	x86_xArgM32and32     // arg m32&32
	x86_xArgM32fp        // arg m32fp
	x86_xArgM32int       // arg m32int
	x86_xArgM512byte     // arg m512byte
	x86_xArgM64          // arg m64
	x86_xArgM64fp        // arg m64fp
	x86_xArgM64int       // arg m64int
	x86_xArgM8           // arg m8
	x86_xArgM80bcd       // arg m80bcd
	x86_xArgM80dec       // arg m80dec
	x86_xArgM80fp        // arg m80fp
	x86_xArgM94108byte   // arg m94/108byte
	x86_xArgMm           // arg mm
	x86_xArgMm1          // arg mm1
	x86_xArgMm2          // arg mm2
	x86_xArgMm2M64       // arg mm2/m64
	x86_xArgMmM32        // arg mm/m32
	x86_xArgMmM64        // arg mm/m64
	x86_xArgMem          // arg mem
	x86_xArgMoffs16      // arg moffs16
	x86_xArgMoffs32      // arg moffs32
	x86_xArgMoffs64      // arg moffs64
	x86_xArgMoffs8       // arg moffs8
	x86_xArgPtr16colon16 // arg ptr16:16
	x86_xArgPtr16colon32 // arg ptr16:32
	x86_xArgR16          // arg r16
	x86_xArgR16op        // arg r16 with +rw in opcode
	x86_xArgR32          // arg r32
	x86_xArgR32M16       // arg r32/m16
	x86_xArgR32M8        // arg r32/m8
	x86_xArgR32op        // arg r32 with +rd in opcode
	x86_xArgR64          // arg r64
	x86_xArgR64M16       // arg r64/m16
	x86_xArgR64op        // arg r64 with +rd in opcode
	x86_xArgR8           // arg r8
	x86_xArgR8op         // arg r8 with +rb in opcode
	x86_xArgRAX          // arg RAX
	x86_xArgRDX          // arg RDX
	x86_xArgRM           // arg r/m
	x86_xArgRM16         // arg r/m16
	x86_xArgRM32         // arg r/m32
	x86_xArgRM64         // arg r/m64
	x86_xArgRM8          // arg r/m8
	x86_xArgReg          // arg reg
	x86_xArgRegM16       // arg reg/m16
	x86_xArgRegM32       // arg reg/m32
	x86_xArgRegM8        // arg reg/m8
	x86_xArgRel16        // arg rel16
	x86_xArgRel32        // arg rel32
	x86_xArgRel8         // arg rel8
	x86_xArgSS           // arg SS
	x86_xArgST           // arg ST, aka ST(0)
	x86_xArgSTi          // arg ST(i) with +i in opcode
	x86_xArgSreg         // arg Sreg
	x86_xArgTR0dashTR7   // arg TR0-TR7
	x86_xArgXmm          // arg xmm
	x86_xArgXMM0         // arg <XMM0>
	x86_xArgXmm1         // arg xmm1
	x86_xArgXmm2         // arg xmm2
	x86_xArgXmm2M128     // arg xmm2/m128
	x86_xArgXmm2M16      // arg xmm2/m16
	x86_xArgXmm2M32      // arg xmm2/m32
	x86_xArgXmm2M64      // arg xmm2/m64
	x86_xArgXmmM128      // arg xmm/m128
	x86_xArgXmmM32       // arg xmm/m32
	x86_xArgXmmM64       // arg xmm/m64
	x86_xArgRmf16        // arg r/m16 but force mod=3
	x86_xArgRmf32        // arg r/m32 but force mod=3
	x86_xArgRmf64        // arg r/m64 but force mod=3
)

// instPrefix returns an Inst describing just one prefix byte.
// It is only used if there is a prefix followed by an unintelligible
// or invalid instruction byte sequence.
func x86_instPrefix(b byte, mode int) (x86_Inst, error) {
	// When tracing it is useful to see what called instPrefix to report an error.
	if x86_trace {
		_, file, line, _ := runtime.Caller(1)
		fmt.Printf("%s:%d\n", file, line)
	}
	p := x86_Prefix(b)
	switch p {
	case x86_PrefixDataSize:
		if mode == 16 {
			p = x86_PrefixData32
		} else {
			p = x86_PrefixData16
		}
	case x86_PrefixAddrSize:
		if mode == 32 {
			p = x86_PrefixAddr16
		} else {
			p = x86_PrefixAddr32
		}
	}
	// Note: using composite literal with Prefix key confuses 'bundle' tool.
	inst := x86_Inst{Len: 1}
	inst.Prefix = x86_Prefixes{p}
	return inst, nil
}

// truncated reports a truncated instruction.
// For now we use instPrefix but perhaps later we will return
// a specific error here.
func x86_truncated(src []byte, mode int) (x86_Inst, error) {
	//	return Inst{}, len(src), ErrTruncated
	return x86_instPrefix(src[0], mode) // too long
}

// These are the errors returned by Decode.
var (
	x86_ErrInvalidMode  = errors.New("invalid x86 mode in Decode")
	x86_ErrTruncated    = errors.New("truncated instruction")
	x86_ErrUnrecognized = errors.New("unrecognized instruction")
)

// decoderCover records coverage information for which parts
// of the byte code have been executed.
// TODO(rsc): This is for testing. Only use this if a flag is given.
var x86_decoderCover []bool

// Decode decodes the leading bytes in src as a single instruction.
// The mode arguments specifies the assumed processor mode:
// 16, 32, or 64 for 16-, 32-, and 64-bit execution modes.
func x86_Decode(src []byte, mode int) (inst x86_Inst, err error) {
	return x86_decode1(src, mode, false)
}

// decode1 is the implementation of Decode but takes an extra
// gnuCompat flag to cause it to change its behavior to mimic
// bugs (or at least unique features) of GNU libopcodes as used
// by objdump. We don't believe that logic is the right thing to do
// in general, but when testing against libopcodes it simplifies the
// comparison if we adjust a few small pieces of logic.
// The affected logic is in the conditional branch for "mandatory" prefixes,
// case xCondPrefix.
func x86_decode1(src []byte, mode int, gnuCompat bool) (x86_Inst, error) {
	switch mode {
	case 16, 32, 64:
		// ok
		// TODO(rsc): 64-bit mode not tested, probably not working.
	default:
		return x86_Inst{}, x86_ErrInvalidMode
	}

	// Maximum instruction size is 15 bytes.
	// If we need to read more, return 'truncated instruction.
	if len(src) > 15 {
		src = src[:15]
	}

	var (
		// prefix decoding information
		pos           = 0        // position reading src
		nprefix       = 0        // number of prefixes
		lockIndex     = -1       // index of LOCK prefix in src and inst.Prefix
		repIndex      = -1       // index of REP/REPN prefix in src and inst.Prefix
		segIndex      = -1       // index of Group 2 prefix in src and inst.Prefix
		dataSizeIndex = -1       // index of Group 3 prefix in src and inst.Prefix
		addrSizeIndex = -1       // index of Group 4 prefix in src and inst.Prefix
		rex           x86_Prefix // rex byte if present (or 0)
		rexUsed       x86_Prefix // bits used in rex byte
		rexIndex      = -1       // index of rex byte

		addrMode = mode // address mode (width in bits)
		dataMode = mode // operand mode (width in bits)

		// decoded ModR/M fields
		haveModrm bool
		modrm     int
		mod       int
		regop     int
		rm        int

		// if ModR/M is memory reference, Mem form
		mem     x86_Mem
		haveMem bool

		// decoded SIB fields
		haveSIB bool
		sib     int
		scale   int
		index   int
		base    int

		// decoded immediate values
		imm  int64
		imm8 int8
		immc int64

		// output
		opshift int
		inst    x86_Inst
		narg    int // number of arguments written to inst
	)

	if mode == 64 {
		dataMode = 32
	}

	// Prefixes are certainly the most complex and underspecified part of
	// decoding x86 instructions. Although the manuals say things like
	// up to four prefixes, one from each group, nearly everyone seems to
	// agree that in practice as many prefixes as possible, including multiple
	// from a particular group or repetitions of a given prefix, can be used on
	// an instruction, provided the total instruction length including prefixes
	// does not exceed the agreed-upon maximum of 15 bytes.
	// Everyone also agrees that if one of these prefixes is the LOCK prefix
	// and the instruction is not one of the instructions that can be used with
	// the LOCK prefix or if the destination is not a memory operand,
	// then the instruction is invalid and produces the #UD exception.
	// However, that is the end of any semblance of agreement.
	//
	// What happens if prefixes are given that conflict with other prefixes?
	// For example, the memory segment overrides CS, DS, ES, FS, GS, SS
	// conflict with each other: only one segment can be in effect.
	// Disassemblers seem to agree that later prefixes take priority over
	// earlier ones. I have not taken the time to write assembly programs
	// to check to see if the hardware agrees.
	//
	// What happens if prefixes are given that have no meaning for the
	// specific instruction to which they are attached? It depends.
	// If they really have no meaning, they are ignored. However, a future
	// processor may assign a different meaning. As a disassembler, we
	// don't really know whether we're seeing a meaningless prefix or one
	// whose meaning we simply haven't been told yet.
	//
	// Combining the two questions, what happens when conflicting
	// extension prefixes are given? No one seems to know for sure.
	// For example, MOVQ is 66 0F D6 /r, MOVDQ2Q is F2 0F D6 /r,
	// and MOVQ2DQ is F3 0F D6 /r. What is '66 F2 F3 0F D6 /r'?
	// Which prefix wins? See the xCondPrefix prefix for more.
	//
	// Writing assembly test cases to divine which interpretation the
	// CPU uses might clarify the situation, but more likely it would
	// make the situation even less clear.

	// Read non-REX prefixes.
ReadPrefixes:
	for ; pos < len(src); pos++ {
		p := x86_Prefix(src[pos])
		switch p {
		default:
			nprefix = pos
			break ReadPrefixes

		// Group 1 - lock and repeat prefixes
		// According to Intel, there should only be one from this set,
		// but according to AMD both can be present.
		case 0xF0:
			if lockIndex >= 0 {
				inst.Prefix[lockIndex] |= x86_PrefixIgnored
			}
			lockIndex = pos
		case 0xF2, 0xF3:
			if repIndex >= 0 {
				inst.Prefix[repIndex] |= x86_PrefixIgnored
			}
			repIndex = pos

		// Group 2 - segment override / branch hints
		case 0x26, 0x2E, 0x36, 0x3E:
			if mode == 64 {
				p |= x86_PrefixIgnored
				break
			}
			fallthrough
		case 0x64, 0x65:
			if segIndex >= 0 {
				inst.Prefix[segIndex] |= x86_PrefixIgnored
			}
			segIndex = pos

		// Group 3 - operand size override
		case 0x66:
			if mode == 16 {
				dataMode = 32
				p = x86_PrefixData32
			} else {
				dataMode = 16
				p = x86_PrefixData16
			}
			if dataSizeIndex >= 0 {
				inst.Prefix[dataSizeIndex] |= x86_PrefixIgnored
			}
			dataSizeIndex = pos

		// Group 4 - address size override
		case 0x67:
			if mode == 32 {
				addrMode = 16
				p = x86_PrefixAddr16
			} else {
				addrMode = 32
				p = x86_PrefixAddr32
			}
			if addrSizeIndex >= 0 {
				inst.Prefix[addrSizeIndex] |= x86_PrefixIgnored
			}
			addrSizeIndex = pos
		}

		if pos >= len(inst.Prefix) {
			return x86_instPrefix(src[0], mode) // too long
		}

		inst.Prefix[pos] = p
	}

	// Read REX prefix.
	if pos < len(src) && mode == 64 && x86_Prefix(src[pos]).IsREX() {
		rex = x86_Prefix(src[pos])
		rexIndex = pos
		if pos >= len(inst.Prefix) {
			return x86_instPrefix(src[0], mode) // too long
		}
		inst.Prefix[pos] = rex
		pos++
		if rex&x86_PrefixREXW != 0 {
			dataMode = 64
			if dataSizeIndex >= 0 {
				inst.Prefix[dataSizeIndex] |= x86_PrefixIgnored
			}
		}
	}

	// Decode instruction stream, interpreting decoding instructions.
	// opshift gives the shift to use when saving the next
	// opcode byte into inst.Opcode.
	opshift = 24
	if x86_decoderCover == nil {
		x86_decoderCover = make([]bool, len(x86_decoder))
	}

	// Decode loop, executing decoder program.
	var oldPC, prevPC int
Decode:
	for pc := 1; ; { // TODO uint
		oldPC = prevPC
		prevPC = pc
		if x86_trace {
			println("run", pc)
		}
		x := x86_decoder[pc]
		x86_decoderCover[pc] = true
		pc++

		// Read and decode ModR/M if needed by opcode.
		switch x86_decodeOp(x) {
		case x86_xCondSlashR, x86_xReadSlashR:
			if haveModrm {
				return x86_Inst{Len: pos}, x86_errInternal
			}
			haveModrm = true
			if pos >= len(src) {
				return x86_truncated(src, mode)
			}
			modrm = int(src[pos])
			pos++
			if opshift >= 0 {
				inst.Opcode |= uint32(modrm) << uint(opshift)
				opshift -= 8
			}
			mod = modrm >> 6
			regop = (modrm >> 3) & 07
			rm = modrm & 07
			if rex&x86_PrefixREXR != 0 {
				rexUsed |= x86_PrefixREXR
				regop |= 8
			}
			if addrMode == 16 {
				// 16-bit modrm form
				if mod != 3 {
					haveMem = true
					mem = x86_addr16[rm]
					if rm == 6 && mod == 0 {
						mem.Base = 0
					}

					// Consume disp16 if present.
					if mod == 0 && rm == 6 || mod == 2 {
						if pos+2 > len(src) {
							return x86_truncated(src, mode)
						}
						mem.Disp = int64(binary.LittleEndian.Uint16(src[pos:]))
						pos += 2
					}

					// Consume disp8 if present.
					if mod == 1 {
						if pos >= len(src) {
							return x86_truncated(src, mode)
						}
						mem.Disp = int64(int8(src[pos]))
						pos++
					}
				}
			} else {
				haveMem = mod != 3

				// 32-bit or 64-bit form
				// Consume SIB encoding if present.
				if rm == 4 && mod != 3 {
					haveSIB = true
					if pos >= len(src) {
						return x86_truncated(src, mode)
					}
					sib = int(src[pos])
					pos++
					if opshift >= 0 {
						inst.Opcode |= uint32(sib) << uint(opshift)
						opshift -= 8
					}
					scale = sib >> 6
					index = (sib >> 3) & 07
					base = sib & 07
					if rex&x86_PrefixREXB != 0 {
						rexUsed |= x86_PrefixREXB
						base |= 8
					}
					if rex&x86_PrefixREXX != 0 {
						rexUsed |= x86_PrefixREXX
						index |= 8
					}

					mem.Scale = 1 << uint(scale)
					if index == 4 {
						// no mem.Index
					} else {
						mem.Index = x86_baseRegForBits(addrMode) + x86_Reg(index)
					}
					if base&7 == 5 && mod == 0 {
						// no mem.Base
					} else {
						mem.Base = x86_baseRegForBits(addrMode) + x86_Reg(base)
					}
				} else {
					if rex&x86_PrefixREXB != 0 {
						rexUsed |= x86_PrefixREXB
						rm |= 8
					}
					if mod == 0 && rm&7 == 5 || rm&7 == 4 {
						// base omitted
					} else if mod != 3 {
						mem.Base = x86_baseRegForBits(addrMode) + x86_Reg(rm)
					}
				}

				// Consume disp32 if present.
				if mod == 0 && (rm&7 == 5 || haveSIB && base&7 == 5) || mod == 2 {
					if pos+4 > len(src) {
						return x86_truncated(src, mode)
					}
					mem.Disp = int64(binary.LittleEndian.Uint32(src[pos:]))
					pos += 4
				}

				// Consume disp8 if present.
				if mod == 1 {
					if pos >= len(src) {
						return x86_truncated(src, mode)
					}
					mem.Disp = int64(int8(src[pos]))
					pos++
				}

				// In 64-bit, mod=0 rm=5 is PC-relative instead of just disp.
				// See Vol 2A. Table 2-7.
				if mode == 64 && mod == 0 && rm&7 == 5 {
					if addrMode == 32 {
						mem.Base = x86_EIP
					} else {
						mem.Base = x86_RIP
					}
				}
			}

			if segIndex >= 0 {
				mem.Segment = x86_prefixToSegment(inst.Prefix[segIndex])
			}
		}

		// Execute single opcode.
		switch x86_decodeOp(x) {
		default:
			println("bad op", x, "at", pc-1, "from", oldPC)
			return x86_Inst{Len: pos}, x86_errInternal

		case x86_xFail:
			inst.Op = 0
			break Decode

		case x86_xMatch:
			break Decode

		case x86_xJump:
			pc = int(x86_decoder[pc])

		// Conditional branches.

		case x86_xCondByte:
			if pos >= len(src) {
				return x86_truncated(src, mode)
			}
			b := src[pos]
			n := int(x86_decoder[pc])
			pc++
			for i := 0; i < n; i++ {
				xb, xpc := x86_decoder[pc], int(x86_decoder[pc+1])
				pc += 2
				if b == byte(xb) {
					pc = xpc
					pos++
					if opshift >= 0 {
						inst.Opcode |= uint32(b) << uint(opshift)
						opshift -= 8
					}
					continue Decode
				}
			}
			// xCondByte is the only conditional with a fall through,
			// so that it can be used to pick off special cases before
			// an xCondSlash. If the fallthrough instruction is xFail,
			// advance the position so that the decoded instruction
			// size includes the byte we just compared against.
			if x86_decodeOp(x86_decoder[pc]) == x86_xJump {
				pc = int(x86_decoder[pc+1])
			}
			if x86_decodeOp(x86_decoder[pc]) == x86_xFail {
				pos++
			}

		case x86_xCondIs64:
			if mode == 64 {
				pc = int(x86_decoder[pc+1])
			} else {
				pc = int(x86_decoder[pc])
			}

		case x86_xCondIsMem:
			mem := haveMem
			if !haveModrm {
				if pos >= len(src) {
					return x86_instPrefix(src[0], mode) // too long
				}
				mem = src[pos]>>6 != 3
			}
			if mem {
				pc = int(x86_decoder[pc+1])
			} else {
				pc = int(x86_decoder[pc])
			}

		case x86_xCondDataSize:
			switch dataMode {
			case 16:
				if dataSizeIndex >= 0 {
					inst.Prefix[dataSizeIndex] |= x86_PrefixImplicit
				}
				pc = int(x86_decoder[pc])
			case 32:
				if dataSizeIndex >= 0 {
					inst.Prefix[dataSizeIndex] |= x86_PrefixImplicit
				}
				pc = int(x86_decoder[pc+1])
			case 64:
				rexUsed |= x86_PrefixREXW
				pc = int(x86_decoder[pc+2])
			}

		case x86_xCondAddrSize:
			switch addrMode {
			case 16:
				if addrSizeIndex >= 0 {
					inst.Prefix[addrSizeIndex] |= x86_PrefixImplicit
				}
				pc = int(x86_decoder[pc])
			case 32:
				if addrSizeIndex >= 0 {
					inst.Prefix[addrSizeIndex] |= x86_PrefixImplicit
				}
				pc = int(x86_decoder[pc+1])
			case 64:
				pc = int(x86_decoder[pc+2])
			}

		case x86_xCondPrefix:
			// Conditional branch based on presence or absence of prefixes.
			// The conflict cases here are completely undocumented and
			// differ significantly between GNU libopcodes and Intel xed.
			// I have not written assembly code to divine what various CPUs
			// do, but it wouldn't surprise me if they are not consistent either.
			//
			// The basic idea is to switch on the presence of a prefix, so that
			// for example:
			//
			//	xCondPrefix, 4
			//	0xF3, 123,
			//	0xF2, 234,
			//	0x66, 345,
			//	0, 456
			//
			// branch to 123 if the F3 prefix is present, 234 if the F2 prefix
			// is present, 66 if the 345 prefix is present, and 456 otherwise.
			// The prefixes are given in descending order so that the 0 will be last.
			//
			// It is unclear what should happen if multiple conditions are
			// satisfied: what if F2 and F3 are both present, or if 66 and F2
			// are present, or if all three are present? The one chosen becomes
			// part of the opcode and the others do not. Perhaps the answer
			// depends on the specific opcodes in question.
			//
			// The only clear example is that CRC32 is F2 0F 38 F1 /r, and
			// it comes in 16-bit and 32-bit forms based on the 66 prefix,
			// so 66 F2 0F 38 F1 /r should be treated as F2 taking priority,
			// with the 66 being only an operand size override, and probably
			// F2 66 0F 38 F1 /r should be treated the same.
			// Perhaps that rule is specific to the case of CRC32, since no
			// 66 0F 38 F1 instruction is defined (today) (that we know of).
			// However, both libopcodes and xed seem to generalize this
			// example and choose F2/F3 in preference to 66, and we
			// do the same.
			//
			// Next, what if both F2 and F3 are present? Which wins?
			// The Intel xed rule, and ours, is that the one that occurs last wins.
			// The GNU libopcodes rule, which we implement only in gnuCompat mode,
			// is that F3 beats F2 unless F3 has no special meaning, in which
			// case F3 can be a modified on an F2 special meaning.
			//
			// Concretely,
			//	66 0F D6 /r is MOVQ
			//	F2 0F D6 /r is MOVDQ2Q
			//	F3 0F D6 /r is MOVQ2DQ.
			//
			//	F2 66 0F D6 /r is 66 + MOVDQ2Q always.
			//	66 F2 0F D6 /r is 66 + MOVDQ2Q always.
			//	F3 66 0F D6 /r is 66 + MOVQ2DQ always.
			//	66 F3 0F D6 /r is 66 + MOVQ2DQ always.
			//	F2 F3 0F D6 /r is F2 + MOVQ2DQ always.
			//	F3 F2 0F D6 /r is F3 + MOVQ2DQ in Intel xed, but F2 + MOVQ2DQ in GNU libopcodes.
			//	Adding 66 anywhere in the prefix section of the
			//	last two cases does not change the outcome.
			//
			// Finally, what if there is a variant in which 66 is a mandatory
			// prefix rather than an operand size override, but we know of
			// no corresponding F2/F3 form, and we see both F2/F3 and 66.
			// Does F2/F3 still take priority, so that the result is an unknown
			// instruction, or does the 66 take priority, so that the extended
			// 66 instruction should be interpreted as having a REP/REPN prefix?
			// Intel xed does the former and GNU libopcodes does the latter.
			// We side with Intel xed, unless we are trying to match libopcodes
			// more closely during the comparison-based test suite.
			//
			// In 64-bit mode REX.W is another valid prefix to test for, but
			// there is less ambiguity about that. When present, REX.W is
			// always the first entry in the table.
			n := int(x86_decoder[pc])
			pc++
			sawF3 := false
			for j := 0; j < n; j++ {
				prefix := x86_Prefix(x86_decoder[pc+2*j])
				if prefix.IsREX() {
					rexUsed |= prefix
					if rex&prefix == prefix {
						pc = int(x86_decoder[pc+2*j+1])
						continue Decode
					}
					continue
				}
				ok := false
				if prefix == 0 {
					ok = true
				} else if prefix.IsREX() {
					rexUsed |= prefix
					if rex&prefix == prefix {
						ok = true
					}
				} else {
					if prefix == 0xF3 {
						sawF3 = true
					}
					switch prefix {
					case x86_PrefixLOCK:
						if lockIndex >= 0 {
							inst.Prefix[lockIndex] |= x86_PrefixImplicit
							ok = true
						}
					case x86_PrefixREP, x86_PrefixREPN:
						if repIndex >= 0 && inst.Prefix[repIndex]&0xFF == prefix {
							inst.Prefix[repIndex] |= x86_PrefixImplicit
							ok = true
						}
						if gnuCompat && !ok && prefix == 0xF3 && repIndex >= 0 && (j+1 >= n || x86_decoder[pc+2*(j+1)] != 0xF2) {
							// Check to see if earlier prefix F3 is present.
							for i := repIndex - 1; i >= 0; i-- {
								if inst.Prefix[i]&0xFF == prefix {
									inst.Prefix[i] |= x86_PrefixImplicit
									ok = true
								}
							}
						}
						if gnuCompat && !ok && prefix == 0xF2 && repIndex >= 0 && !sawF3 && inst.Prefix[repIndex]&0xFF == 0xF3 {
							// Check to see if earlier prefix F2 is present.
							for i := repIndex - 1; i >= 0; i-- {
								if inst.Prefix[i]&0xFF == prefix {
									inst.Prefix[i] |= x86_PrefixImplicit
									ok = true
								}
							}
						}
					case x86_PrefixCS, x86_PrefixDS, x86_PrefixES, x86_PrefixFS, x86_PrefixGS, x86_PrefixSS:
						if segIndex >= 0 && inst.Prefix[segIndex]&0xFF == prefix {
							inst.Prefix[segIndex] |= x86_PrefixImplicit
							ok = true
						}
					case x86_PrefixDataSize:
						// Looking for 66 mandatory prefix.
						// The F2/F3 mandatory prefixes take priority when both are present.
						// If we got this far in the xCondPrefix table and an F2/F3 is present,
						// it means the table didn't have any entry for that prefix. But if 66 has
						// special meaning, perhaps F2/F3 have special meaning that we don't know.
						// Intel xed works this way, treating the F2/F3 as inhibiting the 66.
						// GNU libopcodes allows the 66 to match. We do what Intel xed does
						// except in gnuCompat mode.
						if repIndex >= 0 && !gnuCompat {
							inst.Op = 0
							break Decode
						}
						if dataSizeIndex >= 0 {
							inst.Prefix[dataSizeIndex] |= x86_PrefixImplicit
							ok = true
						}
					case x86_PrefixAddrSize:
						if addrSizeIndex >= 0 {
							inst.Prefix[addrSizeIndex] |= x86_PrefixImplicit
							ok = true
						}
					}
				}
				if ok {
					pc = int(x86_decoder[pc+2*j+1])
					continue Decode
				}
			}
			inst.Op = 0
			break Decode

		case x86_xCondSlashR:
			pc = int(x86_decoder[pc+regop&7])

		// Input.

		case x86_xReadSlashR:
			// done above

		case x86_xReadIb:
			if pos >= len(src) {
				return x86_truncated(src, mode)
			}
			imm8 = int8(src[pos])
			pos++

		case x86_xReadIw:
			if pos+2 > len(src) {
				return x86_truncated(src, mode)
			}
			imm = int64(binary.LittleEndian.Uint16(src[pos:]))
			pos += 2

		case x86_xReadId:
			if pos+4 > len(src) {
				return x86_truncated(src, mode)
			}
			imm = int64(binary.LittleEndian.Uint32(src[pos:]))
			pos += 4

		case x86_xReadIo:
			if pos+8 > len(src) {
				return x86_truncated(src, mode)
			}
			imm = int64(binary.LittleEndian.Uint64(src[pos:]))
			pos += 8

		case x86_xReadCb:
			if pos >= len(src) {
				return x86_truncated(src, mode)
			}
			immc = int64(src[pos])
			pos++

		case x86_xReadCw:
			if pos+2 > len(src) {
				return x86_truncated(src, mode)
			}
			immc = int64(binary.LittleEndian.Uint16(src[pos:]))
			pos += 2

		case x86_xReadCm:
			if addrMode == 16 {
				if pos+2 > len(src) {
					return x86_truncated(src, mode)
				}
				immc = int64(binary.LittleEndian.Uint16(src[pos:]))
				pos += 2
			} else if addrMode == 32 {
				if pos+4 > len(src) {
					return x86_truncated(src, mode)
				}
				immc = int64(binary.LittleEndian.Uint32(src[pos:]))
				pos += 4
			} else {
				if pos+8 > len(src) {
					return x86_truncated(src, mode)
				}
				immc = int64(binary.LittleEndian.Uint64(src[pos:]))
				pos += 8
			}
		case x86_xReadCd:
			if pos+4 > len(src) {
				return x86_truncated(src, mode)
			}
			immc = int64(binary.LittleEndian.Uint32(src[pos:]))
			pos += 4

		case x86_xReadCp:
			if pos+6 > len(src) {
				return x86_truncated(src, mode)
			}
			w := binary.LittleEndian.Uint32(src[pos:])
			w2 := binary.LittleEndian.Uint16(src[pos+4:])
			immc = int64(w2)<<32 | int64(w)
			pos += 6

		// Output.

		case x86_xSetOp:
			inst.Op = x86_Op(x86_decoder[pc])
			pc++

		case x86_xArg1,
			x86_xArg3,
			x86_xArgAL,
			x86_xArgAX,
			x86_xArgCL,
			x86_xArgCS,
			x86_xArgDS,
			x86_xArgDX,
			x86_xArgEAX,
			x86_xArgEDX,
			x86_xArgES,
			x86_xArgFS,
			x86_xArgGS,
			x86_xArgRAX,
			x86_xArgRDX,
			x86_xArgSS,
			x86_xArgST,
			x86_xArgXMM0:
			inst.Args[narg] = x86_fixedArg[x]
			narg++

		case x86_xArgImm8:
			inst.Args[narg] = x86_Imm(imm8)
			narg++

		case x86_xArgImm8u:
			inst.Args[narg] = x86_Imm(uint8(imm8))
			narg++

		case x86_xArgImm16:
			inst.Args[narg] = x86_Imm(int16(imm))
			narg++

		case x86_xArgImm16u:
			inst.Args[narg] = x86_Imm(uint16(imm))
			narg++

		case x86_xArgImm32:
			inst.Args[narg] = x86_Imm(int32(imm))
			narg++

		case x86_xArgImm64:
			inst.Args[narg] = x86_Imm(imm)
			narg++

		case x86_xArgM,
			x86_xArgM128,
			x86_xArgM1428byte,
			x86_xArgM16,
			x86_xArgM16and16,
			x86_xArgM16and32,
			x86_xArgM16and64,
			x86_xArgM16colon16,
			x86_xArgM16colon32,
			x86_xArgM16colon64,
			x86_xArgM16int,
			x86_xArgM2byte,
			x86_xArgM32,
			x86_xArgM32and32,
			x86_xArgM32fp,
			x86_xArgM32int,
			x86_xArgM512byte,
			x86_xArgM64,
			x86_xArgM64fp,
			x86_xArgM64int,
			x86_xArgM8,
			x86_xArgM80bcd,
			x86_xArgM80dec,
			x86_xArgM80fp,
			x86_xArgM94108byte,
			x86_xArgMem:
			if !haveMem {
				inst.Op = 0
				break Decode
			}
			inst.Args[narg] = mem
			inst.MemBytes = int(x86_memBytes[x86_decodeOp(x)])
			narg++

		case x86_xArgPtr16colon16:
			inst.Args[narg] = x86_Imm(immc >> 16)
			inst.Args[narg+1] = x86_Imm(immc & (1<<16 - 1))
			narg += 2

		case x86_xArgPtr16colon32:
			inst.Args[narg] = x86_Imm(immc >> 32)
			inst.Args[narg+1] = x86_Imm(immc & (1<<32 - 1))
			narg += 2

		case x86_xArgMoffs8, x86_xArgMoffs16, x86_xArgMoffs32, x86_xArgMoffs64:
			// TODO(rsc): Can address be 64 bits?
			mem = x86_Mem{Disp: int64(immc)}
			if segIndex >= 0 {
				mem.Segment = x86_prefixToSegment(inst.Prefix[segIndex])
				inst.Prefix[segIndex] |= x86_PrefixImplicit
			}
			inst.Args[narg] = mem
			inst.MemBytes = int(x86_memBytes[x86_decodeOp(x)])
			narg++

		case x86_xArgR8, x86_xArgR16, x86_xArgR32, x86_xArgR64, x86_xArgXmm, x86_xArgXmm1, x86_xArgDR0dashDR7:
			base := x86_baseReg[x]
			index := x86_Reg(regop)
			if rex != 0 && base == x86_AL && index >= 4 {
				rexUsed |= x86_PrefixREX
				index -= 4
				base = x86_SPB
			}
			inst.Args[narg] = base + index
			narg++

		case x86_xArgMm, x86_xArgMm1, x86_xArgTR0dashTR7:
			inst.Args[narg] = x86_baseReg[x] + x86_Reg(regop&7)
			narg++

		case x86_xArgCR0dashCR7:
			// AMD documents an extension that the LOCK prefix
			// can be used in place of a REX prefix in order to access
			// CR8 from 32-bit mode. The LOCK prefix is allowed in
			// all modes, provided the corresponding CPUID bit is set.
			if lockIndex >= 0 {
				inst.Prefix[lockIndex] |= x86_PrefixImplicit
				regop += 8
			}
			inst.Args[narg] = x86_CR0 + x86_Reg(regop)
			narg++

		case x86_xArgSreg:
			regop &= 7
			if regop >= 6 {
				inst.Op = 0
				break Decode
			}
			inst.Args[narg] = x86_ES + x86_Reg(regop)
			narg++

		case x86_xArgRmf16, x86_xArgRmf32, x86_xArgRmf64:
			base := x86_baseReg[x]
			index := x86_Reg(modrm & 07)
			if rex&x86_PrefixREXB != 0 {
				rexUsed |= x86_PrefixREXB
				index += 8
			}
			inst.Args[narg] = base + index
			narg++

		case x86_xArgR8op, x86_xArgR16op, x86_xArgR32op, x86_xArgR64op, x86_xArgSTi:
			n := inst.Opcode >> uint(opshift+8) & 07
			base := x86_baseReg[x]
			index := x86_Reg(n)
			if rex&x86_PrefixREXB != 0 && x86_decodeOp(x) != x86_xArgSTi {
				rexUsed |= x86_PrefixREXB
				index += 8
			}
			if rex != 0 && base == x86_AL && index >= 4 {
				rexUsed |= x86_PrefixREX
				index -= 4
				base = x86_SPB
			}
			inst.Args[narg] = base + index
			narg++

		case x86_xArgRM8, x86_xArgRM16, x86_xArgRM32, x86_xArgRM64, x86_xArgR32M16, x86_xArgR32M8, x86_xArgR64M16,
			x86_xArgMmM32, x86_xArgMmM64, x86_xArgMm2M64,
			x86_xArgXmm2M16, x86_xArgXmm2M32, x86_xArgXmm2M64, x86_xArgXmmM64, x86_xArgXmmM128, x86_xArgXmmM32, x86_xArgXmm2M128:
			if haveMem {
				inst.Args[narg] = mem
				inst.MemBytes = int(x86_memBytes[x86_decodeOp(x)])
			} else {
				base := x86_baseReg[x]
				index := x86_Reg(rm)
				switch x86_decodeOp(x) {
				case x86_xArgMmM32, x86_xArgMmM64, x86_xArgMm2M64:
					// There are only 8 MMX registers, so these ignore the REX.X bit.
					index &= 7
				case x86_xArgRM8:
					if rex != 0 && index >= 4 {
						rexUsed |= x86_PrefixREX
						index -= 4
						base = x86_SPB
					}
				}
				inst.Args[narg] = base + index
			}
			narg++

		case x86_xArgMm2: // register only; TODO(rsc): Handle with tag modrm_regonly tag
			if haveMem {
				inst.Op = 0
				break Decode
			}
			inst.Args[narg] = x86_baseReg[x] + x86_Reg(rm&7)
			narg++

		case x86_xArgXmm2: // register only; TODO(rsc): Handle with tag modrm_regonly tag
			if haveMem {
				inst.Op = 0
				break Decode
			}
			inst.Args[narg] = x86_baseReg[x] + x86_Reg(rm)
			narg++

		case x86_xArgRel8:
			inst.Args[narg] = x86_Rel(int8(immc))
			narg++

		case x86_xArgRel16:
			inst.Args[narg] = x86_Rel(int16(immc))
			narg++

		case x86_xArgRel32:
			inst.Args[narg] = x86_Rel(int32(immc))
			narg++
		}
	}

	if inst.Op == 0 {
		// Invalid instruction.
		if nprefix > 0 {
			return x86_instPrefix(src[0], mode) // invalid instruction
		}
		return x86_Inst{Len: pos}, x86_ErrUnrecognized
	}

	// Matched! Hooray!

	// 90 decodes as XCHG EAX, EAX but is NOP.
	// 66 90 decodes as XCHG AX, AX and is NOP too.
	// 48 90 decodes as XCHG RAX, RAX and is NOP too.
	// 43 90 decodes as XCHG R8D, EAX and is *not* NOP.
	// F3 90 decodes as REP XCHG EAX, EAX but is PAUSE.
	// It's all too special to handle in the decoding tables, at least for now.
	if inst.Op == x86_XCHG && inst.Opcode>>24 == 0x90 {
		if inst.Args[0] == x86_RAX || inst.Args[0] == x86_EAX || inst.Args[0] == x86_AX {
			inst.Op = x86_NOP
			if dataSizeIndex >= 0 {
				inst.Prefix[dataSizeIndex] &^= x86_PrefixImplicit
			}
			inst.Args[0] = nil
			inst.Args[1] = nil
		}
		if repIndex >= 0 && inst.Prefix[repIndex] == 0xF3 {
			inst.Prefix[repIndex] |= x86_PrefixImplicit
			inst.Op = x86_PAUSE
			inst.Args[0] = nil
			inst.Args[1] = nil
		} else if gnuCompat {
			for i := nprefix - 1; i >= 0; i-- {
				if inst.Prefix[i]&0xFF == 0xF3 {
					inst.Prefix[i] |= x86_PrefixImplicit
					inst.Op = x86_PAUSE
					inst.Args[0] = nil
					inst.Args[1] = nil
					break
				}
			}
		}
	}

	// defaultSeg returns the default segment for an implicit
	// memory reference: the final override if present, or else DS.
	defaultSeg := func() x86_Reg {
		if segIndex >= 0 {
			inst.Prefix[segIndex] |= x86_PrefixImplicit
			return x86_prefixToSegment(inst.Prefix[segIndex])
		}
		return x86_DS
	}

	// Add implicit arguments not present in the tables.
	// Normally we shy away from making implicit arguments explicit,
	// following the Intel manuals, but adding the arguments seems
	// the best way to express the effect of the segment override prefixes.
	// TODO(rsc): Perhaps add these to the tables and
	// create bytecode instructions for them.
	usedAddrSize := false
	switch inst.Op {
	case x86_INSB, x86_INSW, x86_INSD:
		inst.Args[0] = x86_Mem{Segment: x86_ES, Base: x86_baseRegForBits(addrMode) + x86_DI - x86_AX}
		inst.Args[1] = x86_DX
		usedAddrSize = true

	case x86_OUTSB, x86_OUTSW, x86_OUTSD:
		inst.Args[0] = x86_DX
		inst.Args[1] = x86_Mem{Segment: defaultSeg(), Base: x86_baseRegForBits(addrMode) + x86_SI - x86_AX}
		usedAddrSize = true

	case x86_MOVSB, x86_MOVSW, x86_MOVSD, x86_MOVSQ:
		inst.Args[0] = x86_Mem{Segment: x86_ES, Base: x86_baseRegForBits(addrMode) + x86_DI - x86_AX}
		inst.Args[1] = x86_Mem{Segment: defaultSeg(), Base: x86_baseRegForBits(addrMode) + x86_SI - x86_AX}
		usedAddrSize = true

	case x86_CMPSB, x86_CMPSW, x86_CMPSD, x86_CMPSQ:
		inst.Args[0] = x86_Mem{Segment: defaultSeg(), Base: x86_baseRegForBits(addrMode) + x86_SI - x86_AX}
		inst.Args[1] = x86_Mem{Segment: x86_ES, Base: x86_baseRegForBits(addrMode) + x86_DI - x86_AX}
		usedAddrSize = true

	case x86_LODSB, x86_LODSW, x86_LODSD, x86_LODSQ:
		switch inst.Op {
		case x86_LODSB:
			inst.Args[0] = x86_AL
		case x86_LODSW:
			inst.Args[0] = x86_AX
		case x86_LODSD:
			inst.Args[0] = x86_EAX
		case x86_LODSQ:
			inst.Args[0] = x86_RAX
		}
		inst.Args[1] = x86_Mem{Segment: defaultSeg(), Base: x86_baseRegForBits(addrMode) + x86_SI - x86_AX}
		usedAddrSize = true

	case x86_STOSB, x86_STOSW, x86_STOSD, x86_STOSQ:
		inst.Args[0] = x86_Mem{Segment: x86_ES, Base: x86_baseRegForBits(addrMode) + x86_DI - x86_AX}
		switch inst.Op {
		case x86_STOSB:
			inst.Args[1] = x86_AL
		case x86_STOSW:
			inst.Args[1] = x86_AX
		case x86_STOSD:
			inst.Args[1] = x86_EAX
		case x86_STOSQ:
			inst.Args[1] = x86_RAX
		}
		usedAddrSize = true

	case x86_SCASB, x86_SCASW, x86_SCASD, x86_SCASQ:
		inst.Args[1] = x86_Mem{Segment: x86_ES, Base: x86_baseRegForBits(addrMode) + x86_DI - x86_AX}
		switch inst.Op {
		case x86_SCASB:
			inst.Args[0] = x86_AL
		case x86_SCASW:
			inst.Args[0] = x86_AX
		case x86_SCASD:
			inst.Args[0] = x86_EAX
		case x86_SCASQ:
			inst.Args[0] = x86_RAX
		}
		usedAddrSize = true

	case x86_XLATB:
		inst.Args[0] = x86_Mem{Segment: defaultSeg(), Base: x86_baseRegForBits(addrMode) + x86_BX - x86_AX}
		usedAddrSize = true
	}

	// If we used the address size annotation to construct the
	// argument list, mark that prefix as implicit: it doesn't need
	// to be shown when printing the instruction.
	if haveMem || usedAddrSize {
		if addrSizeIndex >= 0 {
			inst.Prefix[addrSizeIndex] |= x86_PrefixImplicit
		}
	}

	// Similarly, if there's some memory operand, the segment
	// will be shown there and doesn't need to be shown as an
	// explicit prefix.
	if haveMem {
		if segIndex >= 0 {
			inst.Prefix[segIndex] |= x86_PrefixImplicit
		}
	}

	// Branch predict prefixes are overloaded segment prefixes,
	// since segment prefixes don't make sense on conditional jumps.
	// Rewrite final instance to prediction prefix.
	// The set of instructions to which the prefixes apply (other then the
	// Jcc conditional jumps) is not 100% clear from the manuals, but
	// the disassemblers seem to agree about the LOOP and JCXZ instructions,
	// so we'll follow along.
	// TODO(rsc): Perhaps this instruction class should be derived from the CSV.
	if x86_isCondJmp[inst.Op] || x86_isLoop[inst.Op] || inst.Op == x86_JCXZ || inst.Op == x86_JECXZ || inst.Op == x86_JRCXZ {
	PredictLoop:
		for i := nprefix - 1; i >= 0; i-- {
			p := inst.Prefix[i]
			switch p & 0xFF {
			case x86_PrefixCS:
				inst.Prefix[i] = x86_PrefixPN
				break PredictLoop
			case x86_PrefixDS:
				inst.Prefix[i] = x86_PrefixPT
				break PredictLoop
			}
		}
	}

	// The BND prefix is part of the Intel Memory Protection Extensions (MPX).
	// A REPN applied to certain control transfers is a BND prefix to bound
	// the range of possible destinations. There's surprisingly little documentation
	// about this, so we just do what libopcodes and xed agree on.
	// In particular, it's unclear why a REPN applied to LOOP or JCXZ instructions
	// does not turn into a BND.
	// TODO(rsc): Perhaps this instruction class should be derived from the CSV.
	if x86_isCondJmp[inst.Op] || inst.Op == x86_JMP || inst.Op == x86_CALL || inst.Op == x86_RET {
		for i := nprefix - 1; i >= 0; i-- {
			p := inst.Prefix[i]
			if p&^x86_PrefixIgnored == x86_PrefixREPN {
				inst.Prefix[i] = x86_PrefixBND
				break
			}
		}
	}

	// The LOCK prefix only applies to certain instructions, and then only
	// to instances of the instruction with a memory destination.
	// Other uses of LOCK are invalid and cause a processor exception,
	// in contrast to the "just ignore it" spirit applied to all other prefixes.
	// Mark invalid lock prefixes.
	hasLock := false
	if lockIndex >= 0 && inst.Prefix[lockIndex]&x86_PrefixImplicit == 0 {
		switch inst.Op {
		// TODO(rsc): Perhaps this instruction class should be derived from the CSV.
		case x86_ADD, x86_ADC, x86_AND, x86_BTC, x86_BTR, x86_BTS, x86_CMPXCHG, x86_CMPXCHG8B, x86_CMPXCHG16B, x86_DEC, x86_INC, x86_NEG, x86_NOT, x86_OR, x86_SBB, x86_SUB, x86_XOR, x86_XADD, x86_XCHG:
			if x86_isMem(inst.Args[0]) {
				hasLock = true
				break
			}
			fallthrough
		default:
			inst.Prefix[lockIndex] |= x86_PrefixInvalid
		}
	}

	// In certain cases, all of which require a memory destination,
	// the REPN and REP prefixes are interpreted as XACQUIRE and XRELEASE
	// from the Intel Transactional Synchroniation Extensions (TSX).
	//
	// The specific rules are:
	// (1) Any instruction with a valid LOCK prefix can have XACQUIRE or XRELEASE.
	// (2) Any XCHG, which always has an implicit LOCK, can have XACQUIRE or XRELEASE.
	// (3) Any 0x88-, 0x89-, 0xC6-, or 0xC7-opcode MOV can have XRELEASE.
	if x86_isMem(inst.Args[0]) {
		if inst.Op == x86_XCHG {
			hasLock = true
		}

		for i := len(inst.Prefix) - 1; i >= 0; i-- {
			p := inst.Prefix[i] &^ x86_PrefixIgnored
			switch p {
			case x86_PrefixREPN:
				if hasLock {
					inst.Prefix[i] = inst.Prefix[i]&x86_PrefixIgnored | x86_PrefixXACQUIRE
				}

			case x86_PrefixREP:
				if hasLock {
					inst.Prefix[i] = inst.Prefix[i]&x86_PrefixIgnored | x86_PrefixXRELEASE
				}

				if inst.Op == x86_MOV {
					op := (inst.Opcode >> 24) &^ 1
					if op == 0x88 || op == 0xC6 {
						inst.Prefix[i] = inst.Prefix[i]&x86_PrefixIgnored | x86_PrefixXRELEASE
					}
				}
			}
		}
	}

	// If REP is used on a non-REP-able instruction, mark the prefix as ignored.
	if repIndex >= 0 {
		switch inst.Prefix[repIndex] {
		case x86_PrefixREP, x86_PrefixREPN:
			switch inst.Op {
			// According to the manuals, the REP/REPE prefix applies to all of these,
			// while the REPN applies only to some of them. However, both libopcodes
			// and xed show both prefixes explicitly for all instructions, so we do the same.
			// TODO(rsc): Perhaps this instruction class should be derived from the CSV.
			case x86_INSB, x86_INSW, x86_INSD,
				x86_MOVSB, x86_MOVSW, x86_MOVSD, x86_MOVSQ,
				x86_OUTSB, x86_OUTSW, x86_OUTSD,
				x86_LODSB, x86_LODSW, x86_LODSD, x86_LODSQ,
				x86_CMPSB, x86_CMPSW, x86_CMPSD, x86_CMPSQ,
				x86_SCASB, x86_SCASW, x86_SCASD, x86_SCASQ,
				x86_STOSB, x86_STOSW, x86_STOSD, x86_STOSQ:
				// ok
			default:
				inst.Prefix[repIndex] |= x86_PrefixIgnored
			}
		}
	}

	// If REX was present, mark implicit if all the 1 bits were consumed.
	if rexIndex >= 0 {
		if rexUsed != 0 {
			rexUsed |= x86_PrefixREX
		}
		if rex&^rexUsed == 0 {
			inst.Prefix[rexIndex] |= x86_PrefixImplicit
		}
	}

	inst.DataSize = dataMode
	inst.AddrSize = addrMode
	inst.Mode = mode
	inst.Len = pos
	return inst, nil
}

var x86_errInternal = errors.New("internal error")

// addr16 records the eight 16-bit addressing modes.
var x86_addr16 = [8]x86_Mem{
	{Base: x86_BX, Scale: 1, Index: x86_SI},
	{Base: x86_BX, Scale: 1, Index: x86_DI},
	{Base: x86_BP, Scale: 1, Index: x86_SI},
	{Base: x86_BP, Scale: 1, Index: x86_DI},
	{Base: x86_SI},
	{Base: x86_DI},
	{Base: x86_BP},
	{Base: x86_BX},
}

// baseReg returns the base register for a given register size in bits.
func x86_baseRegForBits(bits int) x86_Reg {
	switch bits {
	case 8:
		return x86_AL
	case 16:
		return x86_AX
	case 32:
		return x86_EAX
	case 64:
		return x86_RAX
	}
	return 0
}

// baseReg records the base register for argument types that specify
// a range of registers indexed by op, regop, or rm.
var x86_baseReg = [...]x86_Reg{
	x86_xArgDR0dashDR7: x86_DR0,
	x86_xArgMm1:        x86_M0,
	x86_xArgMm2:        x86_M0,
	x86_xArgMm2M64:     x86_M0,
	x86_xArgMm:         x86_M0,
	x86_xArgMmM32:      x86_M0,
	x86_xArgMmM64:      x86_M0,
	x86_xArgR16:        x86_AX,
	x86_xArgR16op:      x86_AX,
	x86_xArgR32:        x86_EAX,
	x86_xArgR32M16:     x86_EAX,
	x86_xArgR32M8:      x86_EAX,
	x86_xArgR32op:      x86_EAX,
	x86_xArgR64:        x86_RAX,
	x86_xArgR64M16:     x86_RAX,
	x86_xArgR64op:      x86_RAX,
	x86_xArgR8:         x86_AL,
	x86_xArgR8op:       x86_AL,
	x86_xArgRM16:       x86_AX,
	x86_xArgRM32:       x86_EAX,
	x86_xArgRM64:       x86_RAX,
	x86_xArgRM8:        x86_AL,
	x86_xArgRmf16:      x86_AX,
	x86_xArgRmf32:      x86_EAX,
	x86_xArgRmf64:      x86_RAX,
	x86_xArgSTi:        x86_F0,
	x86_xArgTR0dashTR7: x86_TR0,
	x86_xArgXmm1:       x86_X0,
	x86_xArgXmm2:       x86_X0,
	x86_xArgXmm2M128:   x86_X0,
	x86_xArgXmm2M16:    x86_X0,
	x86_xArgXmm2M32:    x86_X0,
	x86_xArgXmm2M64:    x86_X0,
	x86_xArgXmm:        x86_X0,
	x86_xArgXmmM128:    x86_X0,
	x86_xArgXmmM32:     x86_X0,
	x86_xArgXmmM64:     x86_X0,
}

// prefixToSegment returns the segment register
// corresponding to a particular segment prefix.
func x86_prefixToSegment(p x86_Prefix) x86_Reg {
	switch p &^ x86_PrefixImplicit {
	case x86_PrefixCS:
		return x86_CS
	case x86_PrefixDS:
		return x86_DS
	case x86_PrefixES:
		return x86_ES
	case x86_PrefixFS:
		return x86_FS
	case x86_PrefixGS:
		return x86_GS
	case x86_PrefixSS:
		return x86_SS
	}
	return 0
}

// fixedArg records the fixed arguments corresponding to the given bytecodes.
var x86_fixedArg = [...]x86_Arg{
	x86_xArg1:    x86_Imm(1),
	x86_xArg3:    x86_Imm(3),
	x86_xArgAL:   x86_AL,
	x86_xArgAX:   x86_AX,
	x86_xArgDX:   x86_DX,
	x86_xArgEAX:  x86_EAX,
	x86_xArgEDX:  x86_EDX,
	x86_xArgRAX:  x86_RAX,
	x86_xArgRDX:  x86_RDX,
	x86_xArgCL:   x86_CL,
	x86_xArgCS:   x86_CS,
	x86_xArgDS:   x86_DS,
	x86_xArgES:   x86_ES,
	x86_xArgFS:   x86_FS,
	x86_xArgGS:   x86_GS,
	x86_xArgSS:   x86_SS,
	x86_xArgST:   x86_F0,
	x86_xArgXMM0: x86_X0,
}

// memBytes records the size of the memory pointed at
// by a memory argument of the given form.
var x86_memBytes = [...]int8{
	x86_xArgM128:       128 / 8,
	x86_xArgM16:        16 / 8,
	x86_xArgM16and16:   (16 + 16) / 8,
	x86_xArgM16colon16: (16 + 16) / 8,
	x86_xArgM16colon32: (16 + 32) / 8,
	x86_xArgM16int:     16 / 8,
	x86_xArgM2byte:     2,
	x86_xArgM32:        32 / 8,
	x86_xArgM32and32:   (32 + 32) / 8,
	x86_xArgM32fp:      32 / 8,
	x86_xArgM32int:     32 / 8,
	x86_xArgM64:        64 / 8,
	x86_xArgM64fp:      64 / 8,
	x86_xArgM64int:     64 / 8,
	x86_xArgMm2M64:     64 / 8,
	x86_xArgMmM32:      32 / 8,
	x86_xArgMmM64:      64 / 8,
	x86_xArgMoffs16:    16 / 8,
	x86_xArgMoffs32:    32 / 8,
	x86_xArgMoffs64:    64 / 8,
	x86_xArgMoffs8:     8 / 8,
	x86_xArgR32M16:     16 / 8,
	x86_xArgR32M8:      8 / 8,
	x86_xArgR64M16:     16 / 8,
	x86_xArgRM16:       16 / 8,
	x86_xArgRM32:       32 / 8,
	x86_xArgRM64:       64 / 8,
	x86_xArgRM8:        8 / 8,
	x86_xArgXmm2M128:   128 / 8,
	x86_xArgXmm2M16:    16 / 8,
	x86_xArgXmm2M32:    32 / 8,
	x86_xArgXmm2M64:    64 / 8,
	x86_xArgXmm:        128 / 8,
	x86_xArgXmmM128:    128 / 8,
	x86_xArgXmmM32:     32 / 8,
	x86_xArgXmmM64:     64 / 8,
}

// isCondJmp records the conditional jumps.
var x86_isCondJmp = [x86_maxOp + 1]bool{
	x86_JA:  true,
	x86_JAE: true,
	x86_JB:  true,
	x86_JBE: true,
	x86_JE:  true,
	x86_JG:  true,
	x86_JGE: true,
	x86_JL:  true,
	x86_JLE: true,
	x86_JNE: true,
	x86_JNO: true,
	x86_JNP: true,
	x86_JNS: true,
	x86_JO:  true,
	x86_JP:  true,
	x86_JS:  true,
}

// isLoop records the loop operators.
var x86_isLoop = [x86_maxOp + 1]bool{
	x86_LOOP:   true,
	x86_LOOPE:  true,
	x86_LOOPNE: true,
	x86_JECXZ:  true,
	x86_JRCXZ:  true,
}

/* gnu.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// GNUSyntax returns the GNU assembler syntax for the instruction, as defined by GNU binutils.
// This general form is often called ``AT&T syntax'' as a reference to AT&T System V Unix.
func x86_GNUSyntax(inst x86_Inst) string {
	// Rewrite instruction to mimic GNU peculiarities.
	// Note that inst has been passed by value and contains
	// no pointers, so any changes we make here are local
	// and will not propagate back out to the caller.

	// Adjust opcode [sic].
	switch inst.Op {
	case x86_FDIV, x86_FDIVR, x86_FSUB, x86_FSUBR, x86_FDIVP, x86_FDIVRP, x86_FSUBP, x86_FSUBRP:
		// DC E0, DC F0: libopcodes swaps FSUBR/FSUB and FDIVR/FDIV, at least
		// if you believe the Intel manual is correct (the encoding is irregular as given;
		// libopcodes uses the more regular expected encoding).
		// TODO(rsc): Test to ensure Intel manuals are correct and report to libopcodes maintainers?
		// NOTE: iant thinks this is deliberate, but we can't find the history.
		_, reg1 := inst.Args[0].(x86_Reg)
		_, reg2 := inst.Args[1].(x86_Reg)
		if reg1 && reg2 && (inst.Opcode>>24 == 0xDC || inst.Opcode>>24 == 0xDE) {
			switch inst.Op {
			case x86_FDIV:
				inst.Op = x86_FDIVR
			case x86_FDIVR:
				inst.Op = x86_FDIV
			case x86_FSUB:
				inst.Op = x86_FSUBR
			case x86_FSUBR:
				inst.Op = x86_FSUB
			case x86_FDIVP:
				inst.Op = x86_FDIVRP
			case x86_FDIVRP:
				inst.Op = x86_FDIVP
			case x86_FSUBP:
				inst.Op = x86_FSUBRP
			case x86_FSUBRP:
				inst.Op = x86_FSUBP
			}
		}

	case x86_MOVNTSD:
		// MOVNTSD is F2 0F 2B /r.
		// MOVNTSS is F3 0F 2B /r (supposedly; not in manuals).
		// Usually inner prefixes win for display,
		// so that F3 F2 0F 2B 11 is REP MOVNTSD
		// and F2 F3 0F 2B 11 is REPN MOVNTSS.
		// Libopcodes always prefers MOVNTSS regardless of prefix order.
		if x86_countPrefix(&inst, 0xF3) > 0 {
			found := false
			for i := len(inst.Prefix) - 1; i >= 0; i-- {
				switch inst.Prefix[i] & 0xFF {
				case 0xF3:
					if !found {
						found = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case 0xF2:
					inst.Prefix[i] &^= x86_PrefixImplicit
				}
			}
			inst.Op = x86_MOVNTSS
		}
	}

	// Add implicit arguments.
	switch inst.Op {
	case x86_MONITOR:
		inst.Args[0] = x86_EDX
		inst.Args[1] = x86_ECX
		inst.Args[2] = x86_EAX
		if inst.AddrSize == 16 {
			inst.Args[2] = x86_AX
		}

	case x86_MWAIT:
		if inst.Mode == 64 {
			inst.Args[0] = x86_RCX
			inst.Args[1] = x86_RAX
		} else {
			inst.Args[0] = x86_ECX
			inst.Args[1] = x86_EAX
		}
	}

	// Adjust which prefixes will be displayed.
	// The rule is to display all the prefixes not implied by
	// the usual instruction display, that is, all the prefixes
	// except the ones with PrefixImplicit set.
	// However, of course, there are exceptions to the rule.
	switch inst.Op {
	case x86_CRC32:
		// CRC32 has a mandatory F2 prefix.
		// If there are multiple F2s and no F3s, the extra F2s do not print.
		// (And Decode has already marked them implicit.)
		// However, if there is an F3 anywhere, then the extra F2s do print.
		// If there are multiple F2 prefixes *and* an (ignored) F3,
		// then libopcodes prints the extra F2s as REPNs.
		if x86_countPrefix(&inst, 0xF2) > 1 {
			x86_unmarkImplicit(&inst, 0xF2)
			x86_markLastImplicit(&inst, 0xF2)
		}

		// An unused data size override should probably be shown,
		// to distinguish DATA16 CRC32B from plain CRC32B,
		// but libopcodes always treats the final override as implicit
		// and the others as explicit.
		x86_unmarkImplicit(&inst, x86_PrefixDataSize)
		x86_markLastImplicit(&inst, x86_PrefixDataSize)

	case x86_CVTSI2SD, x86_CVTSI2SS:
		if !x86_isMem(inst.Args[1]) {
			x86_markLastImplicit(&inst, x86_PrefixDataSize)
		}

	case x86_CVTSD2SI, x86_CVTSS2SI, x86_CVTTSD2SI, x86_CVTTSS2SI,
		x86_ENTER, x86_FLDENV, x86_FNSAVE, x86_FNSTENV, x86_FRSTOR, x86_LGDT, x86_LIDT, x86_LRET,
		x86_POP, x86_PUSH, x86_RET, x86_SGDT, x86_SIDT, x86_SYSRET, x86_XBEGIN:
		x86_markLastImplicit(&inst, x86_PrefixDataSize)

	case x86_LOOP, x86_LOOPE, x86_LOOPNE, x86_MONITOR:
		x86_markLastImplicit(&inst, x86_PrefixAddrSize)

	case x86_MOV:
		// The 16-bit and 32-bit forms of MOV Sreg, dst and MOV src, Sreg
		// cannot be distinguished when src or dst refers to memory, because
		// Sreg is always a 16-bit value, even when we're doing a 32-bit
		// instruction. Because the instruction tables distinguished these two,
		// any operand size prefix has been marked as used (to decide which
		// branch to take). Unmark it, so that it will show up in disassembly,
		// so that the reader can tell the size of memory operand.
		// up with the same arguments
		dst, _ := inst.Args[0].(x86_Reg)
		src, _ := inst.Args[1].(x86_Reg)
		if x86_ES <= src && src <= x86_GS && x86_isMem(inst.Args[0]) || x86_ES <= dst && dst <= x86_GS && x86_isMem(inst.Args[1]) {
			x86_unmarkImplicit(&inst, x86_PrefixDataSize)
		}

	case x86_MOVDQU:
		if x86_countPrefix(&inst, 0xF3) > 1 {
			x86_unmarkImplicit(&inst, 0xF3)
			x86_markLastImplicit(&inst, 0xF3)
		}

	case x86_MOVQ2DQ:
		x86_markLastImplicit(&inst, x86_PrefixDataSize)

	case x86_SLDT, x86_SMSW, x86_STR, x86_FXRSTOR, x86_XRSTOR, x86_XSAVE, x86_XSAVEOPT, x86_CMPXCHG8B:
		if x86_isMem(inst.Args[0]) {
			x86_unmarkImplicit(&inst, x86_PrefixDataSize)
		}

	case x86_SYSEXIT:
		x86_unmarkImplicit(&inst, x86_PrefixDataSize)
	}

	if x86_isCondJmp[inst.Op] || x86_isLoop[inst.Op] || inst.Op == x86_JCXZ || inst.Op == x86_JECXZ || inst.Op == x86_JRCXZ {
		if x86_countPrefix(&inst, x86_PrefixCS) > 0 && x86_countPrefix(&inst, x86_PrefixDS) > 0 {
			for i, p := range inst.Prefix {
				switch p & 0xFFF {
				case x86_PrefixPN, x86_PrefixPT:
					inst.Prefix[i] &= 0xF0FF // cut interpretation bits, producing original segment prefix
				}
			}
		}
	}

	// XACQUIRE/XRELEASE adjustment.
	if inst.Op == x86_MOV {
		// MOV into memory is a candidate for turning REP into XRELEASE.
		// However, if the REP is followed by a REPN, that REPN blocks the
		// conversion.
		haveREPN := false
		for i := len(inst.Prefix) - 1; i >= 0; i-- {
			switch inst.Prefix[i] &^ x86_PrefixIgnored {
			case x86_PrefixREPN:
				haveREPN = true
			case x86_PrefixXRELEASE:
				if haveREPN {
					inst.Prefix[i] = x86_PrefixREP
				}
			}
		}
	}

	// We only format the final F2/F3 as XRELEASE/XACQUIRE.
	haveXA := false
	haveXR := false
	for i := len(inst.Prefix) - 1; i >= 0; i-- {
		switch inst.Prefix[i] &^ x86_PrefixIgnored {
		case x86_PrefixXRELEASE:
			if !haveXR {
				haveXR = true
			} else {
				inst.Prefix[i] = x86_PrefixREP
			}

		case x86_PrefixXACQUIRE:
			if !haveXA {
				haveXA = true
			} else {
				inst.Prefix[i] = x86_PrefixREPN
			}
		}
	}

	// Determine opcode.
	op := strings.ToLower(inst.Op.String())
	if alt := x86_gnuOp[inst.Op]; alt != "" {
		op = alt
	}

	// Determine opcode suffix.
	// Libopcodes omits the suffix if the width of the operation
	// can be inferred from a register arguments. For example,
	// add $1, %ebx has no suffix because you can tell from the
	// 32-bit register destination that it is a 32-bit add,
	// but in addl $1, (%ebx), the destination is memory, so the
	// size is not evident without the l suffix.
	needSuffix := true
SuffixLoop:
	for i, a := range inst.Args {
		if a == nil {
			break
		}
		switch a := a.(type) {
		case x86_Reg:
			switch inst.Op {
			case x86_MOVSX, x86_MOVZX:
				continue

			case x86_SHL, x86_SHR, x86_RCL, x86_RCR, x86_ROL, x86_ROR, x86_SAR:
				if i == 1 {
					// shift count does not tell us operand size
					continue
				}

			case x86_CRC32:
				// The source argument does tell us operand size,
				// but libopcodes still always puts a suffix on crc32.
				continue

			case x86_PUSH, x86_POP:
				// Even though segment registers are 16-bit, push and pop
				// can save/restore them from 32-bit slots, so they
				// do not imply operand size.
				if x86_ES <= a && a <= x86_GS {
					continue
				}

			case x86_CVTSI2SD, x86_CVTSI2SS:
				// The integer register argument takes priority.
				if x86_X0 <= a && a <= x86_X15 {
					continue
				}
			}

			if x86_AL <= a && a <= x86_R15 || x86_ES <= a && a <= x86_GS || x86_X0 <= a && a <= x86_X15 || x86_M0 <= a && a <= x86_M7 {
				needSuffix = false
				break SuffixLoop
			}
		}
	}

	if needSuffix {
		switch inst.Op {
		case x86_CMPXCHG8B, x86_FLDCW, x86_FNSTCW, x86_FNSTSW, x86_LDMXCSR, x86_LLDT, x86_LMSW, x86_LTR, x86_PCLMULQDQ,
			x86_SETA, x86_SETAE, x86_SETB, x86_SETBE, x86_SETE, x86_SETG, x86_SETGE, x86_SETL, x86_SETLE, x86_SETNE, x86_SETNO, x86_SETNP, x86_SETNS, x86_SETO, x86_SETP, x86_SETS,
			x86_SLDT, x86_SMSW, x86_STMXCSR, x86_STR, x86_VERR, x86_VERW:
			// For various reasons, libopcodes emits no suffix for these instructions.

		case x86_CRC32:
			op += x86_byteSizeSuffix(x86_argBytes(&inst, inst.Args[1]))

		case x86_LGDT, x86_LIDT, x86_SGDT, x86_SIDT:
			op += x86_byteSizeSuffix(inst.DataSize / 8)

		case x86_MOVZX, x86_MOVSX:
			// Integer size conversions get two suffixes.
			op = op[:4] + x86_byteSizeSuffix(x86_argBytes(&inst, inst.Args[1])) + x86_byteSizeSuffix(x86_argBytes(&inst, inst.Args[0]))

		case x86_LOOP, x86_LOOPE, x86_LOOPNE:
			// Add w suffix to indicate use of CX register instead of ECX.
			if inst.AddrSize == 16 {
				op += "w"
			}

		case x86_CALL, x86_ENTER, x86_JMP, x86_LCALL, x86_LEAVE, x86_LJMP, x86_LRET, x86_RET, x86_SYSRET, x86_XBEGIN:
			// Add w suffix to indicate use of 16-bit target.
			// Exclude JMP rel8.
			if inst.Opcode>>24 == 0xEB {
				break
			}
			if inst.DataSize == 16 && inst.Mode != 16 {
				x86_markLastImplicit(&inst, x86_PrefixDataSize)
				op += "w"
			} else if inst.Mode == 64 {
				op += "q"
			}

		case x86_FRSTOR, x86_FNSAVE, x86_FNSTENV, x86_FLDENV:
			// Add s suffix to indicate shortened FPU state (I guess).
			if inst.DataSize == 16 {
				op += "s"
			}

		case x86_PUSH, x86_POP:
			if x86_markLastImplicit(&inst, x86_PrefixDataSize) {
				op += x86_byteSizeSuffix(inst.DataSize / 8)
			} else if inst.Mode == 64 {
				op += "q"
			} else {
				op += x86_byteSizeSuffix(inst.MemBytes)
			}

		default:
			if x86_isFloat(inst.Op) {
				// I can't explain any of this, but it's what libopcodes does.
				switch inst.MemBytes {
				default:
					if (inst.Op == x86_FLD || inst.Op == x86_FSTP) && x86_isMem(inst.Args[0]) {
						op += "t"
					}
				case 4:
					if x86_isFloatInt(inst.Op) {
						op += "l"
					} else {
						op += "s"
					}
				case 8:
					if x86_isFloatInt(inst.Op) {
						op += "ll"
					} else {
						op += "l"
					}
				}
				break
			}

			op += x86_byteSizeSuffix(inst.MemBytes)
		}
	}

	// Adjust special case opcodes.
	switch inst.Op {
	case 0:
		if inst.Prefix[0] != 0 {
			return strings.ToLower(inst.Prefix[0].String())
		}

	case x86_INT:
		if inst.Opcode>>24 == 0xCC {
			inst.Args[0] = nil
			op = "int3"
		}

	case x86_CMPPS, x86_CMPPD, x86_CMPSD_XMM, x86_CMPSS:
		imm, ok := inst.Args[2].(x86_Imm)
		if ok && 0 <= imm && imm < 8 {
			inst.Args[2] = nil
			op = x86_cmppsOps[imm] + op[3:]
		}

	case x86_PCLMULQDQ:
		imm, ok := inst.Args[2].(x86_Imm)
		if ok && imm&^0x11 == 0 {
			inst.Args[2] = nil
			op = x86_pclmulqOps[(imm&0x10)>>3|(imm&1)]
		}

	case x86_XLATB:
		if x86_markLastImplicit(&inst, x86_PrefixAddrSize) {
			op = "xlat" // not xlatb
		}
	}

	// Build list of argument strings.
	var (
		usedPrefixes bool     // segment prefixes consumed by Mem formatting
		args         []string // formatted arguments
	)
	for i, a := range inst.Args {
		if a == nil {
			break
		}
		switch inst.Op {
		case x86_MOVSB, x86_MOVSW, x86_MOVSD, x86_MOVSQ, x86_OUTSB, x86_OUTSW, x86_OUTSD:
			if i == 0 {
				usedPrefixes = true // disable use of prefixes for first argument
			} else {
				usedPrefixes = false
			}
		}
		if a == x86_Imm(1) && (inst.Opcode>>24)&^1 == 0xD0 {
			continue
		}
		args = append(args, x86_gnuArg(&inst, a, &usedPrefixes))
	}

	// The default is to print the arguments in reverse Intel order.
	// A few instructions inhibit this behavior.
	switch inst.Op {
	case x86_BOUND, x86_LCALL, x86_ENTER, x86_LJMP:
		// no reverse
	default:
		// reverse args
		for i, j := 0, len(args)-1; i < j; i, j = i+1, j-1 {
			args[i], args[j] = args[j], args[i]
		}
	}

	// Build prefix string.
	// Must be after argument formatting, which can turn off segment prefixes.
	var (
		prefix       = "" // output string
		numAddr      = 0
		numData      = 0
		implicitData = false
	)
	for _, p := range inst.Prefix {
		if p&0xFF == x86_PrefixDataSize && p&x86_PrefixImplicit != 0 {
			implicitData = true
		}
	}
	for _, p := range inst.Prefix {
		if p == 0 {
			break
		}
		if p&x86_PrefixImplicit != 0 {
			continue
		}
		switch p &^ (x86_PrefixIgnored | x86_PrefixInvalid) {
		default:
			if p.IsREX() {
				if p&0xFF == x86_PrefixREX {
					prefix += "rex "
				} else {
					prefix += "rex." + p.String()[4:] + " "
				}
				break
			}
			prefix += strings.ToLower(p.String()) + " "

		case x86_PrefixPN:
			op += ",pn"
			continue

		case x86_PrefixPT:
			op += ",pt"
			continue

		case x86_PrefixAddrSize, x86_PrefixAddr16, x86_PrefixAddr32:
			// For unknown reasons, if the addr16 prefix is repeated,
			// libopcodes displays all but the last as addr32, even though
			// the addressing form used in a memory reference is clearly
			// still 16-bit.
			n := 32
			if inst.Mode == 32 {
				n = 16
			}
			numAddr++
			if x86_countPrefix(&inst, x86_PrefixAddrSize) > numAddr {
				n = inst.Mode
			}
			prefix += fmt.Sprintf("addr%d ", n)
			continue

		case x86_PrefixData16, x86_PrefixData32:
			if implicitData && x86_countPrefix(&inst, x86_PrefixDataSize) > 1 {
				// Similar to the addr32 logic above, but it only kicks in
				// when something used the data size prefix (one is implicit).
				n := 16
				if inst.Mode == 16 {
					n = 32
				}
				numData++
				if x86_countPrefix(&inst, x86_PrefixDataSize) > numData {
					if inst.Mode == 16 {
						n = 16
					} else {
						n = 32
					}
				}
				prefix += fmt.Sprintf("data%d ", n)
				continue
			}
			prefix += strings.ToLower(p.String()) + " "
		}
	}

	// Finally! Put it all together.
	text := prefix + op
	if args != nil {
		text += " "
		// Indirect call/jmp gets a star to distinguish from direct jump address.
		if (inst.Op == x86_CALL || inst.Op == x86_JMP || inst.Op == x86_LJMP || inst.Op == x86_LCALL) && (x86_isMem(inst.Args[0]) || x86_isReg(inst.Args[0])) {
			text += "*"
		}
		text += strings.Join(args, ",")
	}
	return text
}

// gnuArg returns the GNU syntax for the argument x from the instruction inst.
// If *usedPrefixes is false and x is a Mem, then the formatting
// includes any segment prefixes and sets *usedPrefixes to true.
func x86_gnuArg(inst *x86_Inst, x x86_Arg, usedPrefixes *bool) string {
	if x == nil {
		return "<nil>"
	}
	switch x := x.(type) {
	case x86_Reg:
		switch inst.Op {
		case x86_CVTSI2SS, x86_CVTSI2SD, x86_CVTSS2SI, x86_CVTSD2SI, x86_CVTTSD2SI, x86_CVTTSS2SI:
			if inst.DataSize == 16 && x86_EAX <= x && x <= x86_R15L {
				x -= x86_EAX - x86_AX
			}

		case x86_IN, x86_INSB, x86_INSW, x86_INSD, x86_OUT, x86_OUTSB, x86_OUTSW, x86_OUTSD:
			// DX is the port, but libopcodes prints it as if it were a memory reference.
			if x == x86_DX {
				return "(%dx)"
			}
		}
		return x86_gccRegName[x]
	case x86_Mem:
		seg := ""
		var haveCS, haveDS, haveES, haveFS, haveGS, haveSS bool
		switch x.Segment {
		case x86_CS:
			haveCS = true
		case x86_DS:
			haveDS = true
		case x86_ES:
			haveES = true
		case x86_FS:
			haveFS = true
		case x86_GS:
			haveGS = true
		case x86_SS:
			haveSS = true
		}
		switch inst.Op {
		case x86_INSB, x86_INSW, x86_INSD, x86_STOSB, x86_STOSW, x86_STOSD, x86_STOSQ, x86_SCASB, x86_SCASW, x86_SCASD, x86_SCASQ:
			// These do not accept segment prefixes, at least in the GNU rendering.
		default:
			if *usedPrefixes {
				break
			}
			for i := len(inst.Prefix) - 1; i >= 0; i-- {
				p := inst.Prefix[i] &^ x86_PrefixIgnored
				if p == 0 {
					continue
				}
				switch p {
				case x86_PrefixCS:
					if !haveCS {
						haveCS = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case x86_PrefixDS:
					if !haveDS {
						haveDS = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case x86_PrefixES:
					if !haveES {
						haveES = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case x86_PrefixFS:
					if !haveFS {
						haveFS = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case x86_PrefixGS:
					if !haveGS {
						haveGS = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				case x86_PrefixSS:
					if !haveSS {
						haveSS = true
						inst.Prefix[i] |= x86_PrefixImplicit
					}
				}
			}
			*usedPrefixes = true
		}
		if haveCS {
			seg += "%cs:"
		}
		if haveDS {
			seg += "%ds:"
		}
		if haveSS {
			seg += "%ss:"
		}
		if haveES {
			seg += "%es:"
		}
		if haveFS {
			seg += "%fs:"
		}
		if haveGS {
			seg += "%gs:"
		}
		disp := ""
		if x.Disp != 0 {
			disp = fmt.Sprintf("%#x", x.Disp)
		}
		if x.Scale == 0 || x.Index == 0 && x.Scale == 1 && (x.Base == x86_ESP || x.Base == x86_RSP || x.Base == 0 && inst.Mode == 64) {
			if x.Base == 0 {
				return seg + disp
			}
			return fmt.Sprintf("%s%s(%s)", seg, disp, x86_gccRegName[x.Base])
		}
		base := x86_gccRegName[x.Base]
		if x.Base == 0 {
			base = ""
		}
		index := x86_gccRegName[x.Index]
		if x.Index == 0 {
			if inst.AddrSize == 64 {
				index = "%riz"
			} else {
				index = "%eiz"
			}
		}
		if x86_AX <= x.Base && x.Base <= x86_DI {
			// 16-bit addressing - no scale
			return fmt.Sprintf("%s%s(%s,%s)", seg, disp, base, index)
		}
		return fmt.Sprintf("%s%s(%s,%s,%d)", seg, disp, base, index, x.Scale)
	case x86_Rel:
		return fmt.Sprintf(".%+#x", int32(x))
	case x86_Imm:
		if inst.Mode == 32 {
			return fmt.Sprintf("$%#x", uint32(x))
		}
		return fmt.Sprintf("$%#x", int64(x))
	}
	return x.String()
}

var x86_gccRegName = [...]string{
	0:        "REG0",
	x86_AL:   "%al",
	x86_CL:   "%cl",
	x86_BL:   "%bl",
	x86_DL:   "%dl",
	x86_AH:   "%ah",
	x86_CH:   "%ch",
	x86_BH:   "%bh",
	x86_DH:   "%dh",
	x86_SPB:  "%spl",
	x86_BPB:  "%bpl",
	x86_SIB:  "%sil",
	x86_DIB:  "%dil",
	x86_R8B:  "%r8b",
	x86_R9B:  "%r9b",
	x86_R10B: "%r10b",
	x86_R11B: "%r11b",
	x86_R12B: "%r12b",
	x86_R13B: "%r13b",
	x86_R14B: "%r14b",
	x86_R15B: "%r15b",
	x86_AX:   "%ax",
	x86_CX:   "%cx",
	x86_BX:   "%bx",
	x86_DX:   "%dx",
	x86_SP:   "%sp",
	x86_BP:   "%bp",
	x86_SI:   "%si",
	x86_DI:   "%di",
	x86_R8W:  "%r8w",
	x86_R9W:  "%r9w",
	x86_R10W: "%r10w",
	x86_R11W: "%r11w",
	x86_R12W: "%r12w",
	x86_R13W: "%r13w",
	x86_R14W: "%r14w",
	x86_R15W: "%r15w",
	x86_EAX:  "%eax",
	x86_ECX:  "%ecx",
	x86_EDX:  "%edx",
	x86_EBX:  "%ebx",
	x86_ESP:  "%esp",
	x86_EBP:  "%ebp",
	x86_ESI:  "%esi",
	x86_EDI:  "%edi",
	x86_R8L:  "%r8d",
	x86_R9L:  "%r9d",
	x86_R10L: "%r10d",
	x86_R11L: "%r11d",
	x86_R12L: "%r12d",
	x86_R13L: "%r13d",
	x86_R14L: "%r14d",
	x86_R15L: "%r15d",
	x86_RAX:  "%rax",
	x86_RCX:  "%rcx",
	x86_RDX:  "%rdx",
	x86_RBX:  "%rbx",
	x86_RSP:  "%rsp",
	x86_RBP:  "%rbp",
	x86_RSI:  "%rsi",
	x86_RDI:  "%rdi",
	x86_R8:   "%r8",
	x86_R9:   "%r9",
	x86_R10:  "%r10",
	x86_R11:  "%r11",
	x86_R12:  "%r12",
	x86_R13:  "%r13",
	x86_R14:  "%r14",
	x86_R15:  "%r15",
	x86_IP:   "%ip",
	x86_EIP:  "%eip",
	x86_RIP:  "%rip",
	x86_F0:   "%st",
	x86_F1:   "%st(1)",
	x86_F2:   "%st(2)",
	x86_F3:   "%st(3)",
	x86_F4:   "%st(4)",
	x86_F5:   "%st(5)",
	x86_F6:   "%st(6)",
	x86_F7:   "%st(7)",
	x86_M0:   "%mm0",
	x86_M1:   "%mm1",
	x86_M2:   "%mm2",
	x86_M3:   "%mm3",
	x86_M4:   "%mm4",
	x86_M5:   "%mm5",
	x86_M6:   "%mm6",
	x86_M7:   "%mm7",
	x86_X0:   "%xmm0",
	x86_X1:   "%xmm1",
	x86_X2:   "%xmm2",
	x86_X3:   "%xmm3",
	x86_X4:   "%xmm4",
	x86_X5:   "%xmm5",
	x86_X6:   "%xmm6",
	x86_X7:   "%xmm7",
	x86_X8:   "%xmm8",
	x86_X9:   "%xmm9",
	x86_X10:  "%xmm10",
	x86_X11:  "%xmm11",
	x86_X12:  "%xmm12",
	x86_X13:  "%xmm13",
	x86_X14:  "%xmm14",
	x86_X15:  "%xmm15",
	x86_CS:   "%cs",
	x86_SS:   "%ss",
	x86_DS:   "%ds",
	x86_ES:   "%es",
	x86_FS:   "%fs",
	x86_GS:   "%gs",
	x86_GDTR: "%gdtr",
	x86_IDTR: "%idtr",
	x86_LDTR: "%ldtr",
	x86_MSW:  "%msw",
	x86_TASK: "%task",
	x86_CR0:  "%cr0",
	x86_CR1:  "%cr1",
	x86_CR2:  "%cr2",
	x86_CR3:  "%cr3",
	x86_CR4:  "%cr4",
	x86_CR5:  "%cr5",
	x86_CR6:  "%cr6",
	x86_CR7:  "%cr7",
	x86_CR8:  "%cr8",
	x86_CR9:  "%cr9",
	x86_CR10: "%cr10",
	x86_CR11: "%cr11",
	x86_CR12: "%cr12",
	x86_CR13: "%cr13",
	x86_CR14: "%cr14",
	x86_CR15: "%cr15",
	x86_DR0:  "%db0",
	x86_DR1:  "%db1",
	x86_DR2:  "%db2",
	x86_DR3:  "%db3",
	x86_DR4:  "%db4",
	x86_DR5:  "%db5",
	x86_DR6:  "%db6",
	x86_DR7:  "%db7",
	x86_TR0:  "%tr0",
	x86_TR1:  "%tr1",
	x86_TR2:  "%tr2",
	x86_TR3:  "%tr3",
	x86_TR4:  "%tr4",
	x86_TR5:  "%tr5",
	x86_TR6:  "%tr6",
	x86_TR7:  "%tr7",
}

var x86_gnuOp = map[x86_Op]string{
	x86_CBW:       "cbtw",
	x86_CDQ:       "cltd",
	x86_CMPSD:     "cmpsl",
	x86_CMPSD_XMM: "cmpsd",
	x86_CWD:       "cwtd",
	x86_CWDE:      "cwtl",
	x86_CQO:       "cqto",
	x86_INSD:      "insl",
	x86_IRET:      "iretw",
	x86_IRETD:     "iret",
	x86_IRETQ:     "iretq",
	x86_LODSB:     "lods",
	x86_LODSD:     "lods",
	x86_LODSQ:     "lods",
	x86_LODSW:     "lods",
	x86_MOVSD:     "movsl",
	x86_MOVSD_XMM: "movsd",
	x86_OUTSD:     "outsl",
	x86_POPA:      "popaw",
	x86_POPAD:     "popa",
	x86_POPF:      "popfw",
	x86_POPFD:     "popf",
	x86_PUSHA:     "pushaw",
	x86_PUSHAD:    "pusha",
	x86_PUSHF:     "pushfw",
	x86_PUSHFD:    "pushf",
	x86_SCASB:     "scas",
	x86_SCASD:     "scas",
	x86_SCASQ:     "scas",
	x86_SCASW:     "scas",
	x86_STOSB:     "stos",
	x86_STOSD:     "stos",
	x86_STOSQ:     "stos",
	x86_STOSW:     "stos",
	x86_XLATB:     "xlat",
}

var x86_cmppsOps = []string{
	"cmpeq",
	"cmplt",
	"cmple",
	"cmpunord",
	"cmpneq",
	"cmpnlt",
	"cmpnle",
	"cmpord",
}

var x86_pclmulqOps = []string{
	"pclmullqlqdq",
	"pclmulhqlqdq",
	"pclmullqhqdq",
	"pclmulhqhqdq",
}

func x86_countPrefix(inst *x86_Inst, target x86_Prefix) int {
	n := 0
	for _, p := range inst.Prefix {
		if p&0xFF == target&0xFF {
			n++
		}
	}
	return n
}

func x86_markLastImplicit(inst *x86_Inst, prefix x86_Prefix) bool {
	for i := len(inst.Prefix) - 1; i >= 0; i-- {
		p := inst.Prefix[i]
		if p&0xFF == prefix {
			inst.Prefix[i] |= x86_PrefixImplicit
			return true
		}
	}
	return false
}

func x86_unmarkImplicit(inst *x86_Inst, prefix x86_Prefix) {
	for i := len(inst.Prefix) - 1; i >= 0; i-- {
		p := inst.Prefix[i]
		if p&0xFF == prefix {
			inst.Prefix[i] &^= x86_PrefixImplicit
		}
	}
}

func x86_byteSizeSuffix(b int) string {
	switch b {
	case 1:
		return "b"
	case 2:
		return "w"
	case 4:
		return "l"
	case 8:
		return "q"
	}
	return ""
}

func x86_argBytes(inst *x86_Inst, arg x86_Arg) int {
	if x86_isMem(arg) {
		return inst.MemBytes
	}
	return x86_regBytes(arg)
}

func x86_isFloat(op x86_Op) bool {
	switch op {
	case x86_FADD, x86_FCOM, x86_FCOMP, x86_FDIV, x86_FDIVR, x86_FIADD, x86_FICOM, x86_FICOMP, x86_FIDIV, x86_FIDIVR, x86_FILD, x86_FIMUL, x86_FIST, x86_FISTP, x86_FISTTP, x86_FISUB, x86_FISUBR, x86_FLD, x86_FMUL, x86_FST, x86_FSTP, x86_FSUB, x86_FSUBR:
		return true
	}
	return false
}

func x86_isFloatInt(op x86_Op) bool {
	switch op {
	case x86_FIADD, x86_FICOM, x86_FICOMP, x86_FIDIV, x86_FIDIVR, x86_FILD, x86_FIMUL, x86_FIST, x86_FISTP, x86_FISTTP, x86_FISUB, x86_FISUBR:
		return true
	}
	return false
}

/* inst.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Package x86asm implements decoding of x86 machine code.

// An Inst is a single instruction.
type x86_Inst struct {
	Prefix   x86_Prefixes // Prefixes applied to the instruction.
	Op       x86_Op       // Opcode mnemonic
	Opcode   uint32       // Encoded opcode bits, left aligned (first byte is Opcode>>24, etc)
	Args     x86_Args     // Instruction arguments, in Intel order
	Mode     int          // processor mode in bits: 16, 32, or 64
	AddrSize int          // address size in bits: 16, 32, or 64
	DataSize int          // operand size in bits: 16, 32, or 64
	MemBytes int          // size of memory argument in bytes: 1, 2, 4, 8, 16, and so on.
	Len      int          // length of encoded instruction in bytes
}

// Prefixes is an array of prefixes associated with a single instruction.
// The prefixes are listed in the same order as found in the instruction:
// each prefix byte corresponds to one slot in the array. The first zero
// in the array marks the end of the prefixes.
type x86_Prefixes [14]x86_Prefix

// A Prefix represents an Intel instruction prefix.
// The low 8 bits are the actual prefix byte encoding,
// and the top 8 bits contain distinguishing bits and metadata.
type x86_Prefix uint16

const (
	// Metadata about the role of a prefix in an instruction.
	x86_PrefixImplicit x86_Prefix = 0x8000 // prefix is implied by instruction text
	x86_PrefixIgnored  x86_Prefix = 0x4000 // prefix is ignored: either irrelevant or overridden by a later prefix
	x86_PrefixInvalid  x86_Prefix = 0x2000 // prefix makes entire instruction invalid (bad LOCK)

	// Memory segment overrides.
	x86_PrefixES x86_Prefix = 0x26 // ES segment override
	x86_PrefixCS x86_Prefix = 0x2E // CS segment override
	x86_PrefixSS x86_Prefix = 0x36 // SS segment override
	x86_PrefixDS x86_Prefix = 0x3E // DS segment override
	x86_PrefixFS x86_Prefix = 0x64 // FS segment override
	x86_PrefixGS x86_Prefix = 0x65 // GS segment override

	// Branch prediction.
	x86_PrefixPN x86_Prefix = 0x12E // predict not taken (conditional branch only)
	x86_PrefixPT x86_Prefix = 0x13E // predict taken (conditional branch only)

	// Size attributes.
	x86_PrefixDataSize x86_Prefix = 0x66 // operand size override
	x86_PrefixData16   x86_Prefix = 0x166
	x86_PrefixData32   x86_Prefix = 0x266
	x86_PrefixAddrSize x86_Prefix = 0x67 // address size override
	x86_PrefixAddr16   x86_Prefix = 0x167
	x86_PrefixAddr32   x86_Prefix = 0x267

	// One of a kind.
	x86_PrefixLOCK     x86_Prefix = 0xF0 // lock
	x86_PrefixREPN     x86_Prefix = 0xF2 // repeat not zero
	x86_PrefixXACQUIRE x86_Prefix = 0x1F2
	x86_PrefixBND      x86_Prefix = 0x2F2
	x86_PrefixREP      x86_Prefix = 0xF3 // repeat
	x86_PrefixXRELEASE x86_Prefix = 0x1F3

	// The REX prefixes must be in the range [PrefixREX, PrefixREX+0x10).
	// the other bits are set or not according to the intended use.
	x86_PrefixREX  x86_Prefix = 0x40 // REX 64-bit extension prefix
	x86_PrefixREXW x86_Prefix = 0x08 // extension bit W (64-bit instruction width)
	x86_PrefixREXR x86_Prefix = 0x04 // extension bit R (r field in modrm)
	x86_PrefixREXX x86_Prefix = 0x02 // extension bit X (index field in sib)
	x86_PrefixREXB x86_Prefix = 0x01 // extension bit B (r/m field in modrm or base field in sib)
)

// IsREX reports whether p is a REX prefix byte.
func (p x86_Prefix) IsREX() bool {
	return p&0xF0 == x86_PrefixREX
}

func (p x86_Prefix) String() string {
	p &^= x86_PrefixImplicit | x86_PrefixIgnored | x86_PrefixInvalid
	if s := x86_prefixNames[p]; s != "" {
		return s
	}

	if p.IsREX() {
		s := "REX."
		if p&x86_PrefixREXW != 0 {
			s += "W"
		}
		if p&x86_PrefixREXR != 0 {
			s += "R"
		}
		if p&x86_PrefixREXX != 0 {
			s += "X"
		}
		if p&x86_PrefixREXB != 0 {
			s += "B"
		}
		return s
	}

	return fmt.Sprintf("Prefix(%#x)", int(p))
}

// An Op is an x86 opcode.
type x86_Op uint32

func (op x86_Op) String() string {
	i := int(op)
	if i < 0 || i >= len(x86_opNames) || x86_opNames[i] == "" {
		return fmt.Sprintf("Op(%d)", i)
	}
	return x86_opNames[i]
}

// An Args holds the instruction arguments.
// If an instruction has fewer than 4 arguments,
// the final elements in the array are nil.
type x86_Args [4]x86_Arg

// An Arg is a single instruction argument,
// one of these types: Reg, Mem, Imm, Rel.
type x86_Arg interface {
	String() string
	isArg()
}

// Note that the implements of Arg that follow are all sized
// so that on a 64-bit machine the data can be inlined in
// the interface value instead of requiring an allocation.

// A Reg is a single register.
// The zero Reg value has no name but indicates ``no register.''
type x86_Reg uint8

const (
	_ x86_Reg = iota

	// 8-bit
	x86_AL
	x86_CL
	x86_DL
	x86_BL
	x86_AH
	x86_CH
	x86_DH
	x86_BH
	x86_SPB
	x86_BPB
	x86_SIB
	x86_DIB
	x86_R8B
	x86_R9B
	x86_R10B
	x86_R11B
	x86_R12B
	x86_R13B
	x86_R14B
	x86_R15B

	// 16-bit
	x86_AX
	x86_CX
	x86_DX
	x86_BX
	x86_SP
	x86_BP
	x86_SI
	x86_DI
	x86_R8W
	x86_R9W
	x86_R10W
	x86_R11W
	x86_R12W
	x86_R13W
	x86_R14W
	x86_R15W

	// 32-bit
	x86_EAX
	x86_ECX
	x86_EDX
	x86_EBX
	x86_ESP
	x86_EBP
	x86_ESI
	x86_EDI
	x86_R8L
	x86_R9L
	x86_R10L
	x86_R11L
	x86_R12L
	x86_R13L
	x86_R14L
	x86_R15L

	// 64-bit
	x86_RAX
	x86_RCX
	x86_RDX
	x86_RBX
	x86_RSP
	x86_RBP
	x86_RSI
	x86_RDI
	x86_R8
	x86_R9
	x86_R10
	x86_R11
	x86_R12
	x86_R13
	x86_R14
	x86_R15

	// Instruction pointer.
	x86_IP  // 16-bit
	x86_EIP // 32-bit
	x86_RIP // 64-bit

	// 387 floating point registers.
	x86_F0
	x86_F1
	x86_F2
	x86_F3
	x86_F4
	x86_F5
	x86_F6
	x86_F7

	// MMX registers.
	x86_M0
	x86_M1
	x86_M2
	x86_M3
	x86_M4
	x86_M5
	x86_M6
	x86_M7

	// XMM registers.
	x86_X0
	x86_X1
	x86_X2
	x86_X3
	x86_X4
	x86_X5
	x86_X6
	x86_X7
	x86_X8
	x86_X9
	x86_X10
	x86_X11
	x86_X12
	x86_X13
	x86_X14
	x86_X15

	// Segment registers.
	x86_ES
	x86_CS
	x86_SS
	x86_DS
	x86_FS
	x86_GS

	// System registers.
	x86_GDTR
	x86_IDTR
	x86_LDTR
	x86_MSW
	x86_TASK

	// Control registers.
	x86_CR0
	x86_CR1
	x86_CR2
	x86_CR3
	x86_CR4
	x86_CR5
	x86_CR6
	x86_CR7
	x86_CR8
	x86_CR9
	x86_CR10
	x86_CR11
	x86_CR12
	x86_CR13
	x86_CR14
	x86_CR15

	// Debug registers.
	x86_DR0
	x86_DR1
	x86_DR2
	x86_DR3
	x86_DR4
	x86_DR5
	x86_DR6
	x86_DR7
	x86_DR8
	x86_DR9
	x86_DR10
	x86_DR11
	x86_DR12
	x86_DR13
	x86_DR14
	x86_DR15

	// Task registers.
	x86_TR0
	x86_TR1
	x86_TR2
	x86_TR3
	x86_TR4
	x86_TR5
	x86_TR6
	x86_TR7
)

const x86_regMax = x86_TR7

func (x86_Reg) isArg() {}

func (r x86_Reg) String() string {
	i := int(r)
	if i < 0 || i >= len(x86_regNames) || x86_regNames[i] == "" {
		return fmt.Sprintf("Reg(%d)", i)
	}
	return x86_regNames[i]
}

// A Mem is a memory reference.
// The general form is Segment:[Base+Scale*Index+Disp].
type x86_Mem struct {
	Segment x86_Reg
	Base    x86_Reg
	Scale   uint8
	Index   x86_Reg
	Disp    int64
}

func (x86_Mem) isArg() {}

func (m x86_Mem) String() string {
	var base, plus, scale, index, disp string

	if m.Base != 0 {
		base = m.Base.String()
	}
	if m.Scale != 0 {
		if m.Base != 0 {
			plus = "+"
		}
		if m.Scale > 1 {
			scale = fmt.Sprintf("%d*", m.Scale)
		}
		index = m.Index.String()
	}
	if m.Disp != 0 || m.Base == 0 && m.Scale == 0 {
		disp = fmt.Sprintf("%+#x", m.Disp)
	}
	return "[" + base + plus + scale + index + disp + "]"
}

// A Rel is an offset relative to the current instruction pointer.
type x86_Rel int32

func (x86_Rel) isArg() {}

func (r x86_Rel) String() string {
	return fmt.Sprintf(".%+d", r)
}

// An Imm is an integer constant.
type x86_Imm int64

func (x86_Imm) isArg() {}

func (i x86_Imm) String() string {
	return fmt.Sprintf("%#x", int64(i))
}

func (i x86_Inst) String() string {
	var buf bytes.Buffer
	for _, p := range i.Prefix {
		if p == 0 {
			break
		}
		if p&x86_PrefixImplicit != 0 {
			continue
		}
		fmt.Fprintf(&buf, "%v ", p)
	}
	fmt.Fprintf(&buf, "%v", i.Op)
	sep := " "
	for _, v := range i.Args {
		if v == nil {
			break
		}
		fmt.Fprintf(&buf, "%s%v", sep, v)
		sep = ", "
	}
	return buf.String()
}

func x86_isReg(a x86_Arg) bool {
	_, ok := a.(x86_Reg)
	return ok
}

func x86_isSegReg(a x86_Arg) bool {
	r, ok := a.(x86_Reg)
	return ok && x86_ES <= r && r <= x86_GS
}

func x86_isMem(a x86_Arg) bool {
	_, ok := a.(x86_Mem)
	return ok
}

func x86_isImm(a x86_Arg) bool {
	_, ok := a.(x86_Imm)
	return ok
}

func x86_regBytes(a x86_Arg) int {
	r, ok := a.(x86_Reg)
	if !ok {
		return 0
	}
	if x86_AL <= r && r <= x86_R15B {
		return 1
	}
	if x86_AX <= r && r <= x86_R15W {
		return 2
	}
	if x86_EAX <= r && r <= x86_R15L {
		return 4
	}
	if x86_RAX <= r && r <= x86_R15 {
		return 8
	}
	return 0
}

func x86_isSegment(p x86_Prefix) bool {
	switch p {
	case x86_PrefixCS, x86_PrefixDS, x86_PrefixES, x86_PrefixFS, x86_PrefixGS, x86_PrefixSS:
		return true
	}
	return false
}

// The Op definitions and string list are in tables.go.

var x86_prefixNames = map[x86_Prefix]string{
	x86_PrefixCS:       "CS",
	x86_PrefixDS:       "DS",
	x86_PrefixES:       "ES",
	x86_PrefixFS:       "FS",
	x86_PrefixGS:       "GS",
	x86_PrefixSS:       "SS",
	x86_PrefixLOCK:     "LOCK",
	x86_PrefixREP:      "REP",
	x86_PrefixREPN:     "REPN",
	x86_PrefixAddrSize: "ADDRSIZE",
	x86_PrefixDataSize: "DATASIZE",
	x86_PrefixAddr16:   "ADDR16",
	x86_PrefixData16:   "DATA16",
	x86_PrefixAddr32:   "ADDR32",
	x86_PrefixData32:   "DATA32",
	x86_PrefixBND:      "BND",
	x86_PrefixXACQUIRE: "XACQUIRE",
	x86_PrefixXRELEASE: "XRELEASE",
	x86_PrefixREX:      "REX",
	x86_PrefixPT:       "PT",
	x86_PrefixPN:       "PN",
}

var x86_regNames = [...]string{
	x86_AL:   "AL",
	x86_CL:   "CL",
	x86_BL:   "BL",
	x86_DL:   "DL",
	x86_AH:   "AH",
	x86_CH:   "CH",
	x86_BH:   "BH",
	x86_DH:   "DH",
	x86_SPB:  "SPB",
	x86_BPB:  "BPB",
	x86_SIB:  "SIB",
	x86_DIB:  "DIB",
	x86_R8B:  "R8B",
	x86_R9B:  "R9B",
	x86_R10B: "R10B",
	x86_R11B: "R11B",
	x86_R12B: "R12B",
	x86_R13B: "R13B",
	x86_R14B: "R14B",
	x86_R15B: "R15B",
	x86_AX:   "AX",
	x86_CX:   "CX",
	x86_BX:   "BX",
	x86_DX:   "DX",
	x86_SP:   "SP",
	x86_BP:   "BP",
	x86_SI:   "SI",
	x86_DI:   "DI",
	x86_R8W:  "R8W",
	x86_R9W:  "R9W",
	x86_R10W: "R10W",
	x86_R11W: "R11W",
	x86_R12W: "R12W",
	x86_R13W: "R13W",
	x86_R14W: "R14W",
	x86_R15W: "R15W",
	x86_EAX:  "EAX",
	x86_ECX:  "ECX",
	x86_EDX:  "EDX",
	x86_EBX:  "EBX",
	x86_ESP:  "ESP",
	x86_EBP:  "EBP",
	x86_ESI:  "ESI",
	x86_EDI:  "EDI",
	x86_R8L:  "R8L",
	x86_R9L:  "R9L",
	x86_R10L: "R10L",
	x86_R11L: "R11L",
	x86_R12L: "R12L",
	x86_R13L: "R13L",
	x86_R14L: "R14L",
	x86_R15L: "R15L",
	x86_RAX:  "RAX",
	x86_RCX:  "RCX",
	x86_RDX:  "RDX",
	x86_RBX:  "RBX",
	x86_RSP:  "RSP",
	x86_RBP:  "RBP",
	x86_RSI:  "RSI",
	x86_RDI:  "RDI",
	x86_R8:   "R8",
	x86_R9:   "R9",
	x86_R10:  "R10",
	x86_R11:  "R11",
	x86_R12:  "R12",
	x86_R13:  "R13",
	x86_R14:  "R14",
	x86_R15:  "R15",
	x86_IP:   "IP",
	x86_EIP:  "EIP",
	x86_RIP:  "RIP",
	x86_F0:   "F0",
	x86_F1:   "F1",
	x86_F2:   "F2",
	x86_F3:   "F3",
	x86_F4:   "F4",
	x86_F5:   "F5",
	x86_F6:   "F6",
	x86_F7:   "F7",
	x86_M0:   "M0",
	x86_M1:   "M1",
	x86_M2:   "M2",
	x86_M3:   "M3",
	x86_M4:   "M4",
	x86_M5:   "M5",
	x86_M6:   "M6",
	x86_M7:   "M7",
	x86_X0:   "X0",
	x86_X1:   "X1",
	x86_X2:   "X2",
	x86_X3:   "X3",
	x86_X4:   "X4",
	x86_X5:   "X5",
	x86_X6:   "X6",
	x86_X7:   "X7",
	x86_X8:   "X8",
	x86_X9:   "X9",
	x86_X10:  "X10",
	x86_X11:  "X11",
	x86_X12:  "X12",
	x86_X13:  "X13",
	x86_X14:  "X14",
	x86_X15:  "X15",
	x86_CS:   "CS",
	x86_SS:   "SS",
	x86_DS:   "DS",
	x86_ES:   "ES",
	x86_FS:   "FS",
	x86_GS:   "GS",
	x86_GDTR: "GDTR",
	x86_IDTR: "IDTR",
	x86_LDTR: "LDTR",
	x86_MSW:  "MSW",
	x86_TASK: "TASK",
	x86_CR0:  "CR0",
	x86_CR1:  "CR1",
	x86_CR2:  "CR2",
	x86_CR3:  "CR3",
	x86_CR4:  "CR4",
	x86_CR5:  "CR5",
	x86_CR6:  "CR6",
	x86_CR7:  "CR7",
	x86_CR8:  "CR8",
	x86_CR9:  "CR9",
	x86_CR10: "CR10",
	x86_CR11: "CR11",
	x86_CR12: "CR12",
	x86_CR13: "CR13",
	x86_CR14: "CR14",
	x86_CR15: "CR15",
	x86_DR0:  "DR0",
	x86_DR1:  "DR1",
	x86_DR2:  "DR2",
	x86_DR3:  "DR3",
	x86_DR4:  "DR4",
	x86_DR5:  "DR5",
	x86_DR6:  "DR6",
	x86_DR7:  "DR7",
	x86_DR8:  "DR8",
	x86_DR9:  "DR9",
	x86_DR10: "DR10",
	x86_DR11: "DR11",
	x86_DR12: "DR12",
	x86_DR13: "DR13",
	x86_DR14: "DR14",
	x86_DR15: "DR15",
	x86_TR0:  "TR0",
	x86_TR1:  "TR1",
	x86_TR2:  "TR2",
	x86_TR3:  "TR3",
	x86_TR4:  "TR4",
	x86_TR5:  "TR5",
	x86_TR6:  "TR6",
	x86_TR7:  "TR7",
}

/* intel.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// IntelSyntax returns the Intel assembler syntax for the instruction, as defined by Intel's XED tool.
func x86_IntelSyntax(inst x86_Inst) string {
	var iargs []x86_Arg
	for _, a := range inst.Args {
		if a == nil {
			break
		}
		iargs = append(iargs, a)
	}

	switch inst.Op {
	case x86_INSB, x86_INSD, x86_INSW, x86_OUTSB, x86_OUTSD, x86_OUTSW, x86_LOOPNE, x86_JCXZ, x86_JECXZ, x86_JRCXZ, x86_LOOP, x86_LOOPE, x86_MOV, x86_XLATB:
		if inst.Op == x86_MOV && (inst.Opcode>>16)&0xFFFC != 0x0F20 {
			break
		}
		for i, p := range inst.Prefix {
			if p&0xFF == x86_PrefixAddrSize {
				inst.Prefix[i] &^= x86_PrefixImplicit
			}
		}
	}

	switch inst.Op {
	case x86_MOV:
		dst, _ := inst.Args[0].(x86_Reg)
		src, _ := inst.Args[1].(x86_Reg)
		if x86_ES <= dst && dst <= x86_GS && x86_EAX <= src && src <= x86_R15L {
			src -= x86_EAX - x86_AX
			iargs[1] = src
		}
		if x86_ES <= dst && dst <= x86_GS && x86_RAX <= src && src <= x86_R15 {
			src -= x86_RAX - x86_AX
			iargs[1] = src
		}

		if inst.Opcode>>24&^3 == 0xA0 {
			for i, p := range inst.Prefix {
				if p&0xFF == x86_PrefixAddrSize {
					inst.Prefix[i] |= x86_PrefixImplicit
				}
			}
		}
	}

	switch inst.Op {
	case x86_AAM, x86_AAD:
		if imm, ok := iargs[0].(x86_Imm); ok {
			if inst.DataSize == 32 {
				iargs[0] = x86_Imm(uint32(int8(imm)))
			} else if inst.DataSize == 16 {
				iargs[0] = x86_Imm(uint16(int8(imm)))
			}
		}

	case x86_PUSH:
		if imm, ok := iargs[0].(x86_Imm); ok {
			iargs[0] = x86_Imm(uint32(imm))
		}
	}

	for _, p := range inst.Prefix {
		if p&x86_PrefixImplicit != 0 {
			for j, pj := range inst.Prefix {
				if pj&0xFF == p&0xFF {
					inst.Prefix[j] |= x86_PrefixImplicit
				}
			}
		}
	}

	if inst.Op != 0 {
		for i, p := range inst.Prefix {
			switch p &^ x86_PrefixIgnored {
			case x86_PrefixData16, x86_PrefixData32, x86_PrefixCS, x86_PrefixDS, x86_PrefixES, x86_PrefixSS:
				inst.Prefix[i] |= x86_PrefixImplicit
			}
			if p.IsREX() {
				inst.Prefix[i] |= x86_PrefixImplicit
			}
		}
	}

	if x86_isLoop[inst.Op] || inst.Op == x86_JCXZ || inst.Op == x86_JECXZ || inst.Op == x86_JRCXZ {
		for i, p := range inst.Prefix {
			if p == x86_PrefixPT || p == x86_PrefixPN {
				inst.Prefix[i] |= x86_PrefixImplicit
			}
		}
	}

	switch inst.Op {
	case x86_AAA, x86_AAS, x86_CBW, x86_CDQE, x86_CLC, x86_CLD, x86_CLI, x86_CLTS, x86_CMC, x86_CPUID, x86_CQO, x86_CWD, x86_DAA, x86_DAS,
		x86_FDECSTP, x86_FINCSTP, x86_FNCLEX, x86_FNINIT, x86_FNOP, x86_FWAIT, x86_HLT,
		x86_ICEBP, x86_INSB, x86_INSD, x86_INSW, x86_INT, x86_INTO, x86_INVD, x86_IRET, x86_IRETQ,
		x86_LAHF, x86_LEAVE, x86_LRET, x86_MONITOR, x86_MWAIT, x86_NOP, x86_OUTSB, x86_OUTSD, x86_OUTSW,
		x86_PAUSE, x86_POPA, x86_POPF, x86_POPFQ, x86_PUSHA, x86_PUSHF, x86_PUSHFQ,
		x86_RDMSR, x86_RDPMC, x86_RDTSC, x86_RDTSCP, x86_RET, x86_RSM,
		x86_SAHF, x86_STC, x86_STD, x86_STI, x86_SYSENTER, x86_SYSEXIT, x86_SYSRET,
		x86_UD2, x86_WBINVD, x86_WRMSR, x86_XEND, x86_XLATB, x86_XTEST:

		if inst.Op == x86_NOP && inst.Opcode>>24 != 0x90 {
			break
		}
		if inst.Op == x86_RET && inst.Opcode>>24 != 0xC3 {
			break
		}
		if inst.Op == x86_INT && inst.Opcode>>24 != 0xCC {
			break
		}
		if inst.Op == x86_LRET && inst.Opcode>>24 != 0xcb {
			break
		}
		for i, p := range inst.Prefix {
			if p&0xFF == x86_PrefixDataSize {
				inst.Prefix[i] &^= x86_PrefixImplicit | x86_PrefixIgnored
			}
		}

	case 0:
		// ok
	}

	switch inst.Op {
	case x86_INSB, x86_INSD, x86_INSW, x86_OUTSB, x86_OUTSD, x86_OUTSW, x86_MONITOR, x86_MWAIT, x86_XLATB:
		iargs = nil

	case x86_STOSB, x86_STOSW, x86_STOSD, x86_STOSQ:
		iargs = iargs[:1]

	case x86_LODSB, x86_LODSW, x86_LODSD, x86_LODSQ, x86_SCASB, x86_SCASW, x86_SCASD, x86_SCASQ:
		iargs = iargs[1:]
	}

	const (
		haveData16 = 1 << iota
		haveData32
		haveAddr16
		haveAddr32
		haveXacquire
		haveXrelease
		haveLock
		haveHintTaken
		haveHintNotTaken
		haveBnd
	)
	var prefixBits uint32
	prefix := ""
	for _, p := range inst.Prefix {
		if p == 0 {
			break
		}
		if p&0xFF == 0xF3 {
			prefixBits &^= haveBnd
		}
		if p&(x86_PrefixImplicit|x86_PrefixIgnored) != 0 {
			continue
		}
		switch p {
		default:
			prefix += strings.ToLower(p.String()) + " "
		case x86_PrefixCS, x86_PrefixDS, x86_PrefixES, x86_PrefixFS, x86_PrefixGS, x86_PrefixSS:
			if inst.Op == 0 {
				prefix += strings.ToLower(p.String()) + " "
			}
		case x86_PrefixREPN:
			prefix += "repne "
		case x86_PrefixLOCK:
			prefixBits |= haveLock
		case x86_PrefixData16, x86_PrefixDataSize:
			prefixBits |= haveData16
		case x86_PrefixData32:
			prefixBits |= haveData32
		case x86_PrefixAddrSize, x86_PrefixAddr16:
			prefixBits |= haveAddr16
		case x86_PrefixAddr32:
			prefixBits |= haveAddr32
		case x86_PrefixXACQUIRE:
			prefixBits |= haveXacquire
		case x86_PrefixXRELEASE:
			prefixBits |= haveXrelease
		case x86_PrefixPT:
			prefixBits |= haveHintTaken
		case x86_PrefixPN:
			prefixBits |= haveHintNotTaken
		case x86_PrefixBND:
			prefixBits |= haveBnd
		}
	}
	switch inst.Op {
	case x86_JMP:
		if inst.Opcode>>24 == 0xEB {
			prefixBits &^= haveBnd
		}
	case x86_RET, x86_LRET:
		prefixBits &^= haveData16 | haveData32
	}

	if prefixBits&haveXacquire != 0 {
		prefix += "xacquire "
	}
	if prefixBits&haveXrelease != 0 {
		prefix += "xrelease "
	}
	if prefixBits&haveLock != 0 {
		prefix += "lock "
	}
	if prefixBits&haveBnd != 0 {
		prefix += "bnd "
	}
	if prefixBits&haveHintTaken != 0 {
		prefix += "hint-taken "
	}
	if prefixBits&haveHintNotTaken != 0 {
		prefix += "hint-not-taken "
	}
	if prefixBits&haveAddr16 != 0 {
		prefix += "addr16 "
	}
	if prefixBits&haveAddr32 != 0 {
		prefix += "addr32 "
	}
	if prefixBits&haveData16 != 0 {
		prefix += "data16 "
	}
	if prefixBits&haveData32 != 0 {
		prefix += "data32 "
	}

	if inst.Op == 0 {
		if prefix == "" {
			return "<no instruction>"
		}
		return prefix[:len(prefix)-1]
	}

	var args []string
	for _, a := range iargs {
		if a == nil {
			break
		}
		args = append(args, x86_intelArg(&inst, a))
	}

	var op string
	switch inst.Op {
	case x86_NOP:
		if inst.Opcode>>24 == 0x0F {
			if inst.DataSize == 16 {
				args = append(args, "ax")
			} else {
				args = append(args, "eax")
			}
		}

	case x86_BLENDVPD, x86_BLENDVPS, x86_PBLENDVB:
		args = args[:2]

	case x86_INT:
		if inst.Opcode>>24 == 0xCC {
			args = nil
			op = "int3"
		}

	case x86_LCALL, x86_LJMP:
		if len(args) == 2 {
			args[0], args[1] = args[1], args[0]
		}

	case x86_FCHS, x86_FABS, x86_FTST, x86_FLDPI, x86_FLDL2E, x86_FLDLG2, x86_F2XM1, x86_FXAM, x86_FLD1, x86_FLDL2T, x86_FSQRT, x86_FRNDINT, x86_FCOS, x86_FSIN:
		if len(args) == 0 {
			args = append(args, "st0")
		}

	case x86_FPTAN, x86_FSINCOS, x86_FUCOMPP, x86_FCOMPP, x86_FYL2X, x86_FPATAN, x86_FXTRACT, x86_FPREM1, x86_FPREM, x86_FYL2XP1, x86_FSCALE:
		if len(args) == 0 {
			args = []string{"st0", "st1"}
		}

	case x86_FST, x86_FSTP, x86_FISTTP, x86_FIST, x86_FISTP, x86_FBSTP:
		if len(args) == 1 {
			args = append(args, "st0")
		}

	case x86_FLD, x86_FXCH, x86_FCOM, x86_FCOMP, x86_FIADD, x86_FIMUL, x86_FICOM, x86_FICOMP, x86_FISUBR, x86_FIDIV, x86_FUCOM, x86_FUCOMP, x86_FILD, x86_FBLD, x86_FADD, x86_FMUL, x86_FSUB, x86_FSUBR, x86_FISUB, x86_FDIV, x86_FDIVR, x86_FIDIVR:
		if len(args) == 1 {
			args = []string{"st0", args[0]}
		}

	case x86_MASKMOVDQU, x86_MASKMOVQ, x86_XLATB, x86_OUTSB, x86_OUTSW, x86_OUTSD:
	FixSegment:
		for i := len(inst.Prefix) - 1; i >= 0; i-- {
			p := inst.Prefix[i] & 0xFF
			switch p {
			case x86_PrefixCS, x86_PrefixES, x86_PrefixFS, x86_PrefixGS, x86_PrefixSS:
				if inst.Mode != 64 || p == x86_PrefixFS || p == x86_PrefixGS {
					args = append(args, strings.ToLower((inst.Prefix[i] & 0xFF).String()))
					break FixSegment
				}
			case x86_PrefixDS:
				if inst.Mode != 64 {
					break FixSegment
				}
			}
		}
	}

	if op == "" {
		op = x86_intelOp[inst.Op]
	}
	if op == "" {
		op = strings.ToLower(inst.Op.String())
	}
	if args != nil {
		op += " " + strings.Join(args, ", ")
	}
	return prefix + op
}

func x86_intelArg(inst *x86_Inst, arg x86_Arg) string {
	switch a := arg.(type) {
	case x86_Imm:
		if inst.Mode == 32 {
			return fmt.Sprintf("%#x", uint32(a))
		}
		if x86_Imm(int32(a)) == a {
			return fmt.Sprintf("%#x", int64(a))
		}
		return fmt.Sprintf("%#x", uint64(a))
	case x86_Mem:
		if a.Base == x86_EIP {
			a.Base = x86_RIP
		}
		prefix := ""
		switch inst.MemBytes {
		case 1:
			prefix = "byte "
		case 2:
			prefix = "word "
		case 4:
			prefix = "dword "
		case 8:
			prefix = "qword "
		case 16:
			prefix = "xmmword "
		}
		switch inst.Op {
		case x86_INVLPG:
			prefix = "byte "
		case x86_STOSB, x86_MOVSB, x86_CMPSB, x86_LODSB, x86_SCASB:
			prefix = "byte "
		case x86_STOSW, x86_MOVSW, x86_CMPSW, x86_LODSW, x86_SCASW:
			prefix = "word "
		case x86_STOSD, x86_MOVSD, x86_CMPSD, x86_LODSD, x86_SCASD:
			prefix = "dword "
		case x86_STOSQ, x86_MOVSQ, x86_CMPSQ, x86_LODSQ, x86_SCASQ:
			prefix = "qword "
		case x86_LAR:
			prefix = "word "
		case x86_BOUND:
			if inst.Mode == 32 {
				prefix = "qword "
			} else {
				prefix = "dword "
			}
		case x86_PREFETCHW, x86_PREFETCHNTA, x86_PREFETCHT0, x86_PREFETCHT1, x86_PREFETCHT2, x86_CLFLUSH:
			prefix = "zmmword "
		}
		switch inst.Op {
		case x86_MOVSB, x86_MOVSW, x86_MOVSD, x86_MOVSQ, x86_CMPSB, x86_CMPSW, x86_CMPSD, x86_CMPSQ, x86_STOSB, x86_STOSW, x86_STOSD, x86_STOSQ, x86_SCASB, x86_SCASW, x86_SCASD, x86_SCASQ, x86_LODSB, x86_LODSW, x86_LODSD, x86_LODSQ:
			switch a.Base {
			case x86_DI, x86_EDI, x86_RDI:
				if a.Segment == x86_ES {
					a.Segment = 0
				}
			case x86_SI, x86_ESI, x86_RSI:
				if a.Segment == x86_DS {
					a.Segment = 0
				}
			}
		case x86_LEA:
			a.Segment = 0
		default:
			switch a.Base {
			case x86_SP, x86_ESP, x86_RSP, x86_BP, x86_EBP, x86_RBP:
				if a.Segment == x86_SS {
					a.Segment = 0
				}
			default:
				if a.Segment == x86_DS {
					a.Segment = 0
				}
			}
		}

		if inst.Mode == 64 && a.Segment != x86_FS && a.Segment != x86_GS {
			a.Segment = 0
		}

		prefix += "ptr "
		if a.Segment != 0 {
			prefix += strings.ToLower(a.Segment.String()) + ":"
		}
		prefix += "["
		if a.Base != 0 {
			prefix += x86_intelArg(inst, a.Base)
		}
		if a.Scale != 0 && a.Index != 0 {
			if a.Base != 0 {
				prefix += "+"
			}
			prefix += fmt.Sprintf("%s*%d", x86_intelArg(inst, a.Index), a.Scale)
		}
		if a.Disp != 0 {
			if prefix[len(prefix)-1] == '[' && (a.Disp >= 0 || int64(int32(a.Disp)) != a.Disp) {
				prefix += fmt.Sprintf("%#x", uint64(a.Disp))
			} else {
				prefix += fmt.Sprintf("%+#x", a.Disp)
			}
		}
		prefix += "]"
		return prefix
	case x86_Rel:
		return fmt.Sprintf(".%+#x", int64(a))
	case x86_Reg:
		if int(a) < len(x86_intelReg) && x86_intelReg[a] != "" {
			return x86_intelReg[a]
		}
	}
	return strings.ToLower(arg.String())
}

var x86_intelOp = map[x86_Op]string{
	x86_JAE:       "jnb",
	x86_JA:        "jnbe",
	x86_JGE:       "jnl",
	x86_JNE:       "jnz",
	x86_JG:        "jnle",
	x86_JE:        "jz",
	x86_SETAE:     "setnb",
	x86_SETA:      "setnbe",
	x86_SETGE:     "setnl",
	x86_SETNE:     "setnz",
	x86_SETG:      "setnle",
	x86_SETE:      "setz",
	x86_CMOVAE:    "cmovnb",
	x86_CMOVA:     "cmovnbe",
	x86_CMOVGE:    "cmovnl",
	x86_CMOVNE:    "cmovnz",
	x86_CMOVG:     "cmovnle",
	x86_CMOVE:     "cmovz",
	x86_LCALL:     "call far",
	x86_LJMP:      "jmp far",
	x86_LRET:      "ret far",
	x86_ICEBP:     "int1",
	x86_MOVSD_XMM: "movsd",
	x86_XLATB:     "xlat",
}

var x86_intelReg = [...]string{
	x86_F0:  "st0",
	x86_F1:  "st1",
	x86_F2:  "st2",
	x86_F3:  "st3",
	x86_F4:  "st4",
	x86_F5:  "st5",
	x86_F6:  "st6",
	x86_F7:  "st7",
	x86_M0:  "mmx0",
	x86_M1:  "mmx1",
	x86_M2:  "mmx2",
	x86_M3:  "mmx3",
	x86_M4:  "mmx4",
	x86_M5:  "mmx5",
	x86_M6:  "mmx6",
	x86_M7:  "mmx7",
	x86_X0:  "xmm0",
	x86_X1:  "xmm1",
	x86_X2:  "xmm2",
	x86_X3:  "xmm3",
	x86_X4:  "xmm4",
	x86_X5:  "xmm5",
	x86_X6:  "xmm6",
	x86_X7:  "xmm7",
	x86_X8:  "xmm8",
	x86_X9:  "xmm9",
	x86_X10: "xmm10",
	x86_X11: "xmm11",
	x86_X12: "xmm12",
	x86_X13: "xmm13",
	x86_X14: "xmm14",
	x86_X15: "xmm15",

	// TODO: Maybe the constants are named wrong.
	x86_SPB: "spl",
	x86_BPB: "bpl",
	x86_SIB: "sil",
	x86_DIB: "dil",

	x86_R8L:  "r8d",
	x86_R9L:  "r9d",
	x86_R10L: "r10d",
	x86_R11L: "r11d",
	x86_R12L: "r12d",
	x86_R13L: "r13d",
	x86_R14L: "r14d",
	x86_R15L: "r15d",
}

/* plan9x.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// plan9Syntax returns the Go assembler syntax for the instruction.
// The syntax was originally defined by Plan 9.
// The pc is the program counter of the instruction, used for expanding
// PC-relative addresses into absolute ones.
// The symname function queries the symbol table for the program
// being disassembled. Given a target address it returns the name and base
// address of the symbol containing the target, if any; otherwise it returns "", 0.
func x86_plan9Syntax(inst x86_Inst, pc uint64, symname func(uint64) (string, uint64)) string {
	if symname == nil {
		symname = func(uint64) (string, uint64) { return "", 0 }
	}
	var args []string
	for i := len(inst.Args) - 1; i >= 0; i-- {
		a := inst.Args[i]
		if a == nil {
			continue
		}
		args = append(args, x86_plan9Arg(&inst, pc, symname, a))
	}

	var last x86_Prefix
	for _, p := range inst.Prefix {
		if p == 0 || p.IsREX() {
			break
		}
		last = p
	}

	prefix := ""
	switch last & 0xFF {
	case 0, 0x66, 0x67:
		// ignore
	case x86_PrefixREPN:
		prefix += "REPNE "
	default:
		prefix += last.String() + " "
	}

	op := inst.Op.String()
	if x86_plan9Suffix[inst.Op] {
		switch inst.DataSize {
		case 8:
			op += "B"
		case 16:
			op += "W"
		case 32:
			op += "L"
		case 64:
			op += "Q"
		}
	}

	if args != nil {
		op += " " + strings.Join(args, ", ")
	}

	return prefix + op
}

func x86_plan9Arg(inst *x86_Inst, pc uint64, symname func(uint64) (string, uint64), arg x86_Arg) string {
	switch a := arg.(type) {
	case x86_Reg:
		return x86_plan9Reg[a]
	case x86_Rel:
		if pc == 0 {
			break
		}
		// If the absolute address is the start of a symbol, use the name.
		// Otherwise use the raw address, so that things like relative
		// jumps show up as JMP 0x123 instead of JMP f+10(SB).
		// It is usually easier to search for 0x123 than to do the mental
		// arithmetic to find f+10.
		addr := pc + uint64(inst.Len) + uint64(a)
		if s, base := symname(addr); s != "" && addr == base {
			return fmt.Sprintf("%s(SB)", s)
		}
		return fmt.Sprintf("%#x", addr)

	case x86_Imm:
		if s, base := symname(uint64(a)); s != "" {
			suffix := ""
			if uint64(a) != base {
				suffix = fmt.Sprintf("%+d", uint64(a)-base)
			}
			return fmt.Sprintf("$%s%s(SB)", s, suffix)
		}
		if inst.Mode == 32 {
			return fmt.Sprintf("$%#x", uint32(a))
		}
		if x86_Imm(int32(a)) == a {
			return fmt.Sprintf("$%#x", int64(a))
		}
		return fmt.Sprintf("$%#x", uint64(a))
	case x86_Mem:
		if a.Segment == 0 && a.Disp != 0 && a.Base == 0 && (a.Index == 0 || a.Scale == 0) {
			if s, base := symname(uint64(a.Disp)); s != "" {
				suffix := ""
				if uint64(a.Disp) != base {
					suffix = fmt.Sprintf("%+d", uint64(a.Disp)-base)
				}
				return fmt.Sprintf("%s%s(SB)", s, suffix)
			}
		}
		s := ""
		if a.Segment != 0 {
			s += fmt.Sprintf("%s:", x86_plan9Reg[a.Segment])
		}
		if a.Disp != 0 {
			s += fmt.Sprintf("%#x", a.Disp)
		} else {
			s += "0"
		}
		if a.Base != 0 {
			s += fmt.Sprintf("(%s)", x86_plan9Reg[a.Base])
		}
		if a.Index != 0 && a.Scale != 0 {
			s += fmt.Sprintf("(%s*%d)", x86_plan9Reg[a.Index], a.Scale)
		}
		return s
	}
	return arg.String()
}

var x86_plan9Suffix = [x86_maxOp + 1]bool{
	x86_ADC:       true,
	x86_ADD:       true,
	x86_AND:       true,
	x86_BSF:       true,
	x86_BSR:       true,
	x86_BT:        true,
	x86_BTC:       true,
	x86_BTR:       true,
	x86_BTS:       true,
	x86_CMP:       true,
	x86_CMPXCHG:   true,
	x86_CVTSI2SD:  true,
	x86_CVTSI2SS:  true,
	x86_CVTSD2SI:  true,
	x86_CVTSS2SI:  true,
	x86_CVTTSD2SI: true,
	x86_CVTTSS2SI: true,
	x86_DEC:       true,
	x86_DIV:       true,
	x86_FLDENV:    true,
	x86_FRSTOR:    true,
	x86_IDIV:      true,
	x86_IMUL:      true,
	x86_IN:        true,
	x86_INC:       true,
	x86_LEA:       true,
	x86_MOV:       true,
	x86_MOVNTI:    true,
	x86_MUL:       true,
	x86_NEG:       true,
	x86_NOP:       true,
	x86_NOT:       true,
	x86_OR:        true,
	x86_OUT:       true,
	x86_POP:       true,
	x86_POPA:      true,
	x86_PUSH:      true,
	x86_PUSHA:     true,
	x86_RCL:       true,
	x86_RCR:       true,
	x86_ROL:       true,
	x86_ROR:       true,
	x86_SAR:       true,
	x86_SBB:       true,
	x86_SHL:       true,
	x86_SHLD:      true,
	x86_SHR:       true,
	x86_SHRD:      true,
	x86_SUB:       true,
	x86_TEST:      true,
	x86_XADD:      true,
	x86_XCHG:      true,
	x86_XOR:       true,
}

var x86_plan9Reg = [...]string{
	x86_AL:   "AL",
	x86_CL:   "CL",
	x86_BL:   "BL",
	x86_DL:   "DL",
	x86_AH:   "AH",
	x86_CH:   "CH",
	x86_BH:   "BH",
	x86_DH:   "DH",
	x86_SPB:  "SP",
	x86_BPB:  "BP",
	x86_SIB:  "SI",
	x86_DIB:  "DI",
	x86_R8B:  "R8",
	x86_R9B:  "R9",
	x86_R10B: "R10",
	x86_R11B: "R11",
	x86_R12B: "R12",
	x86_R13B: "R13",
	x86_R14B: "R14",
	x86_R15B: "R15",
	x86_AX:   "AX",
	x86_CX:   "CX",
	x86_BX:   "BX",
	x86_DX:   "DX",
	x86_SP:   "SP",
	x86_BP:   "BP",
	x86_SI:   "SI",
	x86_DI:   "DI",
	x86_R8W:  "R8",
	x86_R9W:  "R9",
	x86_R10W: "R10",
	x86_R11W: "R11",
	x86_R12W: "R12",
	x86_R13W: "R13",
	x86_R14W: "R14",
	x86_R15W: "R15",
	x86_EAX:  "AX",
	x86_ECX:  "CX",
	x86_EDX:  "DX",
	x86_EBX:  "BX",
	x86_ESP:  "SP",
	x86_EBP:  "BP",
	x86_ESI:  "SI",
	x86_EDI:  "DI",
	x86_R8L:  "R8",
	x86_R9L:  "R9",
	x86_R10L: "R10",
	x86_R11L: "R11",
	x86_R12L: "R12",
	x86_R13L: "R13",
	x86_R14L: "R14",
	x86_R15L: "R15",
	x86_RAX:  "AX",
	x86_RCX:  "CX",
	x86_RDX:  "DX",
	x86_RBX:  "BX",
	x86_RSP:  "SP",
	x86_RBP:  "BP",
	x86_RSI:  "SI",
	x86_RDI:  "DI",
	x86_R8:   "R8",
	x86_R9:   "R9",
	x86_R10:  "R10",
	x86_R11:  "R11",
	x86_R12:  "R12",
	x86_R13:  "R13",
	x86_R14:  "R14",
	x86_R15:  "R15",
	x86_IP:   "IP",
	x86_EIP:  "IP",
	x86_RIP:  "IP",
	x86_F0:   "F0",
	x86_F1:   "F1",
	x86_F2:   "F2",
	x86_F3:   "F3",
	x86_F4:   "F4",
	x86_F5:   "F5",
	x86_F6:   "F6",
	x86_F7:   "F7",
	x86_M0:   "M0",
	x86_M1:   "M1",
	x86_M2:   "M2",
	x86_M3:   "M3",
	x86_M4:   "M4",
	x86_M5:   "M5",
	x86_M6:   "M6",
	x86_M7:   "M7",
	x86_X0:   "X0",
	x86_X1:   "X1",
	x86_X2:   "X2",
	x86_X3:   "X3",
	x86_X4:   "X4",
	x86_X5:   "X5",
	x86_X6:   "X6",
	x86_X7:   "X7",
	x86_X8:   "X8",
	x86_X9:   "X9",
	x86_X10:  "X10",
	x86_X11:  "X11",
	x86_X12:  "X12",
	x86_X13:  "X13",
	x86_X14:  "X14",
	x86_X15:  "X15",
	x86_CS:   "CS",
	x86_SS:   "SS",
	x86_DS:   "DS",
	x86_ES:   "ES",
	x86_FS:   "FS",
	x86_GS:   "GS",
	x86_GDTR: "GDTR",
	x86_IDTR: "IDTR",
	x86_LDTR: "LDTR",
	x86_MSW:  "MSW",
	x86_TASK: "TASK",
	x86_CR0:  "CR0",
	x86_CR1:  "CR1",
	x86_CR2:  "CR2",
	x86_CR3:  "CR3",
	x86_CR4:  "CR4",
	x86_CR5:  "CR5",
	x86_CR6:  "CR6",
	x86_CR7:  "CR7",
	x86_CR8:  "CR8",
	x86_CR9:  "CR9",
	x86_CR10: "CR10",
	x86_CR11: "CR11",
	x86_CR12: "CR12",
	x86_CR13: "CR13",
	x86_CR14: "CR14",
	x86_CR15: "CR15",
	x86_DR0:  "DR0",
	x86_DR1:  "DR1",
	x86_DR2:  "DR2",
	x86_DR3:  "DR3",
	x86_DR4:  "DR4",
	x86_DR5:  "DR5",
	x86_DR6:  "DR6",
	x86_DR7:  "DR7",
	x86_DR8:  "DR8",
	x86_DR9:  "DR9",
	x86_DR10: "DR10",
	x86_DR11: "DR11",
	x86_DR12: "DR12",
	x86_DR13: "DR13",
	x86_DR14: "DR14",
	x86_DR15: "DR15",
	x86_TR0:  "TR0",
	x86_TR1:  "TR1",
	x86_TR2:  "TR2",
	x86_TR3:  "TR3",
	x86_TR4:  "TR4",
	x86_TR5:  "TR5",
	x86_TR6:  "TR6",
	x86_TR7:  "TR7",
}

/* tables.go */

// DO NOT EDIT
// generated by: x86map -fmt=decoder ../x86.csv

var x86_decoder = [...]uint16{
	uint16(x86_xFail),
	/*1*/ uint16(x86_xCondByte), 243,
	0x00, 490,
	0x01, 496,
	0x02, 525,
	0x03, 531,
	0x04, 560,
	0x05, 566,
	0x06, 595,
	0x07, 602,
	0x08, 609,
	0x09, 615,
	0x0A, 644,
	0x0B, 650,
	0x0C, 679,
	0x0D, 685,
	0x0E, 714,
	0x0F, 721,
	0x10, 8026,
	0x11, 8032,
	0x12, 8061,
	0x13, 8067,
	0x14, 8096,
	0x15, 8102,
	0x16, 8131,
	0x17, 8138,
	0x18, 8145,
	0x19, 8151,
	0x1A, 8180,
	0x1B, 8186,
	0x1C, 8215,
	0x1D, 8221,
	0x1E, 8250,
	0x1F, 8257,
	0x20, 8264,
	0x21, 8270,
	0x22, 8299,
	0x23, 8305,
	0x24, 8334,
	0x25, 8340,
	0x27, 8369,
	0x28, 8375,
	0x29, 8381,
	0x2A, 8410,
	0x2B, 8416,
	0x2C, 8445,
	0x2D, 8451,
	0x2F, 8480,
	0x30, 8486,
	0x31, 8492,
	0x32, 8521,
	0x33, 8527,
	0x34, 8556,
	0x35, 8562,
	0x37, 8591,
	0x38, 8597,
	0x39, 8603,
	0x3A, 8632,
	0x3B, 8638,
	0x3C, 8667,
	0x3D, 8673,
	0x3F, 8702,
	0x40, 8708,
	0x41, 8708,
	0x42, 8708,
	0x43, 8708,
	0x44, 8708,
	0x45, 8708,
	0x46, 8708,
	0x47, 8708,
	0x48, 8723,
	0x49, 8723,
	0x4a, 8723,
	0x4b, 8723,
	0x4c, 8723,
	0x4d, 8723,
	0x4e, 8723,
	0x4f, 8723,
	0x50, 8738,
	0x51, 8738,
	0x52, 8738,
	0x53, 8738,
	0x54, 8738,
	0x55, 8738,
	0x56, 8738,
	0x57, 8738,
	0x58, 8765,
	0x59, 8765,
	0x5a, 8765,
	0x5b, 8765,
	0x5c, 8765,
	0x5d, 8765,
	0x5e, 8765,
	0x5f, 8765,
	0x60, 8792,
	0x61, 8805,
	0x62, 8818,
	0x63, 8837,
	0x68, 8868,
	0x69, 8887,
	0x6A, 8922,
	0x6B, 8927,
	0x6C, 8962,
	0x6D, 8965,
	0x6E, 8978,
	0x6F, 8981,
	0x70, 8994,
	0x71, 8999,
	0x72, 9004,
	0x73, 9009,
	0x74, 9014,
	0x75, 9019,
	0x76, 9024,
	0x77, 9029,
	0x78, 9034,
	0x79, 9039,
	0x7A, 9044,
	0x7B, 9049,
	0x7C, 9054,
	0x7D, 9059,
	0x7E, 9064,
	0x7F, 9069,
	0x80, 9074,
	0x81, 9131,
	0x83, 9372,
	0x84, 9613,
	0x85, 9619,
	0x86, 9648,
	0x87, 9654,
	0x88, 9683,
	0x89, 9689,
	0x8A, 9711,
	0x8B, 9717,
	0x8C, 9739,
	0x8D, 9768,
	0x8E, 9797,
	0x8F, 9826,
	0x90, 9862,
	0x91, 9862,
	0x92, 9862,
	0x93, 9862,
	0x94, 9862,
	0x95, 9862,
	0x96, 9862,
	0x97, 9862,
	0x98, 9888,
	0x99, 9908,
	0x9A, 9928,
	0x9B, 9945,
	0x9C, 9948,
	0x9D, 9971,
	0x9E, 9994,
	0x9F, 9997,
	0xA0, 10000,
	0xA1, 10019,
	0xA2, 10041,
	0xA3, 10060,
	0xA4, 10082,
	0xA5, 10085,
	0xA6, 10105,
	0xA7, 10108,
	0xA8, 10128,
	0xA9, 10134,
	0xAA, 10163,
	0xAB, 10166,
	0xAC, 10186,
	0xAD, 10189,
	0xAE, 10209,
	0xAF, 10212,
	0xb0, 10232,
	0xb1, 10232,
	0xb2, 10232,
	0xb3, 10232,
	0xb4, 10232,
	0xb5, 10232,
	0xb6, 10232,
	0xb7, 10232,
	0xb8, 10238,
	0xb9, 10238,
	0xba, 10238,
	0xbb, 10238,
	0xbc, 10238,
	0xbd, 10238,
	0xbe, 10238,
	0xbf, 10238,
	0xC0, 10267,
	0xC1, 10318,
	0xC2, 10516,
	0xC3, 10521,
	0xC4, 10524,
	0xC5, 10543,
	0xC6, 10562,
	0xC7, 10586,
	0xC8, 10647,
	0xC9, 10654,
	0xCA, 10677,
	0xCB, 10682,
	0xCC, 10685,
	0xCD, 10689,
	0xCE, 10694,
	0xCF, 10700,
	0xD0, 10720,
	0xD1, 10764,
	0xD2, 10955,
	0xD3, 10999,
	0xD4, 11190,
	0xD5, 11198,
	0xD7, 11206,
	0xD8, 11219,
	0xD9, 11428,
	0xDA, 11637,
	0xDB, 11769,
	0xDC, 11940,
	0xDD, 12109,
	0xDE, 12248,
	0xDF, 12422,
	0xE0, 12533,
	0xE1, 12538,
	0xE2, 12543,
	0xE3, 12548,
	0xE4, 12574,
	0xE5, 12580,
	0xE6, 12602,
	0xE7, 12608,
	0xE8, 12630,
	0xE9, 12661,
	0xEA, 12692,
	0xEB, 12709,
	0xEC, 12714,
	0xED, 12719,
	0xEE, 12738,
	0xEF, 12743,
	0xF1, 12762,
	0xF4, 12765,
	0xF5, 12768,
	0xF6, 12771,
	0xF7, 12810,
	0xF8, 12986,
	0xF9, 12989,
	0xFA, 12992,
	0xFB, 12995,
	0xFC, 12998,
	0xFD, 13001,
	0xFE, 13004,
	0xFF, 13021,
	uint16(x86_xFail),
	/*490*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*492*/ uint16(x86_xReadSlashR),
	/*493*/ uint16(x86_xArgRM8),
	/*494*/ uint16(x86_xArgR8),
	/*495*/ uint16(x86_xMatch),
	/*496*/ uint16(x86_xCondIs64), 499, 515,
	/*499*/ uint16(x86_xCondDataSize), 503, 509, 0,
	/*503*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*505*/ uint16(x86_xReadSlashR),
	/*506*/ uint16(x86_xArgRM16),
	/*507*/ uint16(x86_xArgR16),
	/*508*/ uint16(x86_xMatch),
	/*509*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*511*/ uint16(x86_xReadSlashR),
	/*512*/ uint16(x86_xArgRM32),
	/*513*/ uint16(x86_xArgR32),
	/*514*/ uint16(x86_xMatch),
	/*515*/ uint16(x86_xCondDataSize), 503, 509, 519,
	/*519*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*521*/ uint16(x86_xReadSlashR),
	/*522*/ uint16(x86_xArgRM64),
	/*523*/ uint16(x86_xArgR64),
	/*524*/ uint16(x86_xMatch),
	/*525*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*527*/ uint16(x86_xReadSlashR),
	/*528*/ uint16(x86_xArgR8),
	/*529*/ uint16(x86_xArgRM8),
	/*530*/ uint16(x86_xMatch),
	/*531*/ uint16(x86_xCondIs64), 534, 550,
	/*534*/ uint16(x86_xCondDataSize), 538, 544, 0,
	/*538*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*540*/ uint16(x86_xReadSlashR),
	/*541*/ uint16(x86_xArgR16),
	/*542*/ uint16(x86_xArgRM16),
	/*543*/ uint16(x86_xMatch),
	/*544*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*546*/ uint16(x86_xReadSlashR),
	/*547*/ uint16(x86_xArgR32),
	/*548*/ uint16(x86_xArgRM32),
	/*549*/ uint16(x86_xMatch),
	/*550*/ uint16(x86_xCondDataSize), 538, 544, 554,
	/*554*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*556*/ uint16(x86_xReadSlashR),
	/*557*/ uint16(x86_xArgR64),
	/*558*/ uint16(x86_xArgRM64),
	/*559*/ uint16(x86_xMatch),
	/*560*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*562*/ uint16(x86_xReadIb),
	/*563*/ uint16(x86_xArgAL),
	/*564*/ uint16(x86_xArgImm8u),
	/*565*/ uint16(x86_xMatch),
	/*566*/ uint16(x86_xCondIs64), 569, 585,
	/*569*/ uint16(x86_xCondDataSize), 573, 579, 0,
	/*573*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*575*/ uint16(x86_xReadIw),
	/*576*/ uint16(x86_xArgAX),
	/*577*/ uint16(x86_xArgImm16),
	/*578*/ uint16(x86_xMatch),
	/*579*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*581*/ uint16(x86_xReadId),
	/*582*/ uint16(x86_xArgEAX),
	/*583*/ uint16(x86_xArgImm32),
	/*584*/ uint16(x86_xMatch),
	/*585*/ uint16(x86_xCondDataSize), 573, 579, 589,
	/*589*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*591*/ uint16(x86_xReadId),
	/*592*/ uint16(x86_xArgRAX),
	/*593*/ uint16(x86_xArgImm32),
	/*594*/ uint16(x86_xMatch),
	/*595*/ uint16(x86_xCondIs64), 598, 0,
	/*598*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*600*/ uint16(x86_xArgES),
	/*601*/ uint16(x86_xMatch),
	/*602*/ uint16(x86_xCondIs64), 605, 0,
	/*605*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*607*/ uint16(x86_xArgES),
	/*608*/ uint16(x86_xMatch),
	/*609*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*611*/ uint16(x86_xReadSlashR),
	/*612*/ uint16(x86_xArgRM8),
	/*613*/ uint16(x86_xArgR8),
	/*614*/ uint16(x86_xMatch),
	/*615*/ uint16(x86_xCondIs64), 618, 634,
	/*618*/ uint16(x86_xCondDataSize), 622, 628, 0,
	/*622*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*624*/ uint16(x86_xReadSlashR),
	/*625*/ uint16(x86_xArgRM16),
	/*626*/ uint16(x86_xArgR16),
	/*627*/ uint16(x86_xMatch),
	/*628*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*630*/ uint16(x86_xReadSlashR),
	/*631*/ uint16(x86_xArgRM32),
	/*632*/ uint16(x86_xArgR32),
	/*633*/ uint16(x86_xMatch),
	/*634*/ uint16(x86_xCondDataSize), 622, 628, 638,
	/*638*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*640*/ uint16(x86_xReadSlashR),
	/*641*/ uint16(x86_xArgRM64),
	/*642*/ uint16(x86_xArgR64),
	/*643*/ uint16(x86_xMatch),
	/*644*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*646*/ uint16(x86_xReadSlashR),
	/*647*/ uint16(x86_xArgR8),
	/*648*/ uint16(x86_xArgRM8),
	/*649*/ uint16(x86_xMatch),
	/*650*/ uint16(x86_xCondIs64), 653, 669,
	/*653*/ uint16(x86_xCondDataSize), 657, 663, 0,
	/*657*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*659*/ uint16(x86_xReadSlashR),
	/*660*/ uint16(x86_xArgR16),
	/*661*/ uint16(x86_xArgRM16),
	/*662*/ uint16(x86_xMatch),
	/*663*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*665*/ uint16(x86_xReadSlashR),
	/*666*/ uint16(x86_xArgR32),
	/*667*/ uint16(x86_xArgRM32),
	/*668*/ uint16(x86_xMatch),
	/*669*/ uint16(x86_xCondDataSize), 657, 663, 673,
	/*673*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*675*/ uint16(x86_xReadSlashR),
	/*676*/ uint16(x86_xArgR64),
	/*677*/ uint16(x86_xArgRM64),
	/*678*/ uint16(x86_xMatch),
	/*679*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*681*/ uint16(x86_xReadIb),
	/*682*/ uint16(x86_xArgAL),
	/*683*/ uint16(x86_xArgImm8u),
	/*684*/ uint16(x86_xMatch),
	/*685*/ uint16(x86_xCondIs64), 688, 704,
	/*688*/ uint16(x86_xCondDataSize), 692, 698, 0,
	/*692*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*694*/ uint16(x86_xReadIw),
	/*695*/ uint16(x86_xArgAX),
	/*696*/ uint16(x86_xArgImm16),
	/*697*/ uint16(x86_xMatch),
	/*698*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*700*/ uint16(x86_xReadId),
	/*701*/ uint16(x86_xArgEAX),
	/*702*/ uint16(x86_xArgImm32),
	/*703*/ uint16(x86_xMatch),
	/*704*/ uint16(x86_xCondDataSize), 692, 698, 708,
	/*708*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*710*/ uint16(x86_xReadId),
	/*711*/ uint16(x86_xArgRAX),
	/*712*/ uint16(x86_xArgImm32),
	/*713*/ uint16(x86_xMatch),
	/*714*/ uint16(x86_xCondIs64), 717, 0,
	/*717*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*719*/ uint16(x86_xArgCS),
	/*720*/ uint16(x86_xMatch),
	/*721*/ uint16(x86_xCondByte), 228,
	0x00, 1180,
	0x01, 1237,
	0x02, 1345,
	0x03, 1367,
	0x05, 1389,
	0x06, 1395,
	0x07, 1398,
	0x08, 1404,
	0x09, 1407,
	0x0B, 1410,
	0x0D, 1413,
	0x10, 1426,
	0x11, 1460,
	0x12, 1494,
	0x13, 1537,
	0x14, 1555,
	0x15, 1573,
	0x16, 1591,
	0x17, 1626,
	0x18, 1644,
	0x1F, 1669,
	0x20, 1690,
	0x21, 1705,
	0x22, 1720,
	0x23, 1735,
	0x24, 1750,
	0x26, 1765,
	0x28, 1780,
	0x29, 1798,
	0x2A, 1816,
	0x2B, 1903,
	0x2C, 1937,
	0x2D, 2024,
	0x2E, 2111,
	0x2F, 2129,
	0x30, 2147,
	0x31, 2150,
	0x32, 2153,
	0x33, 2156,
	0x34, 2159,
	0x35, 2162,
	0x38, 2172,
	0x3A, 3073,
	0x40, 3484,
	0x41, 3513,
	0x42, 3542,
	0x43, 3571,
	0x44, 3600,
	0x45, 3629,
	0x46, 3658,
	0x47, 3687,
	0x48, 3716,
	0x49, 3745,
	0x4A, 3774,
	0x4B, 3803,
	0x4C, 3832,
	0x4D, 3861,
	0x4E, 3890,
	0x4F, 3919,
	0x50, 3948,
	0x51, 3966,
	0x52, 4000,
	0x53, 4018,
	0x54, 4036,
	0x55, 4054,
	0x56, 4072,
	0x57, 4090,
	0x58, 4108,
	0x59, 4142,
	0x5A, 4176,
	0x5B, 4210,
	0x5C, 4236,
	0x5D, 4270,
	0x5E, 4304,
	0x5F, 4338,
	0x60, 4372,
	0x61, 4390,
	0x62, 4408,
	0x63, 4426,
	0x64, 4444,
	0x65, 4462,
	0x66, 4480,
	0x67, 4498,
	0x68, 4516,
	0x69, 4534,
	0x6A, 4552,
	0x6B, 4570,
	0x6C, 4588,
	0x6D, 4598,
	0x6E, 4608,
	0x6F, 4675,
	0x70, 4701,
	0x71, 4743,
	0x72, 4806,
	0x73, 4869,
	0x74, 4934,
	0x75, 4952,
	0x76, 4970,
	0x77, 4988,
	0x7C, 4991,
	0x7D, 5009,
	0x7E, 5027,
	0x7F, 5104,
	0x80, 5130,
	0x81, 5161,
	0x82, 5192,
	0x83, 5223,
	0x84, 5254,
	0x85, 5285,
	0x86, 5316,
	0x87, 5347,
	0x88, 5378,
	0x89, 5409,
	0x8A, 5440,
	0x8B, 5471,
	0x8C, 5502,
	0x8D, 5533,
	0x8E, 5564,
	0x8F, 5595,
	0x90, 5626,
	0x91, 5631,
	0x92, 5636,
	0x93, 5641,
	0x94, 5646,
	0x95, 5651,
	0x96, 5656,
	0x97, 5661,
	0x98, 5666,
	0x99, 5671,
	0x9A, 5676,
	0x9B, 5681,
	0x9C, 5686,
	0x9D, 5691,
	0x9E, 5696,
	0x9F, 5701,
	0xA0, 5706,
	0xA1, 5710,
	0xA2, 5737,
	0xA3, 5740,
	0xA4, 5769,
	0xA5, 5804,
	0xA8, 5836,
	0xA9, 5840,
	0xAA, 5867,
	0xAB, 5870,
	0xAC, 5899,
	0xAD, 5934,
	0xAE, 5966,
	0xAF, 6224,
	0xB0, 6253,
	0xB1, 6259,
	0xB2, 6288,
	0xB3, 6317,
	0xB4, 6346,
	0xB5, 6375,
	0xB6, 6404,
	0xB7, 6433,
	0xB8, 6462,
	0xB9, 6499,
	0xBA, 6502,
	0xBB, 6627,
	0xBC, 6656,
	0xBD, 6723,
	0xBE, 6790,
	0xBF, 6819,
	0xC0, 6848,
	0xC1, 6854,
	0xC2, 6883,
	0xC3, 6925,
	0xC4, 6954,
	0xC5, 6976,
	0xC6, 6998,
	0xC7, 7020,
	0xc8, 7149,
	0xc9, 7149,
	0xca, 7149,
	0xcb, 7149,
	0xcc, 7149,
	0xcd, 7149,
	0xce, 7149,
	0xcf, 7149,
	0xD0, 7172,
	0xD1, 7190,
	0xD2, 7208,
	0xD3, 7226,
	0xD4, 7244,
	0xD5, 7262,
	0xD6, 7280,
	0xD7, 7306,
	0xD8, 7324,
	0xD9, 7342,
	0xDA, 7360,
	0xDB, 7378,
	0xDC, 7396,
	0xDD, 7414,
	0xDE, 7432,
	0xDF, 7450,
	0xE0, 7468,
	0xE1, 7486,
	0xE2, 7504,
	0xE3, 7522,
	0xE4, 7540,
	0xE5, 7558,
	0xE6, 7576,
	0xE7, 7602,
	0xE8, 7620,
	0xE9, 7638,
	0xEA, 7656,
	0xEB, 7674,
	0xEC, 7692,
	0xED, 7710,
	0xEE, 7728,
	0xEF, 7746,
	0xF0, 7764,
	0xF1, 7774,
	0xF2, 7792,
	0xF3, 7810,
	0xF4, 7828,
	0xF5, 7846,
	0xF6, 7864,
	0xF7, 7882,
	0xF8, 7900,
	0xF9, 7918,
	0xFA, 7936,
	0xFB, 7954,
	0xFC, 7972,
	0xFD, 7990,
	0xFE, 8008,
	uint16(x86_xFail),
	/*1180*/ uint16(x86_xCondSlashR),
	1189, // 0
	1205, // 1
	1221, // 2
	1225, // 3
	1229, // 4
	1233, // 5
	0,    // 6
	0,    // 7
	/*1189*/ uint16(x86_xCondDataSize), 1193, 1197, 1201,
	/*1193*/ uint16(x86_xSetOp), uint16(x86_SLDT),
	/*1195*/ uint16(x86_xArgRM16),
	/*1196*/ uint16(x86_xMatch),
	/*1197*/ uint16(x86_xSetOp), uint16(x86_SLDT),
	/*1199*/ uint16(x86_xArgR32M16),
	/*1200*/ uint16(x86_xMatch),
	/*1201*/ uint16(x86_xSetOp), uint16(x86_SLDT),
	/*1203*/ uint16(x86_xArgR64M16),
	/*1204*/ uint16(x86_xMatch),
	/*1205*/ uint16(x86_xCondDataSize), 1209, 1213, 1217,
	/*1209*/ uint16(x86_xSetOp), uint16(x86_STR),
	/*1211*/ uint16(x86_xArgRM16),
	/*1212*/ uint16(x86_xMatch),
	/*1213*/ uint16(x86_xSetOp), uint16(x86_STR),
	/*1215*/ uint16(x86_xArgR32M16),
	/*1216*/ uint16(x86_xMatch),
	/*1217*/ uint16(x86_xSetOp), uint16(x86_STR),
	/*1219*/ uint16(x86_xArgR64M16),
	/*1220*/ uint16(x86_xMatch),
	/*1221*/ uint16(x86_xSetOp), uint16(x86_LLDT),
	/*1223*/ uint16(x86_xArgRM16),
	/*1224*/ uint16(x86_xMatch),
	/*1225*/ uint16(x86_xSetOp), uint16(x86_LTR),
	/*1227*/ uint16(x86_xArgRM16),
	/*1228*/ uint16(x86_xMatch),
	/*1229*/ uint16(x86_xSetOp), uint16(x86_VERR),
	/*1231*/ uint16(x86_xArgRM16),
	/*1232*/ uint16(x86_xMatch),
	/*1233*/ uint16(x86_xSetOp), uint16(x86_VERW),
	/*1235*/ uint16(x86_xArgRM16),
	/*1236*/ uint16(x86_xMatch),
	/*1237*/ uint16(x86_xCondByte), 8,
	0xC8, 1318,
	0xC9, 1321,
	0xD0, 1324,
	0xD1, 1327,
	0xD5, 1330,
	0xD6, 1333,
	0xF8, 1336,
	0xF9, 1342,
	/*1255*/ uint16(x86_xCondSlashR),
	1264, // 0
	1268, // 1
	1272, // 2
	1283, // 3
	1294, // 4
	0,    // 5
	1310, // 6
	1314, // 7
	/*1264*/ uint16(x86_xSetOp), uint16(x86_SGDT),
	/*1266*/ uint16(x86_xArgM),
	/*1267*/ uint16(x86_xMatch),
	/*1268*/ uint16(x86_xSetOp), uint16(x86_SIDT),
	/*1270*/ uint16(x86_xArgM),
	/*1271*/ uint16(x86_xMatch),
	/*1272*/ uint16(x86_xCondIs64), 1275, 1279,
	/*1275*/ uint16(x86_xSetOp), uint16(x86_LGDT),
	/*1277*/ uint16(x86_xArgM16and32),
	/*1278*/ uint16(x86_xMatch),
	/*1279*/ uint16(x86_xSetOp), uint16(x86_LGDT),
	/*1281*/ uint16(x86_xArgM16and64),
	/*1282*/ uint16(x86_xMatch),
	/*1283*/ uint16(x86_xCondIs64), 1286, 1290,
	/*1286*/ uint16(x86_xSetOp), uint16(x86_LIDT),
	/*1288*/ uint16(x86_xArgM16and32),
	/*1289*/ uint16(x86_xMatch),
	/*1290*/ uint16(x86_xSetOp), uint16(x86_LIDT),
	/*1292*/ uint16(x86_xArgM16and64),
	/*1293*/ uint16(x86_xMatch),
	/*1294*/ uint16(x86_xCondDataSize), 1298, 1302, 1306,
	/*1298*/ uint16(x86_xSetOp), uint16(x86_SMSW),
	/*1300*/ uint16(x86_xArgRM16),
	/*1301*/ uint16(x86_xMatch),
	/*1302*/ uint16(x86_xSetOp), uint16(x86_SMSW),
	/*1304*/ uint16(x86_xArgR32M16),
	/*1305*/ uint16(x86_xMatch),
	/*1306*/ uint16(x86_xSetOp), uint16(x86_SMSW),
	/*1308*/ uint16(x86_xArgR64M16),
	/*1309*/ uint16(x86_xMatch),
	/*1310*/ uint16(x86_xSetOp), uint16(x86_LMSW),
	/*1312*/ uint16(x86_xArgRM16),
	/*1313*/ uint16(x86_xMatch),
	/*1314*/ uint16(x86_xSetOp), uint16(x86_INVLPG),
	/*1316*/ uint16(x86_xArgM),
	/*1317*/ uint16(x86_xMatch),
	/*1318*/ uint16(x86_xSetOp), uint16(x86_MONITOR),
	/*1320*/ uint16(x86_xMatch),
	/*1321*/ uint16(x86_xSetOp), uint16(x86_MWAIT),
	/*1323*/ uint16(x86_xMatch),
	/*1324*/ uint16(x86_xSetOp), uint16(x86_XGETBV),
	/*1326*/ uint16(x86_xMatch),
	/*1327*/ uint16(x86_xSetOp), uint16(x86_XSETBV),
	/*1329*/ uint16(x86_xMatch),
	/*1330*/ uint16(x86_xSetOp), uint16(x86_XEND),
	/*1332*/ uint16(x86_xMatch),
	/*1333*/ uint16(x86_xSetOp), uint16(x86_XTEST),
	/*1335*/ uint16(x86_xMatch),
	/*1336*/ uint16(x86_xCondIs64), 0, 1339,
	/*1339*/ uint16(x86_xSetOp), uint16(x86_SWAPGS),
	/*1341*/ uint16(x86_xMatch),
	/*1342*/ uint16(x86_xSetOp), uint16(x86_RDTSCP),
	/*1344*/ uint16(x86_xMatch),
	/*1345*/ uint16(x86_xCondDataSize), 1349, 1355, 1361,
	/*1349*/ uint16(x86_xSetOp), uint16(x86_LAR),
	/*1351*/ uint16(x86_xReadSlashR),
	/*1352*/ uint16(x86_xArgR16),
	/*1353*/ uint16(x86_xArgRM16),
	/*1354*/ uint16(x86_xMatch),
	/*1355*/ uint16(x86_xSetOp), uint16(x86_LAR),
	/*1357*/ uint16(x86_xReadSlashR),
	/*1358*/ uint16(x86_xArgR32),
	/*1359*/ uint16(x86_xArgR32M16),
	/*1360*/ uint16(x86_xMatch),
	/*1361*/ uint16(x86_xSetOp), uint16(x86_LAR),
	/*1363*/ uint16(x86_xReadSlashR),
	/*1364*/ uint16(x86_xArgR64),
	/*1365*/ uint16(x86_xArgR64M16),
	/*1366*/ uint16(x86_xMatch),
	/*1367*/ uint16(x86_xCondDataSize), 1371, 1377, 1383,
	/*1371*/ uint16(x86_xSetOp), uint16(x86_LSL),
	/*1373*/ uint16(x86_xReadSlashR),
	/*1374*/ uint16(x86_xArgR16),
	/*1375*/ uint16(x86_xArgRM16),
	/*1376*/ uint16(x86_xMatch),
	/*1377*/ uint16(x86_xSetOp), uint16(x86_LSL),
	/*1379*/ uint16(x86_xReadSlashR),
	/*1380*/ uint16(x86_xArgR32),
	/*1381*/ uint16(x86_xArgR32M16),
	/*1382*/ uint16(x86_xMatch),
	/*1383*/ uint16(x86_xSetOp), uint16(x86_LSL),
	/*1385*/ uint16(x86_xReadSlashR),
	/*1386*/ uint16(x86_xArgR64),
	/*1387*/ uint16(x86_xArgR32M16),
	/*1388*/ uint16(x86_xMatch),
	/*1389*/ uint16(x86_xCondIs64), 0, 1392,
	/*1392*/ uint16(x86_xSetOp), uint16(x86_SYSCALL),
	/*1394*/ uint16(x86_xMatch),
	/*1395*/ uint16(x86_xSetOp), uint16(x86_CLTS),
	/*1397*/ uint16(x86_xMatch),
	/*1398*/ uint16(x86_xCondIs64), 0, 1401,
	/*1401*/ uint16(x86_xSetOp), uint16(x86_SYSRET),
	/*1403*/ uint16(x86_xMatch),
	/*1404*/ uint16(x86_xSetOp), uint16(x86_INVD),
	/*1406*/ uint16(x86_xMatch),
	/*1407*/ uint16(x86_xSetOp), uint16(x86_WBINVD),
	/*1409*/ uint16(x86_xMatch),
	/*1410*/ uint16(x86_xSetOp), uint16(x86_UD2),
	/*1412*/ uint16(x86_xMatch),
	/*1413*/ uint16(x86_xCondSlashR),
	0,    // 0
	1422, // 1
	0,    // 2
	0,    // 3
	0,    // 4
	0,    // 5
	0,    // 6
	0,    // 7
	/*1422*/ uint16(x86_xSetOp), uint16(x86_PREFETCHW),
	/*1424*/ uint16(x86_xArgM8),
	/*1425*/ uint16(x86_xMatch),
	/*1426*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1454,
	0xF2, 1448,
	0x66, 1442,
	0x0, 1436,
	/*1436*/ uint16(x86_xSetOp), uint16(x86_MOVUPS),
	/*1438*/ uint16(x86_xReadSlashR),
	/*1439*/ uint16(x86_xArgXmm1),
	/*1440*/ uint16(x86_xArgXmm2M128),
	/*1441*/ uint16(x86_xMatch),
	/*1442*/ uint16(x86_xSetOp), uint16(x86_MOVUPD),
	/*1444*/ uint16(x86_xReadSlashR),
	/*1445*/ uint16(x86_xArgXmm1),
	/*1446*/ uint16(x86_xArgXmm2M128),
	/*1447*/ uint16(x86_xMatch),
	/*1448*/ uint16(x86_xSetOp), uint16(x86_MOVSD_XMM),
	/*1450*/ uint16(x86_xReadSlashR),
	/*1451*/ uint16(x86_xArgXmm1),
	/*1452*/ uint16(x86_xArgXmm2M64),
	/*1453*/ uint16(x86_xMatch),
	/*1454*/ uint16(x86_xSetOp), uint16(x86_MOVSS),
	/*1456*/ uint16(x86_xReadSlashR),
	/*1457*/ uint16(x86_xArgXmm1),
	/*1458*/ uint16(x86_xArgXmm2M32),
	/*1459*/ uint16(x86_xMatch),
	/*1460*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1488,
	0xF2, 1482,
	0x66, 1476,
	0x0, 1470,
	/*1470*/ uint16(x86_xSetOp), uint16(x86_MOVUPS),
	/*1472*/ uint16(x86_xReadSlashR),
	/*1473*/ uint16(x86_xArgXmm2M128),
	/*1474*/ uint16(x86_xArgXmm1),
	/*1475*/ uint16(x86_xMatch),
	/*1476*/ uint16(x86_xSetOp), uint16(x86_MOVUPD),
	/*1478*/ uint16(x86_xReadSlashR),
	/*1479*/ uint16(x86_xArgXmm2M128),
	/*1480*/ uint16(x86_xArgXmm),
	/*1481*/ uint16(x86_xMatch),
	/*1482*/ uint16(x86_xSetOp), uint16(x86_MOVSD_XMM),
	/*1484*/ uint16(x86_xReadSlashR),
	/*1485*/ uint16(x86_xArgXmm2M64),
	/*1486*/ uint16(x86_xArgXmm1),
	/*1487*/ uint16(x86_xMatch),
	/*1488*/ uint16(x86_xSetOp), uint16(x86_MOVSS),
	/*1490*/ uint16(x86_xReadSlashR),
	/*1491*/ uint16(x86_xArgXmm2M32),
	/*1492*/ uint16(x86_xArgXmm),
	/*1493*/ uint16(x86_xMatch),
	/*1494*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1531,
	0xF2, 1525,
	0x66, 1519,
	0x0, 1504,
	/*1504*/ uint16(x86_xCondIsMem), 1507, 1513,
	/*1507*/ uint16(x86_xSetOp), uint16(x86_MOVHLPS),
	/*1509*/ uint16(x86_xReadSlashR),
	/*1510*/ uint16(x86_xArgXmm1),
	/*1511*/ uint16(x86_xArgXmm2),
	/*1512*/ uint16(x86_xMatch),
	/*1513*/ uint16(x86_xSetOp), uint16(x86_MOVLPS),
	/*1515*/ uint16(x86_xReadSlashR),
	/*1516*/ uint16(x86_xArgXmm),
	/*1517*/ uint16(x86_xArgM64),
	/*1518*/ uint16(x86_xMatch),
	/*1519*/ uint16(x86_xSetOp), uint16(x86_MOVLPD),
	/*1521*/ uint16(x86_xReadSlashR),
	/*1522*/ uint16(x86_xArgXmm),
	/*1523*/ uint16(x86_xArgXmm2M64),
	/*1524*/ uint16(x86_xMatch),
	/*1525*/ uint16(x86_xSetOp), uint16(x86_MOVDDUP),
	/*1527*/ uint16(x86_xReadSlashR),
	/*1528*/ uint16(x86_xArgXmm1),
	/*1529*/ uint16(x86_xArgXmm2M64),
	/*1530*/ uint16(x86_xMatch),
	/*1531*/ uint16(x86_xSetOp), uint16(x86_MOVSLDUP),
	/*1533*/ uint16(x86_xReadSlashR),
	/*1534*/ uint16(x86_xArgXmm1),
	/*1535*/ uint16(x86_xArgXmm2M128),
	/*1536*/ uint16(x86_xMatch),
	/*1537*/ uint16(x86_xCondPrefix), 2,
	0x66, 1549,
	0x0, 1543,
	/*1543*/ uint16(x86_xSetOp), uint16(x86_MOVLPS),
	/*1545*/ uint16(x86_xReadSlashR),
	/*1546*/ uint16(x86_xArgM64),
	/*1547*/ uint16(x86_xArgXmm),
	/*1548*/ uint16(x86_xMatch),
	/*1549*/ uint16(x86_xSetOp), uint16(x86_MOVLPD),
	/*1551*/ uint16(x86_xReadSlashR),
	/*1552*/ uint16(x86_xArgXmm2M64),
	/*1553*/ uint16(x86_xArgXmm),
	/*1554*/ uint16(x86_xMatch),
	/*1555*/ uint16(x86_xCondPrefix), 2,
	0x66, 1567,
	0x0, 1561,
	/*1561*/ uint16(x86_xSetOp), uint16(x86_UNPCKLPS),
	/*1563*/ uint16(x86_xReadSlashR),
	/*1564*/ uint16(x86_xArgXmm1),
	/*1565*/ uint16(x86_xArgXmm2M128),
	/*1566*/ uint16(x86_xMatch),
	/*1567*/ uint16(x86_xSetOp), uint16(x86_UNPCKLPD),
	/*1569*/ uint16(x86_xReadSlashR),
	/*1570*/ uint16(x86_xArgXmm1),
	/*1571*/ uint16(x86_xArgXmm2M128),
	/*1572*/ uint16(x86_xMatch),
	/*1573*/ uint16(x86_xCondPrefix), 2,
	0x66, 1585,
	0x0, 1579,
	/*1579*/ uint16(x86_xSetOp), uint16(x86_UNPCKHPS),
	/*1581*/ uint16(x86_xReadSlashR),
	/*1582*/ uint16(x86_xArgXmm1),
	/*1583*/ uint16(x86_xArgXmm2M128),
	/*1584*/ uint16(x86_xMatch),
	/*1585*/ uint16(x86_xSetOp), uint16(x86_UNPCKHPD),
	/*1587*/ uint16(x86_xReadSlashR),
	/*1588*/ uint16(x86_xArgXmm1),
	/*1589*/ uint16(x86_xArgXmm2M128),
	/*1590*/ uint16(x86_xMatch),
	/*1591*/ uint16(x86_xCondPrefix), 3,
	0xF3, 1620,
	0x66, 1614,
	0x0, 1599,
	/*1599*/ uint16(x86_xCondIsMem), 1602, 1608,
	/*1602*/ uint16(x86_xSetOp), uint16(x86_MOVLHPS),
	/*1604*/ uint16(x86_xReadSlashR),
	/*1605*/ uint16(x86_xArgXmm1),
	/*1606*/ uint16(x86_xArgXmm2),
	/*1607*/ uint16(x86_xMatch),
	/*1608*/ uint16(x86_xSetOp), uint16(x86_MOVHPS),
	/*1610*/ uint16(x86_xReadSlashR),
	/*1611*/ uint16(x86_xArgXmm),
	/*1612*/ uint16(x86_xArgM64),
	/*1613*/ uint16(x86_xMatch),
	/*1614*/ uint16(x86_xSetOp), uint16(x86_MOVHPD),
	/*1616*/ uint16(x86_xReadSlashR),
	/*1617*/ uint16(x86_xArgXmm),
	/*1618*/ uint16(x86_xArgXmm2M64),
	/*1619*/ uint16(x86_xMatch),
	/*1620*/ uint16(x86_xSetOp), uint16(x86_MOVSHDUP),
	/*1622*/ uint16(x86_xReadSlashR),
	/*1623*/ uint16(x86_xArgXmm1),
	/*1624*/ uint16(x86_xArgXmm2M128),
	/*1625*/ uint16(x86_xMatch),
	/*1626*/ uint16(x86_xCondPrefix), 2,
	0x66, 1638,
	0x0, 1632,
	/*1632*/ uint16(x86_xSetOp), uint16(x86_MOVHPS),
	/*1634*/ uint16(x86_xReadSlashR),
	/*1635*/ uint16(x86_xArgM64),
	/*1636*/ uint16(x86_xArgXmm),
	/*1637*/ uint16(x86_xMatch),
	/*1638*/ uint16(x86_xSetOp), uint16(x86_MOVHPD),
	/*1640*/ uint16(x86_xReadSlashR),
	/*1641*/ uint16(x86_xArgXmm2M64),
	/*1642*/ uint16(x86_xArgXmm),
	/*1643*/ uint16(x86_xMatch),
	/*1644*/ uint16(x86_xCondSlashR),
	1653, // 0
	1657, // 1
	1661, // 2
	1665, // 3
	0,    // 4
	0,    // 5
	0,    // 6
	0,    // 7
	/*1653*/ uint16(x86_xSetOp), uint16(x86_PREFETCHNTA),
	/*1655*/ uint16(x86_xArgM8),
	/*1656*/ uint16(x86_xMatch),
	/*1657*/ uint16(x86_xSetOp), uint16(x86_PREFETCHT0),
	/*1659*/ uint16(x86_xArgM8),
	/*1660*/ uint16(x86_xMatch),
	/*1661*/ uint16(x86_xSetOp), uint16(x86_PREFETCHT1),
	/*1663*/ uint16(x86_xArgM8),
	/*1664*/ uint16(x86_xMatch),
	/*1665*/ uint16(x86_xSetOp), uint16(x86_PREFETCHT2),
	/*1667*/ uint16(x86_xArgM8),
	/*1668*/ uint16(x86_xMatch),
	/*1669*/ uint16(x86_xCondSlashR),
	1678, // 0
	0,    // 1
	0,    // 2
	0,    // 3
	0,    // 4
	0,    // 5
	0,    // 6
	0,    // 7
	/*1678*/ uint16(x86_xCondDataSize), 1682, 1686, 0,
	/*1682*/ uint16(x86_xSetOp), uint16(x86_NOP),
	/*1684*/ uint16(x86_xArgRM16),
	/*1685*/ uint16(x86_xMatch),
	/*1686*/ uint16(x86_xSetOp), uint16(x86_NOP),
	/*1688*/ uint16(x86_xArgRM32),
	/*1689*/ uint16(x86_xMatch),
	/*1690*/ uint16(x86_xCondIs64), 1693, 1699,
	/*1693*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1695*/ uint16(x86_xReadSlashR),
	/*1696*/ uint16(x86_xArgRmf32),
	/*1697*/ uint16(x86_xArgCR0dashCR7),
	/*1698*/ uint16(x86_xMatch),
	/*1699*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1701*/ uint16(x86_xReadSlashR),
	/*1702*/ uint16(x86_xArgRmf64),
	/*1703*/ uint16(x86_xArgCR0dashCR7),
	/*1704*/ uint16(x86_xMatch),
	/*1705*/ uint16(x86_xCondIs64), 1708, 1714,
	/*1708*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1710*/ uint16(x86_xReadSlashR),
	/*1711*/ uint16(x86_xArgRmf32),
	/*1712*/ uint16(x86_xArgDR0dashDR7),
	/*1713*/ uint16(x86_xMatch),
	/*1714*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1716*/ uint16(x86_xReadSlashR),
	/*1717*/ uint16(x86_xArgRmf64),
	/*1718*/ uint16(x86_xArgDR0dashDR7),
	/*1719*/ uint16(x86_xMatch),
	/*1720*/ uint16(x86_xCondIs64), 1723, 1729,
	/*1723*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1725*/ uint16(x86_xReadSlashR),
	/*1726*/ uint16(x86_xArgCR0dashCR7),
	/*1727*/ uint16(x86_xArgRmf32),
	/*1728*/ uint16(x86_xMatch),
	/*1729*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1731*/ uint16(x86_xReadSlashR),
	/*1732*/ uint16(x86_xArgCR0dashCR7),
	/*1733*/ uint16(x86_xArgRmf64),
	/*1734*/ uint16(x86_xMatch),
	/*1735*/ uint16(x86_xCondIs64), 1738, 1744,
	/*1738*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1740*/ uint16(x86_xReadSlashR),
	/*1741*/ uint16(x86_xArgDR0dashDR7),
	/*1742*/ uint16(x86_xArgRmf32),
	/*1743*/ uint16(x86_xMatch),
	/*1744*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1746*/ uint16(x86_xReadSlashR),
	/*1747*/ uint16(x86_xArgDR0dashDR7),
	/*1748*/ uint16(x86_xArgRmf64),
	/*1749*/ uint16(x86_xMatch),
	/*1750*/ uint16(x86_xCondIs64), 1753, 1759,
	/*1753*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1755*/ uint16(x86_xReadSlashR),
	/*1756*/ uint16(x86_xArgRmf32),
	/*1757*/ uint16(x86_xArgTR0dashTR7),
	/*1758*/ uint16(x86_xMatch),
	/*1759*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1761*/ uint16(x86_xReadSlashR),
	/*1762*/ uint16(x86_xArgRmf64),
	/*1763*/ uint16(x86_xArgTR0dashTR7),
	/*1764*/ uint16(x86_xMatch),
	/*1765*/ uint16(x86_xCondIs64), 1768, 1774,
	/*1768*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1770*/ uint16(x86_xReadSlashR),
	/*1771*/ uint16(x86_xArgTR0dashTR7),
	/*1772*/ uint16(x86_xArgRmf32),
	/*1773*/ uint16(x86_xMatch),
	/*1774*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*1776*/ uint16(x86_xReadSlashR),
	/*1777*/ uint16(x86_xArgTR0dashTR7),
	/*1778*/ uint16(x86_xArgRmf64),
	/*1779*/ uint16(x86_xMatch),
	/*1780*/ uint16(x86_xCondPrefix), 2,
	0x66, 1792,
	0x0, 1786,
	/*1786*/ uint16(x86_xSetOp), uint16(x86_MOVAPS),
	/*1788*/ uint16(x86_xReadSlashR),
	/*1789*/ uint16(x86_xArgXmm1),
	/*1790*/ uint16(x86_xArgXmm2M128),
	/*1791*/ uint16(x86_xMatch),
	/*1792*/ uint16(x86_xSetOp), uint16(x86_MOVAPD),
	/*1794*/ uint16(x86_xReadSlashR),
	/*1795*/ uint16(x86_xArgXmm1),
	/*1796*/ uint16(x86_xArgXmm2M128),
	/*1797*/ uint16(x86_xMatch),
	/*1798*/ uint16(x86_xCondPrefix), 2,
	0x66, 1810,
	0x0, 1804,
	/*1804*/ uint16(x86_xSetOp), uint16(x86_MOVAPS),
	/*1806*/ uint16(x86_xReadSlashR),
	/*1807*/ uint16(x86_xArgXmm2M128),
	/*1808*/ uint16(x86_xArgXmm1),
	/*1809*/ uint16(x86_xMatch),
	/*1810*/ uint16(x86_xSetOp), uint16(x86_MOVAPD),
	/*1812*/ uint16(x86_xReadSlashR),
	/*1813*/ uint16(x86_xArgXmm2M128),
	/*1814*/ uint16(x86_xArgXmm1),
	/*1815*/ uint16(x86_xMatch),
	/*1816*/ uint16(x86_xCondIs64), 1819, 1873,
	/*1819*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1857,
	0xF2, 1841,
	0x66, 1835,
	0x0, 1829,
	/*1829*/ uint16(x86_xSetOp), uint16(x86_CVTPI2PS),
	/*1831*/ uint16(x86_xReadSlashR),
	/*1832*/ uint16(x86_xArgXmm),
	/*1833*/ uint16(x86_xArgMmM64),
	/*1834*/ uint16(x86_xMatch),
	/*1835*/ uint16(x86_xSetOp), uint16(x86_CVTPI2PD),
	/*1837*/ uint16(x86_xReadSlashR),
	/*1838*/ uint16(x86_xArgXmm),
	/*1839*/ uint16(x86_xArgMmM64),
	/*1840*/ uint16(x86_xMatch),
	/*1841*/ uint16(x86_xCondDataSize), 1845, 1851, 0,
	/*1845*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SD),
	/*1847*/ uint16(x86_xReadSlashR),
	/*1848*/ uint16(x86_xArgXmm),
	/*1849*/ uint16(x86_xArgRM32),
	/*1850*/ uint16(x86_xMatch),
	/*1851*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SD),
	/*1853*/ uint16(x86_xReadSlashR),
	/*1854*/ uint16(x86_xArgXmm),
	/*1855*/ uint16(x86_xArgRM32),
	/*1856*/ uint16(x86_xMatch),
	/*1857*/ uint16(x86_xCondDataSize), 1861, 1867, 0,
	/*1861*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SS),
	/*1863*/ uint16(x86_xReadSlashR),
	/*1864*/ uint16(x86_xArgXmm),
	/*1865*/ uint16(x86_xArgRM32),
	/*1866*/ uint16(x86_xMatch),
	/*1867*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SS),
	/*1869*/ uint16(x86_xReadSlashR),
	/*1870*/ uint16(x86_xArgXmm),
	/*1871*/ uint16(x86_xArgRM32),
	/*1872*/ uint16(x86_xMatch),
	/*1873*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1893,
	0xF2, 1883,
	0x66, 1835,
	0x0, 1829,
	/*1883*/ uint16(x86_xCondDataSize), 1845, 1851, 1887,
	/*1887*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SD),
	/*1889*/ uint16(x86_xReadSlashR),
	/*1890*/ uint16(x86_xArgXmm),
	/*1891*/ uint16(x86_xArgRM64),
	/*1892*/ uint16(x86_xMatch),
	/*1893*/ uint16(x86_xCondDataSize), 1861, 1867, 1897,
	/*1897*/ uint16(x86_xSetOp), uint16(x86_CVTSI2SS),
	/*1899*/ uint16(x86_xReadSlashR),
	/*1900*/ uint16(x86_xArgXmm),
	/*1901*/ uint16(x86_xArgRM64),
	/*1902*/ uint16(x86_xMatch),
	/*1903*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1931,
	0xF2, 1925,
	0x66, 1919,
	0x0, 1913,
	/*1913*/ uint16(x86_xSetOp), uint16(x86_MOVNTPS),
	/*1915*/ uint16(x86_xReadSlashR),
	/*1916*/ uint16(x86_xArgM128),
	/*1917*/ uint16(x86_xArgXmm),
	/*1918*/ uint16(x86_xMatch),
	/*1919*/ uint16(x86_xSetOp), uint16(x86_MOVNTPD),
	/*1921*/ uint16(x86_xReadSlashR),
	/*1922*/ uint16(x86_xArgM128),
	/*1923*/ uint16(x86_xArgXmm),
	/*1924*/ uint16(x86_xMatch),
	/*1925*/ uint16(x86_xSetOp), uint16(x86_MOVNTSD),
	/*1927*/ uint16(x86_xReadSlashR),
	/*1928*/ uint16(x86_xArgM64),
	/*1929*/ uint16(x86_xArgXmm),
	/*1930*/ uint16(x86_xMatch),
	/*1931*/ uint16(x86_xSetOp), uint16(x86_MOVNTSS),
	/*1933*/ uint16(x86_xReadSlashR),
	/*1934*/ uint16(x86_xArgM32),
	/*1935*/ uint16(x86_xArgXmm),
	/*1936*/ uint16(x86_xMatch),
	/*1937*/ uint16(x86_xCondIs64), 1940, 1994,
	/*1940*/ uint16(x86_xCondPrefix), 4,
	0xF3, 1978,
	0xF2, 1962,
	0x66, 1956,
	0x0, 1950,
	/*1950*/ uint16(x86_xSetOp), uint16(x86_CVTTPS2PI),
	/*1952*/ uint16(x86_xReadSlashR),
	/*1953*/ uint16(x86_xArgMm),
	/*1954*/ uint16(x86_xArgXmmM64),
	/*1955*/ uint16(x86_xMatch),
	/*1956*/ uint16(x86_xSetOp), uint16(x86_CVTTPD2PI),
	/*1958*/ uint16(x86_xReadSlashR),
	/*1959*/ uint16(x86_xArgMm),
	/*1960*/ uint16(x86_xArgXmmM128),
	/*1961*/ uint16(x86_xMatch),
	/*1962*/ uint16(x86_xCondDataSize), 1966, 1972, 0,
	/*1966*/ uint16(x86_xSetOp), uint16(x86_CVTTSD2SI),
	/*1968*/ uint16(x86_xReadSlashR),
	/*1969*/ uint16(x86_xArgR32),
	/*1970*/ uint16(x86_xArgXmmM64),
	/*1971*/ uint16(x86_xMatch),
	/*1972*/ uint16(x86_xSetOp), uint16(x86_CVTTSD2SI),
	/*1974*/ uint16(x86_xReadSlashR),
	/*1975*/ uint16(x86_xArgR32),
	/*1976*/ uint16(x86_xArgXmmM64),
	/*1977*/ uint16(x86_xMatch),
	/*1978*/ uint16(x86_xCondDataSize), 1982, 1988, 0,
	/*1982*/ uint16(x86_xSetOp), uint16(x86_CVTTSS2SI),
	/*1984*/ uint16(x86_xReadSlashR),
	/*1985*/ uint16(x86_xArgR32),
	/*1986*/ uint16(x86_xArgXmmM32),
	/*1987*/ uint16(x86_xMatch),
	/*1988*/ uint16(x86_xSetOp), uint16(x86_CVTTSS2SI),
	/*1990*/ uint16(x86_xReadSlashR),
	/*1991*/ uint16(x86_xArgR32),
	/*1992*/ uint16(x86_xArgXmmM32),
	/*1993*/ uint16(x86_xMatch),
	/*1994*/ uint16(x86_xCondPrefix), 4,
	0xF3, 2014,
	0xF2, 2004,
	0x66, 1956,
	0x0, 1950,
	/*2004*/ uint16(x86_xCondDataSize), 1966, 1972, 2008,
	/*2008*/ uint16(x86_xSetOp), uint16(x86_CVTTSD2SI),
	/*2010*/ uint16(x86_xReadSlashR),
	/*2011*/ uint16(x86_xArgR64),
	/*2012*/ uint16(x86_xArgXmmM64),
	/*2013*/ uint16(x86_xMatch),
	/*2014*/ uint16(x86_xCondDataSize), 1982, 1988, 2018,
	/*2018*/ uint16(x86_xSetOp), uint16(x86_CVTTSS2SI),
	/*2020*/ uint16(x86_xReadSlashR),
	/*2021*/ uint16(x86_xArgR64),
	/*2022*/ uint16(x86_xArgXmmM32),
	/*2023*/ uint16(x86_xMatch),
	/*2024*/ uint16(x86_xCondIs64), 2027, 2081,
	/*2027*/ uint16(x86_xCondPrefix), 4,
	0xF3, 2065,
	0xF2, 2049,
	0x66, 2043,
	0x0, 2037,
	/*2037*/ uint16(x86_xSetOp), uint16(x86_CVTPS2PI),
	/*2039*/ uint16(x86_xReadSlashR),
	/*2040*/ uint16(x86_xArgMm),
	/*2041*/ uint16(x86_xArgXmmM64),
	/*2042*/ uint16(x86_xMatch),
	/*2043*/ uint16(x86_xSetOp), uint16(x86_CVTPD2PI),
	/*2045*/ uint16(x86_xReadSlashR),
	/*2046*/ uint16(x86_xArgMm),
	/*2047*/ uint16(x86_xArgXmmM128),
	/*2048*/ uint16(x86_xMatch),
	/*2049*/ uint16(x86_xCondDataSize), 2053, 2059, 0,
	/*2053*/ uint16(x86_xSetOp), uint16(x86_CVTSD2SI),
	/*2055*/ uint16(x86_xReadSlashR),
	/*2056*/ uint16(x86_xArgR32),
	/*2057*/ uint16(x86_xArgXmmM64),
	/*2058*/ uint16(x86_xMatch),
	/*2059*/ uint16(x86_xSetOp), uint16(x86_CVTSD2SI),
	/*2061*/ uint16(x86_xReadSlashR),
	/*2062*/ uint16(x86_xArgR32),
	/*2063*/ uint16(x86_xArgXmmM64),
	/*2064*/ uint16(x86_xMatch),
	/*2065*/ uint16(x86_xCondDataSize), 2069, 2075, 0,
	/*2069*/ uint16(x86_xSetOp), uint16(x86_CVTSS2SI),
	/*2071*/ uint16(x86_xReadSlashR),
	/*2072*/ uint16(x86_xArgR32),
	/*2073*/ uint16(x86_xArgXmmM32),
	/*2074*/ uint16(x86_xMatch),
	/*2075*/ uint16(x86_xSetOp), uint16(x86_CVTSS2SI),
	/*2077*/ uint16(x86_xReadSlashR),
	/*2078*/ uint16(x86_xArgR32),
	/*2079*/ uint16(x86_xArgXmmM32),
	/*2080*/ uint16(x86_xMatch),
	/*2081*/ uint16(x86_xCondPrefix), 4,
	0xF3, 2101,
	0xF2, 2091,
	0x66, 2043,
	0x0, 2037,
	/*2091*/ uint16(x86_xCondDataSize), 2053, 2059, 2095,
	/*2095*/ uint16(x86_xSetOp), uint16(x86_CVTSD2SI),
	/*2097*/ uint16(x86_xReadSlashR),
	/*2098*/ uint16(x86_xArgR64),
	/*2099*/ uint16(x86_xArgXmmM64),
	/*2100*/ uint16(x86_xMatch),
	/*2101*/ uint16(x86_xCondDataSize), 2069, 2075, 2105,
	/*2105*/ uint16(x86_xSetOp), uint16(x86_CVTSS2SI),
	/*2107*/ uint16(x86_xReadSlashR),
	/*2108*/ uint16(x86_xArgR64),
	/*2109*/ uint16(x86_xArgXmmM32),
	/*2110*/ uint16(x86_xMatch),
	/*2111*/ uint16(x86_xCondPrefix), 2,
	0x66, 2123,
	0x0, 2117,
	/*2117*/ uint16(x86_xSetOp), uint16(x86_UCOMISS),
	/*2119*/ uint16(x86_xReadSlashR),
	/*2120*/ uint16(x86_xArgXmm1),
	/*2121*/ uint16(x86_xArgXmm2M32),
	/*2122*/ uint16(x86_xMatch),
	/*2123*/ uint16(x86_xSetOp), uint16(x86_UCOMISD),
	/*2125*/ uint16(x86_xReadSlashR),
	/*2126*/ uint16(x86_xArgXmm1),
	/*2127*/ uint16(x86_xArgXmm2M64),
	/*2128*/ uint16(x86_xMatch),
	/*2129*/ uint16(x86_xCondPrefix), 2,
	0x66, 2141,
	0x0, 2135,
	/*2135*/ uint16(x86_xSetOp), uint16(x86_COMISS),
	/*2137*/ uint16(x86_xReadSlashR),
	/*2138*/ uint16(x86_xArgXmm1),
	/*2139*/ uint16(x86_xArgXmm2M32),
	/*2140*/ uint16(x86_xMatch),
	/*2141*/ uint16(x86_xSetOp), uint16(x86_COMISD),
	/*2143*/ uint16(x86_xReadSlashR),
	/*2144*/ uint16(x86_xArgXmm1),
	/*2145*/ uint16(x86_xArgXmm2M64),
	/*2146*/ uint16(x86_xMatch),
	/*2147*/ uint16(x86_xSetOp), uint16(x86_WRMSR),
	/*2149*/ uint16(x86_xMatch),
	/*2150*/ uint16(x86_xSetOp), uint16(x86_RDTSC),
	/*2152*/ uint16(x86_xMatch),
	/*2153*/ uint16(x86_xSetOp), uint16(x86_RDMSR),
	/*2155*/ uint16(x86_xMatch),
	/*2156*/ uint16(x86_xSetOp), uint16(x86_RDPMC),
	/*2158*/ uint16(x86_xMatch),
	/*2159*/ uint16(x86_xSetOp), uint16(x86_SYSENTER),
	/*2161*/ uint16(x86_xMatch),
	/*2162*/ uint16(x86_xCondDataSize), 2166, 2166, 2169,
	/*2166*/ uint16(x86_xSetOp), uint16(x86_SYSEXIT),
	/*2168*/ uint16(x86_xMatch),
	/*2169*/ uint16(x86_xSetOp), uint16(x86_SYSEXIT),
	/*2171*/ uint16(x86_xMatch),
	/*2172*/ uint16(x86_xCondByte), 54,
	0x00, 2283,
	0x01, 2301,
	0x02, 2319,
	0x03, 2337,
	0x04, 2355,
	0x05, 2373,
	0x06, 2391,
	0x07, 2409,
	0x08, 2427,
	0x09, 2445,
	0x0A, 2463,
	0x0B, 2481,
	0x10, 2499,
	0x14, 2510,
	0x15, 2521,
	0x17, 2532,
	0x1C, 2542,
	0x1D, 2560,
	0x1E, 2578,
	0x20, 2596,
	0x21, 2606,
	0x22, 2616,
	0x23, 2626,
	0x24, 2636,
	0x25, 2646,
	0x28, 2656,
	0x29, 2666,
	0x2A, 2676,
	0x2B, 2686,
	0x30, 2696,
	0x31, 2706,
	0x32, 2716,
	0x33, 2726,
	0x34, 2736,
	0x35, 2746,
	0x37, 2756,
	0x38, 2766,
	0x39, 2776,
	0x3A, 2786,
	0x3B, 2796,
	0x3C, 2806,
	0x3D, 2816,
	0x3E, 2826,
	0x3F, 2836,
	0x40, 2846,
	0x41, 2856,
	0x82, 2866,
	0xDB, 2889,
	0xDC, 2899,
	0xDD, 2909,
	0xDE, 2919,
	0xDF, 2929,
	0xF0, 2939,
	0xF1, 3006,
	uint16(x86_xFail),
	/*2283*/ uint16(x86_xCondPrefix), 2,
	0x66, 2295,
	0x0, 2289,
	/*2289*/ uint16(x86_xSetOp), uint16(x86_PSHUFB),
	/*2291*/ uint16(x86_xReadSlashR),
	/*2292*/ uint16(x86_xArgMm1),
	/*2293*/ uint16(x86_xArgMm2M64),
	/*2294*/ uint16(x86_xMatch),
	/*2295*/ uint16(x86_xSetOp), uint16(x86_PSHUFB),
	/*2297*/ uint16(x86_xReadSlashR),
	/*2298*/ uint16(x86_xArgXmm1),
	/*2299*/ uint16(x86_xArgXmm2M128),
	/*2300*/ uint16(x86_xMatch),
	/*2301*/ uint16(x86_xCondPrefix), 2,
	0x66, 2313,
	0x0, 2307,
	/*2307*/ uint16(x86_xSetOp), uint16(x86_PHADDW),
	/*2309*/ uint16(x86_xReadSlashR),
	/*2310*/ uint16(x86_xArgMm1),
	/*2311*/ uint16(x86_xArgMm2M64),
	/*2312*/ uint16(x86_xMatch),
	/*2313*/ uint16(x86_xSetOp), uint16(x86_PHADDW),
	/*2315*/ uint16(x86_xReadSlashR),
	/*2316*/ uint16(x86_xArgXmm1),
	/*2317*/ uint16(x86_xArgXmm2M128),
	/*2318*/ uint16(x86_xMatch),
	/*2319*/ uint16(x86_xCondPrefix), 2,
	0x66, 2331,
	0x0, 2325,
	/*2325*/ uint16(x86_xSetOp), uint16(x86_PHADDD),
	/*2327*/ uint16(x86_xReadSlashR),
	/*2328*/ uint16(x86_xArgMm1),
	/*2329*/ uint16(x86_xArgMm2M64),
	/*2330*/ uint16(x86_xMatch),
	/*2331*/ uint16(x86_xSetOp), uint16(x86_PHADDD),
	/*2333*/ uint16(x86_xReadSlashR),
	/*2334*/ uint16(x86_xArgXmm1),
	/*2335*/ uint16(x86_xArgXmm2M128),
	/*2336*/ uint16(x86_xMatch),
	/*2337*/ uint16(x86_xCondPrefix), 2,
	0x66, 2349,
	0x0, 2343,
	/*2343*/ uint16(x86_xSetOp), uint16(x86_PHADDSW),
	/*2345*/ uint16(x86_xReadSlashR),
	/*2346*/ uint16(x86_xArgMm1),
	/*2347*/ uint16(x86_xArgMm2M64),
	/*2348*/ uint16(x86_xMatch),
	/*2349*/ uint16(x86_xSetOp), uint16(x86_PHADDSW),
	/*2351*/ uint16(x86_xReadSlashR),
	/*2352*/ uint16(x86_xArgXmm1),
	/*2353*/ uint16(x86_xArgXmm2M128),
	/*2354*/ uint16(x86_xMatch),
	/*2355*/ uint16(x86_xCondPrefix), 2,
	0x66, 2367,
	0x0, 2361,
	/*2361*/ uint16(x86_xSetOp), uint16(x86_PMADDUBSW),
	/*2363*/ uint16(x86_xReadSlashR),
	/*2364*/ uint16(x86_xArgMm1),
	/*2365*/ uint16(x86_xArgMm2M64),
	/*2366*/ uint16(x86_xMatch),
	/*2367*/ uint16(x86_xSetOp), uint16(x86_PMADDUBSW),
	/*2369*/ uint16(x86_xReadSlashR),
	/*2370*/ uint16(x86_xArgXmm1),
	/*2371*/ uint16(x86_xArgXmm2M128),
	/*2372*/ uint16(x86_xMatch),
	/*2373*/ uint16(x86_xCondPrefix), 2,
	0x66, 2385,
	0x0, 2379,
	/*2379*/ uint16(x86_xSetOp), uint16(x86_PHSUBW),
	/*2381*/ uint16(x86_xReadSlashR),
	/*2382*/ uint16(x86_xArgMm1),
	/*2383*/ uint16(x86_xArgMm2M64),
	/*2384*/ uint16(x86_xMatch),
	/*2385*/ uint16(x86_xSetOp), uint16(x86_PHSUBW),
	/*2387*/ uint16(x86_xReadSlashR),
	/*2388*/ uint16(x86_xArgXmm1),
	/*2389*/ uint16(x86_xArgXmm2M128),
	/*2390*/ uint16(x86_xMatch),
	/*2391*/ uint16(x86_xCondPrefix), 2,
	0x66, 2403,
	0x0, 2397,
	/*2397*/ uint16(x86_xSetOp), uint16(x86_PHSUBD),
	/*2399*/ uint16(x86_xReadSlashR),
	/*2400*/ uint16(x86_xArgMm1),
	/*2401*/ uint16(x86_xArgMm2M64),
	/*2402*/ uint16(x86_xMatch),
	/*2403*/ uint16(x86_xSetOp), uint16(x86_PHSUBD),
	/*2405*/ uint16(x86_xReadSlashR),
	/*2406*/ uint16(x86_xArgXmm1),
	/*2407*/ uint16(x86_xArgXmm2M128),
	/*2408*/ uint16(x86_xMatch),
	/*2409*/ uint16(x86_xCondPrefix), 2,
	0x66, 2421,
	0x0, 2415,
	/*2415*/ uint16(x86_xSetOp), uint16(x86_PHSUBSW),
	/*2417*/ uint16(x86_xReadSlashR),
	/*2418*/ uint16(x86_xArgMm1),
	/*2419*/ uint16(x86_xArgMm2M64),
	/*2420*/ uint16(x86_xMatch),
	/*2421*/ uint16(x86_xSetOp), uint16(x86_PHSUBSW),
	/*2423*/ uint16(x86_xReadSlashR),
	/*2424*/ uint16(x86_xArgXmm1),
	/*2425*/ uint16(x86_xArgXmm2M128),
	/*2426*/ uint16(x86_xMatch),
	/*2427*/ uint16(x86_xCondPrefix), 2,
	0x66, 2439,
	0x0, 2433,
	/*2433*/ uint16(x86_xSetOp), uint16(x86_PSIGNB),
	/*2435*/ uint16(x86_xReadSlashR),
	/*2436*/ uint16(x86_xArgMm1),
	/*2437*/ uint16(x86_xArgMm2M64),
	/*2438*/ uint16(x86_xMatch),
	/*2439*/ uint16(x86_xSetOp), uint16(x86_PSIGNB),
	/*2441*/ uint16(x86_xReadSlashR),
	/*2442*/ uint16(x86_xArgXmm1),
	/*2443*/ uint16(x86_xArgXmm2M128),
	/*2444*/ uint16(x86_xMatch),
	/*2445*/ uint16(x86_xCondPrefix), 2,
	0x66, 2457,
	0x0, 2451,
	/*2451*/ uint16(x86_xSetOp), uint16(x86_PSIGNW),
	/*2453*/ uint16(x86_xReadSlashR),
	/*2454*/ uint16(x86_xArgMm1),
	/*2455*/ uint16(x86_xArgMm2M64),
	/*2456*/ uint16(x86_xMatch),
	/*2457*/ uint16(x86_xSetOp), uint16(x86_PSIGNW),
	/*2459*/ uint16(x86_xReadSlashR),
	/*2460*/ uint16(x86_xArgXmm1),
	/*2461*/ uint16(x86_xArgXmm2M128),
	/*2462*/ uint16(x86_xMatch),
	/*2463*/ uint16(x86_xCondPrefix), 2,
	0x66, 2475,
	0x0, 2469,
	/*2469*/ uint16(x86_xSetOp), uint16(x86_PSIGND),
	/*2471*/ uint16(x86_xReadSlashR),
	/*2472*/ uint16(x86_xArgMm1),
	/*2473*/ uint16(x86_xArgMm2M64),
	/*2474*/ uint16(x86_xMatch),
	/*2475*/ uint16(x86_xSetOp), uint16(x86_PSIGND),
	/*2477*/ uint16(x86_xReadSlashR),
	/*2478*/ uint16(x86_xArgXmm1),
	/*2479*/ uint16(x86_xArgXmm2M128),
	/*2480*/ uint16(x86_xMatch),
	/*2481*/ uint16(x86_xCondPrefix), 2,
	0x66, 2493,
	0x0, 2487,
	/*2487*/ uint16(x86_xSetOp), uint16(x86_PMULHRSW),
	/*2489*/ uint16(x86_xReadSlashR),
	/*2490*/ uint16(x86_xArgMm1),
	/*2491*/ uint16(x86_xArgMm2M64),
	/*2492*/ uint16(x86_xMatch),
	/*2493*/ uint16(x86_xSetOp), uint16(x86_PMULHRSW),
	/*2495*/ uint16(x86_xReadSlashR),
	/*2496*/ uint16(x86_xArgXmm1),
	/*2497*/ uint16(x86_xArgXmm2M128),
	/*2498*/ uint16(x86_xMatch),
	/*2499*/ uint16(x86_xCondPrefix), 1,
	0x66, 2503,
	/*2503*/ uint16(x86_xSetOp), uint16(x86_PBLENDVB),
	/*2505*/ uint16(x86_xReadSlashR),
	/*2506*/ uint16(x86_xArgXmm1),
	/*2507*/ uint16(x86_xArgXmm2M128),
	/*2508*/ uint16(x86_xArgXMM0),
	/*2509*/ uint16(x86_xMatch),
	/*2510*/ uint16(x86_xCondPrefix), 1,
	0x66, 2514,
	/*2514*/ uint16(x86_xSetOp), uint16(x86_BLENDVPS),
	/*2516*/ uint16(x86_xReadSlashR),
	/*2517*/ uint16(x86_xArgXmm1),
	/*2518*/ uint16(x86_xArgXmm2M128),
	/*2519*/ uint16(x86_xArgXMM0),
	/*2520*/ uint16(x86_xMatch),
	/*2521*/ uint16(x86_xCondPrefix), 1,
	0x66, 2525,
	/*2525*/ uint16(x86_xSetOp), uint16(x86_BLENDVPD),
	/*2527*/ uint16(x86_xReadSlashR),
	/*2528*/ uint16(x86_xArgXmm1),
	/*2529*/ uint16(x86_xArgXmm2M128),
	/*2530*/ uint16(x86_xArgXMM0),
	/*2531*/ uint16(x86_xMatch),
	/*2532*/ uint16(x86_xCondPrefix), 1,
	0x66, 2536,
	/*2536*/ uint16(x86_xSetOp), uint16(x86_PTEST),
	/*2538*/ uint16(x86_xReadSlashR),
	/*2539*/ uint16(x86_xArgXmm1),
	/*2540*/ uint16(x86_xArgXmm2M128),
	/*2541*/ uint16(x86_xMatch),
	/*2542*/ uint16(x86_xCondPrefix), 2,
	0x66, 2554,
	0x0, 2548,
	/*2548*/ uint16(x86_xSetOp), uint16(x86_PABSB),
	/*2550*/ uint16(x86_xReadSlashR),
	/*2551*/ uint16(x86_xArgMm1),
	/*2552*/ uint16(x86_xArgMm2M64),
	/*2553*/ uint16(x86_xMatch),
	/*2554*/ uint16(x86_xSetOp), uint16(x86_PABSB),
	/*2556*/ uint16(x86_xReadSlashR),
	/*2557*/ uint16(x86_xArgXmm1),
	/*2558*/ uint16(x86_xArgXmm2M128),
	/*2559*/ uint16(x86_xMatch),
	/*2560*/ uint16(x86_xCondPrefix), 2,
	0x66, 2572,
	0x0, 2566,
	/*2566*/ uint16(x86_xSetOp), uint16(x86_PABSW),
	/*2568*/ uint16(x86_xReadSlashR),
	/*2569*/ uint16(x86_xArgMm1),
	/*2570*/ uint16(x86_xArgMm2M64),
	/*2571*/ uint16(x86_xMatch),
	/*2572*/ uint16(x86_xSetOp), uint16(x86_PABSW),
	/*2574*/ uint16(x86_xReadSlashR),
	/*2575*/ uint16(x86_xArgXmm1),
	/*2576*/ uint16(x86_xArgXmm2M128),
	/*2577*/ uint16(x86_xMatch),
	/*2578*/ uint16(x86_xCondPrefix), 2,
	0x66, 2590,
	0x0, 2584,
	/*2584*/ uint16(x86_xSetOp), uint16(x86_PABSD),
	/*2586*/ uint16(x86_xReadSlashR),
	/*2587*/ uint16(x86_xArgMm1),
	/*2588*/ uint16(x86_xArgMm2M64),
	/*2589*/ uint16(x86_xMatch),
	/*2590*/ uint16(x86_xSetOp), uint16(x86_PABSD),
	/*2592*/ uint16(x86_xReadSlashR),
	/*2593*/ uint16(x86_xArgXmm1),
	/*2594*/ uint16(x86_xArgXmm2M128),
	/*2595*/ uint16(x86_xMatch),
	/*2596*/ uint16(x86_xCondPrefix), 1,
	0x66, 2600,
	/*2600*/ uint16(x86_xSetOp), uint16(x86_PMOVSXBW),
	/*2602*/ uint16(x86_xReadSlashR),
	/*2603*/ uint16(x86_xArgXmm1),
	/*2604*/ uint16(x86_xArgXmm2M64),
	/*2605*/ uint16(x86_xMatch),
	/*2606*/ uint16(x86_xCondPrefix), 1,
	0x66, 2610,
	/*2610*/ uint16(x86_xSetOp), uint16(x86_PMOVSXBD),
	/*2612*/ uint16(x86_xReadSlashR),
	/*2613*/ uint16(x86_xArgXmm1),
	/*2614*/ uint16(x86_xArgXmm2M32),
	/*2615*/ uint16(x86_xMatch),
	/*2616*/ uint16(x86_xCondPrefix), 1,
	0x66, 2620,
	/*2620*/ uint16(x86_xSetOp), uint16(x86_PMOVSXBQ),
	/*2622*/ uint16(x86_xReadSlashR),
	/*2623*/ uint16(x86_xArgXmm1),
	/*2624*/ uint16(x86_xArgXmm2M16),
	/*2625*/ uint16(x86_xMatch),
	/*2626*/ uint16(x86_xCondPrefix), 1,
	0x66, 2630,
	/*2630*/ uint16(x86_xSetOp), uint16(x86_PMOVSXWD),
	/*2632*/ uint16(x86_xReadSlashR),
	/*2633*/ uint16(x86_xArgXmm1),
	/*2634*/ uint16(x86_xArgXmm2M64),
	/*2635*/ uint16(x86_xMatch),
	/*2636*/ uint16(x86_xCondPrefix), 1,
	0x66, 2640,
	/*2640*/ uint16(x86_xSetOp), uint16(x86_PMOVSXWQ),
	/*2642*/ uint16(x86_xReadSlashR),
	/*2643*/ uint16(x86_xArgXmm1),
	/*2644*/ uint16(x86_xArgXmm2M32),
	/*2645*/ uint16(x86_xMatch),
	/*2646*/ uint16(x86_xCondPrefix), 1,
	0x66, 2650,
	/*2650*/ uint16(x86_xSetOp), uint16(x86_PMOVSXDQ),
	/*2652*/ uint16(x86_xReadSlashR),
	/*2653*/ uint16(x86_xArgXmm1),
	/*2654*/ uint16(x86_xArgXmm2M64),
	/*2655*/ uint16(x86_xMatch),
	/*2656*/ uint16(x86_xCondPrefix), 1,
	0x66, 2660,
	/*2660*/ uint16(x86_xSetOp), uint16(x86_PMULDQ),
	/*2662*/ uint16(x86_xReadSlashR),
	/*2663*/ uint16(x86_xArgXmm1),
	/*2664*/ uint16(x86_xArgXmm2M128),
	/*2665*/ uint16(x86_xMatch),
	/*2666*/ uint16(x86_xCondPrefix), 1,
	0x66, 2670,
	/*2670*/ uint16(x86_xSetOp), uint16(x86_PCMPEQQ),
	/*2672*/ uint16(x86_xReadSlashR),
	/*2673*/ uint16(x86_xArgXmm1),
	/*2674*/ uint16(x86_xArgXmm2M128),
	/*2675*/ uint16(x86_xMatch),
	/*2676*/ uint16(x86_xCondPrefix), 1,
	0x66, 2680,
	/*2680*/ uint16(x86_xSetOp), uint16(x86_MOVNTDQA),
	/*2682*/ uint16(x86_xReadSlashR),
	/*2683*/ uint16(x86_xArgXmm1),
	/*2684*/ uint16(x86_xArgM128),
	/*2685*/ uint16(x86_xMatch),
	/*2686*/ uint16(x86_xCondPrefix), 1,
	0x66, 2690,
	/*2690*/ uint16(x86_xSetOp), uint16(x86_PACKUSDW),
	/*2692*/ uint16(x86_xReadSlashR),
	/*2693*/ uint16(x86_xArgXmm1),
	/*2694*/ uint16(x86_xArgXmm2M128),
	/*2695*/ uint16(x86_xMatch),
	/*2696*/ uint16(x86_xCondPrefix), 1,
	0x66, 2700,
	/*2700*/ uint16(x86_xSetOp), uint16(x86_PMOVZXBW),
	/*2702*/ uint16(x86_xReadSlashR),
	/*2703*/ uint16(x86_xArgXmm1),
	/*2704*/ uint16(x86_xArgXmm2M64),
	/*2705*/ uint16(x86_xMatch),
	/*2706*/ uint16(x86_xCondPrefix), 1,
	0x66, 2710,
	/*2710*/ uint16(x86_xSetOp), uint16(x86_PMOVZXBD),
	/*2712*/ uint16(x86_xReadSlashR),
	/*2713*/ uint16(x86_xArgXmm1),
	/*2714*/ uint16(x86_xArgXmm2M32),
	/*2715*/ uint16(x86_xMatch),
	/*2716*/ uint16(x86_xCondPrefix), 1,
	0x66, 2720,
	/*2720*/ uint16(x86_xSetOp), uint16(x86_PMOVZXBQ),
	/*2722*/ uint16(x86_xReadSlashR),
	/*2723*/ uint16(x86_xArgXmm1),
	/*2724*/ uint16(x86_xArgXmm2M16),
	/*2725*/ uint16(x86_xMatch),
	/*2726*/ uint16(x86_xCondPrefix), 1,
	0x66, 2730,
	/*2730*/ uint16(x86_xSetOp), uint16(x86_PMOVZXWD),
	/*2732*/ uint16(x86_xReadSlashR),
	/*2733*/ uint16(x86_xArgXmm1),
	/*2734*/ uint16(x86_xArgXmm2M64),
	/*2735*/ uint16(x86_xMatch),
	/*2736*/ uint16(x86_xCondPrefix), 1,
	0x66, 2740,
	/*2740*/ uint16(x86_xSetOp), uint16(x86_PMOVZXWQ),
	/*2742*/ uint16(x86_xReadSlashR),
	/*2743*/ uint16(x86_xArgXmm1),
	/*2744*/ uint16(x86_xArgXmm2M32),
	/*2745*/ uint16(x86_xMatch),
	/*2746*/ uint16(x86_xCondPrefix), 1,
	0x66, 2750,
	/*2750*/ uint16(x86_xSetOp), uint16(x86_PMOVZXDQ),
	/*2752*/ uint16(x86_xReadSlashR),
	/*2753*/ uint16(x86_xArgXmm1),
	/*2754*/ uint16(x86_xArgXmm2M64),
	/*2755*/ uint16(x86_xMatch),
	/*2756*/ uint16(x86_xCondPrefix), 1,
	0x66, 2760,
	/*2760*/ uint16(x86_xSetOp), uint16(x86_PCMPGTQ),
	/*2762*/ uint16(x86_xReadSlashR),
	/*2763*/ uint16(x86_xArgXmm1),
	/*2764*/ uint16(x86_xArgXmm2M128),
	/*2765*/ uint16(x86_xMatch),
	/*2766*/ uint16(x86_xCondPrefix), 1,
	0x66, 2770,
	/*2770*/ uint16(x86_xSetOp), uint16(x86_PMINSB),
	/*2772*/ uint16(x86_xReadSlashR),
	/*2773*/ uint16(x86_xArgXmm1),
	/*2774*/ uint16(x86_xArgXmm2M128),
	/*2775*/ uint16(x86_xMatch),
	/*2776*/ uint16(x86_xCondPrefix), 1,
	0x66, 2780,
	/*2780*/ uint16(x86_xSetOp), uint16(x86_PMINSD),
	/*2782*/ uint16(x86_xReadSlashR),
	/*2783*/ uint16(x86_xArgXmm1),
	/*2784*/ uint16(x86_xArgXmm2M128),
	/*2785*/ uint16(x86_xMatch),
	/*2786*/ uint16(x86_xCondPrefix), 1,
	0x66, 2790,
	/*2790*/ uint16(x86_xSetOp), uint16(x86_PMINUW),
	/*2792*/ uint16(x86_xReadSlashR),
	/*2793*/ uint16(x86_xArgXmm1),
	/*2794*/ uint16(x86_xArgXmm2M128),
	/*2795*/ uint16(x86_xMatch),
	/*2796*/ uint16(x86_xCondPrefix), 1,
	0x66, 2800,
	/*2800*/ uint16(x86_xSetOp), uint16(x86_PMINUD),
	/*2802*/ uint16(x86_xReadSlashR),
	/*2803*/ uint16(x86_xArgXmm1),
	/*2804*/ uint16(x86_xArgXmm2M128),
	/*2805*/ uint16(x86_xMatch),
	/*2806*/ uint16(x86_xCondPrefix), 1,
	0x66, 2810,
	/*2810*/ uint16(x86_xSetOp), uint16(x86_PMAXSB),
	/*2812*/ uint16(x86_xReadSlashR),
	/*2813*/ uint16(x86_xArgXmm1),
	/*2814*/ uint16(x86_xArgXmm2M128),
	/*2815*/ uint16(x86_xMatch),
	/*2816*/ uint16(x86_xCondPrefix), 1,
	0x66, 2820,
	/*2820*/ uint16(x86_xSetOp), uint16(x86_PMAXSD),
	/*2822*/ uint16(x86_xReadSlashR),
	/*2823*/ uint16(x86_xArgXmm1),
	/*2824*/ uint16(x86_xArgXmm2M128),
	/*2825*/ uint16(x86_xMatch),
	/*2826*/ uint16(x86_xCondPrefix), 1,
	0x66, 2830,
	/*2830*/ uint16(x86_xSetOp), uint16(x86_PMAXUW),
	/*2832*/ uint16(x86_xReadSlashR),
	/*2833*/ uint16(x86_xArgXmm1),
	/*2834*/ uint16(x86_xArgXmm2M128),
	/*2835*/ uint16(x86_xMatch),
	/*2836*/ uint16(x86_xCondPrefix), 1,
	0x66, 2840,
	/*2840*/ uint16(x86_xSetOp), uint16(x86_PMAXUD),
	/*2842*/ uint16(x86_xReadSlashR),
	/*2843*/ uint16(x86_xArgXmm1),
	/*2844*/ uint16(x86_xArgXmm2M128),
	/*2845*/ uint16(x86_xMatch),
	/*2846*/ uint16(x86_xCondPrefix), 1,
	0x66, 2850,
	/*2850*/ uint16(x86_xSetOp), uint16(x86_PMULLD),
	/*2852*/ uint16(x86_xReadSlashR),
	/*2853*/ uint16(x86_xArgXmm1),
	/*2854*/ uint16(x86_xArgXmm2M128),
	/*2855*/ uint16(x86_xMatch),
	/*2856*/ uint16(x86_xCondPrefix), 1,
	0x66, 2860,
	/*2860*/ uint16(x86_xSetOp), uint16(x86_PHMINPOSUW),
	/*2862*/ uint16(x86_xReadSlashR),
	/*2863*/ uint16(x86_xArgXmm1),
	/*2864*/ uint16(x86_xArgXmm2M128),
	/*2865*/ uint16(x86_xMatch),
	/*2866*/ uint16(x86_xCondIs64), 2869, 2879,
	/*2869*/ uint16(x86_xCondPrefix), 1,
	0x66, 2873,
	/*2873*/ uint16(x86_xSetOp), uint16(x86_INVPCID),
	/*2875*/ uint16(x86_xReadSlashR),
	/*2876*/ uint16(x86_xArgR32),
	/*2877*/ uint16(x86_xArgM128),
	/*2878*/ uint16(x86_xMatch),
	/*2879*/ uint16(x86_xCondPrefix), 1,
	0x66, 2883,
	/*2883*/ uint16(x86_xSetOp), uint16(x86_INVPCID),
	/*2885*/ uint16(x86_xReadSlashR),
	/*2886*/ uint16(x86_xArgR64),
	/*2887*/ uint16(x86_xArgM128),
	/*2888*/ uint16(x86_xMatch),
	/*2889*/ uint16(x86_xCondPrefix), 1,
	0x66, 2893,
	/*2893*/ uint16(x86_xSetOp), uint16(x86_AESIMC),
	/*2895*/ uint16(x86_xReadSlashR),
	/*2896*/ uint16(x86_xArgXmm1),
	/*2897*/ uint16(x86_xArgXmm2M128),
	/*2898*/ uint16(x86_xMatch),
	/*2899*/ uint16(x86_xCondPrefix), 1,
	0x66, 2903,
	/*2903*/ uint16(x86_xSetOp), uint16(x86_AESENC),
	/*2905*/ uint16(x86_xReadSlashR),
	/*2906*/ uint16(x86_xArgXmm1),
	/*2907*/ uint16(x86_xArgXmm2M128),
	/*2908*/ uint16(x86_xMatch),
	/*2909*/ uint16(x86_xCondPrefix), 1,
	0x66, 2913,
	/*2913*/ uint16(x86_xSetOp), uint16(x86_AESENCLAST),
	/*2915*/ uint16(x86_xReadSlashR),
	/*2916*/ uint16(x86_xArgXmm1),
	/*2917*/ uint16(x86_xArgXmm2M128),
	/*2918*/ uint16(x86_xMatch),
	/*2919*/ uint16(x86_xCondPrefix), 1,
	0x66, 2923,
	/*2923*/ uint16(x86_xSetOp), uint16(x86_AESDEC),
	/*2925*/ uint16(x86_xReadSlashR),
	/*2926*/ uint16(x86_xArgXmm1),
	/*2927*/ uint16(x86_xArgXmm2M128),
	/*2928*/ uint16(x86_xMatch),
	/*2929*/ uint16(x86_xCondPrefix), 1,
	0x66, 2933,
	/*2933*/ uint16(x86_xSetOp), uint16(x86_AESDECLAST),
	/*2935*/ uint16(x86_xReadSlashR),
	/*2936*/ uint16(x86_xArgXmm1),
	/*2937*/ uint16(x86_xArgXmm2M128),
	/*2938*/ uint16(x86_xMatch),
	/*2939*/ uint16(x86_xCondIs64), 2942, 2980,
	/*2942*/ uint16(x86_xCondPrefix), 2,
	0xF2, 2964,
	0x0, 2948,
	/*2948*/ uint16(x86_xCondDataSize), 2952, 2958, 0,
	/*2952*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*2954*/ uint16(x86_xReadSlashR),
	/*2955*/ uint16(x86_xArgR16),
	/*2956*/ uint16(x86_xArgM16),
	/*2957*/ uint16(x86_xMatch),
	/*2958*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*2960*/ uint16(x86_xReadSlashR),
	/*2961*/ uint16(x86_xArgR32),
	/*2962*/ uint16(x86_xArgM32),
	/*2963*/ uint16(x86_xMatch),
	/*2964*/ uint16(x86_xCondDataSize), 2968, 2974, 0,
	/*2968*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*2970*/ uint16(x86_xReadSlashR),
	/*2971*/ uint16(x86_xArgR32),
	/*2972*/ uint16(x86_xArgRM8),
	/*2973*/ uint16(x86_xMatch),
	/*2974*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*2976*/ uint16(x86_xReadSlashR),
	/*2977*/ uint16(x86_xArgR32),
	/*2978*/ uint16(x86_xArgRM8),
	/*2979*/ uint16(x86_xMatch),
	/*2980*/ uint16(x86_xCondPrefix), 2,
	0xF2, 2996,
	0x0, 2986,
	/*2986*/ uint16(x86_xCondDataSize), 2952, 2958, 2990,
	/*2990*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*2992*/ uint16(x86_xReadSlashR),
	/*2993*/ uint16(x86_xArgR64),
	/*2994*/ uint16(x86_xArgM64),
	/*2995*/ uint16(x86_xMatch),
	/*2996*/ uint16(x86_xCondDataSize), 2968, 2974, 3000,
	/*3000*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*3002*/ uint16(x86_xReadSlashR),
	/*3003*/ uint16(x86_xArgR64),
	/*3004*/ uint16(x86_xArgRM8),
	/*3005*/ uint16(x86_xMatch),
	/*3006*/ uint16(x86_xCondIs64), 3009, 3047,
	/*3009*/ uint16(x86_xCondPrefix), 2,
	0xF2, 3031,
	0x0, 3015,
	/*3015*/ uint16(x86_xCondDataSize), 3019, 3025, 0,
	/*3019*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*3021*/ uint16(x86_xReadSlashR),
	/*3022*/ uint16(x86_xArgM16),
	/*3023*/ uint16(x86_xArgR16),
	/*3024*/ uint16(x86_xMatch),
	/*3025*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*3027*/ uint16(x86_xReadSlashR),
	/*3028*/ uint16(x86_xArgM32),
	/*3029*/ uint16(x86_xArgR32),
	/*3030*/ uint16(x86_xMatch),
	/*3031*/ uint16(x86_xCondDataSize), 3035, 3041, 0,
	/*3035*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*3037*/ uint16(x86_xReadSlashR),
	/*3038*/ uint16(x86_xArgR32),
	/*3039*/ uint16(x86_xArgRM16),
	/*3040*/ uint16(x86_xMatch),
	/*3041*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*3043*/ uint16(x86_xReadSlashR),
	/*3044*/ uint16(x86_xArgR32),
	/*3045*/ uint16(x86_xArgRM32),
	/*3046*/ uint16(x86_xMatch),
	/*3047*/ uint16(x86_xCondPrefix), 2,
	0xF2, 3063,
	0x0, 3053,
	/*3053*/ uint16(x86_xCondDataSize), 3019, 3025, 3057,
	/*3057*/ uint16(x86_xSetOp), uint16(x86_MOVBE),
	/*3059*/ uint16(x86_xReadSlashR),
	/*3060*/ uint16(x86_xArgM64),
	/*3061*/ uint16(x86_xArgR64),
	/*3062*/ uint16(x86_xMatch),
	/*3063*/ uint16(x86_xCondDataSize), 3035, 3041, 3067,
	/*3067*/ uint16(x86_xSetOp), uint16(x86_CRC32),
	/*3069*/ uint16(x86_xReadSlashR),
	/*3070*/ uint16(x86_xArgR64),
	/*3071*/ uint16(x86_xArgRM64),
	/*3072*/ uint16(x86_xMatch),
	/*3073*/ uint16(x86_xCondByte), 24,
	0x08, 3124,
	0x09, 3136,
	0x0A, 3148,
	0x0B, 3160,
	0x0C, 3172,
	0x0D, 3184,
	0x0E, 3196,
	0x0F, 3208,
	0x14, 3230,
	0x15, 3242,
	0x16, 3254,
	0x17, 3297,
	0x20, 3309,
	0x21, 3321,
	0x22, 3333,
	0x40, 3376,
	0x41, 3388,
	0x42, 3400,
	0x44, 3412,
	0x60, 3424,
	0x61, 3436,
	0x62, 3448,
	0x63, 3460,
	0xDF, 3472,
	uint16(x86_xFail),
	/*3124*/ uint16(x86_xCondPrefix), 1,
	0x66, 3128,
	/*3128*/ uint16(x86_xSetOp), uint16(x86_ROUNDPS),
	/*3130*/ uint16(x86_xReadSlashR),
	/*3131*/ uint16(x86_xReadIb),
	/*3132*/ uint16(x86_xArgXmm1),
	/*3133*/ uint16(x86_xArgXmm2M128),
	/*3134*/ uint16(x86_xArgImm8u),
	/*3135*/ uint16(x86_xMatch),
	/*3136*/ uint16(x86_xCondPrefix), 1,
	0x66, 3140,
	/*3140*/ uint16(x86_xSetOp), uint16(x86_ROUNDPD),
	/*3142*/ uint16(x86_xReadSlashR),
	/*3143*/ uint16(x86_xReadIb),
	/*3144*/ uint16(x86_xArgXmm1),
	/*3145*/ uint16(x86_xArgXmm2M128),
	/*3146*/ uint16(x86_xArgImm8u),
	/*3147*/ uint16(x86_xMatch),
	/*3148*/ uint16(x86_xCondPrefix), 1,
	0x66, 3152,
	/*3152*/ uint16(x86_xSetOp), uint16(x86_ROUNDSS),
	/*3154*/ uint16(x86_xReadSlashR),
	/*3155*/ uint16(x86_xReadIb),
	/*3156*/ uint16(x86_xArgXmm1),
	/*3157*/ uint16(x86_xArgXmm2M32),
	/*3158*/ uint16(x86_xArgImm8u),
	/*3159*/ uint16(x86_xMatch),
	/*3160*/ uint16(x86_xCondPrefix), 1,
	0x66, 3164,
	/*3164*/ uint16(x86_xSetOp), uint16(x86_ROUNDSD),
	/*3166*/ uint16(x86_xReadSlashR),
	/*3167*/ uint16(x86_xReadIb),
	/*3168*/ uint16(x86_xArgXmm1),
	/*3169*/ uint16(x86_xArgXmm2M64),
	/*3170*/ uint16(x86_xArgImm8u),
	/*3171*/ uint16(x86_xMatch),
	/*3172*/ uint16(x86_xCondPrefix), 1,
	0x66, 3176,
	/*3176*/ uint16(x86_xSetOp), uint16(x86_BLENDPS),
	/*3178*/ uint16(x86_xReadSlashR),
	/*3179*/ uint16(x86_xReadIb),
	/*3180*/ uint16(x86_xArgXmm1),
	/*3181*/ uint16(x86_xArgXmm2M128),
	/*3182*/ uint16(x86_xArgImm8u),
	/*3183*/ uint16(x86_xMatch),
	/*3184*/ uint16(x86_xCondPrefix), 1,
	0x66, 3188,
	/*3188*/ uint16(x86_xSetOp), uint16(x86_BLENDPD),
	/*3190*/ uint16(x86_xReadSlashR),
	/*3191*/ uint16(x86_xReadIb),
	/*3192*/ uint16(x86_xArgXmm1),
	/*3193*/ uint16(x86_xArgXmm2M128),
	/*3194*/ uint16(x86_xArgImm8u),
	/*3195*/ uint16(x86_xMatch),
	/*3196*/ uint16(x86_xCondPrefix), 1,
	0x66, 3200,
	/*3200*/ uint16(x86_xSetOp), uint16(x86_PBLENDW),
	/*3202*/ uint16(x86_xReadSlashR),
	/*3203*/ uint16(x86_xReadIb),
	/*3204*/ uint16(x86_xArgXmm1),
	/*3205*/ uint16(x86_xArgXmm2M128),
	/*3206*/ uint16(x86_xArgImm8u),
	/*3207*/ uint16(x86_xMatch),
	/*3208*/ uint16(x86_xCondPrefix), 2,
	0x66, 3222,
	0x0, 3214,
	/*3214*/ uint16(x86_xSetOp), uint16(x86_PALIGNR),
	/*3216*/ uint16(x86_xReadSlashR),
	/*3217*/ uint16(x86_xReadIb),
	/*3218*/ uint16(x86_xArgMm1),
	/*3219*/ uint16(x86_xArgMm2M64),
	/*3220*/ uint16(x86_xArgImm8u),
	/*3221*/ uint16(x86_xMatch),
	/*3222*/ uint16(x86_xSetOp), uint16(x86_PALIGNR),
	/*3224*/ uint16(x86_xReadSlashR),
	/*3225*/ uint16(x86_xReadIb),
	/*3226*/ uint16(x86_xArgXmm1),
	/*3227*/ uint16(x86_xArgXmm2M128),
	/*3228*/ uint16(x86_xArgImm8u),
	/*3229*/ uint16(x86_xMatch),
	/*3230*/ uint16(x86_xCondPrefix), 1,
	0x66, 3234,
	/*3234*/ uint16(x86_xSetOp), uint16(x86_PEXTRB),
	/*3236*/ uint16(x86_xReadSlashR),
	/*3237*/ uint16(x86_xReadIb),
	/*3238*/ uint16(x86_xArgR32M8),
	/*3239*/ uint16(x86_xArgXmm1),
	/*3240*/ uint16(x86_xArgImm8u),
	/*3241*/ uint16(x86_xMatch),
	/*3242*/ uint16(x86_xCondPrefix), 1,
	0x66, 3246,
	/*3246*/ uint16(x86_xSetOp), uint16(x86_PEXTRW),
	/*3248*/ uint16(x86_xReadSlashR),
	/*3249*/ uint16(x86_xReadIb),
	/*3250*/ uint16(x86_xArgR32M16),
	/*3251*/ uint16(x86_xArgXmm1),
	/*3252*/ uint16(x86_xArgImm8u),
	/*3253*/ uint16(x86_xMatch),
	/*3254*/ uint16(x86_xCondIs64), 3257, 3281,
	/*3257*/ uint16(x86_xCondPrefix), 1,
	0x66, 3261,
	/*3261*/ uint16(x86_xCondDataSize), 3265, 3273, 0,
	/*3265*/ uint16(x86_xSetOp), uint16(x86_PEXTRD),
	/*3267*/ uint16(x86_xReadSlashR),
	/*3268*/ uint16(x86_xReadIb),
	/*3269*/ uint16(x86_xArgRM32),
	/*3270*/ uint16(x86_xArgXmm1),
	/*3271*/ uint16(x86_xArgImm8u),
	/*3272*/ uint16(x86_xMatch),
	/*3273*/ uint16(x86_xSetOp), uint16(x86_PEXTRD),
	/*3275*/ uint16(x86_xReadSlashR),
	/*3276*/ uint16(x86_xReadIb),
	/*3277*/ uint16(x86_xArgRM32),
	/*3278*/ uint16(x86_xArgXmm1),
	/*3279*/ uint16(x86_xArgImm8u),
	/*3280*/ uint16(x86_xMatch),
	/*3281*/ uint16(x86_xCondPrefix), 1,
	0x66, 3285,
	/*3285*/ uint16(x86_xCondDataSize), 3265, 3273, 3289,
	/*3289*/ uint16(x86_xSetOp), uint16(x86_PEXTRQ),
	/*3291*/ uint16(x86_xReadSlashR),
	/*3292*/ uint16(x86_xReadIb),
	/*3293*/ uint16(x86_xArgRM64),
	/*3294*/ uint16(x86_xArgXmm1),
	/*3295*/ uint16(x86_xArgImm8u),
	/*3296*/ uint16(x86_xMatch),
	/*3297*/ uint16(x86_xCondPrefix), 1,
	0x66, 3301,
	/*3301*/ uint16(x86_xSetOp), uint16(x86_EXTRACTPS),
	/*3303*/ uint16(x86_xReadSlashR),
	/*3304*/ uint16(x86_xReadIb),
	/*3305*/ uint16(x86_xArgRM32),
	/*3306*/ uint16(x86_xArgXmm1),
	/*3307*/ uint16(x86_xArgImm8u),
	/*3308*/ uint16(x86_xMatch),
	/*3309*/ uint16(x86_xCondPrefix), 1,
	0x66, 3313,
	/*3313*/ uint16(x86_xSetOp), uint16(x86_PINSRB),
	/*3315*/ uint16(x86_xReadSlashR),
	/*3316*/ uint16(x86_xReadIb),
	/*3317*/ uint16(x86_xArgXmm1),
	/*3318*/ uint16(x86_xArgR32M8),
	/*3319*/ uint16(x86_xArgImm8u),
	/*3320*/ uint16(x86_xMatch),
	/*3321*/ uint16(x86_xCondPrefix), 1,
	0x66, 3325,
	/*3325*/ uint16(x86_xSetOp), uint16(x86_INSERTPS),
	/*3327*/ uint16(x86_xReadSlashR),
	/*3328*/ uint16(x86_xReadIb),
	/*3329*/ uint16(x86_xArgXmm1),
	/*3330*/ uint16(x86_xArgXmm2M32),
	/*3331*/ uint16(x86_xArgImm8u),
	/*3332*/ uint16(x86_xMatch),
	/*3333*/ uint16(x86_xCondIs64), 3336, 3360,
	/*3336*/ uint16(x86_xCondPrefix), 1,
	0x66, 3340,
	/*3340*/ uint16(x86_xCondDataSize), 3344, 3352, 0,
	/*3344*/ uint16(x86_xSetOp), uint16(x86_PINSRD),
	/*3346*/ uint16(x86_xReadSlashR),
	/*3347*/ uint16(x86_xReadIb),
	/*3348*/ uint16(x86_xArgXmm1),
	/*3349*/ uint16(x86_xArgRM32),
	/*3350*/ uint16(x86_xArgImm8u),
	/*3351*/ uint16(x86_xMatch),
	/*3352*/ uint16(x86_xSetOp), uint16(x86_PINSRD),
	/*3354*/ uint16(x86_xReadSlashR),
	/*3355*/ uint16(x86_xReadIb),
	/*3356*/ uint16(x86_xArgXmm1),
	/*3357*/ uint16(x86_xArgRM32),
	/*3358*/ uint16(x86_xArgImm8u),
	/*3359*/ uint16(x86_xMatch),
	/*3360*/ uint16(x86_xCondPrefix), 1,
	0x66, 3364,
	/*3364*/ uint16(x86_xCondDataSize), 3344, 3352, 3368,
	/*3368*/ uint16(x86_xSetOp), uint16(x86_PINSRQ),
	/*3370*/ uint16(x86_xReadSlashR),
	/*3371*/ uint16(x86_xReadIb),
	/*3372*/ uint16(x86_xArgXmm1),
	/*3373*/ uint16(x86_xArgRM64),
	/*3374*/ uint16(x86_xArgImm8u),
	/*3375*/ uint16(x86_xMatch),
	/*3376*/ uint16(x86_xCondPrefix), 1,
	0x66, 3380,
	/*3380*/ uint16(x86_xSetOp), uint16(x86_DPPS),
	/*3382*/ uint16(x86_xReadSlashR),
	/*3383*/ uint16(x86_xReadIb),
	/*3384*/ uint16(x86_xArgXmm1),
	/*3385*/ uint16(x86_xArgXmm2M128),
	/*3386*/ uint16(x86_xArgImm8u),
	/*3387*/ uint16(x86_xMatch),
	/*3388*/ uint16(x86_xCondPrefix), 1,
	0x66, 3392,
	/*3392*/ uint16(x86_xSetOp), uint16(x86_DPPD),
	/*3394*/ uint16(x86_xReadSlashR),
	/*3395*/ uint16(x86_xReadIb),
	/*3396*/ uint16(x86_xArgXmm1),
	/*3397*/ uint16(x86_xArgXmm2M128),
	/*3398*/ uint16(x86_xArgImm8u),
	/*3399*/ uint16(x86_xMatch),
	/*3400*/ uint16(x86_xCondPrefix), 1,
	0x66, 3404,
	/*3404*/ uint16(x86_xSetOp), uint16(x86_MPSADBW),
	/*3406*/ uint16(x86_xReadSlashR),
	/*3407*/ uint16(x86_xReadIb),
	/*3408*/ uint16(x86_xArgXmm1),
	/*3409*/ uint16(x86_xArgXmm2M128),
	/*3410*/ uint16(x86_xArgImm8u),
	/*3411*/ uint16(x86_xMatch),
	/*3412*/ uint16(x86_xCondPrefix), 1,
	0x66, 3416,
	/*3416*/ uint16(x86_xSetOp), uint16(x86_PCLMULQDQ),
	/*3418*/ uint16(x86_xReadSlashR),
	/*3419*/ uint16(x86_xReadIb),
	/*3420*/ uint16(x86_xArgXmm1),
	/*3421*/ uint16(x86_xArgXmm2M128),
	/*3422*/ uint16(x86_xArgImm8u),
	/*3423*/ uint16(x86_xMatch),
	/*3424*/ uint16(x86_xCondPrefix), 1,
	0x66, 3428,
	/*3428*/ uint16(x86_xSetOp), uint16(x86_PCMPESTRM),
	/*3430*/ uint16(x86_xReadSlashR),
	/*3431*/ uint16(x86_xReadIb),
	/*3432*/ uint16(x86_xArgXmm1),
	/*3433*/ uint16(x86_xArgXmm2M128),
	/*3434*/ uint16(x86_xArgImm8u),
	/*3435*/ uint16(x86_xMatch),
	/*3436*/ uint16(x86_xCondPrefix), 1,
	0x66, 3440,
	/*3440*/ uint16(x86_xSetOp), uint16(x86_PCMPESTRI),
	/*3442*/ uint16(x86_xReadSlashR),
	/*3443*/ uint16(x86_xReadIb),
	/*3444*/ uint16(x86_xArgXmm1),
	/*3445*/ uint16(x86_xArgXmm2M128),
	/*3446*/ uint16(x86_xArgImm8u),
	/*3447*/ uint16(x86_xMatch),
	/*3448*/ uint16(x86_xCondPrefix), 1,
	0x66, 3452,
	/*3452*/ uint16(x86_xSetOp), uint16(x86_PCMPISTRM),
	/*3454*/ uint16(x86_xReadSlashR),
	/*3455*/ uint16(x86_xReadIb),
	/*3456*/ uint16(x86_xArgXmm1),
	/*3457*/ uint16(x86_xArgXmm2M128),
	/*3458*/ uint16(x86_xArgImm8u),
	/*3459*/ uint16(x86_xMatch),
	/*3460*/ uint16(x86_xCondPrefix), 1,
	0x66, 3464,
	/*3464*/ uint16(x86_xSetOp), uint16(x86_PCMPISTRI),
	/*3466*/ uint16(x86_xReadSlashR),
	/*3467*/ uint16(x86_xReadIb),
	/*3468*/ uint16(x86_xArgXmm1),
	/*3469*/ uint16(x86_xArgXmm2M128),
	/*3470*/ uint16(x86_xArgImm8u),
	/*3471*/ uint16(x86_xMatch),
	/*3472*/ uint16(x86_xCondPrefix), 1,
	0x66, 3476,
	/*3476*/ uint16(x86_xSetOp), uint16(x86_AESKEYGENASSIST),
	/*3478*/ uint16(x86_xReadSlashR),
	/*3479*/ uint16(x86_xReadIb),
	/*3480*/ uint16(x86_xArgXmm1),
	/*3481*/ uint16(x86_xArgXmm2M128),
	/*3482*/ uint16(x86_xArgImm8u),
	/*3483*/ uint16(x86_xMatch),
	/*3484*/ uint16(x86_xCondIs64), 3487, 3503,
	/*3487*/ uint16(x86_xCondDataSize), 3491, 3497, 0,
	/*3491*/ uint16(x86_xSetOp), uint16(x86_CMOVO),
	/*3493*/ uint16(x86_xReadSlashR),
	/*3494*/ uint16(x86_xArgR16),
	/*3495*/ uint16(x86_xArgRM16),
	/*3496*/ uint16(x86_xMatch),
	/*3497*/ uint16(x86_xSetOp), uint16(x86_CMOVO),
	/*3499*/ uint16(x86_xReadSlashR),
	/*3500*/ uint16(x86_xArgR32),
	/*3501*/ uint16(x86_xArgRM32),
	/*3502*/ uint16(x86_xMatch),
	/*3503*/ uint16(x86_xCondDataSize), 3491, 3497, 3507,
	/*3507*/ uint16(x86_xSetOp), uint16(x86_CMOVO),
	/*3509*/ uint16(x86_xReadSlashR),
	/*3510*/ uint16(x86_xArgR64),
	/*3511*/ uint16(x86_xArgRM64),
	/*3512*/ uint16(x86_xMatch),
	/*3513*/ uint16(x86_xCondIs64), 3516, 3532,
	/*3516*/ uint16(x86_xCondDataSize), 3520, 3526, 0,
	/*3520*/ uint16(x86_xSetOp), uint16(x86_CMOVNO),
	/*3522*/ uint16(x86_xReadSlashR),
	/*3523*/ uint16(x86_xArgR16),
	/*3524*/ uint16(x86_xArgRM16),
	/*3525*/ uint16(x86_xMatch),
	/*3526*/ uint16(x86_xSetOp), uint16(x86_CMOVNO),
	/*3528*/ uint16(x86_xReadSlashR),
	/*3529*/ uint16(x86_xArgR32),
	/*3530*/ uint16(x86_xArgRM32),
	/*3531*/ uint16(x86_xMatch),
	/*3532*/ uint16(x86_xCondDataSize), 3520, 3526, 3536,
	/*3536*/ uint16(x86_xSetOp), uint16(x86_CMOVNO),
	/*3538*/ uint16(x86_xReadSlashR),
	/*3539*/ uint16(x86_xArgR64),
	/*3540*/ uint16(x86_xArgRM64),
	/*3541*/ uint16(x86_xMatch),
	/*3542*/ uint16(x86_xCondIs64), 3545, 3561,
	/*3545*/ uint16(x86_xCondDataSize), 3549, 3555, 0,
	/*3549*/ uint16(x86_xSetOp), uint16(x86_CMOVB),
	/*3551*/ uint16(x86_xReadSlashR),
	/*3552*/ uint16(x86_xArgR16),
	/*3553*/ uint16(x86_xArgRM16),
	/*3554*/ uint16(x86_xMatch),
	/*3555*/ uint16(x86_xSetOp), uint16(x86_CMOVB),
	/*3557*/ uint16(x86_xReadSlashR),
	/*3558*/ uint16(x86_xArgR32),
	/*3559*/ uint16(x86_xArgRM32),
	/*3560*/ uint16(x86_xMatch),
	/*3561*/ uint16(x86_xCondDataSize), 3549, 3555, 3565,
	/*3565*/ uint16(x86_xSetOp), uint16(x86_CMOVB),
	/*3567*/ uint16(x86_xReadSlashR),
	/*3568*/ uint16(x86_xArgR64),
	/*3569*/ uint16(x86_xArgRM64),
	/*3570*/ uint16(x86_xMatch),
	/*3571*/ uint16(x86_xCondIs64), 3574, 3590,
	/*3574*/ uint16(x86_xCondDataSize), 3578, 3584, 0,
	/*3578*/ uint16(x86_xSetOp), uint16(x86_CMOVAE),
	/*3580*/ uint16(x86_xReadSlashR),
	/*3581*/ uint16(x86_xArgR16),
	/*3582*/ uint16(x86_xArgRM16),
	/*3583*/ uint16(x86_xMatch),
	/*3584*/ uint16(x86_xSetOp), uint16(x86_CMOVAE),
	/*3586*/ uint16(x86_xReadSlashR),
	/*3587*/ uint16(x86_xArgR32),
	/*3588*/ uint16(x86_xArgRM32),
	/*3589*/ uint16(x86_xMatch),
	/*3590*/ uint16(x86_xCondDataSize), 3578, 3584, 3594,
	/*3594*/ uint16(x86_xSetOp), uint16(x86_CMOVAE),
	/*3596*/ uint16(x86_xReadSlashR),
	/*3597*/ uint16(x86_xArgR64),
	/*3598*/ uint16(x86_xArgRM64),
	/*3599*/ uint16(x86_xMatch),
	/*3600*/ uint16(x86_xCondIs64), 3603, 3619,
	/*3603*/ uint16(x86_xCondDataSize), 3607, 3613, 0,
	/*3607*/ uint16(x86_xSetOp), uint16(x86_CMOVE),
	/*3609*/ uint16(x86_xReadSlashR),
	/*3610*/ uint16(x86_xArgR16),
	/*3611*/ uint16(x86_xArgRM16),
	/*3612*/ uint16(x86_xMatch),
	/*3613*/ uint16(x86_xSetOp), uint16(x86_CMOVE),
	/*3615*/ uint16(x86_xReadSlashR),
	/*3616*/ uint16(x86_xArgR32),
	/*3617*/ uint16(x86_xArgRM32),
	/*3618*/ uint16(x86_xMatch),
	/*3619*/ uint16(x86_xCondDataSize), 3607, 3613, 3623,
	/*3623*/ uint16(x86_xSetOp), uint16(x86_CMOVE),
	/*3625*/ uint16(x86_xReadSlashR),
	/*3626*/ uint16(x86_xArgR64),
	/*3627*/ uint16(x86_xArgRM64),
	/*3628*/ uint16(x86_xMatch),
	/*3629*/ uint16(x86_xCondIs64), 3632, 3648,
	/*3632*/ uint16(x86_xCondDataSize), 3636, 3642, 0,
	/*3636*/ uint16(x86_xSetOp), uint16(x86_CMOVNE),
	/*3638*/ uint16(x86_xReadSlashR),
	/*3639*/ uint16(x86_xArgR16),
	/*3640*/ uint16(x86_xArgRM16),
	/*3641*/ uint16(x86_xMatch),
	/*3642*/ uint16(x86_xSetOp), uint16(x86_CMOVNE),
	/*3644*/ uint16(x86_xReadSlashR),
	/*3645*/ uint16(x86_xArgR32),
	/*3646*/ uint16(x86_xArgRM32),
	/*3647*/ uint16(x86_xMatch),
	/*3648*/ uint16(x86_xCondDataSize), 3636, 3642, 3652,
	/*3652*/ uint16(x86_xSetOp), uint16(x86_CMOVNE),
	/*3654*/ uint16(x86_xReadSlashR),
	/*3655*/ uint16(x86_xArgR64),
	/*3656*/ uint16(x86_xArgRM64),
	/*3657*/ uint16(x86_xMatch),
	/*3658*/ uint16(x86_xCondIs64), 3661, 3677,
	/*3661*/ uint16(x86_xCondDataSize), 3665, 3671, 0,
	/*3665*/ uint16(x86_xSetOp), uint16(x86_CMOVBE),
	/*3667*/ uint16(x86_xReadSlashR),
	/*3668*/ uint16(x86_xArgR16),
	/*3669*/ uint16(x86_xArgRM16),
	/*3670*/ uint16(x86_xMatch),
	/*3671*/ uint16(x86_xSetOp), uint16(x86_CMOVBE),
	/*3673*/ uint16(x86_xReadSlashR),
	/*3674*/ uint16(x86_xArgR32),
	/*3675*/ uint16(x86_xArgRM32),
	/*3676*/ uint16(x86_xMatch),
	/*3677*/ uint16(x86_xCondDataSize), 3665, 3671, 3681,
	/*3681*/ uint16(x86_xSetOp), uint16(x86_CMOVBE),
	/*3683*/ uint16(x86_xReadSlashR),
	/*3684*/ uint16(x86_xArgR64),
	/*3685*/ uint16(x86_xArgRM64),
	/*3686*/ uint16(x86_xMatch),
	/*3687*/ uint16(x86_xCondIs64), 3690, 3706,
	/*3690*/ uint16(x86_xCondDataSize), 3694, 3700, 0,
	/*3694*/ uint16(x86_xSetOp), uint16(x86_CMOVA),
	/*3696*/ uint16(x86_xReadSlashR),
	/*3697*/ uint16(x86_xArgR16),
	/*3698*/ uint16(x86_xArgRM16),
	/*3699*/ uint16(x86_xMatch),
	/*3700*/ uint16(x86_xSetOp), uint16(x86_CMOVA),
	/*3702*/ uint16(x86_xReadSlashR),
	/*3703*/ uint16(x86_xArgR32),
	/*3704*/ uint16(x86_xArgRM32),
	/*3705*/ uint16(x86_xMatch),
	/*3706*/ uint16(x86_xCondDataSize), 3694, 3700, 3710,
	/*3710*/ uint16(x86_xSetOp), uint16(x86_CMOVA),
	/*3712*/ uint16(x86_xReadSlashR),
	/*3713*/ uint16(x86_xArgR64),
	/*3714*/ uint16(x86_xArgRM64),
	/*3715*/ uint16(x86_xMatch),
	/*3716*/ uint16(x86_xCondIs64), 3719, 3735,
	/*3719*/ uint16(x86_xCondDataSize), 3723, 3729, 0,
	/*3723*/ uint16(x86_xSetOp), uint16(x86_CMOVS),
	/*3725*/ uint16(x86_xReadSlashR),
	/*3726*/ uint16(x86_xArgR16),
	/*3727*/ uint16(x86_xArgRM16),
	/*3728*/ uint16(x86_xMatch),
	/*3729*/ uint16(x86_xSetOp), uint16(x86_CMOVS),
	/*3731*/ uint16(x86_xReadSlashR),
	/*3732*/ uint16(x86_xArgR32),
	/*3733*/ uint16(x86_xArgRM32),
	/*3734*/ uint16(x86_xMatch),
	/*3735*/ uint16(x86_xCondDataSize), 3723, 3729, 3739,
	/*3739*/ uint16(x86_xSetOp), uint16(x86_CMOVS),
	/*3741*/ uint16(x86_xReadSlashR),
	/*3742*/ uint16(x86_xArgR64),
	/*3743*/ uint16(x86_xArgRM64),
	/*3744*/ uint16(x86_xMatch),
	/*3745*/ uint16(x86_xCondIs64), 3748, 3764,
	/*3748*/ uint16(x86_xCondDataSize), 3752, 3758, 0,
	/*3752*/ uint16(x86_xSetOp), uint16(x86_CMOVNS),
	/*3754*/ uint16(x86_xReadSlashR),
	/*3755*/ uint16(x86_xArgR16),
	/*3756*/ uint16(x86_xArgRM16),
	/*3757*/ uint16(x86_xMatch),
	/*3758*/ uint16(x86_xSetOp), uint16(x86_CMOVNS),
	/*3760*/ uint16(x86_xReadSlashR),
	/*3761*/ uint16(x86_xArgR32),
	/*3762*/ uint16(x86_xArgRM32),
	/*3763*/ uint16(x86_xMatch),
	/*3764*/ uint16(x86_xCondDataSize), 3752, 3758, 3768,
	/*3768*/ uint16(x86_xSetOp), uint16(x86_CMOVNS),
	/*3770*/ uint16(x86_xReadSlashR),
	/*3771*/ uint16(x86_xArgR64),
	/*3772*/ uint16(x86_xArgRM64),
	/*3773*/ uint16(x86_xMatch),
	/*3774*/ uint16(x86_xCondIs64), 3777, 3793,
	/*3777*/ uint16(x86_xCondDataSize), 3781, 3787, 0,
	/*3781*/ uint16(x86_xSetOp), uint16(x86_CMOVP),
	/*3783*/ uint16(x86_xReadSlashR),
	/*3784*/ uint16(x86_xArgR16),
	/*3785*/ uint16(x86_xArgRM16),
	/*3786*/ uint16(x86_xMatch),
	/*3787*/ uint16(x86_xSetOp), uint16(x86_CMOVP),
	/*3789*/ uint16(x86_xReadSlashR),
	/*3790*/ uint16(x86_xArgR32),
	/*3791*/ uint16(x86_xArgRM32),
	/*3792*/ uint16(x86_xMatch),
	/*3793*/ uint16(x86_xCondDataSize), 3781, 3787, 3797,
	/*3797*/ uint16(x86_xSetOp), uint16(x86_CMOVP),
	/*3799*/ uint16(x86_xReadSlashR),
	/*3800*/ uint16(x86_xArgR64),
	/*3801*/ uint16(x86_xArgRM64),
	/*3802*/ uint16(x86_xMatch),
	/*3803*/ uint16(x86_xCondIs64), 3806, 3822,
	/*3806*/ uint16(x86_xCondDataSize), 3810, 3816, 0,
	/*3810*/ uint16(x86_xSetOp), uint16(x86_CMOVNP),
	/*3812*/ uint16(x86_xReadSlashR),
	/*3813*/ uint16(x86_xArgR16),
	/*3814*/ uint16(x86_xArgRM16),
	/*3815*/ uint16(x86_xMatch),
	/*3816*/ uint16(x86_xSetOp), uint16(x86_CMOVNP),
	/*3818*/ uint16(x86_xReadSlashR),
	/*3819*/ uint16(x86_xArgR32),
	/*3820*/ uint16(x86_xArgRM32),
	/*3821*/ uint16(x86_xMatch),
	/*3822*/ uint16(x86_xCondDataSize), 3810, 3816, 3826,
	/*3826*/ uint16(x86_xSetOp), uint16(x86_CMOVNP),
	/*3828*/ uint16(x86_xReadSlashR),
	/*3829*/ uint16(x86_xArgR64),
	/*3830*/ uint16(x86_xArgRM64),
	/*3831*/ uint16(x86_xMatch),
	/*3832*/ uint16(x86_xCondIs64), 3835, 3851,
	/*3835*/ uint16(x86_xCondDataSize), 3839, 3845, 0,
	/*3839*/ uint16(x86_xSetOp), uint16(x86_CMOVL),
	/*3841*/ uint16(x86_xReadSlashR),
	/*3842*/ uint16(x86_xArgR16),
	/*3843*/ uint16(x86_xArgRM16),
	/*3844*/ uint16(x86_xMatch),
	/*3845*/ uint16(x86_xSetOp), uint16(x86_CMOVL),
	/*3847*/ uint16(x86_xReadSlashR),
	/*3848*/ uint16(x86_xArgR32),
	/*3849*/ uint16(x86_xArgRM32),
	/*3850*/ uint16(x86_xMatch),
	/*3851*/ uint16(x86_xCondDataSize), 3839, 3845, 3855,
	/*3855*/ uint16(x86_xSetOp), uint16(x86_CMOVL),
	/*3857*/ uint16(x86_xReadSlashR),
	/*3858*/ uint16(x86_xArgR64),
	/*3859*/ uint16(x86_xArgRM64),
	/*3860*/ uint16(x86_xMatch),
	/*3861*/ uint16(x86_xCondIs64), 3864, 3880,
	/*3864*/ uint16(x86_xCondDataSize), 3868, 3874, 0,
	/*3868*/ uint16(x86_xSetOp), uint16(x86_CMOVGE),
	/*3870*/ uint16(x86_xReadSlashR),
	/*3871*/ uint16(x86_xArgR16),
	/*3872*/ uint16(x86_xArgRM16),
	/*3873*/ uint16(x86_xMatch),
	/*3874*/ uint16(x86_xSetOp), uint16(x86_CMOVGE),
	/*3876*/ uint16(x86_xReadSlashR),
	/*3877*/ uint16(x86_xArgR32),
	/*3878*/ uint16(x86_xArgRM32),
	/*3879*/ uint16(x86_xMatch),
	/*3880*/ uint16(x86_xCondDataSize), 3868, 3874, 3884,
	/*3884*/ uint16(x86_xSetOp), uint16(x86_CMOVGE),
	/*3886*/ uint16(x86_xReadSlashR),
	/*3887*/ uint16(x86_xArgR64),
	/*3888*/ uint16(x86_xArgRM64),
	/*3889*/ uint16(x86_xMatch),
	/*3890*/ uint16(x86_xCondIs64), 3893, 3909,
	/*3893*/ uint16(x86_xCondDataSize), 3897, 3903, 0,
	/*3897*/ uint16(x86_xSetOp), uint16(x86_CMOVLE),
	/*3899*/ uint16(x86_xReadSlashR),
	/*3900*/ uint16(x86_xArgR16),
	/*3901*/ uint16(x86_xArgRM16),
	/*3902*/ uint16(x86_xMatch),
	/*3903*/ uint16(x86_xSetOp), uint16(x86_CMOVLE),
	/*3905*/ uint16(x86_xReadSlashR),
	/*3906*/ uint16(x86_xArgR32),
	/*3907*/ uint16(x86_xArgRM32),
	/*3908*/ uint16(x86_xMatch),
	/*3909*/ uint16(x86_xCondDataSize), 3897, 3903, 3913,
	/*3913*/ uint16(x86_xSetOp), uint16(x86_CMOVLE),
	/*3915*/ uint16(x86_xReadSlashR),
	/*3916*/ uint16(x86_xArgR64),
	/*3917*/ uint16(x86_xArgRM64),
	/*3918*/ uint16(x86_xMatch),
	/*3919*/ uint16(x86_xCondIs64), 3922, 3938,
	/*3922*/ uint16(x86_xCondDataSize), 3926, 3932, 0,
	/*3926*/ uint16(x86_xSetOp), uint16(x86_CMOVG),
	/*3928*/ uint16(x86_xReadSlashR),
	/*3929*/ uint16(x86_xArgR16),
	/*3930*/ uint16(x86_xArgRM16),
	/*3931*/ uint16(x86_xMatch),
	/*3932*/ uint16(x86_xSetOp), uint16(x86_CMOVG),
	/*3934*/ uint16(x86_xReadSlashR),
	/*3935*/ uint16(x86_xArgR32),
	/*3936*/ uint16(x86_xArgRM32),
	/*3937*/ uint16(x86_xMatch),
	/*3938*/ uint16(x86_xCondDataSize), 3926, 3932, 3942,
	/*3942*/ uint16(x86_xSetOp), uint16(x86_CMOVG),
	/*3944*/ uint16(x86_xReadSlashR),
	/*3945*/ uint16(x86_xArgR64),
	/*3946*/ uint16(x86_xArgRM64),
	/*3947*/ uint16(x86_xMatch),
	/*3948*/ uint16(x86_xCondPrefix), 2,
	0x66, 3960,
	0x0, 3954,
	/*3954*/ uint16(x86_xSetOp), uint16(x86_MOVMSKPS),
	/*3956*/ uint16(x86_xReadSlashR),
	/*3957*/ uint16(x86_xArgR32),
	/*3958*/ uint16(x86_xArgXmm2),
	/*3959*/ uint16(x86_xMatch),
	/*3960*/ uint16(x86_xSetOp), uint16(x86_MOVMSKPD),
	/*3962*/ uint16(x86_xReadSlashR),
	/*3963*/ uint16(x86_xArgR32),
	/*3964*/ uint16(x86_xArgXmm2),
	/*3965*/ uint16(x86_xMatch),
	/*3966*/ uint16(x86_xCondPrefix), 4,
	0xF3, 3994,
	0xF2, 3988,
	0x66, 3982,
	0x0, 3976,
	/*3976*/ uint16(x86_xSetOp), uint16(x86_SQRTPS),
	/*3978*/ uint16(x86_xReadSlashR),
	/*3979*/ uint16(x86_xArgXmm1),
	/*3980*/ uint16(x86_xArgXmm2M128),
	/*3981*/ uint16(x86_xMatch),
	/*3982*/ uint16(x86_xSetOp), uint16(x86_SQRTPD),
	/*3984*/ uint16(x86_xReadSlashR),
	/*3985*/ uint16(x86_xArgXmm1),
	/*3986*/ uint16(x86_xArgXmm2M128),
	/*3987*/ uint16(x86_xMatch),
	/*3988*/ uint16(x86_xSetOp), uint16(x86_SQRTSD),
	/*3990*/ uint16(x86_xReadSlashR),
	/*3991*/ uint16(x86_xArgXmm1),
	/*3992*/ uint16(x86_xArgXmm2M64),
	/*3993*/ uint16(x86_xMatch),
	/*3994*/ uint16(x86_xSetOp), uint16(x86_SQRTSS),
	/*3996*/ uint16(x86_xReadSlashR),
	/*3997*/ uint16(x86_xArgXmm1),
	/*3998*/ uint16(x86_xArgXmm2M32),
	/*3999*/ uint16(x86_xMatch),
	/*4000*/ uint16(x86_xCondPrefix), 2,
	0xF3, 4012,
	0x0, 4006,
	/*4006*/ uint16(x86_xSetOp), uint16(x86_RSQRTPS),
	/*4008*/ uint16(x86_xReadSlashR),
	/*4009*/ uint16(x86_xArgXmm1),
	/*4010*/ uint16(x86_xArgXmm2M128),
	/*4011*/ uint16(x86_xMatch),
	/*4012*/ uint16(x86_xSetOp), uint16(x86_RSQRTSS),
	/*4014*/ uint16(x86_xReadSlashR),
	/*4015*/ uint16(x86_xArgXmm1),
	/*4016*/ uint16(x86_xArgXmm2M32),
	/*4017*/ uint16(x86_xMatch),
	/*4018*/ uint16(x86_xCondPrefix), 2,
	0xF3, 4030,
	0x0, 4024,
	/*4024*/ uint16(x86_xSetOp), uint16(x86_RCPPS),
	/*4026*/ uint16(x86_xReadSlashR),
	/*4027*/ uint16(x86_xArgXmm1),
	/*4028*/ uint16(x86_xArgXmm2M128),
	/*4029*/ uint16(x86_xMatch),
	/*4030*/ uint16(x86_xSetOp), uint16(x86_RCPSS),
	/*4032*/ uint16(x86_xReadSlashR),
	/*4033*/ uint16(x86_xArgXmm1),
	/*4034*/ uint16(x86_xArgXmm2M32),
	/*4035*/ uint16(x86_xMatch),
	/*4036*/ uint16(x86_xCondPrefix), 2,
	0x66, 4048,
	0x0, 4042,
	/*4042*/ uint16(x86_xSetOp), uint16(x86_ANDPS),
	/*4044*/ uint16(x86_xReadSlashR),
	/*4045*/ uint16(x86_xArgXmm1),
	/*4046*/ uint16(x86_xArgXmm2M128),
	/*4047*/ uint16(x86_xMatch),
	/*4048*/ uint16(x86_xSetOp), uint16(x86_ANDPD),
	/*4050*/ uint16(x86_xReadSlashR),
	/*4051*/ uint16(x86_xArgXmm1),
	/*4052*/ uint16(x86_xArgXmm2M128),
	/*4053*/ uint16(x86_xMatch),
	/*4054*/ uint16(x86_xCondPrefix), 2,
	0x66, 4066,
	0x0, 4060,
	/*4060*/ uint16(x86_xSetOp), uint16(x86_ANDNPS),
	/*4062*/ uint16(x86_xReadSlashR),
	/*4063*/ uint16(x86_xArgXmm1),
	/*4064*/ uint16(x86_xArgXmm2M128),
	/*4065*/ uint16(x86_xMatch),
	/*4066*/ uint16(x86_xSetOp), uint16(x86_ANDNPD),
	/*4068*/ uint16(x86_xReadSlashR),
	/*4069*/ uint16(x86_xArgXmm1),
	/*4070*/ uint16(x86_xArgXmm2M128),
	/*4071*/ uint16(x86_xMatch),
	/*4072*/ uint16(x86_xCondPrefix), 2,
	0x66, 4084,
	0x0, 4078,
	/*4078*/ uint16(x86_xSetOp), uint16(x86_ORPS),
	/*4080*/ uint16(x86_xReadSlashR),
	/*4081*/ uint16(x86_xArgXmm1),
	/*4082*/ uint16(x86_xArgXmm2M128),
	/*4083*/ uint16(x86_xMatch),
	/*4084*/ uint16(x86_xSetOp), uint16(x86_ORPD),
	/*4086*/ uint16(x86_xReadSlashR),
	/*4087*/ uint16(x86_xArgXmm1),
	/*4088*/ uint16(x86_xArgXmm2M128),
	/*4089*/ uint16(x86_xMatch),
	/*4090*/ uint16(x86_xCondPrefix), 2,
	0x66, 4102,
	0x0, 4096,
	/*4096*/ uint16(x86_xSetOp), uint16(x86_XORPS),
	/*4098*/ uint16(x86_xReadSlashR),
	/*4099*/ uint16(x86_xArgXmm1),
	/*4100*/ uint16(x86_xArgXmm2M128),
	/*4101*/ uint16(x86_xMatch),
	/*4102*/ uint16(x86_xSetOp), uint16(x86_XORPD),
	/*4104*/ uint16(x86_xReadSlashR),
	/*4105*/ uint16(x86_xArgXmm1),
	/*4106*/ uint16(x86_xArgXmm2M128),
	/*4107*/ uint16(x86_xMatch),
	/*4108*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4136,
	0xF2, 4130,
	0x66, 4124,
	0x0, 4118,
	/*4118*/ uint16(x86_xSetOp), uint16(x86_ADDPS),
	/*4120*/ uint16(x86_xReadSlashR),
	/*4121*/ uint16(x86_xArgXmm1),
	/*4122*/ uint16(x86_xArgXmm2M128),
	/*4123*/ uint16(x86_xMatch),
	/*4124*/ uint16(x86_xSetOp), uint16(x86_ADDPD),
	/*4126*/ uint16(x86_xReadSlashR),
	/*4127*/ uint16(x86_xArgXmm1),
	/*4128*/ uint16(x86_xArgXmm2M128),
	/*4129*/ uint16(x86_xMatch),
	/*4130*/ uint16(x86_xSetOp), uint16(x86_ADDSD),
	/*4132*/ uint16(x86_xReadSlashR),
	/*4133*/ uint16(x86_xArgXmm1),
	/*4134*/ uint16(x86_xArgXmm2M64),
	/*4135*/ uint16(x86_xMatch),
	/*4136*/ uint16(x86_xSetOp), uint16(x86_ADDSS),
	/*4138*/ uint16(x86_xReadSlashR),
	/*4139*/ uint16(x86_xArgXmm1),
	/*4140*/ uint16(x86_xArgXmm2M32),
	/*4141*/ uint16(x86_xMatch),
	/*4142*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4170,
	0xF2, 4164,
	0x66, 4158,
	0x0, 4152,
	/*4152*/ uint16(x86_xSetOp), uint16(x86_MULPS),
	/*4154*/ uint16(x86_xReadSlashR),
	/*4155*/ uint16(x86_xArgXmm1),
	/*4156*/ uint16(x86_xArgXmm2M128),
	/*4157*/ uint16(x86_xMatch),
	/*4158*/ uint16(x86_xSetOp), uint16(x86_MULPD),
	/*4160*/ uint16(x86_xReadSlashR),
	/*4161*/ uint16(x86_xArgXmm1),
	/*4162*/ uint16(x86_xArgXmm2M128),
	/*4163*/ uint16(x86_xMatch),
	/*4164*/ uint16(x86_xSetOp), uint16(x86_MULSD),
	/*4166*/ uint16(x86_xReadSlashR),
	/*4167*/ uint16(x86_xArgXmm1),
	/*4168*/ uint16(x86_xArgXmm2M64),
	/*4169*/ uint16(x86_xMatch),
	/*4170*/ uint16(x86_xSetOp), uint16(x86_MULSS),
	/*4172*/ uint16(x86_xReadSlashR),
	/*4173*/ uint16(x86_xArgXmm1),
	/*4174*/ uint16(x86_xArgXmm2M32),
	/*4175*/ uint16(x86_xMatch),
	/*4176*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4204,
	0xF2, 4198,
	0x66, 4192,
	0x0, 4186,
	/*4186*/ uint16(x86_xSetOp), uint16(x86_CVTPS2PD),
	/*4188*/ uint16(x86_xReadSlashR),
	/*4189*/ uint16(x86_xArgXmm1),
	/*4190*/ uint16(x86_xArgXmm2M64),
	/*4191*/ uint16(x86_xMatch),
	/*4192*/ uint16(x86_xSetOp), uint16(x86_CVTPD2PS),
	/*4194*/ uint16(x86_xReadSlashR),
	/*4195*/ uint16(x86_xArgXmm1),
	/*4196*/ uint16(x86_xArgXmm2M128),
	/*4197*/ uint16(x86_xMatch),
	/*4198*/ uint16(x86_xSetOp), uint16(x86_CVTSD2SS),
	/*4200*/ uint16(x86_xReadSlashR),
	/*4201*/ uint16(x86_xArgXmm1),
	/*4202*/ uint16(x86_xArgXmm2M64),
	/*4203*/ uint16(x86_xMatch),
	/*4204*/ uint16(x86_xSetOp), uint16(x86_CVTSS2SD),
	/*4206*/ uint16(x86_xReadSlashR),
	/*4207*/ uint16(x86_xArgXmm1),
	/*4208*/ uint16(x86_xArgXmm2M32),
	/*4209*/ uint16(x86_xMatch),
	/*4210*/ uint16(x86_xCondPrefix), 3,
	0xF3, 4230,
	0x66, 4224,
	0x0, 4218,
	/*4218*/ uint16(x86_xSetOp), uint16(x86_CVTDQ2PS),
	/*4220*/ uint16(x86_xReadSlashR),
	/*4221*/ uint16(x86_xArgXmm1),
	/*4222*/ uint16(x86_xArgXmm2M128),
	/*4223*/ uint16(x86_xMatch),
	/*4224*/ uint16(x86_xSetOp), uint16(x86_CVTPS2DQ),
	/*4226*/ uint16(x86_xReadSlashR),
	/*4227*/ uint16(x86_xArgXmm1),
	/*4228*/ uint16(x86_xArgXmm2M128),
	/*4229*/ uint16(x86_xMatch),
	/*4230*/ uint16(x86_xSetOp), uint16(x86_CVTTPS2DQ),
	/*4232*/ uint16(x86_xReadSlashR),
	/*4233*/ uint16(x86_xArgXmm1),
	/*4234*/ uint16(x86_xArgXmm2M128),
	/*4235*/ uint16(x86_xMatch),
	/*4236*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4264,
	0xF2, 4258,
	0x66, 4252,
	0x0, 4246,
	/*4246*/ uint16(x86_xSetOp), uint16(x86_SUBPS),
	/*4248*/ uint16(x86_xReadSlashR),
	/*4249*/ uint16(x86_xArgXmm1),
	/*4250*/ uint16(x86_xArgXmm2M128),
	/*4251*/ uint16(x86_xMatch),
	/*4252*/ uint16(x86_xSetOp), uint16(x86_SUBPD),
	/*4254*/ uint16(x86_xReadSlashR),
	/*4255*/ uint16(x86_xArgXmm1),
	/*4256*/ uint16(x86_xArgXmm2M128),
	/*4257*/ uint16(x86_xMatch),
	/*4258*/ uint16(x86_xSetOp), uint16(x86_SUBSD),
	/*4260*/ uint16(x86_xReadSlashR),
	/*4261*/ uint16(x86_xArgXmm1),
	/*4262*/ uint16(x86_xArgXmm2M64),
	/*4263*/ uint16(x86_xMatch),
	/*4264*/ uint16(x86_xSetOp), uint16(x86_SUBSS),
	/*4266*/ uint16(x86_xReadSlashR),
	/*4267*/ uint16(x86_xArgXmm1),
	/*4268*/ uint16(x86_xArgXmm2M32),
	/*4269*/ uint16(x86_xMatch),
	/*4270*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4298,
	0xF2, 4292,
	0x66, 4286,
	0x0, 4280,
	/*4280*/ uint16(x86_xSetOp), uint16(x86_MINPS),
	/*4282*/ uint16(x86_xReadSlashR),
	/*4283*/ uint16(x86_xArgXmm1),
	/*4284*/ uint16(x86_xArgXmm2M128),
	/*4285*/ uint16(x86_xMatch),
	/*4286*/ uint16(x86_xSetOp), uint16(x86_MINPD),
	/*4288*/ uint16(x86_xReadSlashR),
	/*4289*/ uint16(x86_xArgXmm1),
	/*4290*/ uint16(x86_xArgXmm2M128),
	/*4291*/ uint16(x86_xMatch),
	/*4292*/ uint16(x86_xSetOp), uint16(x86_MINSD),
	/*4294*/ uint16(x86_xReadSlashR),
	/*4295*/ uint16(x86_xArgXmm1),
	/*4296*/ uint16(x86_xArgXmm2M64),
	/*4297*/ uint16(x86_xMatch),
	/*4298*/ uint16(x86_xSetOp), uint16(x86_MINSS),
	/*4300*/ uint16(x86_xReadSlashR),
	/*4301*/ uint16(x86_xArgXmm1),
	/*4302*/ uint16(x86_xArgXmm2M32),
	/*4303*/ uint16(x86_xMatch),
	/*4304*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4332,
	0xF2, 4326,
	0x66, 4320,
	0x0, 4314,
	/*4314*/ uint16(x86_xSetOp), uint16(x86_DIVPS),
	/*4316*/ uint16(x86_xReadSlashR),
	/*4317*/ uint16(x86_xArgXmm1),
	/*4318*/ uint16(x86_xArgXmm2M128),
	/*4319*/ uint16(x86_xMatch),
	/*4320*/ uint16(x86_xSetOp), uint16(x86_DIVPD),
	/*4322*/ uint16(x86_xReadSlashR),
	/*4323*/ uint16(x86_xArgXmm1),
	/*4324*/ uint16(x86_xArgXmm2M128),
	/*4325*/ uint16(x86_xMatch),
	/*4326*/ uint16(x86_xSetOp), uint16(x86_DIVSD),
	/*4328*/ uint16(x86_xReadSlashR),
	/*4329*/ uint16(x86_xArgXmm1),
	/*4330*/ uint16(x86_xArgXmm2M64),
	/*4331*/ uint16(x86_xMatch),
	/*4332*/ uint16(x86_xSetOp), uint16(x86_DIVSS),
	/*4334*/ uint16(x86_xReadSlashR),
	/*4335*/ uint16(x86_xArgXmm1),
	/*4336*/ uint16(x86_xArgXmm2M32),
	/*4337*/ uint16(x86_xMatch),
	/*4338*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4366,
	0xF2, 4360,
	0x66, 4354,
	0x0, 4348,
	/*4348*/ uint16(x86_xSetOp), uint16(x86_MAXPS),
	/*4350*/ uint16(x86_xReadSlashR),
	/*4351*/ uint16(x86_xArgXmm1),
	/*4352*/ uint16(x86_xArgXmm2M128),
	/*4353*/ uint16(x86_xMatch),
	/*4354*/ uint16(x86_xSetOp), uint16(x86_MAXPD),
	/*4356*/ uint16(x86_xReadSlashR),
	/*4357*/ uint16(x86_xArgXmm1),
	/*4358*/ uint16(x86_xArgXmm2M128),
	/*4359*/ uint16(x86_xMatch),
	/*4360*/ uint16(x86_xSetOp), uint16(x86_MAXSD),
	/*4362*/ uint16(x86_xReadSlashR),
	/*4363*/ uint16(x86_xArgXmm1),
	/*4364*/ uint16(x86_xArgXmm2M64),
	/*4365*/ uint16(x86_xMatch),
	/*4366*/ uint16(x86_xSetOp), uint16(x86_MAXSS),
	/*4368*/ uint16(x86_xReadSlashR),
	/*4369*/ uint16(x86_xArgXmm1),
	/*4370*/ uint16(x86_xArgXmm2M32),
	/*4371*/ uint16(x86_xMatch),
	/*4372*/ uint16(x86_xCondPrefix), 2,
	0x66, 4384,
	0x0, 4378,
	/*4378*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLBW),
	/*4380*/ uint16(x86_xReadSlashR),
	/*4381*/ uint16(x86_xArgMm),
	/*4382*/ uint16(x86_xArgMmM32),
	/*4383*/ uint16(x86_xMatch),
	/*4384*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLBW),
	/*4386*/ uint16(x86_xReadSlashR),
	/*4387*/ uint16(x86_xArgXmm1),
	/*4388*/ uint16(x86_xArgXmm2M128),
	/*4389*/ uint16(x86_xMatch),
	/*4390*/ uint16(x86_xCondPrefix), 2,
	0x66, 4402,
	0x0, 4396,
	/*4396*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLWD),
	/*4398*/ uint16(x86_xReadSlashR),
	/*4399*/ uint16(x86_xArgMm),
	/*4400*/ uint16(x86_xArgMmM32),
	/*4401*/ uint16(x86_xMatch),
	/*4402*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLWD),
	/*4404*/ uint16(x86_xReadSlashR),
	/*4405*/ uint16(x86_xArgXmm1),
	/*4406*/ uint16(x86_xArgXmm2M128),
	/*4407*/ uint16(x86_xMatch),
	/*4408*/ uint16(x86_xCondPrefix), 2,
	0x66, 4420,
	0x0, 4414,
	/*4414*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLDQ),
	/*4416*/ uint16(x86_xReadSlashR),
	/*4417*/ uint16(x86_xArgMm),
	/*4418*/ uint16(x86_xArgMmM32),
	/*4419*/ uint16(x86_xMatch),
	/*4420*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLDQ),
	/*4422*/ uint16(x86_xReadSlashR),
	/*4423*/ uint16(x86_xArgXmm1),
	/*4424*/ uint16(x86_xArgXmm2M128),
	/*4425*/ uint16(x86_xMatch),
	/*4426*/ uint16(x86_xCondPrefix), 2,
	0x66, 4438,
	0x0, 4432,
	/*4432*/ uint16(x86_xSetOp), uint16(x86_PACKSSWB),
	/*4434*/ uint16(x86_xReadSlashR),
	/*4435*/ uint16(x86_xArgMm1),
	/*4436*/ uint16(x86_xArgMm2M64),
	/*4437*/ uint16(x86_xMatch),
	/*4438*/ uint16(x86_xSetOp), uint16(x86_PACKSSWB),
	/*4440*/ uint16(x86_xReadSlashR),
	/*4441*/ uint16(x86_xArgXmm1),
	/*4442*/ uint16(x86_xArgXmm2M128),
	/*4443*/ uint16(x86_xMatch),
	/*4444*/ uint16(x86_xCondPrefix), 2,
	0x66, 4456,
	0x0, 4450,
	/*4450*/ uint16(x86_xSetOp), uint16(x86_PCMPGTB),
	/*4452*/ uint16(x86_xReadSlashR),
	/*4453*/ uint16(x86_xArgMm),
	/*4454*/ uint16(x86_xArgMmM64),
	/*4455*/ uint16(x86_xMatch),
	/*4456*/ uint16(x86_xSetOp), uint16(x86_PCMPGTB),
	/*4458*/ uint16(x86_xReadSlashR),
	/*4459*/ uint16(x86_xArgXmm1),
	/*4460*/ uint16(x86_xArgXmm2M128),
	/*4461*/ uint16(x86_xMatch),
	/*4462*/ uint16(x86_xCondPrefix), 2,
	0x66, 4474,
	0x0, 4468,
	/*4468*/ uint16(x86_xSetOp), uint16(x86_PCMPGTW),
	/*4470*/ uint16(x86_xReadSlashR),
	/*4471*/ uint16(x86_xArgMm),
	/*4472*/ uint16(x86_xArgMmM64),
	/*4473*/ uint16(x86_xMatch),
	/*4474*/ uint16(x86_xSetOp), uint16(x86_PCMPGTW),
	/*4476*/ uint16(x86_xReadSlashR),
	/*4477*/ uint16(x86_xArgXmm1),
	/*4478*/ uint16(x86_xArgXmm2M128),
	/*4479*/ uint16(x86_xMatch),
	/*4480*/ uint16(x86_xCondPrefix), 2,
	0x66, 4492,
	0x0, 4486,
	/*4486*/ uint16(x86_xSetOp), uint16(x86_PCMPGTD),
	/*4488*/ uint16(x86_xReadSlashR),
	/*4489*/ uint16(x86_xArgMm),
	/*4490*/ uint16(x86_xArgMmM64),
	/*4491*/ uint16(x86_xMatch),
	/*4492*/ uint16(x86_xSetOp), uint16(x86_PCMPGTD),
	/*4494*/ uint16(x86_xReadSlashR),
	/*4495*/ uint16(x86_xArgXmm1),
	/*4496*/ uint16(x86_xArgXmm2M128),
	/*4497*/ uint16(x86_xMatch),
	/*4498*/ uint16(x86_xCondPrefix), 2,
	0x66, 4510,
	0x0, 4504,
	/*4504*/ uint16(x86_xSetOp), uint16(x86_PACKUSWB),
	/*4506*/ uint16(x86_xReadSlashR),
	/*4507*/ uint16(x86_xArgMm),
	/*4508*/ uint16(x86_xArgMmM64),
	/*4509*/ uint16(x86_xMatch),
	/*4510*/ uint16(x86_xSetOp), uint16(x86_PACKUSWB),
	/*4512*/ uint16(x86_xReadSlashR),
	/*4513*/ uint16(x86_xArgXmm1),
	/*4514*/ uint16(x86_xArgXmm2M128),
	/*4515*/ uint16(x86_xMatch),
	/*4516*/ uint16(x86_xCondPrefix), 2,
	0x66, 4528,
	0x0, 4522,
	/*4522*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHBW),
	/*4524*/ uint16(x86_xReadSlashR),
	/*4525*/ uint16(x86_xArgMm),
	/*4526*/ uint16(x86_xArgMmM64),
	/*4527*/ uint16(x86_xMatch),
	/*4528*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHBW),
	/*4530*/ uint16(x86_xReadSlashR),
	/*4531*/ uint16(x86_xArgXmm1),
	/*4532*/ uint16(x86_xArgXmm2M128),
	/*4533*/ uint16(x86_xMatch),
	/*4534*/ uint16(x86_xCondPrefix), 2,
	0x66, 4546,
	0x0, 4540,
	/*4540*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHWD),
	/*4542*/ uint16(x86_xReadSlashR),
	/*4543*/ uint16(x86_xArgMm),
	/*4544*/ uint16(x86_xArgMmM64),
	/*4545*/ uint16(x86_xMatch),
	/*4546*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHWD),
	/*4548*/ uint16(x86_xReadSlashR),
	/*4549*/ uint16(x86_xArgXmm1),
	/*4550*/ uint16(x86_xArgXmm2M128),
	/*4551*/ uint16(x86_xMatch),
	/*4552*/ uint16(x86_xCondPrefix), 2,
	0x66, 4564,
	0x0, 4558,
	/*4558*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHDQ),
	/*4560*/ uint16(x86_xReadSlashR),
	/*4561*/ uint16(x86_xArgMm),
	/*4562*/ uint16(x86_xArgMmM64),
	/*4563*/ uint16(x86_xMatch),
	/*4564*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHDQ),
	/*4566*/ uint16(x86_xReadSlashR),
	/*4567*/ uint16(x86_xArgXmm1),
	/*4568*/ uint16(x86_xArgXmm2M128),
	/*4569*/ uint16(x86_xMatch),
	/*4570*/ uint16(x86_xCondPrefix), 2,
	0x66, 4582,
	0x0, 4576,
	/*4576*/ uint16(x86_xSetOp), uint16(x86_PACKSSDW),
	/*4578*/ uint16(x86_xReadSlashR),
	/*4579*/ uint16(x86_xArgMm1),
	/*4580*/ uint16(x86_xArgMm2M64),
	/*4581*/ uint16(x86_xMatch),
	/*4582*/ uint16(x86_xSetOp), uint16(x86_PACKSSDW),
	/*4584*/ uint16(x86_xReadSlashR),
	/*4585*/ uint16(x86_xArgXmm1),
	/*4586*/ uint16(x86_xArgXmm2M128),
	/*4587*/ uint16(x86_xMatch),
	/*4588*/ uint16(x86_xCondPrefix), 1,
	0x66, 4592,
	/*4592*/ uint16(x86_xSetOp), uint16(x86_PUNPCKLQDQ),
	/*4594*/ uint16(x86_xReadSlashR),
	/*4595*/ uint16(x86_xArgXmm1),
	/*4596*/ uint16(x86_xArgXmm2M128),
	/*4597*/ uint16(x86_xMatch),
	/*4598*/ uint16(x86_xCondPrefix), 1,
	0x66, 4602,
	/*4602*/ uint16(x86_xSetOp), uint16(x86_PUNPCKHQDQ),
	/*4604*/ uint16(x86_xReadSlashR),
	/*4605*/ uint16(x86_xArgXmm1),
	/*4606*/ uint16(x86_xArgXmm2M128),
	/*4607*/ uint16(x86_xMatch),
	/*4608*/ uint16(x86_xCondIs64), 4611, 4649,
	/*4611*/ uint16(x86_xCondPrefix), 2,
	0x66, 4633,
	0x0, 4617,
	/*4617*/ uint16(x86_xCondDataSize), 4621, 4627, 0,
	/*4621*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*4623*/ uint16(x86_xReadSlashR),
	/*4624*/ uint16(x86_xArgMm),
	/*4625*/ uint16(x86_xArgRM32),
	/*4626*/ uint16(x86_xMatch),
	/*4627*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*4629*/ uint16(x86_xReadSlashR),
	/*4630*/ uint16(x86_xArgMm),
	/*4631*/ uint16(x86_xArgRM32),
	/*4632*/ uint16(x86_xMatch),
	/*4633*/ uint16(x86_xCondDataSize), 4637, 4643, 0,
	/*4637*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*4639*/ uint16(x86_xReadSlashR),
	/*4640*/ uint16(x86_xArgXmm),
	/*4641*/ uint16(x86_xArgRM32),
	/*4642*/ uint16(x86_xMatch),
	/*4643*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*4645*/ uint16(x86_xReadSlashR),
	/*4646*/ uint16(x86_xArgXmm),
	/*4647*/ uint16(x86_xArgRM32),
	/*4648*/ uint16(x86_xMatch),
	/*4649*/ uint16(x86_xCondPrefix), 2,
	0x66, 4665,
	0x0, 4655,
	/*4655*/ uint16(x86_xCondDataSize), 4621, 4627, 4659,
	/*4659*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*4661*/ uint16(x86_xReadSlashR),
	/*4662*/ uint16(x86_xArgMm),
	/*4663*/ uint16(x86_xArgRM64),
	/*4664*/ uint16(x86_xMatch),
	/*4665*/ uint16(x86_xCondDataSize), 4637, 4643, 4669,
	/*4669*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*4671*/ uint16(x86_xReadSlashR),
	/*4672*/ uint16(x86_xArgXmm),
	/*4673*/ uint16(x86_xArgRM64),
	/*4674*/ uint16(x86_xMatch),
	/*4675*/ uint16(x86_xCondPrefix), 3,
	0xF3, 4695,
	0x66, 4689,
	0x0, 4683,
	/*4683*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*4685*/ uint16(x86_xReadSlashR),
	/*4686*/ uint16(x86_xArgMm),
	/*4687*/ uint16(x86_xArgMmM64),
	/*4688*/ uint16(x86_xMatch),
	/*4689*/ uint16(x86_xSetOp), uint16(x86_MOVDQA),
	/*4691*/ uint16(x86_xReadSlashR),
	/*4692*/ uint16(x86_xArgXmm1),
	/*4693*/ uint16(x86_xArgXmm2M128),
	/*4694*/ uint16(x86_xMatch),
	/*4695*/ uint16(x86_xSetOp), uint16(x86_MOVDQU),
	/*4697*/ uint16(x86_xReadSlashR),
	/*4698*/ uint16(x86_xArgXmm1),
	/*4699*/ uint16(x86_xArgXmm2M128),
	/*4700*/ uint16(x86_xMatch),
	/*4701*/ uint16(x86_xCondPrefix), 4,
	0xF3, 4735,
	0xF2, 4727,
	0x66, 4719,
	0x0, 4711,
	/*4711*/ uint16(x86_xSetOp), uint16(x86_PSHUFW),
	/*4713*/ uint16(x86_xReadSlashR),
	/*4714*/ uint16(x86_xReadIb),
	/*4715*/ uint16(x86_xArgMm1),
	/*4716*/ uint16(x86_xArgMm2M64),
	/*4717*/ uint16(x86_xArgImm8u),
	/*4718*/ uint16(x86_xMatch),
	/*4719*/ uint16(x86_xSetOp), uint16(x86_PSHUFD),
	/*4721*/ uint16(x86_xReadSlashR),
	/*4722*/ uint16(x86_xReadIb),
	/*4723*/ uint16(x86_xArgXmm1),
	/*4724*/ uint16(x86_xArgXmm2M128),
	/*4725*/ uint16(x86_xArgImm8u),
	/*4726*/ uint16(x86_xMatch),
	/*4727*/ uint16(x86_xSetOp), uint16(x86_PSHUFLW),
	/*4729*/ uint16(x86_xReadSlashR),
	/*4730*/ uint16(x86_xReadIb),
	/*4731*/ uint16(x86_xArgXmm1),
	/*4732*/ uint16(x86_xArgXmm2M128),
	/*4733*/ uint16(x86_xArgImm8u),
	/*4734*/ uint16(x86_xMatch),
	/*4735*/ uint16(x86_xSetOp), uint16(x86_PSHUFHW),
	/*4737*/ uint16(x86_xReadSlashR),
	/*4738*/ uint16(x86_xReadIb),
	/*4739*/ uint16(x86_xArgXmm1),
	/*4740*/ uint16(x86_xArgXmm2M128),
	/*4741*/ uint16(x86_xArgImm8u),
	/*4742*/ uint16(x86_xMatch),
	/*4743*/ uint16(x86_xCondSlashR),
	0,    // 0
	0,    // 1
	4752, // 2
	0,    // 3
	4770, // 4
	0,    // 5
	4788, // 6
	0,    // 7
	/*4752*/ uint16(x86_xCondPrefix), 2,
	0x66, 4764,
	0x0, 4758,
	/*4758*/ uint16(x86_xSetOp), uint16(x86_PSRLW),
	/*4760*/ uint16(x86_xReadIb),
	/*4761*/ uint16(x86_xArgMm2),
	/*4762*/ uint16(x86_xArgImm8u),
	/*4763*/ uint16(x86_xMatch),
	/*4764*/ uint16(x86_xSetOp), uint16(x86_PSRLW),
	/*4766*/ uint16(x86_xReadIb),
	/*4767*/ uint16(x86_xArgXmm2),
	/*4768*/ uint16(x86_xArgImm8u),
	/*4769*/ uint16(x86_xMatch),
	/*4770*/ uint16(x86_xCondPrefix), 2,
	0x66, 4782,
	0x0, 4776,
	/*4776*/ uint16(x86_xSetOp), uint16(x86_PSRAW),
	/*4778*/ uint16(x86_xReadIb),
	/*4779*/ uint16(x86_xArgMm2),
	/*4780*/ uint16(x86_xArgImm8u),
	/*4781*/ uint16(x86_xMatch),
	/*4782*/ uint16(x86_xSetOp), uint16(x86_PSRAW),
	/*4784*/ uint16(x86_xReadIb),
	/*4785*/ uint16(x86_xArgXmm2),
	/*4786*/ uint16(x86_xArgImm8u),
	/*4787*/ uint16(x86_xMatch),
	/*4788*/ uint16(x86_xCondPrefix), 2,
	0x66, 4800,
	0x0, 4794,
	/*4794*/ uint16(x86_xSetOp), uint16(x86_PSLLW),
	/*4796*/ uint16(x86_xReadIb),
	/*4797*/ uint16(x86_xArgMm2),
	/*4798*/ uint16(x86_xArgImm8u),
	/*4799*/ uint16(x86_xMatch),
	/*4800*/ uint16(x86_xSetOp), uint16(x86_PSLLW),
	/*4802*/ uint16(x86_xReadIb),
	/*4803*/ uint16(x86_xArgXmm2),
	/*4804*/ uint16(x86_xArgImm8u),
	/*4805*/ uint16(x86_xMatch),
	/*4806*/ uint16(x86_xCondSlashR),
	0,    // 0
	0,    // 1
	4815, // 2
	0,    // 3
	4833, // 4
	0,    // 5
	4851, // 6
	0,    // 7
	/*4815*/ uint16(x86_xCondPrefix), 2,
	0x66, 4827,
	0x0, 4821,
	/*4821*/ uint16(x86_xSetOp), uint16(x86_PSRLD),
	/*4823*/ uint16(x86_xReadIb),
	/*4824*/ uint16(x86_xArgMm2),
	/*4825*/ uint16(x86_xArgImm8u),
	/*4826*/ uint16(x86_xMatch),
	/*4827*/ uint16(x86_xSetOp), uint16(x86_PSRLD),
	/*4829*/ uint16(x86_xReadIb),
	/*4830*/ uint16(x86_xArgXmm2),
	/*4831*/ uint16(x86_xArgImm8u),
	/*4832*/ uint16(x86_xMatch),
	/*4833*/ uint16(x86_xCondPrefix), 2,
	0x66, 4845,
	0x0, 4839,
	/*4839*/ uint16(x86_xSetOp), uint16(x86_PSRAD),
	/*4841*/ uint16(x86_xReadIb),
	/*4842*/ uint16(x86_xArgMm2),
	/*4843*/ uint16(x86_xArgImm8u),
	/*4844*/ uint16(x86_xMatch),
	/*4845*/ uint16(x86_xSetOp), uint16(x86_PSRAD),
	/*4847*/ uint16(x86_xReadIb),
	/*4848*/ uint16(x86_xArgXmm2),
	/*4849*/ uint16(x86_xArgImm8u),
	/*4850*/ uint16(x86_xMatch),
	/*4851*/ uint16(x86_xCondPrefix), 2,
	0x66, 4863,
	0x0, 4857,
	/*4857*/ uint16(x86_xSetOp), uint16(x86_PSLLD),
	/*4859*/ uint16(x86_xReadIb),
	/*4860*/ uint16(x86_xArgMm2),
	/*4861*/ uint16(x86_xArgImm8u),
	/*4862*/ uint16(x86_xMatch),
	/*4863*/ uint16(x86_xSetOp), uint16(x86_PSLLD),
	/*4865*/ uint16(x86_xReadIb),
	/*4866*/ uint16(x86_xArgXmm2),
	/*4867*/ uint16(x86_xArgImm8u),
	/*4868*/ uint16(x86_xMatch),
	/*4869*/ uint16(x86_xCondSlashR),
	0,    // 0
	0,    // 1
	4878, // 2
	4896, // 3
	0,    // 4
	0,    // 5
	4906, // 6
	4924, // 7
	/*4878*/ uint16(x86_xCondPrefix), 2,
	0x66, 4890,
	0x0, 4884,
	/*4884*/ uint16(x86_xSetOp), uint16(x86_PSRLQ),
	/*4886*/ uint16(x86_xReadIb),
	/*4887*/ uint16(x86_xArgMm2),
	/*4888*/ uint16(x86_xArgImm8u),
	/*4889*/ uint16(x86_xMatch),
	/*4890*/ uint16(x86_xSetOp), uint16(x86_PSRLQ),
	/*4892*/ uint16(x86_xReadIb),
	/*4893*/ uint16(x86_xArgXmm2),
	/*4894*/ uint16(x86_xArgImm8u),
	/*4895*/ uint16(x86_xMatch),
	/*4896*/ uint16(x86_xCondPrefix), 1,
	0x66, 4900,
	/*4900*/ uint16(x86_xSetOp), uint16(x86_PSRLDQ),
	/*4902*/ uint16(x86_xReadIb),
	/*4903*/ uint16(x86_xArgXmm2),
	/*4904*/ uint16(x86_xArgImm8u),
	/*4905*/ uint16(x86_xMatch),
	/*4906*/ uint16(x86_xCondPrefix), 2,
	0x66, 4918,
	0x0, 4912,
	/*4912*/ uint16(x86_xSetOp), uint16(x86_PSLLQ),
	/*4914*/ uint16(x86_xReadIb),
	/*4915*/ uint16(x86_xArgMm2),
	/*4916*/ uint16(x86_xArgImm8u),
	/*4917*/ uint16(x86_xMatch),
	/*4918*/ uint16(x86_xSetOp), uint16(x86_PSLLQ),
	/*4920*/ uint16(x86_xReadIb),
	/*4921*/ uint16(x86_xArgXmm2),
	/*4922*/ uint16(x86_xArgImm8u),
	/*4923*/ uint16(x86_xMatch),
	/*4924*/ uint16(x86_xCondPrefix), 1,
	0x66, 4928,
	/*4928*/ uint16(x86_xSetOp), uint16(x86_PSLLDQ),
	/*4930*/ uint16(x86_xReadIb),
	/*4931*/ uint16(x86_xArgXmm2),
	/*4932*/ uint16(x86_xArgImm8u),
	/*4933*/ uint16(x86_xMatch),
	/*4934*/ uint16(x86_xCondPrefix), 2,
	0x66, 4946,
	0x0, 4940,
	/*4940*/ uint16(x86_xSetOp), uint16(x86_PCMPEQB),
	/*4942*/ uint16(x86_xReadSlashR),
	/*4943*/ uint16(x86_xArgMm),
	/*4944*/ uint16(x86_xArgMmM64),
	/*4945*/ uint16(x86_xMatch),
	/*4946*/ uint16(x86_xSetOp), uint16(x86_PCMPEQB),
	/*4948*/ uint16(x86_xReadSlashR),
	/*4949*/ uint16(x86_xArgXmm1),
	/*4950*/ uint16(x86_xArgXmm2M128),
	/*4951*/ uint16(x86_xMatch),
	/*4952*/ uint16(x86_xCondPrefix), 2,
	0x66, 4964,
	0x0, 4958,
	/*4958*/ uint16(x86_xSetOp), uint16(x86_PCMPEQW),
	/*4960*/ uint16(x86_xReadSlashR),
	/*4961*/ uint16(x86_xArgMm),
	/*4962*/ uint16(x86_xArgMmM64),
	/*4963*/ uint16(x86_xMatch),
	/*4964*/ uint16(x86_xSetOp), uint16(x86_PCMPEQW),
	/*4966*/ uint16(x86_xReadSlashR),
	/*4967*/ uint16(x86_xArgXmm1),
	/*4968*/ uint16(x86_xArgXmm2M128),
	/*4969*/ uint16(x86_xMatch),
	/*4970*/ uint16(x86_xCondPrefix), 2,
	0x66, 4982,
	0x0, 4976,
	/*4976*/ uint16(x86_xSetOp), uint16(x86_PCMPEQD),
	/*4978*/ uint16(x86_xReadSlashR),
	/*4979*/ uint16(x86_xArgMm),
	/*4980*/ uint16(x86_xArgMmM64),
	/*4981*/ uint16(x86_xMatch),
	/*4982*/ uint16(x86_xSetOp), uint16(x86_PCMPEQD),
	/*4984*/ uint16(x86_xReadSlashR),
	/*4985*/ uint16(x86_xArgXmm1),
	/*4986*/ uint16(x86_xArgXmm2M128),
	/*4987*/ uint16(x86_xMatch),
	/*4988*/ uint16(x86_xSetOp), uint16(x86_EMMS),
	/*4990*/ uint16(x86_xMatch),
	/*4991*/ uint16(x86_xCondPrefix), 2,
	0xF2, 5003,
	0x66, 4997,
	/*4997*/ uint16(x86_xSetOp), uint16(x86_HADDPD),
	/*4999*/ uint16(x86_xReadSlashR),
	/*5000*/ uint16(x86_xArgXmm1),
	/*5001*/ uint16(x86_xArgXmm2M128),
	/*5002*/ uint16(x86_xMatch),
	/*5003*/ uint16(x86_xSetOp), uint16(x86_HADDPS),
	/*5005*/ uint16(x86_xReadSlashR),
	/*5006*/ uint16(x86_xArgXmm1),
	/*5007*/ uint16(x86_xArgXmm2M128),
	/*5008*/ uint16(x86_xMatch),
	/*5009*/ uint16(x86_xCondPrefix), 2,
	0xF2, 5021,
	0x66, 5015,
	/*5015*/ uint16(x86_xSetOp), uint16(x86_HSUBPD),
	/*5017*/ uint16(x86_xReadSlashR),
	/*5018*/ uint16(x86_xArgXmm1),
	/*5019*/ uint16(x86_xArgXmm2M128),
	/*5020*/ uint16(x86_xMatch),
	/*5021*/ uint16(x86_xSetOp), uint16(x86_HSUBPS),
	/*5023*/ uint16(x86_xReadSlashR),
	/*5024*/ uint16(x86_xArgXmm1),
	/*5025*/ uint16(x86_xArgXmm2M128),
	/*5026*/ uint16(x86_xMatch),
	/*5027*/ uint16(x86_xCondIs64), 5030, 5076,
	/*5030*/ uint16(x86_xCondPrefix), 3,
	0xF3, 5070,
	0x66, 5054,
	0x0, 5038,
	/*5038*/ uint16(x86_xCondDataSize), 5042, 5048, 0,
	/*5042*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*5044*/ uint16(x86_xReadSlashR),
	/*5045*/ uint16(x86_xArgRM32),
	/*5046*/ uint16(x86_xArgMm),
	/*5047*/ uint16(x86_xMatch),
	/*5048*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*5050*/ uint16(x86_xReadSlashR),
	/*5051*/ uint16(x86_xArgRM32),
	/*5052*/ uint16(x86_xArgMm),
	/*5053*/ uint16(x86_xMatch),
	/*5054*/ uint16(x86_xCondDataSize), 5058, 5064, 0,
	/*5058*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*5060*/ uint16(x86_xReadSlashR),
	/*5061*/ uint16(x86_xArgRM32),
	/*5062*/ uint16(x86_xArgXmm),
	/*5063*/ uint16(x86_xMatch),
	/*5064*/ uint16(x86_xSetOp), uint16(x86_MOVD),
	/*5066*/ uint16(x86_xReadSlashR),
	/*5067*/ uint16(x86_xArgRM32),
	/*5068*/ uint16(x86_xArgXmm),
	/*5069*/ uint16(x86_xMatch),
	/*5070*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*5072*/ uint16(x86_xReadSlashR),
	/*5073*/ uint16(x86_xArgXmm1),
	/*5074*/ uint16(x86_xArgXmm2M64),
	/*5075*/ uint16(x86_xMatch),
	/*5076*/ uint16(x86_xCondPrefix), 3,
	0xF3, 5070,
	0x66, 5094,
	0x0, 5084,
	/*5084*/ uint16(x86_xCondDataSize), 5042, 5048, 5088,
	/*5088*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*5090*/ uint16(x86_xReadSlashR),
	/*5091*/ uint16(x86_xArgRM64),
	/*5092*/ uint16(x86_xArgMm),
	/*5093*/ uint16(x86_xMatch),
	/*5094*/ uint16(x86_xCondDataSize), 5058, 5064, 5098,
	/*5098*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*5100*/ uint16(x86_xReadSlashR),
	/*5101*/ uint16(x86_xArgRM64),
	/*5102*/ uint16(x86_xArgXmm),
	/*5103*/ uint16(x86_xMatch),
	/*5104*/ uint16(x86_xCondPrefix), 3,
	0xF3, 5124,
	0x66, 5118,
	0x0, 5112,
	/*5112*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*5114*/ uint16(x86_xReadSlashR),
	/*5115*/ uint16(x86_xArgMmM64),
	/*5116*/ uint16(x86_xArgMm),
	/*5117*/ uint16(x86_xMatch),
	/*5118*/ uint16(x86_xSetOp), uint16(x86_MOVDQA),
	/*5120*/ uint16(x86_xReadSlashR),
	/*5121*/ uint16(x86_xArgXmm2M128),
	/*5122*/ uint16(x86_xArgXmm1),
	/*5123*/ uint16(x86_xMatch),
	/*5124*/ uint16(x86_xSetOp), uint16(x86_MOVDQU),
	/*5126*/ uint16(x86_xReadSlashR),
	/*5127*/ uint16(x86_xArgXmm2M128),
	/*5128*/ uint16(x86_xArgXmm1),
	/*5129*/ uint16(x86_xMatch),
	/*5130*/ uint16(x86_xCondIs64), 5133, 5147,
	/*5133*/ uint16(x86_xCondDataSize), 5137, 5142, 0,
	/*5137*/ uint16(x86_xSetOp), uint16(x86_JO),
	/*5139*/ uint16(x86_xReadCw),
	/*5140*/ uint16(x86_xArgRel16),
	/*5141*/ uint16(x86_xMatch),
	/*5142*/ uint16(x86_xSetOp), uint16(x86_JO),
	/*5144*/ uint16(x86_xReadCd),
	/*5145*/ uint16(x86_xArgRel32),
	/*5146*/ uint16(x86_xMatch),
	/*5147*/ uint16(x86_xCondDataSize), 5151, 5142, 5156,
	/*5151*/ uint16(x86_xSetOp), uint16(x86_JO),
	/*5153*/ uint16(x86_xReadCd),
	/*5154*/ uint16(x86_xArgRel32),
	/*5155*/ uint16(x86_xMatch),
	/*5156*/ uint16(x86_xSetOp), uint16(x86_JO),
	/*5158*/ uint16(x86_xReadCd),
	/*5159*/ uint16(x86_xArgRel32),
	/*5160*/ uint16(x86_xMatch),
	/*5161*/ uint16(x86_xCondIs64), 5164, 5178,
	/*5164*/ uint16(x86_xCondDataSize), 5168, 5173, 0,
	/*5168*/ uint16(x86_xSetOp), uint16(x86_JNO),
	/*5170*/ uint16(x86_xReadCw),
	/*5171*/ uint16(x86_xArgRel16),
	/*5172*/ uint16(x86_xMatch),
	/*5173*/ uint16(x86_xSetOp), uint16(x86_JNO),
	/*5175*/ uint16(x86_xReadCd),
	/*5176*/ uint16(x86_xArgRel32),
	/*5177*/ uint16(x86_xMatch),
	/*5178*/ uint16(x86_xCondDataSize), 5182, 5173, 5187,
	/*5182*/ uint16(x86_xSetOp), uint16(x86_JNO),
	/*5184*/ uint16(x86_xReadCd),
	/*5185*/ uint16(x86_xArgRel32),
	/*5186*/ uint16(x86_xMatch),
	/*5187*/ uint16(x86_xSetOp), uint16(x86_JNO),
	/*5189*/ uint16(x86_xReadCd),
	/*5190*/ uint16(x86_xArgRel32),
	/*5191*/ uint16(x86_xMatch),
	/*5192*/ uint16(x86_xCondIs64), 5195, 5209,
	/*5195*/ uint16(x86_xCondDataSize), 5199, 5204, 0,
	/*5199*/ uint16(x86_xSetOp), uint16(x86_JB),
	/*5201*/ uint16(x86_xReadCw),
	/*5202*/ uint16(x86_xArgRel16),
	/*5203*/ uint16(x86_xMatch),
	/*5204*/ uint16(x86_xSetOp), uint16(x86_JB),
	/*5206*/ uint16(x86_xReadCd),
	/*5207*/ uint16(x86_xArgRel32),
	/*5208*/ uint16(x86_xMatch),
	/*5209*/ uint16(x86_xCondDataSize), 5213, 5204, 5218,
	/*5213*/ uint16(x86_xSetOp), uint16(x86_JB),
	/*5215*/ uint16(x86_xReadCd),
	/*5216*/ uint16(x86_xArgRel32),
	/*5217*/ uint16(x86_xMatch),
	/*5218*/ uint16(x86_xSetOp), uint16(x86_JB),
	/*5220*/ uint16(x86_xReadCd),
	/*5221*/ uint16(x86_xArgRel32),
	/*5222*/ uint16(x86_xMatch),
	/*5223*/ uint16(x86_xCondIs64), 5226, 5240,
	/*5226*/ uint16(x86_xCondDataSize), 5230, 5235, 0,
	/*5230*/ uint16(x86_xSetOp), uint16(x86_JAE),
	/*5232*/ uint16(x86_xReadCw),
	/*5233*/ uint16(x86_xArgRel16),
	/*5234*/ uint16(x86_xMatch),
	/*5235*/ uint16(x86_xSetOp), uint16(x86_JAE),
	/*5237*/ uint16(x86_xReadCd),
	/*5238*/ uint16(x86_xArgRel32),
	/*5239*/ uint16(x86_xMatch),
	/*5240*/ uint16(x86_xCondDataSize), 5244, 5235, 5249,
	/*5244*/ uint16(x86_xSetOp), uint16(x86_JAE),
	/*5246*/ uint16(x86_xReadCd),
	/*5247*/ uint16(x86_xArgRel32),
	/*5248*/ uint16(x86_xMatch),
	/*5249*/ uint16(x86_xSetOp), uint16(x86_JAE),
	/*5251*/ uint16(x86_xReadCd),
	/*5252*/ uint16(x86_xArgRel32),
	/*5253*/ uint16(x86_xMatch),
	/*5254*/ uint16(x86_xCondIs64), 5257, 5271,
	/*5257*/ uint16(x86_xCondDataSize), 5261, 5266, 0,
	/*5261*/ uint16(x86_xSetOp), uint16(x86_JE),
	/*5263*/ uint16(x86_xReadCw),
	/*5264*/ uint16(x86_xArgRel16),
	/*5265*/ uint16(x86_xMatch),
	/*5266*/ uint16(x86_xSetOp), uint16(x86_JE),
	/*5268*/ uint16(x86_xReadCd),
	/*5269*/ uint16(x86_xArgRel32),
	/*5270*/ uint16(x86_xMatch),
	/*5271*/ uint16(x86_xCondDataSize), 5275, 5266, 5280,
	/*5275*/ uint16(x86_xSetOp), uint16(x86_JE),
	/*5277*/ uint16(x86_xReadCd),
	/*5278*/ uint16(x86_xArgRel32),
	/*5279*/ uint16(x86_xMatch),
	/*5280*/ uint16(x86_xSetOp), uint16(x86_JE),
	/*5282*/ uint16(x86_xReadCd),
	/*5283*/ uint16(x86_xArgRel32),
	/*5284*/ uint16(x86_xMatch),
	/*5285*/ uint16(x86_xCondIs64), 5288, 5302,
	/*5288*/ uint16(x86_xCondDataSize), 5292, 5297, 0,
	/*5292*/ uint16(x86_xSetOp), uint16(x86_JNE),
	/*5294*/ uint16(x86_xReadCw),
	/*5295*/ uint16(x86_xArgRel16),
	/*5296*/ uint16(x86_xMatch),
	/*5297*/ uint16(x86_xSetOp), uint16(x86_JNE),
	/*5299*/ uint16(x86_xReadCd),
	/*5300*/ uint16(x86_xArgRel32),
	/*5301*/ uint16(x86_xMatch),
	/*5302*/ uint16(x86_xCondDataSize), 5306, 5297, 5311,
	/*5306*/ uint16(x86_xSetOp), uint16(x86_JNE),
	/*5308*/ uint16(x86_xReadCd),
	/*5309*/ uint16(x86_xArgRel32),
	/*5310*/ uint16(x86_xMatch),
	/*5311*/ uint16(x86_xSetOp), uint16(x86_JNE),
	/*5313*/ uint16(x86_xReadCd),
	/*5314*/ uint16(x86_xArgRel32),
	/*5315*/ uint16(x86_xMatch),
	/*5316*/ uint16(x86_xCondIs64), 5319, 5333,
	/*5319*/ uint16(x86_xCondDataSize), 5323, 5328, 0,
	/*5323*/ uint16(x86_xSetOp), uint16(x86_JBE),
	/*5325*/ uint16(x86_xReadCw),
	/*5326*/ uint16(x86_xArgRel16),
	/*5327*/ uint16(x86_xMatch),
	/*5328*/ uint16(x86_xSetOp), uint16(x86_JBE),
	/*5330*/ uint16(x86_xReadCd),
	/*5331*/ uint16(x86_xArgRel32),
	/*5332*/ uint16(x86_xMatch),
	/*5333*/ uint16(x86_xCondDataSize), 5337, 5328, 5342,
	/*5337*/ uint16(x86_xSetOp), uint16(x86_JBE),
	/*5339*/ uint16(x86_xReadCd),
	/*5340*/ uint16(x86_xArgRel32),
	/*5341*/ uint16(x86_xMatch),
	/*5342*/ uint16(x86_xSetOp), uint16(x86_JBE),
	/*5344*/ uint16(x86_xReadCd),
	/*5345*/ uint16(x86_xArgRel32),
	/*5346*/ uint16(x86_xMatch),
	/*5347*/ uint16(x86_xCondIs64), 5350, 5364,
	/*5350*/ uint16(x86_xCondDataSize), 5354, 5359, 0,
	/*5354*/ uint16(x86_xSetOp), uint16(x86_JA),
	/*5356*/ uint16(x86_xReadCw),
	/*5357*/ uint16(x86_xArgRel16),
	/*5358*/ uint16(x86_xMatch),
	/*5359*/ uint16(x86_xSetOp), uint16(x86_JA),
	/*5361*/ uint16(x86_xReadCd),
	/*5362*/ uint16(x86_xArgRel32),
	/*5363*/ uint16(x86_xMatch),
	/*5364*/ uint16(x86_xCondDataSize), 5368, 5359, 5373,
	/*5368*/ uint16(x86_xSetOp), uint16(x86_JA),
	/*5370*/ uint16(x86_xReadCd),
	/*5371*/ uint16(x86_xArgRel32),
	/*5372*/ uint16(x86_xMatch),
	/*5373*/ uint16(x86_xSetOp), uint16(x86_JA),
	/*5375*/ uint16(x86_xReadCd),
	/*5376*/ uint16(x86_xArgRel32),
	/*5377*/ uint16(x86_xMatch),
	/*5378*/ uint16(x86_xCondIs64), 5381, 5395,
	/*5381*/ uint16(x86_xCondDataSize), 5385, 5390, 0,
	/*5385*/ uint16(x86_xSetOp), uint16(x86_JS),
	/*5387*/ uint16(x86_xReadCw),
	/*5388*/ uint16(x86_xArgRel16),
	/*5389*/ uint16(x86_xMatch),
	/*5390*/ uint16(x86_xSetOp), uint16(x86_JS),
	/*5392*/ uint16(x86_xReadCd),
	/*5393*/ uint16(x86_xArgRel32),
	/*5394*/ uint16(x86_xMatch),
	/*5395*/ uint16(x86_xCondDataSize), 5399, 5390, 5404,
	/*5399*/ uint16(x86_xSetOp), uint16(x86_JS),
	/*5401*/ uint16(x86_xReadCd),
	/*5402*/ uint16(x86_xArgRel32),
	/*5403*/ uint16(x86_xMatch),
	/*5404*/ uint16(x86_xSetOp), uint16(x86_JS),
	/*5406*/ uint16(x86_xReadCd),
	/*5407*/ uint16(x86_xArgRel32),
	/*5408*/ uint16(x86_xMatch),
	/*5409*/ uint16(x86_xCondIs64), 5412, 5426,
	/*5412*/ uint16(x86_xCondDataSize), 5416, 5421, 0,
	/*5416*/ uint16(x86_xSetOp), uint16(x86_JNS),
	/*5418*/ uint16(x86_xReadCw),
	/*5419*/ uint16(x86_xArgRel16),
	/*5420*/ uint16(x86_xMatch),
	/*5421*/ uint16(x86_xSetOp), uint16(x86_JNS),
	/*5423*/ uint16(x86_xReadCd),
	/*5424*/ uint16(x86_xArgRel32),
	/*5425*/ uint16(x86_xMatch),
	/*5426*/ uint16(x86_xCondDataSize), 5430, 5421, 5435,
	/*5430*/ uint16(x86_xSetOp), uint16(x86_JNS),
	/*5432*/ uint16(x86_xReadCd),
	/*5433*/ uint16(x86_xArgRel32),
	/*5434*/ uint16(x86_xMatch),
	/*5435*/ uint16(x86_xSetOp), uint16(x86_JNS),
	/*5437*/ uint16(x86_xReadCd),
	/*5438*/ uint16(x86_xArgRel32),
	/*5439*/ uint16(x86_xMatch),
	/*5440*/ uint16(x86_xCondIs64), 5443, 5457,
	/*5443*/ uint16(x86_xCondDataSize), 5447, 5452, 0,
	/*5447*/ uint16(x86_xSetOp), uint16(x86_JP),
	/*5449*/ uint16(x86_xReadCw),
	/*5450*/ uint16(x86_xArgRel16),
	/*5451*/ uint16(x86_xMatch),
	/*5452*/ uint16(x86_xSetOp), uint16(x86_JP),
	/*5454*/ uint16(x86_xReadCd),
	/*5455*/ uint16(x86_xArgRel32),
	/*5456*/ uint16(x86_xMatch),
	/*5457*/ uint16(x86_xCondDataSize), 5461, 5452, 5466,
	/*5461*/ uint16(x86_xSetOp), uint16(x86_JP),
	/*5463*/ uint16(x86_xReadCd),
	/*5464*/ uint16(x86_xArgRel32),
	/*5465*/ uint16(x86_xMatch),
	/*5466*/ uint16(x86_xSetOp), uint16(x86_JP),
	/*5468*/ uint16(x86_xReadCd),
	/*5469*/ uint16(x86_xArgRel32),
	/*5470*/ uint16(x86_xMatch),
	/*5471*/ uint16(x86_xCondIs64), 5474, 5488,
	/*5474*/ uint16(x86_xCondDataSize), 5478, 5483, 0,
	/*5478*/ uint16(x86_xSetOp), uint16(x86_JNP),
	/*5480*/ uint16(x86_xReadCw),
	/*5481*/ uint16(x86_xArgRel16),
	/*5482*/ uint16(x86_xMatch),
	/*5483*/ uint16(x86_xSetOp), uint16(x86_JNP),
	/*5485*/ uint16(x86_xReadCd),
	/*5486*/ uint16(x86_xArgRel32),
	/*5487*/ uint16(x86_xMatch),
	/*5488*/ uint16(x86_xCondDataSize), 5492, 5483, 5497,
	/*5492*/ uint16(x86_xSetOp), uint16(x86_JNP),
	/*5494*/ uint16(x86_xReadCd),
	/*5495*/ uint16(x86_xArgRel32),
	/*5496*/ uint16(x86_xMatch),
	/*5497*/ uint16(x86_xSetOp), uint16(x86_JNP),
	/*5499*/ uint16(x86_xReadCd),
	/*5500*/ uint16(x86_xArgRel32),
	/*5501*/ uint16(x86_xMatch),
	/*5502*/ uint16(x86_xCondIs64), 5505, 5519,
	/*5505*/ uint16(x86_xCondDataSize), 5509, 5514, 0,
	/*5509*/ uint16(x86_xSetOp), uint16(x86_JL),
	/*5511*/ uint16(x86_xReadCw),
	/*5512*/ uint16(x86_xArgRel16),
	/*5513*/ uint16(x86_xMatch),
	/*5514*/ uint16(x86_xSetOp), uint16(x86_JL),
	/*5516*/ uint16(x86_xReadCd),
	/*5517*/ uint16(x86_xArgRel32),
	/*5518*/ uint16(x86_xMatch),
	/*5519*/ uint16(x86_xCondDataSize), 5523, 5514, 5528,
	/*5523*/ uint16(x86_xSetOp), uint16(x86_JL),
	/*5525*/ uint16(x86_xReadCd),
	/*5526*/ uint16(x86_xArgRel32),
	/*5527*/ uint16(x86_xMatch),
	/*5528*/ uint16(x86_xSetOp), uint16(x86_JL),
	/*5530*/ uint16(x86_xReadCd),
	/*5531*/ uint16(x86_xArgRel32),
	/*5532*/ uint16(x86_xMatch),
	/*5533*/ uint16(x86_xCondIs64), 5536, 5550,
	/*5536*/ uint16(x86_xCondDataSize), 5540, 5545, 0,
	/*5540*/ uint16(x86_xSetOp), uint16(x86_JGE),
	/*5542*/ uint16(x86_xReadCw),
	/*5543*/ uint16(x86_xArgRel16),
	/*5544*/ uint16(x86_xMatch),
	/*5545*/ uint16(x86_xSetOp), uint16(x86_JGE),
	/*5547*/ uint16(x86_xReadCd),
	/*5548*/ uint16(x86_xArgRel32),
	/*5549*/ uint16(x86_xMatch),
	/*5550*/ uint16(x86_xCondDataSize), 5554, 5545, 5559,
	/*5554*/ uint16(x86_xSetOp), uint16(x86_JGE),
	/*5556*/ uint16(x86_xReadCd),
	/*5557*/ uint16(x86_xArgRel32),
	/*5558*/ uint16(x86_xMatch),
	/*5559*/ uint16(x86_xSetOp), uint16(x86_JGE),
	/*5561*/ uint16(x86_xReadCd),
	/*5562*/ uint16(x86_xArgRel32),
	/*5563*/ uint16(x86_xMatch),
	/*5564*/ uint16(x86_xCondIs64), 5567, 5581,
	/*5567*/ uint16(x86_xCondDataSize), 5571, 5576, 0,
	/*5571*/ uint16(x86_xSetOp), uint16(x86_JLE),
	/*5573*/ uint16(x86_xReadCw),
	/*5574*/ uint16(x86_xArgRel16),
	/*5575*/ uint16(x86_xMatch),
	/*5576*/ uint16(x86_xSetOp), uint16(x86_JLE),
	/*5578*/ uint16(x86_xReadCd),
	/*5579*/ uint16(x86_xArgRel32),
	/*5580*/ uint16(x86_xMatch),
	/*5581*/ uint16(x86_xCondDataSize), 5585, 5576, 5590,
	/*5585*/ uint16(x86_xSetOp), uint16(x86_JLE),
	/*5587*/ uint16(x86_xReadCd),
	/*5588*/ uint16(x86_xArgRel32),
	/*5589*/ uint16(x86_xMatch),
	/*5590*/ uint16(x86_xSetOp), uint16(x86_JLE),
	/*5592*/ uint16(x86_xReadCd),
	/*5593*/ uint16(x86_xArgRel32),
	/*5594*/ uint16(x86_xMatch),
	/*5595*/ uint16(x86_xCondIs64), 5598, 5612,
	/*5598*/ uint16(x86_xCondDataSize), 5602, 5607, 0,
	/*5602*/ uint16(x86_xSetOp), uint16(x86_JG),
	/*5604*/ uint16(x86_xReadCw),
	/*5605*/ uint16(x86_xArgRel16),
	/*5606*/ uint16(x86_xMatch),
	/*5607*/ uint16(x86_xSetOp), uint16(x86_JG),
	/*5609*/ uint16(x86_xReadCd),
	/*5610*/ uint16(x86_xArgRel32),
	/*5611*/ uint16(x86_xMatch),
	/*5612*/ uint16(x86_xCondDataSize), 5616, 5607, 5621,
	/*5616*/ uint16(x86_xSetOp), uint16(x86_JG),
	/*5618*/ uint16(x86_xReadCd),
	/*5619*/ uint16(x86_xArgRel32),
	/*5620*/ uint16(x86_xMatch),
	/*5621*/ uint16(x86_xSetOp), uint16(x86_JG),
	/*5623*/ uint16(x86_xReadCd),
	/*5624*/ uint16(x86_xArgRel32),
	/*5625*/ uint16(x86_xMatch),
	/*5626*/ uint16(x86_xSetOp), uint16(x86_SETO),
	/*5628*/ uint16(x86_xReadSlashR),
	/*5629*/ uint16(x86_xArgRM8),
	/*5630*/ uint16(x86_xMatch),
	/*5631*/ uint16(x86_xSetOp), uint16(x86_SETNO),
	/*5633*/ uint16(x86_xReadSlashR),
	/*5634*/ uint16(x86_xArgRM8),
	/*5635*/ uint16(x86_xMatch),
	/*5636*/ uint16(x86_xSetOp), uint16(x86_SETB),
	/*5638*/ uint16(x86_xReadSlashR),
	/*5639*/ uint16(x86_xArgRM8),
	/*5640*/ uint16(x86_xMatch),
	/*5641*/ uint16(x86_xSetOp), uint16(x86_SETAE),
	/*5643*/ uint16(x86_xReadSlashR),
	/*5644*/ uint16(x86_xArgRM8),
	/*5645*/ uint16(x86_xMatch),
	/*5646*/ uint16(x86_xSetOp), uint16(x86_SETE),
	/*5648*/ uint16(x86_xReadSlashR),
	/*5649*/ uint16(x86_xArgRM8),
	/*5650*/ uint16(x86_xMatch),
	/*5651*/ uint16(x86_xSetOp), uint16(x86_SETNE),
	/*5653*/ uint16(x86_xReadSlashR),
	/*5654*/ uint16(x86_xArgRM8),
	/*5655*/ uint16(x86_xMatch),
	/*5656*/ uint16(x86_xSetOp), uint16(x86_SETBE),
	/*5658*/ uint16(x86_xReadSlashR),
	/*5659*/ uint16(x86_xArgRM8),
	/*5660*/ uint16(x86_xMatch),
	/*5661*/ uint16(x86_xSetOp), uint16(x86_SETA),
	/*5663*/ uint16(x86_xReadSlashR),
	/*5664*/ uint16(x86_xArgRM8),
	/*5665*/ uint16(x86_xMatch),
	/*5666*/ uint16(x86_xSetOp), uint16(x86_SETS),
	/*5668*/ uint16(x86_xReadSlashR),
	/*5669*/ uint16(x86_xArgRM8),
	/*5670*/ uint16(x86_xMatch),
	/*5671*/ uint16(x86_xSetOp), uint16(x86_SETNS),
	/*5673*/ uint16(x86_xReadSlashR),
	/*5674*/ uint16(x86_xArgRM8),
	/*5675*/ uint16(x86_xMatch),
	/*5676*/ uint16(x86_xSetOp), uint16(x86_SETP),
	/*5678*/ uint16(x86_xReadSlashR),
	/*5679*/ uint16(x86_xArgRM8),
	/*5680*/ uint16(x86_xMatch),
	/*5681*/ uint16(x86_xSetOp), uint16(x86_SETNP),
	/*5683*/ uint16(x86_xReadSlashR),
	/*5684*/ uint16(x86_xArgRM8),
	/*5685*/ uint16(x86_xMatch),
	/*5686*/ uint16(x86_xSetOp), uint16(x86_SETL),
	/*5688*/ uint16(x86_xReadSlashR),
	/*5689*/ uint16(x86_xArgRM8),
	/*5690*/ uint16(x86_xMatch),
	/*5691*/ uint16(x86_xSetOp), uint16(x86_SETGE),
	/*5693*/ uint16(x86_xReadSlashR),
	/*5694*/ uint16(x86_xArgRM8),
	/*5695*/ uint16(x86_xMatch),
	/*5696*/ uint16(x86_xSetOp), uint16(x86_SETLE),
	/*5698*/ uint16(x86_xReadSlashR),
	/*5699*/ uint16(x86_xArgRM8),
	/*5700*/ uint16(x86_xMatch),
	/*5701*/ uint16(x86_xSetOp), uint16(x86_SETG),
	/*5703*/ uint16(x86_xReadSlashR),
	/*5704*/ uint16(x86_xArgRM8),
	/*5705*/ uint16(x86_xMatch),
	/*5706*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*5708*/ uint16(x86_xArgFS),
	/*5709*/ uint16(x86_xMatch),
	/*5710*/ uint16(x86_xCondIs64), 5713, 5725,
	/*5713*/ uint16(x86_xCondDataSize), 5717, 5721, 0,
	/*5717*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5719*/ uint16(x86_xArgFS),
	/*5720*/ uint16(x86_xMatch),
	/*5721*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5723*/ uint16(x86_xArgFS),
	/*5724*/ uint16(x86_xMatch),
	/*5725*/ uint16(x86_xCondDataSize), 5717, 5729, 5733,
	/*5729*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5731*/ uint16(x86_xArgFS),
	/*5732*/ uint16(x86_xMatch),
	/*5733*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5735*/ uint16(x86_xArgFS),
	/*5736*/ uint16(x86_xMatch),
	/*5737*/ uint16(x86_xSetOp), uint16(x86_CPUID),
	/*5739*/ uint16(x86_xMatch),
	/*5740*/ uint16(x86_xCondIs64), 5743, 5759,
	/*5743*/ uint16(x86_xCondDataSize), 5747, 5753, 0,
	/*5747*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*5749*/ uint16(x86_xReadSlashR),
	/*5750*/ uint16(x86_xArgRM16),
	/*5751*/ uint16(x86_xArgR16),
	/*5752*/ uint16(x86_xMatch),
	/*5753*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*5755*/ uint16(x86_xReadSlashR),
	/*5756*/ uint16(x86_xArgRM32),
	/*5757*/ uint16(x86_xArgR32),
	/*5758*/ uint16(x86_xMatch),
	/*5759*/ uint16(x86_xCondDataSize), 5747, 5753, 5763,
	/*5763*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*5765*/ uint16(x86_xReadSlashR),
	/*5766*/ uint16(x86_xArgRM64),
	/*5767*/ uint16(x86_xArgR64),
	/*5768*/ uint16(x86_xMatch),
	/*5769*/ uint16(x86_xCondIs64), 5772, 5792,
	/*5772*/ uint16(x86_xCondDataSize), 5776, 5784, 0,
	/*5776*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5778*/ uint16(x86_xReadSlashR),
	/*5779*/ uint16(x86_xReadIb),
	/*5780*/ uint16(x86_xArgRM16),
	/*5781*/ uint16(x86_xArgR16),
	/*5782*/ uint16(x86_xArgImm8u),
	/*5783*/ uint16(x86_xMatch),
	/*5784*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5786*/ uint16(x86_xReadSlashR),
	/*5787*/ uint16(x86_xReadIb),
	/*5788*/ uint16(x86_xArgRM32),
	/*5789*/ uint16(x86_xArgR32),
	/*5790*/ uint16(x86_xArgImm8u),
	/*5791*/ uint16(x86_xMatch),
	/*5792*/ uint16(x86_xCondDataSize), 5776, 5784, 5796,
	/*5796*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5798*/ uint16(x86_xReadSlashR),
	/*5799*/ uint16(x86_xReadIb),
	/*5800*/ uint16(x86_xArgRM64),
	/*5801*/ uint16(x86_xArgR64),
	/*5802*/ uint16(x86_xArgImm8u),
	/*5803*/ uint16(x86_xMatch),
	/*5804*/ uint16(x86_xCondIs64), 5807, 5825,
	/*5807*/ uint16(x86_xCondDataSize), 5811, 5818, 0,
	/*5811*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5813*/ uint16(x86_xReadSlashR),
	/*5814*/ uint16(x86_xArgRM16),
	/*5815*/ uint16(x86_xArgR16),
	/*5816*/ uint16(x86_xArgCL),
	/*5817*/ uint16(x86_xMatch),
	/*5818*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5820*/ uint16(x86_xReadSlashR),
	/*5821*/ uint16(x86_xArgRM32),
	/*5822*/ uint16(x86_xArgR32),
	/*5823*/ uint16(x86_xArgCL),
	/*5824*/ uint16(x86_xMatch),
	/*5825*/ uint16(x86_xCondDataSize), 5811, 5818, 5829,
	/*5829*/ uint16(x86_xSetOp), uint16(x86_SHLD),
	/*5831*/ uint16(x86_xReadSlashR),
	/*5832*/ uint16(x86_xArgRM64),
	/*5833*/ uint16(x86_xArgR64),
	/*5834*/ uint16(x86_xArgCL),
	/*5835*/ uint16(x86_xMatch),
	/*5836*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*5838*/ uint16(x86_xArgGS),
	/*5839*/ uint16(x86_xMatch),
	/*5840*/ uint16(x86_xCondIs64), 5843, 5855,
	/*5843*/ uint16(x86_xCondDataSize), 5847, 5851, 0,
	/*5847*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5849*/ uint16(x86_xArgGS),
	/*5850*/ uint16(x86_xMatch),
	/*5851*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5853*/ uint16(x86_xArgGS),
	/*5854*/ uint16(x86_xMatch),
	/*5855*/ uint16(x86_xCondDataSize), 5847, 5859, 5863,
	/*5859*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5861*/ uint16(x86_xArgGS),
	/*5862*/ uint16(x86_xMatch),
	/*5863*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*5865*/ uint16(x86_xArgGS),
	/*5866*/ uint16(x86_xMatch),
	/*5867*/ uint16(x86_xSetOp), uint16(x86_RSM),
	/*5869*/ uint16(x86_xMatch),
	/*5870*/ uint16(x86_xCondIs64), 5873, 5889,
	/*5873*/ uint16(x86_xCondDataSize), 5877, 5883, 0,
	/*5877*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*5879*/ uint16(x86_xReadSlashR),
	/*5880*/ uint16(x86_xArgRM16),
	/*5881*/ uint16(x86_xArgR16),
	/*5882*/ uint16(x86_xMatch),
	/*5883*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*5885*/ uint16(x86_xReadSlashR),
	/*5886*/ uint16(x86_xArgRM32),
	/*5887*/ uint16(x86_xArgR32),
	/*5888*/ uint16(x86_xMatch),
	/*5889*/ uint16(x86_xCondDataSize), 5877, 5883, 5893,
	/*5893*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*5895*/ uint16(x86_xReadSlashR),
	/*5896*/ uint16(x86_xArgRM64),
	/*5897*/ uint16(x86_xArgR64),
	/*5898*/ uint16(x86_xMatch),
	/*5899*/ uint16(x86_xCondIs64), 5902, 5922,
	/*5902*/ uint16(x86_xCondDataSize), 5906, 5914, 0,
	/*5906*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5908*/ uint16(x86_xReadSlashR),
	/*5909*/ uint16(x86_xReadIb),
	/*5910*/ uint16(x86_xArgRM16),
	/*5911*/ uint16(x86_xArgR16),
	/*5912*/ uint16(x86_xArgImm8u),
	/*5913*/ uint16(x86_xMatch),
	/*5914*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5916*/ uint16(x86_xReadSlashR),
	/*5917*/ uint16(x86_xReadIb),
	/*5918*/ uint16(x86_xArgRM32),
	/*5919*/ uint16(x86_xArgR32),
	/*5920*/ uint16(x86_xArgImm8u),
	/*5921*/ uint16(x86_xMatch),
	/*5922*/ uint16(x86_xCondDataSize), 5906, 5914, 5926,
	/*5926*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5928*/ uint16(x86_xReadSlashR),
	/*5929*/ uint16(x86_xReadIb),
	/*5930*/ uint16(x86_xArgRM64),
	/*5931*/ uint16(x86_xArgR64),
	/*5932*/ uint16(x86_xArgImm8u),
	/*5933*/ uint16(x86_xMatch),
	/*5934*/ uint16(x86_xCondIs64), 5937, 5955,
	/*5937*/ uint16(x86_xCondDataSize), 5941, 5948, 0,
	/*5941*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5943*/ uint16(x86_xReadSlashR),
	/*5944*/ uint16(x86_xArgRM16),
	/*5945*/ uint16(x86_xArgR16),
	/*5946*/ uint16(x86_xArgCL),
	/*5947*/ uint16(x86_xMatch),
	/*5948*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5950*/ uint16(x86_xReadSlashR),
	/*5951*/ uint16(x86_xArgRM32),
	/*5952*/ uint16(x86_xArgR32),
	/*5953*/ uint16(x86_xArgCL),
	/*5954*/ uint16(x86_xMatch),
	/*5955*/ uint16(x86_xCondDataSize), 5941, 5948, 5959,
	/*5959*/ uint16(x86_xSetOp), uint16(x86_SHRD),
	/*5961*/ uint16(x86_xReadSlashR),
	/*5962*/ uint16(x86_xArgRM64),
	/*5963*/ uint16(x86_xArgR64),
	/*5964*/ uint16(x86_xArgCL),
	/*5965*/ uint16(x86_xMatch),
	/*5966*/ uint16(x86_xCondByte), 3,
	0xE8, 6215,
	0xF0, 6218,
	0xF8, 6221,
	/*5974*/ uint16(x86_xCondSlashR),
	5983, // 0
	6037, // 1
	6091, // 2
	6120, // 3
	6149, // 4
	6172, // 5
	6195, // 6
	6211, // 7
	/*5983*/ uint16(x86_xCondIs64), 5986, 5998,
	/*5986*/ uint16(x86_xCondDataSize), 5990, 5994, 0,
	/*5990*/ uint16(x86_xSetOp), uint16(x86_FXSAVE),
	/*5992*/ uint16(x86_xArgM512byte),
	/*5993*/ uint16(x86_xMatch),
	/*5994*/ uint16(x86_xSetOp), uint16(x86_FXSAVE),
	/*5996*/ uint16(x86_xArgM512byte),
	/*5997*/ uint16(x86_xMatch),
	/*5998*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6012,
	0x0, 6004,
	/*6004*/ uint16(x86_xCondDataSize), 5990, 5994, 6008,
	/*6008*/ uint16(x86_xSetOp), uint16(x86_FXSAVE64),
	/*6010*/ uint16(x86_xArgM512byte),
	/*6011*/ uint16(x86_xMatch),
	/*6012*/ uint16(x86_xCondDataSize), 6016, 6023, 6030,
	/*6016*/ uint16(x86_xCondIsMem), 6019, 0,
	/*6019*/ uint16(x86_xSetOp), uint16(x86_RDFSBASE),
	/*6021*/ uint16(x86_xArgRM32),
	/*6022*/ uint16(x86_xMatch),
	/*6023*/ uint16(x86_xCondIsMem), 6026, 0,
	/*6026*/ uint16(x86_xSetOp), uint16(x86_RDFSBASE),
	/*6028*/ uint16(x86_xArgRM32),
	/*6029*/ uint16(x86_xMatch),
	/*6030*/ uint16(x86_xCondIsMem), 6033, 0,
	/*6033*/ uint16(x86_xSetOp), uint16(x86_RDFSBASE),
	/*6035*/ uint16(x86_xArgRM64),
	/*6036*/ uint16(x86_xMatch),
	/*6037*/ uint16(x86_xCondIs64), 6040, 6052,
	/*6040*/ uint16(x86_xCondDataSize), 6044, 6048, 0,
	/*6044*/ uint16(x86_xSetOp), uint16(x86_FXRSTOR),
	/*6046*/ uint16(x86_xArgM512byte),
	/*6047*/ uint16(x86_xMatch),
	/*6048*/ uint16(x86_xSetOp), uint16(x86_FXRSTOR),
	/*6050*/ uint16(x86_xArgM512byte),
	/*6051*/ uint16(x86_xMatch),
	/*6052*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6066,
	0x0, 6058,
	/*6058*/ uint16(x86_xCondDataSize), 6044, 6048, 6062,
	/*6062*/ uint16(x86_xSetOp), uint16(x86_FXRSTOR64),
	/*6064*/ uint16(x86_xArgM512byte),
	/*6065*/ uint16(x86_xMatch),
	/*6066*/ uint16(x86_xCondDataSize), 6070, 6077, 6084,
	/*6070*/ uint16(x86_xCondIsMem), 6073, 0,
	/*6073*/ uint16(x86_xSetOp), uint16(x86_RDGSBASE),
	/*6075*/ uint16(x86_xArgRM32),
	/*6076*/ uint16(x86_xMatch),
	/*6077*/ uint16(x86_xCondIsMem), 6080, 0,
	/*6080*/ uint16(x86_xSetOp), uint16(x86_RDGSBASE),
	/*6082*/ uint16(x86_xArgRM32),
	/*6083*/ uint16(x86_xMatch),
	/*6084*/ uint16(x86_xCondIsMem), 6087, 0,
	/*6087*/ uint16(x86_xSetOp), uint16(x86_RDGSBASE),
	/*6089*/ uint16(x86_xArgRM64),
	/*6090*/ uint16(x86_xMatch),
	/*6091*/ uint16(x86_xCondIs64), 6094, 6098,
	/*6094*/ uint16(x86_xSetOp), uint16(x86_LDMXCSR),
	/*6096*/ uint16(x86_xArgM32),
	/*6097*/ uint16(x86_xMatch),
	/*6098*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6104,
	0x0, 6094,
	/*6104*/ uint16(x86_xCondDataSize), 6108, 6112, 6116,
	/*6108*/ uint16(x86_xSetOp), uint16(x86_WRFSBASE),
	/*6110*/ uint16(x86_xArgRM32),
	/*6111*/ uint16(x86_xMatch),
	/*6112*/ uint16(x86_xSetOp), uint16(x86_WRFSBASE),
	/*6114*/ uint16(x86_xArgRM32),
	/*6115*/ uint16(x86_xMatch),
	/*6116*/ uint16(x86_xSetOp), uint16(x86_WRFSBASE),
	/*6118*/ uint16(x86_xArgRM64),
	/*6119*/ uint16(x86_xMatch),
	/*6120*/ uint16(x86_xCondIs64), 6123, 6127,
	/*6123*/ uint16(x86_xSetOp), uint16(x86_STMXCSR),
	/*6125*/ uint16(x86_xArgM32),
	/*6126*/ uint16(x86_xMatch),
	/*6127*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6133,
	0x0, 6123,
	/*6133*/ uint16(x86_xCondDataSize), 6137, 6141, 6145,
	/*6137*/ uint16(x86_xSetOp), uint16(x86_WRGSBASE),
	/*6139*/ uint16(x86_xArgRM32),
	/*6140*/ uint16(x86_xMatch),
	/*6141*/ uint16(x86_xSetOp), uint16(x86_WRGSBASE),
	/*6143*/ uint16(x86_xArgRM32),
	/*6144*/ uint16(x86_xMatch),
	/*6145*/ uint16(x86_xSetOp), uint16(x86_WRGSBASE),
	/*6147*/ uint16(x86_xArgRM64),
	/*6148*/ uint16(x86_xMatch),
	/*6149*/ uint16(x86_xCondIs64), 6152, 6164,
	/*6152*/ uint16(x86_xCondDataSize), 6156, 6160, 0,
	/*6156*/ uint16(x86_xSetOp), uint16(x86_XSAVE),
	/*6158*/ uint16(x86_xArgMem),
	/*6159*/ uint16(x86_xMatch),
	/*6160*/ uint16(x86_xSetOp), uint16(x86_XSAVE),
	/*6162*/ uint16(x86_xArgMem),
	/*6163*/ uint16(x86_xMatch),
	/*6164*/ uint16(x86_xCondDataSize), 6156, 6160, 6168,
	/*6168*/ uint16(x86_xSetOp), uint16(x86_XSAVE64),
	/*6170*/ uint16(x86_xArgMem),
	/*6171*/ uint16(x86_xMatch),
	/*6172*/ uint16(x86_xCondIs64), 6175, 6187,
	/*6175*/ uint16(x86_xCondDataSize), 6179, 6183, 0,
	/*6179*/ uint16(x86_xSetOp), uint16(x86_XRSTOR),
	/*6181*/ uint16(x86_xArgMem),
	/*6182*/ uint16(x86_xMatch),
	/*6183*/ uint16(x86_xSetOp), uint16(x86_XRSTOR),
	/*6185*/ uint16(x86_xArgMem),
	/*6186*/ uint16(x86_xMatch),
	/*6187*/ uint16(x86_xCondDataSize), 6179, 6183, 6191,
	/*6191*/ uint16(x86_xSetOp), uint16(x86_XRSTOR64),
	/*6193*/ uint16(x86_xArgMem),
	/*6194*/ uint16(x86_xMatch),
	/*6195*/ uint16(x86_xCondDataSize), 6199, 6203, 6207,
	/*6199*/ uint16(x86_xSetOp), uint16(x86_XSAVEOPT),
	/*6201*/ uint16(x86_xArgMem),
	/*6202*/ uint16(x86_xMatch),
	/*6203*/ uint16(x86_xSetOp), uint16(x86_XSAVEOPT),
	/*6205*/ uint16(x86_xArgMem),
	/*6206*/ uint16(x86_xMatch),
	/*6207*/ uint16(x86_xSetOp), uint16(x86_XSAVEOPT64),
	/*6209*/ uint16(x86_xArgMem),
	/*6210*/ uint16(x86_xMatch),
	/*6211*/ uint16(x86_xSetOp), uint16(x86_CLFLUSH),
	/*6213*/ uint16(x86_xArgM8),
	/*6214*/ uint16(x86_xMatch),
	/*6215*/ uint16(x86_xSetOp), uint16(x86_LFENCE),
	/*6217*/ uint16(x86_xMatch),
	/*6218*/ uint16(x86_xSetOp), uint16(x86_MFENCE),
	/*6220*/ uint16(x86_xMatch),
	/*6221*/ uint16(x86_xSetOp), uint16(x86_SFENCE),
	/*6223*/ uint16(x86_xMatch),
	/*6224*/ uint16(x86_xCondIs64), 6227, 6243,
	/*6227*/ uint16(x86_xCondDataSize), 6231, 6237, 0,
	/*6231*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*6233*/ uint16(x86_xReadSlashR),
	/*6234*/ uint16(x86_xArgR16),
	/*6235*/ uint16(x86_xArgRM16),
	/*6236*/ uint16(x86_xMatch),
	/*6237*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*6239*/ uint16(x86_xReadSlashR),
	/*6240*/ uint16(x86_xArgR32),
	/*6241*/ uint16(x86_xArgRM32),
	/*6242*/ uint16(x86_xMatch),
	/*6243*/ uint16(x86_xCondDataSize), 6231, 6237, 6247,
	/*6247*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*6249*/ uint16(x86_xReadSlashR),
	/*6250*/ uint16(x86_xArgR64),
	/*6251*/ uint16(x86_xArgRM64),
	/*6252*/ uint16(x86_xMatch),
	/*6253*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG),
	/*6255*/ uint16(x86_xReadSlashR),
	/*6256*/ uint16(x86_xArgRM8),
	/*6257*/ uint16(x86_xArgR8),
	/*6258*/ uint16(x86_xMatch),
	/*6259*/ uint16(x86_xCondIs64), 6262, 6278,
	/*6262*/ uint16(x86_xCondDataSize), 6266, 6272, 0,
	/*6266*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG),
	/*6268*/ uint16(x86_xReadSlashR),
	/*6269*/ uint16(x86_xArgRM16),
	/*6270*/ uint16(x86_xArgR16),
	/*6271*/ uint16(x86_xMatch),
	/*6272*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG),
	/*6274*/ uint16(x86_xReadSlashR),
	/*6275*/ uint16(x86_xArgRM32),
	/*6276*/ uint16(x86_xArgR32),
	/*6277*/ uint16(x86_xMatch),
	/*6278*/ uint16(x86_xCondDataSize), 6266, 6272, 6282,
	/*6282*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG),
	/*6284*/ uint16(x86_xReadSlashR),
	/*6285*/ uint16(x86_xArgRM64),
	/*6286*/ uint16(x86_xArgR64),
	/*6287*/ uint16(x86_xMatch),
	/*6288*/ uint16(x86_xCondIs64), 6291, 6307,
	/*6291*/ uint16(x86_xCondDataSize), 6295, 6301, 0,
	/*6295*/ uint16(x86_xSetOp), uint16(x86_LSS),
	/*6297*/ uint16(x86_xReadSlashR),
	/*6298*/ uint16(x86_xArgR16),
	/*6299*/ uint16(x86_xArgM16colon16),
	/*6300*/ uint16(x86_xMatch),
	/*6301*/ uint16(x86_xSetOp), uint16(x86_LSS),
	/*6303*/ uint16(x86_xReadSlashR),
	/*6304*/ uint16(x86_xArgR32),
	/*6305*/ uint16(x86_xArgM16colon32),
	/*6306*/ uint16(x86_xMatch),
	/*6307*/ uint16(x86_xCondDataSize), 6295, 6301, 6311,
	/*6311*/ uint16(x86_xSetOp), uint16(x86_LSS),
	/*6313*/ uint16(x86_xReadSlashR),
	/*6314*/ uint16(x86_xArgR64),
	/*6315*/ uint16(x86_xArgM16colon64),
	/*6316*/ uint16(x86_xMatch),
	/*6317*/ uint16(x86_xCondIs64), 6320, 6336,
	/*6320*/ uint16(x86_xCondDataSize), 6324, 6330, 0,
	/*6324*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6326*/ uint16(x86_xReadSlashR),
	/*6327*/ uint16(x86_xArgRM16),
	/*6328*/ uint16(x86_xArgR16),
	/*6329*/ uint16(x86_xMatch),
	/*6330*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6332*/ uint16(x86_xReadSlashR),
	/*6333*/ uint16(x86_xArgRM32),
	/*6334*/ uint16(x86_xArgR32),
	/*6335*/ uint16(x86_xMatch),
	/*6336*/ uint16(x86_xCondDataSize), 6324, 6330, 6340,
	/*6340*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6342*/ uint16(x86_xReadSlashR),
	/*6343*/ uint16(x86_xArgRM64),
	/*6344*/ uint16(x86_xArgR64),
	/*6345*/ uint16(x86_xMatch),
	/*6346*/ uint16(x86_xCondIs64), 6349, 6365,
	/*6349*/ uint16(x86_xCondDataSize), 6353, 6359, 0,
	/*6353*/ uint16(x86_xSetOp), uint16(x86_LFS),
	/*6355*/ uint16(x86_xReadSlashR),
	/*6356*/ uint16(x86_xArgR16),
	/*6357*/ uint16(x86_xArgM16colon16),
	/*6358*/ uint16(x86_xMatch),
	/*6359*/ uint16(x86_xSetOp), uint16(x86_LFS),
	/*6361*/ uint16(x86_xReadSlashR),
	/*6362*/ uint16(x86_xArgR32),
	/*6363*/ uint16(x86_xArgM16colon32),
	/*6364*/ uint16(x86_xMatch),
	/*6365*/ uint16(x86_xCondDataSize), 6353, 6359, 6369,
	/*6369*/ uint16(x86_xSetOp), uint16(x86_LFS),
	/*6371*/ uint16(x86_xReadSlashR),
	/*6372*/ uint16(x86_xArgR64),
	/*6373*/ uint16(x86_xArgM16colon64),
	/*6374*/ uint16(x86_xMatch),
	/*6375*/ uint16(x86_xCondIs64), 6378, 6394,
	/*6378*/ uint16(x86_xCondDataSize), 6382, 6388, 0,
	/*6382*/ uint16(x86_xSetOp), uint16(x86_LGS),
	/*6384*/ uint16(x86_xReadSlashR),
	/*6385*/ uint16(x86_xArgR16),
	/*6386*/ uint16(x86_xArgM16colon16),
	/*6387*/ uint16(x86_xMatch),
	/*6388*/ uint16(x86_xSetOp), uint16(x86_LGS),
	/*6390*/ uint16(x86_xReadSlashR),
	/*6391*/ uint16(x86_xArgR32),
	/*6392*/ uint16(x86_xArgM16colon32),
	/*6393*/ uint16(x86_xMatch),
	/*6394*/ uint16(x86_xCondDataSize), 6382, 6388, 6398,
	/*6398*/ uint16(x86_xSetOp), uint16(x86_LGS),
	/*6400*/ uint16(x86_xReadSlashR),
	/*6401*/ uint16(x86_xArgR64),
	/*6402*/ uint16(x86_xArgM16colon64),
	/*6403*/ uint16(x86_xMatch),
	/*6404*/ uint16(x86_xCondIs64), 6407, 6423,
	/*6407*/ uint16(x86_xCondDataSize), 6411, 6417, 0,
	/*6411*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6413*/ uint16(x86_xReadSlashR),
	/*6414*/ uint16(x86_xArgR16),
	/*6415*/ uint16(x86_xArgRM8),
	/*6416*/ uint16(x86_xMatch),
	/*6417*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6419*/ uint16(x86_xReadSlashR),
	/*6420*/ uint16(x86_xArgR32),
	/*6421*/ uint16(x86_xArgRM8),
	/*6422*/ uint16(x86_xMatch),
	/*6423*/ uint16(x86_xCondDataSize), 6411, 6417, 6427,
	/*6427*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6429*/ uint16(x86_xReadSlashR),
	/*6430*/ uint16(x86_xArgR64),
	/*6431*/ uint16(x86_xArgRM8),
	/*6432*/ uint16(x86_xMatch),
	/*6433*/ uint16(x86_xCondIs64), 6436, 6452,
	/*6436*/ uint16(x86_xCondDataSize), 6440, 6446, 0,
	/*6440*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6442*/ uint16(x86_xReadSlashR),
	/*6443*/ uint16(x86_xArgR16),
	/*6444*/ uint16(x86_xArgRM16),
	/*6445*/ uint16(x86_xMatch),
	/*6446*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6448*/ uint16(x86_xReadSlashR),
	/*6449*/ uint16(x86_xArgR32),
	/*6450*/ uint16(x86_xArgRM16),
	/*6451*/ uint16(x86_xMatch),
	/*6452*/ uint16(x86_xCondDataSize), 6440, 6446, 6456,
	/*6456*/ uint16(x86_xSetOp), uint16(x86_MOVZX),
	/*6458*/ uint16(x86_xReadSlashR),
	/*6459*/ uint16(x86_xArgR64),
	/*6460*/ uint16(x86_xArgRM16),
	/*6461*/ uint16(x86_xMatch),
	/*6462*/ uint16(x86_xCondIs64), 6465, 6485,
	/*6465*/ uint16(x86_xCondPrefix), 1,
	0xF3, 6469,
	/*6469*/ uint16(x86_xCondDataSize), 6473, 6479, 0,
	/*6473*/ uint16(x86_xSetOp), uint16(x86_POPCNT),
	/*6475*/ uint16(x86_xReadSlashR),
	/*6476*/ uint16(x86_xArgR16),
	/*6477*/ uint16(x86_xArgRM16),
	/*6478*/ uint16(x86_xMatch),
	/*6479*/ uint16(x86_xSetOp), uint16(x86_POPCNT),
	/*6481*/ uint16(x86_xReadSlashR),
	/*6482*/ uint16(x86_xArgR32),
	/*6483*/ uint16(x86_xArgRM32),
	/*6484*/ uint16(x86_xMatch),
	/*6485*/ uint16(x86_xCondPrefix), 1,
	0xF3, 6489,
	/*6489*/ uint16(x86_xCondDataSize), 6473, 6479, 6493,
	/*6493*/ uint16(x86_xSetOp), uint16(x86_POPCNT),
	/*6495*/ uint16(x86_xReadSlashR),
	/*6496*/ uint16(x86_xArgR64),
	/*6497*/ uint16(x86_xArgRM64),
	/*6498*/ uint16(x86_xMatch),
	/*6499*/ uint16(x86_xSetOp), uint16(x86_UD1),
	/*6501*/ uint16(x86_xMatch),
	/*6502*/ uint16(x86_xCondSlashR),
	0,    // 0
	0,    // 1
	0,    // 2
	0,    // 3
	6511, // 4
	6540, // 5
	6569, // 6
	6598, // 7
	/*6511*/ uint16(x86_xCondIs64), 6514, 6530,
	/*6514*/ uint16(x86_xCondDataSize), 6518, 6524, 0,
	/*6518*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*6520*/ uint16(x86_xReadIb),
	/*6521*/ uint16(x86_xArgRM16),
	/*6522*/ uint16(x86_xArgImm8u),
	/*6523*/ uint16(x86_xMatch),
	/*6524*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*6526*/ uint16(x86_xReadIb),
	/*6527*/ uint16(x86_xArgRM32),
	/*6528*/ uint16(x86_xArgImm8u),
	/*6529*/ uint16(x86_xMatch),
	/*6530*/ uint16(x86_xCondDataSize), 6518, 6524, 6534,
	/*6534*/ uint16(x86_xSetOp), uint16(x86_BT),
	/*6536*/ uint16(x86_xReadIb),
	/*6537*/ uint16(x86_xArgRM64),
	/*6538*/ uint16(x86_xArgImm8u),
	/*6539*/ uint16(x86_xMatch),
	/*6540*/ uint16(x86_xCondIs64), 6543, 6559,
	/*6543*/ uint16(x86_xCondDataSize), 6547, 6553, 0,
	/*6547*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*6549*/ uint16(x86_xReadIb),
	/*6550*/ uint16(x86_xArgRM16),
	/*6551*/ uint16(x86_xArgImm8u),
	/*6552*/ uint16(x86_xMatch),
	/*6553*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*6555*/ uint16(x86_xReadIb),
	/*6556*/ uint16(x86_xArgRM32),
	/*6557*/ uint16(x86_xArgImm8u),
	/*6558*/ uint16(x86_xMatch),
	/*6559*/ uint16(x86_xCondDataSize), 6547, 6553, 6563,
	/*6563*/ uint16(x86_xSetOp), uint16(x86_BTS),
	/*6565*/ uint16(x86_xReadIb),
	/*6566*/ uint16(x86_xArgRM64),
	/*6567*/ uint16(x86_xArgImm8u),
	/*6568*/ uint16(x86_xMatch),
	/*6569*/ uint16(x86_xCondIs64), 6572, 6588,
	/*6572*/ uint16(x86_xCondDataSize), 6576, 6582, 0,
	/*6576*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6578*/ uint16(x86_xReadIb),
	/*6579*/ uint16(x86_xArgRM16),
	/*6580*/ uint16(x86_xArgImm8u),
	/*6581*/ uint16(x86_xMatch),
	/*6582*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6584*/ uint16(x86_xReadIb),
	/*6585*/ uint16(x86_xArgRM32),
	/*6586*/ uint16(x86_xArgImm8u),
	/*6587*/ uint16(x86_xMatch),
	/*6588*/ uint16(x86_xCondDataSize), 6576, 6582, 6592,
	/*6592*/ uint16(x86_xSetOp), uint16(x86_BTR),
	/*6594*/ uint16(x86_xReadIb),
	/*6595*/ uint16(x86_xArgRM64),
	/*6596*/ uint16(x86_xArgImm8u),
	/*6597*/ uint16(x86_xMatch),
	/*6598*/ uint16(x86_xCondIs64), 6601, 6617,
	/*6601*/ uint16(x86_xCondDataSize), 6605, 6611, 0,
	/*6605*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6607*/ uint16(x86_xReadIb),
	/*6608*/ uint16(x86_xArgRM16),
	/*6609*/ uint16(x86_xArgImm8u),
	/*6610*/ uint16(x86_xMatch),
	/*6611*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6613*/ uint16(x86_xReadIb),
	/*6614*/ uint16(x86_xArgRM32),
	/*6615*/ uint16(x86_xArgImm8u),
	/*6616*/ uint16(x86_xMatch),
	/*6617*/ uint16(x86_xCondDataSize), 6605, 6611, 6621,
	/*6621*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6623*/ uint16(x86_xReadIb),
	/*6624*/ uint16(x86_xArgRM64),
	/*6625*/ uint16(x86_xArgImm8u),
	/*6626*/ uint16(x86_xMatch),
	/*6627*/ uint16(x86_xCondIs64), 6630, 6646,
	/*6630*/ uint16(x86_xCondDataSize), 6634, 6640, 0,
	/*6634*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6636*/ uint16(x86_xReadSlashR),
	/*6637*/ uint16(x86_xArgRM16),
	/*6638*/ uint16(x86_xArgR16),
	/*6639*/ uint16(x86_xMatch),
	/*6640*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6642*/ uint16(x86_xReadSlashR),
	/*6643*/ uint16(x86_xArgRM32),
	/*6644*/ uint16(x86_xArgR32),
	/*6645*/ uint16(x86_xMatch),
	/*6646*/ uint16(x86_xCondDataSize), 6634, 6640, 6650,
	/*6650*/ uint16(x86_xSetOp), uint16(x86_BTC),
	/*6652*/ uint16(x86_xReadSlashR),
	/*6653*/ uint16(x86_xArgRM64),
	/*6654*/ uint16(x86_xArgR64),
	/*6655*/ uint16(x86_xMatch),
	/*6656*/ uint16(x86_xCondIs64), 6659, 6697,
	/*6659*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6681,
	0x0, 6665,
	/*6665*/ uint16(x86_xCondDataSize), 6669, 6675, 0,
	/*6669*/ uint16(x86_xSetOp), uint16(x86_BSF),
	/*6671*/ uint16(x86_xReadSlashR),
	/*6672*/ uint16(x86_xArgR16),
	/*6673*/ uint16(x86_xArgRM16),
	/*6674*/ uint16(x86_xMatch),
	/*6675*/ uint16(x86_xSetOp), uint16(x86_BSF),
	/*6677*/ uint16(x86_xReadSlashR),
	/*6678*/ uint16(x86_xArgR32),
	/*6679*/ uint16(x86_xArgRM32),
	/*6680*/ uint16(x86_xMatch),
	/*6681*/ uint16(x86_xCondDataSize), 6685, 6691, 0,
	/*6685*/ uint16(x86_xSetOp), uint16(x86_TZCNT),
	/*6687*/ uint16(x86_xReadSlashR),
	/*6688*/ uint16(x86_xArgR16),
	/*6689*/ uint16(x86_xArgRM16),
	/*6690*/ uint16(x86_xMatch),
	/*6691*/ uint16(x86_xSetOp), uint16(x86_TZCNT),
	/*6693*/ uint16(x86_xReadSlashR),
	/*6694*/ uint16(x86_xArgR32),
	/*6695*/ uint16(x86_xArgRM32),
	/*6696*/ uint16(x86_xMatch),
	/*6697*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6713,
	0x0, 6703,
	/*6703*/ uint16(x86_xCondDataSize), 6669, 6675, 6707,
	/*6707*/ uint16(x86_xSetOp), uint16(x86_BSF),
	/*6709*/ uint16(x86_xReadSlashR),
	/*6710*/ uint16(x86_xArgR64),
	/*6711*/ uint16(x86_xArgRM64),
	/*6712*/ uint16(x86_xMatch),
	/*6713*/ uint16(x86_xCondDataSize), 6685, 6691, 6717,
	/*6717*/ uint16(x86_xSetOp), uint16(x86_TZCNT),
	/*6719*/ uint16(x86_xReadSlashR),
	/*6720*/ uint16(x86_xArgR64),
	/*6721*/ uint16(x86_xArgRM64),
	/*6722*/ uint16(x86_xMatch),
	/*6723*/ uint16(x86_xCondIs64), 6726, 6764,
	/*6726*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6748,
	0x0, 6732,
	/*6732*/ uint16(x86_xCondDataSize), 6736, 6742, 0,
	/*6736*/ uint16(x86_xSetOp), uint16(x86_BSR),
	/*6738*/ uint16(x86_xReadSlashR),
	/*6739*/ uint16(x86_xArgR16),
	/*6740*/ uint16(x86_xArgRM16),
	/*6741*/ uint16(x86_xMatch),
	/*6742*/ uint16(x86_xSetOp), uint16(x86_BSR),
	/*6744*/ uint16(x86_xReadSlashR),
	/*6745*/ uint16(x86_xArgR32),
	/*6746*/ uint16(x86_xArgRM32),
	/*6747*/ uint16(x86_xMatch),
	/*6748*/ uint16(x86_xCondDataSize), 6752, 6758, 0,
	/*6752*/ uint16(x86_xSetOp), uint16(x86_LZCNT),
	/*6754*/ uint16(x86_xReadSlashR),
	/*6755*/ uint16(x86_xArgR16),
	/*6756*/ uint16(x86_xArgRM16),
	/*6757*/ uint16(x86_xMatch),
	/*6758*/ uint16(x86_xSetOp), uint16(x86_LZCNT),
	/*6760*/ uint16(x86_xReadSlashR),
	/*6761*/ uint16(x86_xArgR32),
	/*6762*/ uint16(x86_xArgRM32),
	/*6763*/ uint16(x86_xMatch),
	/*6764*/ uint16(x86_xCondPrefix), 2,
	0xF3, 6780,
	0x0, 6770,
	/*6770*/ uint16(x86_xCondDataSize), 6736, 6742, 6774,
	/*6774*/ uint16(x86_xSetOp), uint16(x86_BSR),
	/*6776*/ uint16(x86_xReadSlashR),
	/*6777*/ uint16(x86_xArgR64),
	/*6778*/ uint16(x86_xArgRM64),
	/*6779*/ uint16(x86_xMatch),
	/*6780*/ uint16(x86_xCondDataSize), 6752, 6758, 6784,
	/*6784*/ uint16(x86_xSetOp), uint16(x86_LZCNT),
	/*6786*/ uint16(x86_xReadSlashR),
	/*6787*/ uint16(x86_xArgR64),
	/*6788*/ uint16(x86_xArgRM64),
	/*6789*/ uint16(x86_xMatch),
	/*6790*/ uint16(x86_xCondIs64), 6793, 6809,
	/*6793*/ uint16(x86_xCondDataSize), 6797, 6803, 0,
	/*6797*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6799*/ uint16(x86_xReadSlashR),
	/*6800*/ uint16(x86_xArgR16),
	/*6801*/ uint16(x86_xArgRM8),
	/*6802*/ uint16(x86_xMatch),
	/*6803*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6805*/ uint16(x86_xReadSlashR),
	/*6806*/ uint16(x86_xArgR32),
	/*6807*/ uint16(x86_xArgRM8),
	/*6808*/ uint16(x86_xMatch),
	/*6809*/ uint16(x86_xCondDataSize), 6797, 6803, 6813,
	/*6813*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6815*/ uint16(x86_xReadSlashR),
	/*6816*/ uint16(x86_xArgR64),
	/*6817*/ uint16(x86_xArgRM8),
	/*6818*/ uint16(x86_xMatch),
	/*6819*/ uint16(x86_xCondIs64), 6822, 6838,
	/*6822*/ uint16(x86_xCondDataSize), 6826, 6832, 0,
	/*6826*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6828*/ uint16(x86_xReadSlashR),
	/*6829*/ uint16(x86_xArgR16),
	/*6830*/ uint16(x86_xArgRM16),
	/*6831*/ uint16(x86_xMatch),
	/*6832*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6834*/ uint16(x86_xReadSlashR),
	/*6835*/ uint16(x86_xArgR32),
	/*6836*/ uint16(x86_xArgRM16),
	/*6837*/ uint16(x86_xMatch),
	/*6838*/ uint16(x86_xCondDataSize), 6826, 6832, 6842,
	/*6842*/ uint16(x86_xSetOp), uint16(x86_MOVSX),
	/*6844*/ uint16(x86_xReadSlashR),
	/*6845*/ uint16(x86_xArgR64),
	/*6846*/ uint16(x86_xArgRM16),
	/*6847*/ uint16(x86_xMatch),
	/*6848*/ uint16(x86_xSetOp), uint16(x86_XADD),
	/*6850*/ uint16(x86_xReadSlashR),
	/*6851*/ uint16(x86_xArgRM8),
	/*6852*/ uint16(x86_xArgR8),
	/*6853*/ uint16(x86_xMatch),
	/*6854*/ uint16(x86_xCondIs64), 6857, 6873,
	/*6857*/ uint16(x86_xCondDataSize), 6861, 6867, 0,
	/*6861*/ uint16(x86_xSetOp), uint16(x86_XADD),
	/*6863*/ uint16(x86_xReadSlashR),
	/*6864*/ uint16(x86_xArgRM16),
	/*6865*/ uint16(x86_xArgR16),
	/*6866*/ uint16(x86_xMatch),
	/*6867*/ uint16(x86_xSetOp), uint16(x86_XADD),
	/*6869*/ uint16(x86_xReadSlashR),
	/*6870*/ uint16(x86_xArgRM32),
	/*6871*/ uint16(x86_xArgR32),
	/*6872*/ uint16(x86_xMatch),
	/*6873*/ uint16(x86_xCondDataSize), 6861, 6867, 6877,
	/*6877*/ uint16(x86_xSetOp), uint16(x86_XADD),
	/*6879*/ uint16(x86_xReadSlashR),
	/*6880*/ uint16(x86_xArgRM64),
	/*6881*/ uint16(x86_xArgR64),
	/*6882*/ uint16(x86_xMatch),
	/*6883*/ uint16(x86_xCondPrefix), 4,
	0xF3, 6917,
	0xF2, 6909,
	0x66, 6901,
	0x0, 6893,
	/*6893*/ uint16(x86_xSetOp), uint16(x86_CMPPS),
	/*6895*/ uint16(x86_xReadSlashR),
	/*6896*/ uint16(x86_xReadIb),
	/*6897*/ uint16(x86_xArgXmm1),
	/*6898*/ uint16(x86_xArgXmm2M128),
	/*6899*/ uint16(x86_xArgImm8u),
	/*6900*/ uint16(x86_xMatch),
	/*6901*/ uint16(x86_xSetOp), uint16(x86_CMPPD),
	/*6903*/ uint16(x86_xReadSlashR),
	/*6904*/ uint16(x86_xReadIb),
	/*6905*/ uint16(x86_xArgXmm1),
	/*6906*/ uint16(x86_xArgXmm2M128),
	/*6907*/ uint16(x86_xArgImm8u),
	/*6908*/ uint16(x86_xMatch),
	/*6909*/ uint16(x86_xSetOp), uint16(x86_CMPSD_XMM),
	/*6911*/ uint16(x86_xReadSlashR),
	/*6912*/ uint16(x86_xReadIb),
	/*6913*/ uint16(x86_xArgXmm1),
	/*6914*/ uint16(x86_xArgXmm2M64),
	/*6915*/ uint16(x86_xArgImm8u),
	/*6916*/ uint16(x86_xMatch),
	/*6917*/ uint16(x86_xSetOp), uint16(x86_CMPSS),
	/*6919*/ uint16(x86_xReadSlashR),
	/*6920*/ uint16(x86_xReadIb),
	/*6921*/ uint16(x86_xArgXmm1),
	/*6922*/ uint16(x86_xArgXmm2M32),
	/*6923*/ uint16(x86_xArgImm8u),
	/*6924*/ uint16(x86_xMatch),
	/*6925*/ uint16(x86_xCondIs64), 6928, 6944,
	/*6928*/ uint16(x86_xCondDataSize), 6932, 6938, 0,
	/*6932*/ uint16(x86_xSetOp), uint16(x86_MOVNTI),
	/*6934*/ uint16(x86_xReadSlashR),
	/*6935*/ uint16(x86_xArgM32),
	/*6936*/ uint16(x86_xArgR32),
	/*6937*/ uint16(x86_xMatch),
	/*6938*/ uint16(x86_xSetOp), uint16(x86_MOVNTI),
	/*6940*/ uint16(x86_xReadSlashR),
	/*6941*/ uint16(x86_xArgM32),
	/*6942*/ uint16(x86_xArgR32),
	/*6943*/ uint16(x86_xMatch),
	/*6944*/ uint16(x86_xCondDataSize), 6932, 6938, 6948,
	/*6948*/ uint16(x86_xSetOp), uint16(x86_MOVNTI),
	/*6950*/ uint16(x86_xReadSlashR),
	/*6951*/ uint16(x86_xArgM64),
	/*6952*/ uint16(x86_xArgR64),
	/*6953*/ uint16(x86_xMatch),
	/*6954*/ uint16(x86_xCondPrefix), 2,
	0x66, 6968,
	0x0, 6960,
	/*6960*/ uint16(x86_xSetOp), uint16(x86_PINSRW),
	/*6962*/ uint16(x86_xReadSlashR),
	/*6963*/ uint16(x86_xReadIb),
	/*6964*/ uint16(x86_xArgMm),
	/*6965*/ uint16(x86_xArgR32M16),
	/*6966*/ uint16(x86_xArgImm8u),
	/*6967*/ uint16(x86_xMatch),
	/*6968*/ uint16(x86_xSetOp), uint16(x86_PINSRW),
	/*6970*/ uint16(x86_xReadSlashR),
	/*6971*/ uint16(x86_xReadIb),
	/*6972*/ uint16(x86_xArgXmm),
	/*6973*/ uint16(x86_xArgR32M16),
	/*6974*/ uint16(x86_xArgImm8u),
	/*6975*/ uint16(x86_xMatch),
	/*6976*/ uint16(x86_xCondPrefix), 2,
	0x66, 6990,
	0x0, 6982,
	/*6982*/ uint16(x86_xSetOp), uint16(x86_PEXTRW),
	/*6984*/ uint16(x86_xReadSlashR),
	/*6985*/ uint16(x86_xReadIb),
	/*6986*/ uint16(x86_xArgR32),
	/*6987*/ uint16(x86_xArgMm2),
	/*6988*/ uint16(x86_xArgImm8u),
	/*6989*/ uint16(x86_xMatch),
	/*6990*/ uint16(x86_xSetOp), uint16(x86_PEXTRW),
	/*6992*/ uint16(x86_xReadSlashR),
	/*6993*/ uint16(x86_xReadIb),
	/*6994*/ uint16(x86_xArgR32),
	/*6995*/ uint16(x86_xArgXmm2),
	/*6996*/ uint16(x86_xArgImm8u),
	/*6997*/ uint16(x86_xMatch),
	/*6998*/ uint16(x86_xCondPrefix), 2,
	0x66, 7012,
	0x0, 7004,
	/*7004*/ uint16(x86_xSetOp), uint16(x86_SHUFPS),
	/*7006*/ uint16(x86_xReadSlashR),
	/*7007*/ uint16(x86_xReadIb),
	/*7008*/ uint16(x86_xArgXmm1),
	/*7009*/ uint16(x86_xArgXmm2M128),
	/*7010*/ uint16(x86_xArgImm8u),
	/*7011*/ uint16(x86_xMatch),
	/*7012*/ uint16(x86_xSetOp), uint16(x86_SHUFPD),
	/*7014*/ uint16(x86_xReadSlashR),
	/*7015*/ uint16(x86_xReadIb),
	/*7016*/ uint16(x86_xArgXmm1),
	/*7017*/ uint16(x86_xArgXmm2M128),
	/*7018*/ uint16(x86_xArgImm8u),
	/*7019*/ uint16(x86_xMatch),
	/*7020*/ uint16(x86_xCondSlashR),
	0,    // 0
	7029, // 1
	0,    // 2
	7052, // 3
	7075, // 4
	7098, // 5
	7121, // 6
	0,    // 7
	/*7029*/ uint16(x86_xCondIs64), 7032, 7044,
	/*7032*/ uint16(x86_xCondDataSize), 7036, 7040, 0,
	/*7036*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG8B),
	/*7038*/ uint16(x86_xArgM64),
	/*7039*/ uint16(x86_xMatch),
	/*7040*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG8B),
	/*7042*/ uint16(x86_xArgM64),
	/*7043*/ uint16(x86_xMatch),
	/*7044*/ uint16(x86_xCondDataSize), 7036, 7040, 7048,
	/*7048*/ uint16(x86_xSetOp), uint16(x86_CMPXCHG16B),
	/*7050*/ uint16(x86_xArgM128),
	/*7051*/ uint16(x86_xMatch),
	/*7052*/ uint16(x86_xCondIs64), 7055, 7067,
	/*7055*/ uint16(x86_xCondDataSize), 7059, 7063, 0,
	/*7059*/ uint16(x86_xSetOp), uint16(x86_XRSTORS),
	/*7061*/ uint16(x86_xArgMem),
	/*7062*/ uint16(x86_xMatch),
	/*7063*/ uint16(x86_xSetOp), uint16(x86_XRSTORS),
	/*7065*/ uint16(x86_xArgMem),
	/*7066*/ uint16(x86_xMatch),
	/*7067*/ uint16(x86_xCondDataSize), 7059, 7063, 7071,
	/*7071*/ uint16(x86_xSetOp), uint16(x86_XRSTORS64),
	/*7073*/ uint16(x86_xArgMem),
	/*7074*/ uint16(x86_xMatch),
	/*7075*/ uint16(x86_xCondIs64), 7078, 7090,
	/*7078*/ uint16(x86_xCondDataSize), 7082, 7086, 0,
	/*7082*/ uint16(x86_xSetOp), uint16(x86_XSAVEC),
	/*7084*/ uint16(x86_xArgMem),
	/*7085*/ uint16(x86_xMatch),
	/*7086*/ uint16(x86_xSetOp), uint16(x86_XSAVEC),
	/*7088*/ uint16(x86_xArgMem),
	/*7089*/ uint16(x86_xMatch),
	/*7090*/ uint16(x86_xCondDataSize), 7082, 7086, 7094,
	/*7094*/ uint16(x86_xSetOp), uint16(x86_XSAVEC64),
	/*7096*/ uint16(x86_xArgMem),
	/*7097*/ uint16(x86_xMatch),
	/*7098*/ uint16(x86_xCondIs64), 7101, 7113,
	/*7101*/ uint16(x86_xCondDataSize), 7105, 7109, 0,
	/*7105*/ uint16(x86_xSetOp), uint16(x86_XSAVES),
	/*7107*/ uint16(x86_xArgMem),
	/*7108*/ uint16(x86_xMatch),
	/*7109*/ uint16(x86_xSetOp), uint16(x86_XSAVES),
	/*7111*/ uint16(x86_xArgMem),
	/*7112*/ uint16(x86_xMatch),
	/*7113*/ uint16(x86_xCondDataSize), 7105, 7109, 7117,
	/*7117*/ uint16(x86_xSetOp), uint16(x86_XSAVES64),
	/*7119*/ uint16(x86_xArgMem),
	/*7120*/ uint16(x86_xMatch),
	/*7121*/ uint16(x86_xCondIs64), 7124, 7142,
	/*7124*/ uint16(x86_xCondDataSize), 7128, 7135, 0,
	/*7128*/ uint16(x86_xCondIsMem), 7131, 0,
	/*7131*/ uint16(x86_xSetOp), uint16(x86_RDRAND),
	/*7133*/ uint16(x86_xArgRmf16),
	/*7134*/ uint16(x86_xMatch),
	/*7135*/ uint16(x86_xCondIsMem), 7138, 0,
	/*7138*/ uint16(x86_xSetOp), uint16(x86_RDRAND),
	/*7140*/ uint16(x86_xArgRmf32),
	/*7141*/ uint16(x86_xMatch),
	/*7142*/ uint16(x86_xCondDataSize), 7128, 7135, 7146,
	/*7146*/ uint16(x86_xSetOp), uint16(x86_RDRAND),
	/*7148*/ uint16(x86_xMatch),
	/*7149*/ uint16(x86_xCondIs64), 7152, 7164,
	/*7152*/ uint16(x86_xCondDataSize), 7156, 7160, 0,
	/*7156*/ uint16(x86_xSetOp), uint16(x86_BSWAP),
	/*7158*/ uint16(x86_xArgR16op),
	/*7159*/ uint16(x86_xMatch),
	/*7160*/ uint16(x86_xSetOp), uint16(x86_BSWAP),
	/*7162*/ uint16(x86_xArgR32op),
	/*7163*/ uint16(x86_xMatch),
	/*7164*/ uint16(x86_xCondDataSize), 7156, 7160, 7168,
	/*7168*/ uint16(x86_xSetOp), uint16(x86_BSWAP),
	/*7170*/ uint16(x86_xArgR64op),
	/*7171*/ uint16(x86_xMatch),
	/*7172*/ uint16(x86_xCondPrefix), 2,
	0xF2, 7184,
	0x66, 7178,
	/*7178*/ uint16(x86_xSetOp), uint16(x86_ADDSUBPD),
	/*7180*/ uint16(x86_xReadSlashR),
	/*7181*/ uint16(x86_xArgXmm1),
	/*7182*/ uint16(x86_xArgXmm2M128),
	/*7183*/ uint16(x86_xMatch),
	/*7184*/ uint16(x86_xSetOp), uint16(x86_ADDSUBPS),
	/*7186*/ uint16(x86_xReadSlashR),
	/*7187*/ uint16(x86_xArgXmm1),
	/*7188*/ uint16(x86_xArgXmm2M128),
	/*7189*/ uint16(x86_xMatch),
	/*7190*/ uint16(x86_xCondPrefix), 2,
	0x66, 7202,
	0x0, 7196,
	/*7196*/ uint16(x86_xSetOp), uint16(x86_PSRLW),
	/*7198*/ uint16(x86_xReadSlashR),
	/*7199*/ uint16(x86_xArgMm),
	/*7200*/ uint16(x86_xArgMmM64),
	/*7201*/ uint16(x86_xMatch),
	/*7202*/ uint16(x86_xSetOp), uint16(x86_PSRLW),
	/*7204*/ uint16(x86_xReadSlashR),
	/*7205*/ uint16(x86_xArgXmm1),
	/*7206*/ uint16(x86_xArgXmm2M128),
	/*7207*/ uint16(x86_xMatch),
	/*7208*/ uint16(x86_xCondPrefix), 2,
	0x66, 7220,
	0x0, 7214,
	/*7214*/ uint16(x86_xSetOp), uint16(x86_PSRLD),
	/*7216*/ uint16(x86_xReadSlashR),
	/*7217*/ uint16(x86_xArgMm),
	/*7218*/ uint16(x86_xArgMmM64),
	/*7219*/ uint16(x86_xMatch),
	/*7220*/ uint16(x86_xSetOp), uint16(x86_PSRLD),
	/*7222*/ uint16(x86_xReadSlashR),
	/*7223*/ uint16(x86_xArgXmm1),
	/*7224*/ uint16(x86_xArgXmm2M128),
	/*7225*/ uint16(x86_xMatch),
	/*7226*/ uint16(x86_xCondPrefix), 2,
	0x66, 7238,
	0x0, 7232,
	/*7232*/ uint16(x86_xSetOp), uint16(x86_PSRLQ),
	/*7234*/ uint16(x86_xReadSlashR),
	/*7235*/ uint16(x86_xArgMm),
	/*7236*/ uint16(x86_xArgMmM64),
	/*7237*/ uint16(x86_xMatch),
	/*7238*/ uint16(x86_xSetOp), uint16(x86_PSRLQ),
	/*7240*/ uint16(x86_xReadSlashR),
	/*7241*/ uint16(x86_xArgXmm1),
	/*7242*/ uint16(x86_xArgXmm2M128),
	/*7243*/ uint16(x86_xMatch),
	/*7244*/ uint16(x86_xCondPrefix), 2,
	0x66, 7256,
	0x0, 7250,
	/*7250*/ uint16(x86_xSetOp), uint16(x86_PADDQ),
	/*7252*/ uint16(x86_xReadSlashR),
	/*7253*/ uint16(x86_xArgMm1),
	/*7254*/ uint16(x86_xArgMm2M64),
	/*7255*/ uint16(x86_xMatch),
	/*7256*/ uint16(x86_xSetOp), uint16(x86_PADDQ),
	/*7258*/ uint16(x86_xReadSlashR),
	/*7259*/ uint16(x86_xArgXmm1),
	/*7260*/ uint16(x86_xArgXmm2M128),
	/*7261*/ uint16(x86_xMatch),
	/*7262*/ uint16(x86_xCondPrefix), 2,
	0x66, 7274,
	0x0, 7268,
	/*7268*/ uint16(x86_xSetOp), uint16(x86_PMULLW),
	/*7270*/ uint16(x86_xReadSlashR),
	/*7271*/ uint16(x86_xArgMm),
	/*7272*/ uint16(x86_xArgMmM64),
	/*7273*/ uint16(x86_xMatch),
	/*7274*/ uint16(x86_xSetOp), uint16(x86_PMULLW),
	/*7276*/ uint16(x86_xReadSlashR),
	/*7277*/ uint16(x86_xArgXmm1),
	/*7278*/ uint16(x86_xArgXmm2M128),
	/*7279*/ uint16(x86_xMatch),
	/*7280*/ uint16(x86_xCondPrefix), 3,
	0xF3, 7300,
	0xF2, 7294,
	0x66, 7288,
	/*7288*/ uint16(x86_xSetOp), uint16(x86_MOVQ),
	/*7290*/ uint16(x86_xReadSlashR),
	/*7291*/ uint16(x86_xArgXmm2M64),
	/*7292*/ uint16(x86_xArgXmm1),
	/*7293*/ uint16(x86_xMatch),
	/*7294*/ uint16(x86_xSetOp), uint16(x86_MOVDQ2Q),
	/*7296*/ uint16(x86_xReadSlashR),
	/*7297*/ uint16(x86_xArgMm),
	/*7298*/ uint16(x86_xArgXmm2),
	/*7299*/ uint16(x86_xMatch),
	/*7300*/ uint16(x86_xSetOp), uint16(x86_MOVQ2DQ),
	/*7302*/ uint16(x86_xReadSlashR),
	/*7303*/ uint16(x86_xArgXmm1),
	/*7304*/ uint16(x86_xArgMm2),
	/*7305*/ uint16(x86_xMatch),
	/*7306*/ uint16(x86_xCondPrefix), 2,
	0x66, 7318,
	0x0, 7312,
	/*7312*/ uint16(x86_xSetOp), uint16(x86_PMOVMSKB),
	/*7314*/ uint16(x86_xReadSlashR),
	/*7315*/ uint16(x86_xArgR32),
	/*7316*/ uint16(x86_xArgMm2),
	/*7317*/ uint16(x86_xMatch),
	/*7318*/ uint16(x86_xSetOp), uint16(x86_PMOVMSKB),
	/*7320*/ uint16(x86_xReadSlashR),
	/*7321*/ uint16(x86_xArgR32),
	/*7322*/ uint16(x86_xArgXmm2),
	/*7323*/ uint16(x86_xMatch),
	/*7324*/ uint16(x86_xCondPrefix), 2,
	0x66, 7336,
	0x0, 7330,
	/*7330*/ uint16(x86_xSetOp), uint16(x86_PSUBUSB),
	/*7332*/ uint16(x86_xReadSlashR),
	/*7333*/ uint16(x86_xArgMm),
	/*7334*/ uint16(x86_xArgMmM64),
	/*7335*/ uint16(x86_xMatch),
	/*7336*/ uint16(x86_xSetOp), uint16(x86_PSUBUSB),
	/*7338*/ uint16(x86_xReadSlashR),
	/*7339*/ uint16(x86_xArgXmm1),
	/*7340*/ uint16(x86_xArgXmm2M128),
	/*7341*/ uint16(x86_xMatch),
	/*7342*/ uint16(x86_xCondPrefix), 2,
	0x66, 7354,
	0x0, 7348,
	/*7348*/ uint16(x86_xSetOp), uint16(x86_PSUBUSW),
	/*7350*/ uint16(x86_xReadSlashR),
	/*7351*/ uint16(x86_xArgMm),
	/*7352*/ uint16(x86_xArgMmM64),
	/*7353*/ uint16(x86_xMatch),
	/*7354*/ uint16(x86_xSetOp), uint16(x86_PSUBUSW),
	/*7356*/ uint16(x86_xReadSlashR),
	/*7357*/ uint16(x86_xArgXmm1),
	/*7358*/ uint16(x86_xArgXmm2M128),
	/*7359*/ uint16(x86_xMatch),
	/*7360*/ uint16(x86_xCondPrefix), 2,
	0x66, 7372,
	0x0, 7366,
	/*7366*/ uint16(x86_xSetOp), uint16(x86_PMINUB),
	/*7368*/ uint16(x86_xReadSlashR),
	/*7369*/ uint16(x86_xArgMm1),
	/*7370*/ uint16(x86_xArgMm2M64),
	/*7371*/ uint16(x86_xMatch),
	/*7372*/ uint16(x86_xSetOp), uint16(x86_PMINUB),
	/*7374*/ uint16(x86_xReadSlashR),
	/*7375*/ uint16(x86_xArgXmm1),
	/*7376*/ uint16(x86_xArgXmm2M128),
	/*7377*/ uint16(x86_xMatch),
	/*7378*/ uint16(x86_xCondPrefix), 2,
	0x66, 7390,
	0x0, 7384,
	/*7384*/ uint16(x86_xSetOp), uint16(x86_PAND),
	/*7386*/ uint16(x86_xReadSlashR),
	/*7387*/ uint16(x86_xArgMm),
	/*7388*/ uint16(x86_xArgMmM64),
	/*7389*/ uint16(x86_xMatch),
	/*7390*/ uint16(x86_xSetOp), uint16(x86_PAND),
	/*7392*/ uint16(x86_xReadSlashR),
	/*7393*/ uint16(x86_xArgXmm1),
	/*7394*/ uint16(x86_xArgXmm2M128),
	/*7395*/ uint16(x86_xMatch),
	/*7396*/ uint16(x86_xCondPrefix), 2,
	0x66, 7408,
	0x0, 7402,
	/*7402*/ uint16(x86_xSetOp), uint16(x86_PADDUSB),
	/*7404*/ uint16(x86_xReadSlashR),
	/*7405*/ uint16(x86_xArgMm),
	/*7406*/ uint16(x86_xArgMmM64),
	/*7407*/ uint16(x86_xMatch),
	/*7408*/ uint16(x86_xSetOp), uint16(x86_PADDUSB),
	/*7410*/ uint16(x86_xReadSlashR),
	/*7411*/ uint16(x86_xArgXmm1),
	/*7412*/ uint16(x86_xArgXmm2M128),
	/*7413*/ uint16(x86_xMatch),
	/*7414*/ uint16(x86_xCondPrefix), 2,
	0x66, 7426,
	0x0, 7420,
	/*7420*/ uint16(x86_xSetOp), uint16(x86_PADDUSW),
	/*7422*/ uint16(x86_xReadSlashR),
	/*7423*/ uint16(x86_xArgMm),
	/*7424*/ uint16(x86_xArgMmM64),
	/*7425*/ uint16(x86_xMatch),
	/*7426*/ uint16(x86_xSetOp), uint16(x86_PADDUSW),
	/*7428*/ uint16(x86_xReadSlashR),
	/*7429*/ uint16(x86_xArgXmm1),
	/*7430*/ uint16(x86_xArgXmm2M128),
	/*7431*/ uint16(x86_xMatch),
	/*7432*/ uint16(x86_xCondPrefix), 2,
	0x66, 7444,
	0x0, 7438,
	/*7438*/ uint16(x86_xSetOp), uint16(x86_PMAXUB),
	/*7440*/ uint16(x86_xReadSlashR),
	/*7441*/ uint16(x86_xArgMm1),
	/*7442*/ uint16(x86_xArgMm2M64),
	/*7443*/ uint16(x86_xMatch),
	/*7444*/ uint16(x86_xSetOp), uint16(x86_PMAXUB),
	/*7446*/ uint16(x86_xReadSlashR),
	/*7447*/ uint16(x86_xArgXmm1),
	/*7448*/ uint16(x86_xArgXmm2M128),
	/*7449*/ uint16(x86_xMatch),
	/*7450*/ uint16(x86_xCondPrefix), 2,
	0x66, 7462,
	0x0, 7456,
	/*7456*/ uint16(x86_xSetOp), uint16(x86_PANDN),
	/*7458*/ uint16(x86_xReadSlashR),
	/*7459*/ uint16(x86_xArgMm),
	/*7460*/ uint16(x86_xArgMmM64),
	/*7461*/ uint16(x86_xMatch),
	/*7462*/ uint16(x86_xSetOp), uint16(x86_PANDN),
	/*7464*/ uint16(x86_xReadSlashR),
	/*7465*/ uint16(x86_xArgXmm1),
	/*7466*/ uint16(x86_xArgXmm2M128),
	/*7467*/ uint16(x86_xMatch),
	/*7468*/ uint16(x86_xCondPrefix), 2,
	0x66, 7480,
	0x0, 7474,
	/*7474*/ uint16(x86_xSetOp), uint16(x86_PAVGB),
	/*7476*/ uint16(x86_xReadSlashR),
	/*7477*/ uint16(x86_xArgMm1),
	/*7478*/ uint16(x86_xArgMm2M64),
	/*7479*/ uint16(x86_xMatch),
	/*7480*/ uint16(x86_xSetOp), uint16(x86_PAVGB),
	/*7482*/ uint16(x86_xReadSlashR),
	/*7483*/ uint16(x86_xArgXmm1),
	/*7484*/ uint16(x86_xArgXmm2M128),
	/*7485*/ uint16(x86_xMatch),
	/*7486*/ uint16(x86_xCondPrefix), 2,
	0x66, 7498,
	0x0, 7492,
	/*7492*/ uint16(x86_xSetOp), uint16(x86_PSRAW),
	/*7494*/ uint16(x86_xReadSlashR),
	/*7495*/ uint16(x86_xArgMm),
	/*7496*/ uint16(x86_xArgMmM64),
	/*7497*/ uint16(x86_xMatch),
	/*7498*/ uint16(x86_xSetOp), uint16(x86_PSRAW),
	/*7500*/ uint16(x86_xReadSlashR),
	/*7501*/ uint16(x86_xArgXmm1),
	/*7502*/ uint16(x86_xArgXmm2M128),
	/*7503*/ uint16(x86_xMatch),
	/*7504*/ uint16(x86_xCondPrefix), 2,
	0x66, 7516,
	0x0, 7510,
	/*7510*/ uint16(x86_xSetOp), uint16(x86_PSRAD),
	/*7512*/ uint16(x86_xReadSlashR),
	/*7513*/ uint16(x86_xArgMm),
	/*7514*/ uint16(x86_xArgMmM64),
	/*7515*/ uint16(x86_xMatch),
	/*7516*/ uint16(x86_xSetOp), uint16(x86_PSRAD),
	/*7518*/ uint16(x86_xReadSlashR),
	/*7519*/ uint16(x86_xArgXmm1),
	/*7520*/ uint16(x86_xArgXmm2M128),
	/*7521*/ uint16(x86_xMatch),
	/*7522*/ uint16(x86_xCondPrefix), 2,
	0x66, 7534,
	0x0, 7528,
	/*7528*/ uint16(x86_xSetOp), uint16(x86_PAVGW),
	/*7530*/ uint16(x86_xReadSlashR),
	/*7531*/ uint16(x86_xArgMm1),
	/*7532*/ uint16(x86_xArgMm2M64),
	/*7533*/ uint16(x86_xMatch),
	/*7534*/ uint16(x86_xSetOp), uint16(x86_PAVGW),
	/*7536*/ uint16(x86_xReadSlashR),
	/*7537*/ uint16(x86_xArgXmm1),
	/*7538*/ uint16(x86_xArgXmm2M128),
	/*7539*/ uint16(x86_xMatch),
	/*7540*/ uint16(x86_xCondPrefix), 2,
	0x66, 7552,
	0x0, 7546,
	/*7546*/ uint16(x86_xSetOp), uint16(x86_PMULHUW),
	/*7548*/ uint16(x86_xReadSlashR),
	/*7549*/ uint16(x86_xArgMm1),
	/*7550*/ uint16(x86_xArgMm2M64),
	/*7551*/ uint16(x86_xMatch),
	/*7552*/ uint16(x86_xSetOp), uint16(x86_PMULHUW),
	/*7554*/ uint16(x86_xReadSlashR),
	/*7555*/ uint16(x86_xArgXmm1),
	/*7556*/ uint16(x86_xArgXmm2M128),
	/*7557*/ uint16(x86_xMatch),
	/*7558*/ uint16(x86_xCondPrefix), 2,
	0x66, 7570,
	0x0, 7564,
	/*7564*/ uint16(x86_xSetOp), uint16(x86_PMULHW),
	/*7566*/ uint16(x86_xReadSlashR),
	/*7567*/ uint16(x86_xArgMm),
	/*7568*/ uint16(x86_xArgMmM64),
	/*7569*/ uint16(x86_xMatch),
	/*7570*/ uint16(x86_xSetOp), uint16(x86_PMULHW),
	/*7572*/ uint16(x86_xReadSlashR),
	/*7573*/ uint16(x86_xArgXmm1),
	/*7574*/ uint16(x86_xArgXmm2M128),
	/*7575*/ uint16(x86_xMatch),
	/*7576*/ uint16(x86_xCondPrefix), 3,
	0xF3, 7596,
	0xF2, 7590,
	0x66, 7584,
	/*7584*/ uint16(x86_xSetOp), uint16(x86_CVTTPD2DQ),
	/*7586*/ uint16(x86_xReadSlashR),
	/*7587*/ uint16(x86_xArgXmm1),
	/*7588*/ uint16(x86_xArgXmm2M128),
	/*7589*/ uint16(x86_xMatch),
	/*7590*/ uint16(x86_xSetOp), uint16(x86_CVTPD2DQ),
	/*7592*/ uint16(x86_xReadSlashR),
	/*7593*/ uint16(x86_xArgXmm1),
	/*7594*/ uint16(x86_xArgXmm2M128),
	/*7595*/ uint16(x86_xMatch),
	/*7596*/ uint16(x86_xSetOp), uint16(x86_CVTDQ2PD),
	/*7598*/ uint16(x86_xReadSlashR),
	/*7599*/ uint16(x86_xArgXmm1),
	/*7600*/ uint16(x86_xArgXmm2M64),
	/*7601*/ uint16(x86_xMatch),
	/*7602*/ uint16(x86_xCondPrefix), 2,
	0x66, 7614,
	0x0, 7608,
	/*7608*/ uint16(x86_xSetOp), uint16(x86_MOVNTQ),
	/*7610*/ uint16(x86_xReadSlashR),
	/*7611*/ uint16(x86_xArgM64),
	/*7612*/ uint16(x86_xArgMm),
	/*7613*/ uint16(x86_xMatch),
	/*7614*/ uint16(x86_xSetOp), uint16(x86_MOVNTDQ),
	/*7616*/ uint16(x86_xReadSlashR),
	/*7617*/ uint16(x86_xArgM128),
	/*7618*/ uint16(x86_xArgXmm),
	/*7619*/ uint16(x86_xMatch),
	/*7620*/ uint16(x86_xCondPrefix), 2,
	0x66, 7632,
	0x0, 7626,
	/*7626*/ uint16(x86_xSetOp), uint16(x86_PSUBSB),
	/*7628*/ uint16(x86_xReadSlashR),
	/*7629*/ uint16(x86_xArgMm),
	/*7630*/ uint16(x86_xArgMmM64),
	/*7631*/ uint16(x86_xMatch),
	/*7632*/ uint16(x86_xSetOp), uint16(x86_PSUBSB),
	/*7634*/ uint16(x86_xReadSlashR),
	/*7635*/ uint16(x86_xArgXmm1),
	/*7636*/ uint16(x86_xArgXmm2M128),
	/*7637*/ uint16(x86_xMatch),
	/*7638*/ uint16(x86_xCondPrefix), 2,
	0x66, 7650,
	0x0, 7644,
	/*7644*/ uint16(x86_xSetOp), uint16(x86_PSUBSW),
	/*7646*/ uint16(x86_xReadSlashR),
	/*7647*/ uint16(x86_xArgMm),
	/*7648*/ uint16(x86_xArgMmM64),
	/*7649*/ uint16(x86_xMatch),
	/*7650*/ uint16(x86_xSetOp), uint16(x86_PSUBSW),
	/*7652*/ uint16(x86_xReadSlashR),
	/*7653*/ uint16(x86_xArgXmm1),
	/*7654*/ uint16(x86_xArgXmm2M128),
	/*7655*/ uint16(x86_xMatch),
	/*7656*/ uint16(x86_xCondPrefix), 2,
	0x66, 7668,
	0x0, 7662,
	/*7662*/ uint16(x86_xSetOp), uint16(x86_PMINSW),
	/*7664*/ uint16(x86_xReadSlashR),
	/*7665*/ uint16(x86_xArgMm1),
	/*7666*/ uint16(x86_xArgMm2M64),
	/*7667*/ uint16(x86_xMatch),
	/*7668*/ uint16(x86_xSetOp), uint16(x86_PMINSW),
	/*7670*/ uint16(x86_xReadSlashR),
	/*7671*/ uint16(x86_xArgXmm1),
	/*7672*/ uint16(x86_xArgXmm2M128),
	/*7673*/ uint16(x86_xMatch),
	/*7674*/ uint16(x86_xCondPrefix), 2,
	0x66, 7686,
	0x0, 7680,
	/*7680*/ uint16(x86_xSetOp), uint16(x86_POR),
	/*7682*/ uint16(x86_xReadSlashR),
	/*7683*/ uint16(x86_xArgMm),
	/*7684*/ uint16(x86_xArgMmM64),
	/*7685*/ uint16(x86_xMatch),
	/*7686*/ uint16(x86_xSetOp), uint16(x86_POR),
	/*7688*/ uint16(x86_xReadSlashR),
	/*7689*/ uint16(x86_xArgXmm1),
	/*7690*/ uint16(x86_xArgXmm2M128),
	/*7691*/ uint16(x86_xMatch),
	/*7692*/ uint16(x86_xCondPrefix), 2,
	0x66, 7704,
	0x0, 7698,
	/*7698*/ uint16(x86_xSetOp), uint16(x86_PADDSB),
	/*7700*/ uint16(x86_xReadSlashR),
	/*7701*/ uint16(x86_xArgMm),
	/*7702*/ uint16(x86_xArgMmM64),
	/*7703*/ uint16(x86_xMatch),
	/*7704*/ uint16(x86_xSetOp), uint16(x86_PADDSB),
	/*7706*/ uint16(x86_xReadSlashR),
	/*7707*/ uint16(x86_xArgXmm1),
	/*7708*/ uint16(x86_xArgXmm2M128),
	/*7709*/ uint16(x86_xMatch),
	/*7710*/ uint16(x86_xCondPrefix), 2,
	0x66, 7722,
	0x0, 7716,
	/*7716*/ uint16(x86_xSetOp), uint16(x86_PADDSW),
	/*7718*/ uint16(x86_xReadSlashR),
	/*7719*/ uint16(x86_xArgMm),
	/*7720*/ uint16(x86_xArgMmM64),
	/*7721*/ uint16(x86_xMatch),
	/*7722*/ uint16(x86_xSetOp), uint16(x86_PADDSW),
	/*7724*/ uint16(x86_xReadSlashR),
	/*7725*/ uint16(x86_xArgXmm1),
	/*7726*/ uint16(x86_xArgXmm2M128),
	/*7727*/ uint16(x86_xMatch),
	/*7728*/ uint16(x86_xCondPrefix), 2,
	0x66, 7740,
	0x0, 7734,
	/*7734*/ uint16(x86_xSetOp), uint16(x86_PMAXSW),
	/*7736*/ uint16(x86_xReadSlashR),
	/*7737*/ uint16(x86_xArgMm1),
	/*7738*/ uint16(x86_xArgMm2M64),
	/*7739*/ uint16(x86_xMatch),
	/*7740*/ uint16(x86_xSetOp), uint16(x86_PMAXSW),
	/*7742*/ uint16(x86_xReadSlashR),
	/*7743*/ uint16(x86_xArgXmm1),
	/*7744*/ uint16(x86_xArgXmm2M128),
	/*7745*/ uint16(x86_xMatch),
	/*7746*/ uint16(x86_xCondPrefix), 2,
	0x66, 7758,
	0x0, 7752,
	/*7752*/ uint16(x86_xSetOp), uint16(x86_PXOR),
	/*7754*/ uint16(x86_xReadSlashR),
	/*7755*/ uint16(x86_xArgMm),
	/*7756*/ uint16(x86_xArgMmM64),
	/*7757*/ uint16(x86_xMatch),
	/*7758*/ uint16(x86_xSetOp), uint16(x86_PXOR),
	/*7760*/ uint16(x86_xReadSlashR),
	/*7761*/ uint16(x86_xArgXmm1),
	/*7762*/ uint16(x86_xArgXmm2M128),
	/*7763*/ uint16(x86_xMatch),
	/*7764*/ uint16(x86_xCondPrefix), 1,
	0xF2, 7768,
	/*7768*/ uint16(x86_xSetOp), uint16(x86_LDDQU),
	/*7770*/ uint16(x86_xReadSlashR),
	/*7771*/ uint16(x86_xArgXmm1),
	/*7772*/ uint16(x86_xArgM128),
	/*7773*/ uint16(x86_xMatch),
	/*7774*/ uint16(x86_xCondPrefix), 2,
	0x66, 7786,
	0x0, 7780,
	/*7780*/ uint16(x86_xSetOp), uint16(x86_PSLLW),
	/*7782*/ uint16(x86_xReadSlashR),
	/*7783*/ uint16(x86_xArgMm),
	/*7784*/ uint16(x86_xArgMmM64),
	/*7785*/ uint16(x86_xMatch),
	/*7786*/ uint16(x86_xSetOp), uint16(x86_PSLLW),
	/*7788*/ uint16(x86_xReadSlashR),
	/*7789*/ uint16(x86_xArgXmm1),
	/*7790*/ uint16(x86_xArgXmm2M128),
	/*7791*/ uint16(x86_xMatch),
	/*7792*/ uint16(x86_xCondPrefix), 2,
	0x66, 7804,
	0x0, 7798,
	/*7798*/ uint16(x86_xSetOp), uint16(x86_PSLLD),
	/*7800*/ uint16(x86_xReadSlashR),
	/*7801*/ uint16(x86_xArgMm),
	/*7802*/ uint16(x86_xArgMmM64),
	/*7803*/ uint16(x86_xMatch),
	/*7804*/ uint16(x86_xSetOp), uint16(x86_PSLLD),
	/*7806*/ uint16(x86_xReadSlashR),
	/*7807*/ uint16(x86_xArgXmm1),
	/*7808*/ uint16(x86_xArgXmm2M128),
	/*7809*/ uint16(x86_xMatch),
	/*7810*/ uint16(x86_xCondPrefix), 2,
	0x66, 7822,
	0x0, 7816,
	/*7816*/ uint16(x86_xSetOp), uint16(x86_PSLLQ),
	/*7818*/ uint16(x86_xReadSlashR),
	/*7819*/ uint16(x86_xArgMm),
	/*7820*/ uint16(x86_xArgMmM64),
	/*7821*/ uint16(x86_xMatch),
	/*7822*/ uint16(x86_xSetOp), uint16(x86_PSLLQ),
	/*7824*/ uint16(x86_xReadSlashR),
	/*7825*/ uint16(x86_xArgXmm1),
	/*7826*/ uint16(x86_xArgXmm2M128),
	/*7827*/ uint16(x86_xMatch),
	/*7828*/ uint16(x86_xCondPrefix), 2,
	0x66, 7840,
	0x0, 7834,
	/*7834*/ uint16(x86_xSetOp), uint16(x86_PMULUDQ),
	/*7836*/ uint16(x86_xReadSlashR),
	/*7837*/ uint16(x86_xArgMm1),
	/*7838*/ uint16(x86_xArgMm2M64),
	/*7839*/ uint16(x86_xMatch),
	/*7840*/ uint16(x86_xSetOp), uint16(x86_PMULUDQ),
	/*7842*/ uint16(x86_xReadSlashR),
	/*7843*/ uint16(x86_xArgXmm1),
	/*7844*/ uint16(x86_xArgXmm2M128),
	/*7845*/ uint16(x86_xMatch),
	/*7846*/ uint16(x86_xCondPrefix), 2,
	0x66, 7858,
	0x0, 7852,
	/*7852*/ uint16(x86_xSetOp), uint16(x86_PMADDWD),
	/*7854*/ uint16(x86_xReadSlashR),
	/*7855*/ uint16(x86_xArgMm),
	/*7856*/ uint16(x86_xArgMmM64),
	/*7857*/ uint16(x86_xMatch),
	/*7858*/ uint16(x86_xSetOp), uint16(x86_PMADDWD),
	/*7860*/ uint16(x86_xReadSlashR),
	/*7861*/ uint16(x86_xArgXmm1),
	/*7862*/ uint16(x86_xArgXmm2M128),
	/*7863*/ uint16(x86_xMatch),
	/*7864*/ uint16(x86_xCondPrefix), 2,
	0x66, 7876,
	0x0, 7870,
	/*7870*/ uint16(x86_xSetOp), uint16(x86_PSADBW),
	/*7872*/ uint16(x86_xReadSlashR),
	/*7873*/ uint16(x86_xArgMm1),
	/*7874*/ uint16(x86_xArgMm2M64),
	/*7875*/ uint16(x86_xMatch),
	/*7876*/ uint16(x86_xSetOp), uint16(x86_PSADBW),
	/*7878*/ uint16(x86_xReadSlashR),
	/*7879*/ uint16(x86_xArgXmm1),
	/*7880*/ uint16(x86_xArgXmm2M128),
	/*7881*/ uint16(x86_xMatch),
	/*7882*/ uint16(x86_xCondPrefix), 2,
	0x66, 7894,
	0x0, 7888,
	/*7888*/ uint16(x86_xSetOp), uint16(x86_MASKMOVQ),
	/*7890*/ uint16(x86_xReadSlashR),
	/*7891*/ uint16(x86_xArgMm1),
	/*7892*/ uint16(x86_xArgMm2),
	/*7893*/ uint16(x86_xMatch),
	/*7894*/ uint16(x86_xSetOp), uint16(x86_MASKMOVDQU),
	/*7896*/ uint16(x86_xReadSlashR),
	/*7897*/ uint16(x86_xArgXmm1),
	/*7898*/ uint16(x86_xArgXmm2),
	/*7899*/ uint16(x86_xMatch),
	/*7900*/ uint16(x86_xCondPrefix), 2,
	0x66, 7912,
	0x0, 7906,
	/*7906*/ uint16(x86_xSetOp), uint16(x86_PSUBB),
	/*7908*/ uint16(x86_xReadSlashR),
	/*7909*/ uint16(x86_xArgMm),
	/*7910*/ uint16(x86_xArgMmM64),
	/*7911*/ uint16(x86_xMatch),
	/*7912*/ uint16(x86_xSetOp), uint16(x86_PSUBB),
	/*7914*/ uint16(x86_xReadSlashR),
	/*7915*/ uint16(x86_xArgXmm1),
	/*7916*/ uint16(x86_xArgXmm2M128),
	/*7917*/ uint16(x86_xMatch),
	/*7918*/ uint16(x86_xCondPrefix), 2,
	0x66, 7930,
	0x0, 7924,
	/*7924*/ uint16(x86_xSetOp), uint16(x86_PSUBW),
	/*7926*/ uint16(x86_xReadSlashR),
	/*7927*/ uint16(x86_xArgMm),
	/*7928*/ uint16(x86_xArgMmM64),
	/*7929*/ uint16(x86_xMatch),
	/*7930*/ uint16(x86_xSetOp), uint16(x86_PSUBW),
	/*7932*/ uint16(x86_xReadSlashR),
	/*7933*/ uint16(x86_xArgXmm1),
	/*7934*/ uint16(x86_xArgXmm2M128),
	/*7935*/ uint16(x86_xMatch),
	/*7936*/ uint16(x86_xCondPrefix), 2,
	0x66, 7948,
	0x0, 7942,
	/*7942*/ uint16(x86_xSetOp), uint16(x86_PSUBD),
	/*7944*/ uint16(x86_xReadSlashR),
	/*7945*/ uint16(x86_xArgMm),
	/*7946*/ uint16(x86_xArgMmM64),
	/*7947*/ uint16(x86_xMatch),
	/*7948*/ uint16(x86_xSetOp), uint16(x86_PSUBD),
	/*7950*/ uint16(x86_xReadSlashR),
	/*7951*/ uint16(x86_xArgXmm1),
	/*7952*/ uint16(x86_xArgXmm2M128),
	/*7953*/ uint16(x86_xMatch),
	/*7954*/ uint16(x86_xCondPrefix), 2,
	0x66, 7966,
	0x0, 7960,
	/*7960*/ uint16(x86_xSetOp), uint16(x86_PSUBQ),
	/*7962*/ uint16(x86_xReadSlashR),
	/*7963*/ uint16(x86_xArgMm1),
	/*7964*/ uint16(x86_xArgMm2M64),
	/*7965*/ uint16(x86_xMatch),
	/*7966*/ uint16(x86_xSetOp), uint16(x86_PSUBQ),
	/*7968*/ uint16(x86_xReadSlashR),
	/*7969*/ uint16(x86_xArgXmm1),
	/*7970*/ uint16(x86_xArgXmm2M128),
	/*7971*/ uint16(x86_xMatch),
	/*7972*/ uint16(x86_xCondPrefix), 2,
	0x66, 7984,
	0x0, 7978,
	/*7978*/ uint16(x86_xSetOp), uint16(x86_PADDB),
	/*7980*/ uint16(x86_xReadSlashR),
	/*7981*/ uint16(x86_xArgMm),
	/*7982*/ uint16(x86_xArgMmM64),
	/*7983*/ uint16(x86_xMatch),
	/*7984*/ uint16(x86_xSetOp), uint16(x86_PADDB),
	/*7986*/ uint16(x86_xReadSlashR),
	/*7987*/ uint16(x86_xArgXmm1),
	/*7988*/ uint16(x86_xArgXmm2M128),
	/*7989*/ uint16(x86_xMatch),
	/*7990*/ uint16(x86_xCondPrefix), 2,
	0x66, 8002,
	0x0, 7996,
	/*7996*/ uint16(x86_xSetOp), uint16(x86_PADDW),
	/*7998*/ uint16(x86_xReadSlashR),
	/*7999*/ uint16(x86_xArgMm),
	/*8000*/ uint16(x86_xArgMmM64),
	/*8001*/ uint16(x86_xMatch),
	/*8002*/ uint16(x86_xSetOp), uint16(x86_PADDW),
	/*8004*/ uint16(x86_xReadSlashR),
	/*8005*/ uint16(x86_xArgXmm1),
	/*8006*/ uint16(x86_xArgXmm2M128),
	/*8007*/ uint16(x86_xMatch),
	/*8008*/ uint16(x86_xCondPrefix), 2,
	0x66, 8020,
	0x0, 8014,
	/*8014*/ uint16(x86_xSetOp), uint16(x86_PADDD),
	/*8016*/ uint16(x86_xReadSlashR),
	/*8017*/ uint16(x86_xArgMm),
	/*8018*/ uint16(x86_xArgMmM64),
	/*8019*/ uint16(x86_xMatch),
	/*8020*/ uint16(x86_xSetOp), uint16(x86_PADDD),
	/*8022*/ uint16(x86_xReadSlashR),
	/*8023*/ uint16(x86_xArgXmm1),
	/*8024*/ uint16(x86_xArgXmm2M128),
	/*8025*/ uint16(x86_xMatch),
	/*8026*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8028*/ uint16(x86_xReadSlashR),
	/*8029*/ uint16(x86_xArgRM8),
	/*8030*/ uint16(x86_xArgR8),
	/*8031*/ uint16(x86_xMatch),
	/*8032*/ uint16(x86_xCondIs64), 8035, 8051,
	/*8035*/ uint16(x86_xCondDataSize), 8039, 8045, 0,
	/*8039*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8041*/ uint16(x86_xReadSlashR),
	/*8042*/ uint16(x86_xArgRM16),
	/*8043*/ uint16(x86_xArgR16),
	/*8044*/ uint16(x86_xMatch),
	/*8045*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8047*/ uint16(x86_xReadSlashR),
	/*8048*/ uint16(x86_xArgRM32),
	/*8049*/ uint16(x86_xArgR32),
	/*8050*/ uint16(x86_xMatch),
	/*8051*/ uint16(x86_xCondDataSize), 8039, 8045, 8055,
	/*8055*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8057*/ uint16(x86_xReadSlashR),
	/*8058*/ uint16(x86_xArgRM64),
	/*8059*/ uint16(x86_xArgR64),
	/*8060*/ uint16(x86_xMatch),
	/*8061*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8063*/ uint16(x86_xReadSlashR),
	/*8064*/ uint16(x86_xArgR8),
	/*8065*/ uint16(x86_xArgRM8),
	/*8066*/ uint16(x86_xMatch),
	/*8067*/ uint16(x86_xCondIs64), 8070, 8086,
	/*8070*/ uint16(x86_xCondDataSize), 8074, 8080, 0,
	/*8074*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8076*/ uint16(x86_xReadSlashR),
	/*8077*/ uint16(x86_xArgR16),
	/*8078*/ uint16(x86_xArgRM16),
	/*8079*/ uint16(x86_xMatch),
	/*8080*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8082*/ uint16(x86_xReadSlashR),
	/*8083*/ uint16(x86_xArgR32),
	/*8084*/ uint16(x86_xArgRM32),
	/*8085*/ uint16(x86_xMatch),
	/*8086*/ uint16(x86_xCondDataSize), 8074, 8080, 8090,
	/*8090*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8092*/ uint16(x86_xReadSlashR),
	/*8093*/ uint16(x86_xArgR64),
	/*8094*/ uint16(x86_xArgRM64),
	/*8095*/ uint16(x86_xMatch),
	/*8096*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8098*/ uint16(x86_xReadIb),
	/*8099*/ uint16(x86_xArgAL),
	/*8100*/ uint16(x86_xArgImm8u),
	/*8101*/ uint16(x86_xMatch),
	/*8102*/ uint16(x86_xCondIs64), 8105, 8121,
	/*8105*/ uint16(x86_xCondDataSize), 8109, 8115, 0,
	/*8109*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8111*/ uint16(x86_xReadIw),
	/*8112*/ uint16(x86_xArgAX),
	/*8113*/ uint16(x86_xArgImm16),
	/*8114*/ uint16(x86_xMatch),
	/*8115*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8117*/ uint16(x86_xReadId),
	/*8118*/ uint16(x86_xArgEAX),
	/*8119*/ uint16(x86_xArgImm32),
	/*8120*/ uint16(x86_xMatch),
	/*8121*/ uint16(x86_xCondDataSize), 8109, 8115, 8125,
	/*8125*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*8127*/ uint16(x86_xReadId),
	/*8128*/ uint16(x86_xArgRAX),
	/*8129*/ uint16(x86_xArgImm32),
	/*8130*/ uint16(x86_xMatch),
	/*8131*/ uint16(x86_xCondIs64), 8134, 0,
	/*8134*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8136*/ uint16(x86_xArgSS),
	/*8137*/ uint16(x86_xMatch),
	/*8138*/ uint16(x86_xCondIs64), 8141, 0,
	/*8141*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8143*/ uint16(x86_xArgSS),
	/*8144*/ uint16(x86_xMatch),
	/*8145*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8147*/ uint16(x86_xReadSlashR),
	/*8148*/ uint16(x86_xArgRM8),
	/*8149*/ uint16(x86_xArgR8),
	/*8150*/ uint16(x86_xMatch),
	/*8151*/ uint16(x86_xCondIs64), 8154, 8170,
	/*8154*/ uint16(x86_xCondDataSize), 8158, 8164, 0,
	/*8158*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8160*/ uint16(x86_xReadSlashR),
	/*8161*/ uint16(x86_xArgRM16),
	/*8162*/ uint16(x86_xArgR16),
	/*8163*/ uint16(x86_xMatch),
	/*8164*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8166*/ uint16(x86_xReadSlashR),
	/*8167*/ uint16(x86_xArgRM32),
	/*8168*/ uint16(x86_xArgR32),
	/*8169*/ uint16(x86_xMatch),
	/*8170*/ uint16(x86_xCondDataSize), 8158, 8164, 8174,
	/*8174*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8176*/ uint16(x86_xReadSlashR),
	/*8177*/ uint16(x86_xArgRM64),
	/*8178*/ uint16(x86_xArgR64),
	/*8179*/ uint16(x86_xMatch),
	/*8180*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8182*/ uint16(x86_xReadSlashR),
	/*8183*/ uint16(x86_xArgR8),
	/*8184*/ uint16(x86_xArgRM8),
	/*8185*/ uint16(x86_xMatch),
	/*8186*/ uint16(x86_xCondIs64), 8189, 8205,
	/*8189*/ uint16(x86_xCondDataSize), 8193, 8199, 0,
	/*8193*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8195*/ uint16(x86_xReadSlashR),
	/*8196*/ uint16(x86_xArgR16),
	/*8197*/ uint16(x86_xArgRM16),
	/*8198*/ uint16(x86_xMatch),
	/*8199*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8201*/ uint16(x86_xReadSlashR),
	/*8202*/ uint16(x86_xArgR32),
	/*8203*/ uint16(x86_xArgRM32),
	/*8204*/ uint16(x86_xMatch),
	/*8205*/ uint16(x86_xCondDataSize), 8193, 8199, 8209,
	/*8209*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8211*/ uint16(x86_xReadSlashR),
	/*8212*/ uint16(x86_xArgR64),
	/*8213*/ uint16(x86_xArgRM64),
	/*8214*/ uint16(x86_xMatch),
	/*8215*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8217*/ uint16(x86_xReadIb),
	/*8218*/ uint16(x86_xArgAL),
	/*8219*/ uint16(x86_xArgImm8u),
	/*8220*/ uint16(x86_xMatch),
	/*8221*/ uint16(x86_xCondIs64), 8224, 8240,
	/*8224*/ uint16(x86_xCondDataSize), 8228, 8234, 0,
	/*8228*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8230*/ uint16(x86_xReadIw),
	/*8231*/ uint16(x86_xArgAX),
	/*8232*/ uint16(x86_xArgImm16),
	/*8233*/ uint16(x86_xMatch),
	/*8234*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8236*/ uint16(x86_xReadId),
	/*8237*/ uint16(x86_xArgEAX),
	/*8238*/ uint16(x86_xArgImm32),
	/*8239*/ uint16(x86_xMatch),
	/*8240*/ uint16(x86_xCondDataSize), 8228, 8234, 8244,
	/*8244*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*8246*/ uint16(x86_xReadId),
	/*8247*/ uint16(x86_xArgRAX),
	/*8248*/ uint16(x86_xArgImm32),
	/*8249*/ uint16(x86_xMatch),
	/*8250*/ uint16(x86_xCondIs64), 8253, 0,
	/*8253*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8255*/ uint16(x86_xArgDS),
	/*8256*/ uint16(x86_xMatch),
	/*8257*/ uint16(x86_xCondIs64), 8260, 0,
	/*8260*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8262*/ uint16(x86_xArgDS),
	/*8263*/ uint16(x86_xMatch),
	/*8264*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8266*/ uint16(x86_xReadSlashR),
	/*8267*/ uint16(x86_xArgRM8),
	/*8268*/ uint16(x86_xArgR8),
	/*8269*/ uint16(x86_xMatch),
	/*8270*/ uint16(x86_xCondIs64), 8273, 8289,
	/*8273*/ uint16(x86_xCondDataSize), 8277, 8283, 0,
	/*8277*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8279*/ uint16(x86_xReadSlashR),
	/*8280*/ uint16(x86_xArgRM16),
	/*8281*/ uint16(x86_xArgR16),
	/*8282*/ uint16(x86_xMatch),
	/*8283*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8285*/ uint16(x86_xReadSlashR),
	/*8286*/ uint16(x86_xArgRM32),
	/*8287*/ uint16(x86_xArgR32),
	/*8288*/ uint16(x86_xMatch),
	/*8289*/ uint16(x86_xCondDataSize), 8277, 8283, 8293,
	/*8293*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8295*/ uint16(x86_xReadSlashR),
	/*8296*/ uint16(x86_xArgRM64),
	/*8297*/ uint16(x86_xArgR64),
	/*8298*/ uint16(x86_xMatch),
	/*8299*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8301*/ uint16(x86_xReadSlashR),
	/*8302*/ uint16(x86_xArgR8),
	/*8303*/ uint16(x86_xArgRM8),
	/*8304*/ uint16(x86_xMatch),
	/*8305*/ uint16(x86_xCondIs64), 8308, 8324,
	/*8308*/ uint16(x86_xCondDataSize), 8312, 8318, 0,
	/*8312*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8314*/ uint16(x86_xReadSlashR),
	/*8315*/ uint16(x86_xArgR16),
	/*8316*/ uint16(x86_xArgRM16),
	/*8317*/ uint16(x86_xMatch),
	/*8318*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8320*/ uint16(x86_xReadSlashR),
	/*8321*/ uint16(x86_xArgR32),
	/*8322*/ uint16(x86_xArgRM32),
	/*8323*/ uint16(x86_xMatch),
	/*8324*/ uint16(x86_xCondDataSize), 8312, 8318, 8328,
	/*8328*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8330*/ uint16(x86_xReadSlashR),
	/*8331*/ uint16(x86_xArgR64),
	/*8332*/ uint16(x86_xArgRM64),
	/*8333*/ uint16(x86_xMatch),
	/*8334*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8336*/ uint16(x86_xReadIb),
	/*8337*/ uint16(x86_xArgAL),
	/*8338*/ uint16(x86_xArgImm8u),
	/*8339*/ uint16(x86_xMatch),
	/*8340*/ uint16(x86_xCondIs64), 8343, 8359,
	/*8343*/ uint16(x86_xCondDataSize), 8347, 8353, 0,
	/*8347*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8349*/ uint16(x86_xReadIw),
	/*8350*/ uint16(x86_xArgAX),
	/*8351*/ uint16(x86_xArgImm16),
	/*8352*/ uint16(x86_xMatch),
	/*8353*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8355*/ uint16(x86_xReadId),
	/*8356*/ uint16(x86_xArgEAX),
	/*8357*/ uint16(x86_xArgImm32),
	/*8358*/ uint16(x86_xMatch),
	/*8359*/ uint16(x86_xCondDataSize), 8347, 8353, 8363,
	/*8363*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*8365*/ uint16(x86_xReadId),
	/*8366*/ uint16(x86_xArgRAX),
	/*8367*/ uint16(x86_xArgImm32),
	/*8368*/ uint16(x86_xMatch),
	/*8369*/ uint16(x86_xCondIs64), 8372, 0,
	/*8372*/ uint16(x86_xSetOp), uint16(x86_DAA),
	/*8374*/ uint16(x86_xMatch),
	/*8375*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8377*/ uint16(x86_xReadSlashR),
	/*8378*/ uint16(x86_xArgRM8),
	/*8379*/ uint16(x86_xArgR8),
	/*8380*/ uint16(x86_xMatch),
	/*8381*/ uint16(x86_xCondIs64), 8384, 8400,
	/*8384*/ uint16(x86_xCondDataSize), 8388, 8394, 0,
	/*8388*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8390*/ uint16(x86_xReadSlashR),
	/*8391*/ uint16(x86_xArgRM16),
	/*8392*/ uint16(x86_xArgR16),
	/*8393*/ uint16(x86_xMatch),
	/*8394*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8396*/ uint16(x86_xReadSlashR),
	/*8397*/ uint16(x86_xArgRM32),
	/*8398*/ uint16(x86_xArgR32),
	/*8399*/ uint16(x86_xMatch),
	/*8400*/ uint16(x86_xCondDataSize), 8388, 8394, 8404,
	/*8404*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8406*/ uint16(x86_xReadSlashR),
	/*8407*/ uint16(x86_xArgRM64),
	/*8408*/ uint16(x86_xArgR64),
	/*8409*/ uint16(x86_xMatch),
	/*8410*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8412*/ uint16(x86_xReadSlashR),
	/*8413*/ uint16(x86_xArgR8),
	/*8414*/ uint16(x86_xArgRM8),
	/*8415*/ uint16(x86_xMatch),
	/*8416*/ uint16(x86_xCondIs64), 8419, 8435,
	/*8419*/ uint16(x86_xCondDataSize), 8423, 8429, 0,
	/*8423*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8425*/ uint16(x86_xReadSlashR),
	/*8426*/ uint16(x86_xArgR16),
	/*8427*/ uint16(x86_xArgRM16),
	/*8428*/ uint16(x86_xMatch),
	/*8429*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8431*/ uint16(x86_xReadSlashR),
	/*8432*/ uint16(x86_xArgR32),
	/*8433*/ uint16(x86_xArgRM32),
	/*8434*/ uint16(x86_xMatch),
	/*8435*/ uint16(x86_xCondDataSize), 8423, 8429, 8439,
	/*8439*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8441*/ uint16(x86_xReadSlashR),
	/*8442*/ uint16(x86_xArgR64),
	/*8443*/ uint16(x86_xArgRM64),
	/*8444*/ uint16(x86_xMatch),
	/*8445*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8447*/ uint16(x86_xReadIb),
	/*8448*/ uint16(x86_xArgAL),
	/*8449*/ uint16(x86_xArgImm8u),
	/*8450*/ uint16(x86_xMatch),
	/*8451*/ uint16(x86_xCondIs64), 8454, 8470,
	/*8454*/ uint16(x86_xCondDataSize), 8458, 8464, 0,
	/*8458*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8460*/ uint16(x86_xReadIw),
	/*8461*/ uint16(x86_xArgAX),
	/*8462*/ uint16(x86_xArgImm16),
	/*8463*/ uint16(x86_xMatch),
	/*8464*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8466*/ uint16(x86_xReadId),
	/*8467*/ uint16(x86_xArgEAX),
	/*8468*/ uint16(x86_xArgImm32),
	/*8469*/ uint16(x86_xMatch),
	/*8470*/ uint16(x86_xCondDataSize), 8458, 8464, 8474,
	/*8474*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*8476*/ uint16(x86_xReadId),
	/*8477*/ uint16(x86_xArgRAX),
	/*8478*/ uint16(x86_xArgImm32),
	/*8479*/ uint16(x86_xMatch),
	/*8480*/ uint16(x86_xCondIs64), 8483, 0,
	/*8483*/ uint16(x86_xSetOp), uint16(x86_DAS),
	/*8485*/ uint16(x86_xMatch),
	/*8486*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8488*/ uint16(x86_xReadSlashR),
	/*8489*/ uint16(x86_xArgRM8),
	/*8490*/ uint16(x86_xArgR8),
	/*8491*/ uint16(x86_xMatch),
	/*8492*/ uint16(x86_xCondIs64), 8495, 8511,
	/*8495*/ uint16(x86_xCondDataSize), 8499, 8505, 0,
	/*8499*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8501*/ uint16(x86_xReadSlashR),
	/*8502*/ uint16(x86_xArgRM16),
	/*8503*/ uint16(x86_xArgR16),
	/*8504*/ uint16(x86_xMatch),
	/*8505*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8507*/ uint16(x86_xReadSlashR),
	/*8508*/ uint16(x86_xArgRM32),
	/*8509*/ uint16(x86_xArgR32),
	/*8510*/ uint16(x86_xMatch),
	/*8511*/ uint16(x86_xCondDataSize), 8499, 8505, 8515,
	/*8515*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8517*/ uint16(x86_xReadSlashR),
	/*8518*/ uint16(x86_xArgRM64),
	/*8519*/ uint16(x86_xArgR64),
	/*8520*/ uint16(x86_xMatch),
	/*8521*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8523*/ uint16(x86_xReadSlashR),
	/*8524*/ uint16(x86_xArgR8),
	/*8525*/ uint16(x86_xArgRM8),
	/*8526*/ uint16(x86_xMatch),
	/*8527*/ uint16(x86_xCondIs64), 8530, 8546,
	/*8530*/ uint16(x86_xCondDataSize), 8534, 8540, 0,
	/*8534*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8536*/ uint16(x86_xReadSlashR),
	/*8537*/ uint16(x86_xArgR16),
	/*8538*/ uint16(x86_xArgRM16),
	/*8539*/ uint16(x86_xMatch),
	/*8540*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8542*/ uint16(x86_xReadSlashR),
	/*8543*/ uint16(x86_xArgR32),
	/*8544*/ uint16(x86_xArgRM32),
	/*8545*/ uint16(x86_xMatch),
	/*8546*/ uint16(x86_xCondDataSize), 8534, 8540, 8550,
	/*8550*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8552*/ uint16(x86_xReadSlashR),
	/*8553*/ uint16(x86_xArgR64),
	/*8554*/ uint16(x86_xArgRM64),
	/*8555*/ uint16(x86_xMatch),
	/*8556*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8558*/ uint16(x86_xReadIb),
	/*8559*/ uint16(x86_xArgAL),
	/*8560*/ uint16(x86_xArgImm8u),
	/*8561*/ uint16(x86_xMatch),
	/*8562*/ uint16(x86_xCondIs64), 8565, 8581,
	/*8565*/ uint16(x86_xCondDataSize), 8569, 8575, 0,
	/*8569*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8571*/ uint16(x86_xReadIw),
	/*8572*/ uint16(x86_xArgAX),
	/*8573*/ uint16(x86_xArgImm16),
	/*8574*/ uint16(x86_xMatch),
	/*8575*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8577*/ uint16(x86_xReadId),
	/*8578*/ uint16(x86_xArgEAX),
	/*8579*/ uint16(x86_xArgImm32),
	/*8580*/ uint16(x86_xMatch),
	/*8581*/ uint16(x86_xCondDataSize), 8569, 8575, 8585,
	/*8585*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*8587*/ uint16(x86_xReadId),
	/*8588*/ uint16(x86_xArgRAX),
	/*8589*/ uint16(x86_xArgImm32),
	/*8590*/ uint16(x86_xMatch),
	/*8591*/ uint16(x86_xCondIs64), 8594, 0,
	/*8594*/ uint16(x86_xSetOp), uint16(x86_AAA),
	/*8596*/ uint16(x86_xMatch),
	/*8597*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8599*/ uint16(x86_xReadSlashR),
	/*8600*/ uint16(x86_xArgRM8),
	/*8601*/ uint16(x86_xArgR8),
	/*8602*/ uint16(x86_xMatch),
	/*8603*/ uint16(x86_xCondIs64), 8606, 8622,
	/*8606*/ uint16(x86_xCondDataSize), 8610, 8616, 0,
	/*8610*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8612*/ uint16(x86_xReadSlashR),
	/*8613*/ uint16(x86_xArgRM16),
	/*8614*/ uint16(x86_xArgR16),
	/*8615*/ uint16(x86_xMatch),
	/*8616*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8618*/ uint16(x86_xReadSlashR),
	/*8619*/ uint16(x86_xArgRM32),
	/*8620*/ uint16(x86_xArgR32),
	/*8621*/ uint16(x86_xMatch),
	/*8622*/ uint16(x86_xCondDataSize), 8610, 8616, 8626,
	/*8626*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8628*/ uint16(x86_xReadSlashR),
	/*8629*/ uint16(x86_xArgRM64),
	/*8630*/ uint16(x86_xArgR64),
	/*8631*/ uint16(x86_xMatch),
	/*8632*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8634*/ uint16(x86_xReadSlashR),
	/*8635*/ uint16(x86_xArgR8),
	/*8636*/ uint16(x86_xArgRM8),
	/*8637*/ uint16(x86_xMatch),
	/*8638*/ uint16(x86_xCondIs64), 8641, 8657,
	/*8641*/ uint16(x86_xCondDataSize), 8645, 8651, 0,
	/*8645*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8647*/ uint16(x86_xReadSlashR),
	/*8648*/ uint16(x86_xArgR16),
	/*8649*/ uint16(x86_xArgRM16),
	/*8650*/ uint16(x86_xMatch),
	/*8651*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8653*/ uint16(x86_xReadSlashR),
	/*8654*/ uint16(x86_xArgR32),
	/*8655*/ uint16(x86_xArgRM32),
	/*8656*/ uint16(x86_xMatch),
	/*8657*/ uint16(x86_xCondDataSize), 8645, 8651, 8661,
	/*8661*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8663*/ uint16(x86_xReadSlashR),
	/*8664*/ uint16(x86_xArgR64),
	/*8665*/ uint16(x86_xArgRM64),
	/*8666*/ uint16(x86_xMatch),
	/*8667*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8669*/ uint16(x86_xReadIb),
	/*8670*/ uint16(x86_xArgAL),
	/*8671*/ uint16(x86_xArgImm8u),
	/*8672*/ uint16(x86_xMatch),
	/*8673*/ uint16(x86_xCondIs64), 8676, 8692,
	/*8676*/ uint16(x86_xCondDataSize), 8680, 8686, 0,
	/*8680*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8682*/ uint16(x86_xReadIw),
	/*8683*/ uint16(x86_xArgAX),
	/*8684*/ uint16(x86_xArgImm16),
	/*8685*/ uint16(x86_xMatch),
	/*8686*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8688*/ uint16(x86_xReadId),
	/*8689*/ uint16(x86_xArgEAX),
	/*8690*/ uint16(x86_xArgImm32),
	/*8691*/ uint16(x86_xMatch),
	/*8692*/ uint16(x86_xCondDataSize), 8680, 8686, 8696,
	/*8696*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*8698*/ uint16(x86_xReadId),
	/*8699*/ uint16(x86_xArgRAX),
	/*8700*/ uint16(x86_xArgImm32),
	/*8701*/ uint16(x86_xMatch),
	/*8702*/ uint16(x86_xCondIs64), 8705, 0,
	/*8705*/ uint16(x86_xSetOp), uint16(x86_AAS),
	/*8707*/ uint16(x86_xMatch),
	/*8708*/ uint16(x86_xCondIs64), 8711, 0,
	/*8711*/ uint16(x86_xCondDataSize), 8715, 8719, 0,
	/*8715*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*8717*/ uint16(x86_xArgR16op),
	/*8718*/ uint16(x86_xMatch),
	/*8719*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*8721*/ uint16(x86_xArgR32op),
	/*8722*/ uint16(x86_xMatch),
	/*8723*/ uint16(x86_xCondIs64), 8726, 0,
	/*8726*/ uint16(x86_xCondDataSize), 8730, 8734, 0,
	/*8730*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*8732*/ uint16(x86_xArgR16op),
	/*8733*/ uint16(x86_xMatch),
	/*8734*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*8736*/ uint16(x86_xArgR32op),
	/*8737*/ uint16(x86_xMatch),
	/*8738*/ uint16(x86_xCondIs64), 8741, 8753,
	/*8741*/ uint16(x86_xCondDataSize), 8745, 8749, 0,
	/*8745*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8747*/ uint16(x86_xArgR16op),
	/*8748*/ uint16(x86_xMatch),
	/*8749*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8751*/ uint16(x86_xArgR32op),
	/*8752*/ uint16(x86_xMatch),
	/*8753*/ uint16(x86_xCondDataSize), 8745, 8757, 8761,
	/*8757*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8759*/ uint16(x86_xArgR64op),
	/*8760*/ uint16(x86_xMatch),
	/*8761*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8763*/ uint16(x86_xArgR64op),
	/*8764*/ uint16(x86_xMatch),
	/*8765*/ uint16(x86_xCondIs64), 8768, 8780,
	/*8768*/ uint16(x86_xCondDataSize), 8772, 8776, 0,
	/*8772*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8774*/ uint16(x86_xArgR16op),
	/*8775*/ uint16(x86_xMatch),
	/*8776*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8778*/ uint16(x86_xArgR32op),
	/*8779*/ uint16(x86_xMatch),
	/*8780*/ uint16(x86_xCondDataSize), 8772, 8784, 8788,
	/*8784*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8786*/ uint16(x86_xArgR64op),
	/*8787*/ uint16(x86_xMatch),
	/*8788*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*8790*/ uint16(x86_xArgR64op),
	/*8791*/ uint16(x86_xMatch),
	/*8792*/ uint16(x86_xCondIs64), 8795, 0,
	/*8795*/ uint16(x86_xCondDataSize), 8799, 8802, 0,
	/*8799*/ uint16(x86_xSetOp), uint16(x86_PUSHA),
	/*8801*/ uint16(x86_xMatch),
	/*8802*/ uint16(x86_xSetOp), uint16(x86_PUSHAD),
	/*8804*/ uint16(x86_xMatch),
	/*8805*/ uint16(x86_xCondIs64), 8808, 0,
	/*8808*/ uint16(x86_xCondDataSize), 8812, 8815, 0,
	/*8812*/ uint16(x86_xSetOp), uint16(x86_POPA),
	/*8814*/ uint16(x86_xMatch),
	/*8815*/ uint16(x86_xSetOp), uint16(x86_POPAD),
	/*8817*/ uint16(x86_xMatch),
	/*8818*/ uint16(x86_xCondIs64), 8821, 0,
	/*8821*/ uint16(x86_xCondDataSize), 8825, 8831, 0,
	/*8825*/ uint16(x86_xSetOp), uint16(x86_BOUND),
	/*8827*/ uint16(x86_xReadSlashR),
	/*8828*/ uint16(x86_xArgR16),
	/*8829*/ uint16(x86_xArgM16and16),
	/*8830*/ uint16(x86_xMatch),
	/*8831*/ uint16(x86_xSetOp), uint16(x86_BOUND),
	/*8833*/ uint16(x86_xReadSlashR),
	/*8834*/ uint16(x86_xArgR32),
	/*8835*/ uint16(x86_xArgM32and32),
	/*8836*/ uint16(x86_xMatch),
	/*8837*/ uint16(x86_xCondIs64), 8840, 8846,
	/*8840*/ uint16(x86_xSetOp), uint16(x86_ARPL),
	/*8842*/ uint16(x86_xReadSlashR),
	/*8843*/ uint16(x86_xArgRM16),
	/*8844*/ uint16(x86_xArgR16),
	/*8845*/ uint16(x86_xMatch),
	/*8846*/ uint16(x86_xCondDataSize), 8850, 8856, 8862,
	/*8850*/ uint16(x86_xSetOp), uint16(x86_MOVSXD),
	/*8852*/ uint16(x86_xReadSlashR),
	/*8853*/ uint16(x86_xArgR16),
	/*8854*/ uint16(x86_xArgRM32),
	/*8855*/ uint16(x86_xMatch),
	/*8856*/ uint16(x86_xSetOp), uint16(x86_MOVSXD),
	/*8858*/ uint16(x86_xReadSlashR),
	/*8859*/ uint16(x86_xArgR32),
	/*8860*/ uint16(x86_xArgRM32),
	/*8861*/ uint16(x86_xMatch),
	/*8862*/ uint16(x86_xSetOp), uint16(x86_MOVSXD),
	/*8864*/ uint16(x86_xReadSlashR),
	/*8865*/ uint16(x86_xArgR64),
	/*8866*/ uint16(x86_xArgRM32),
	/*8867*/ uint16(x86_xMatch),
	/*8868*/ uint16(x86_xCondDataSize), 8872, 8877, 8882,
	/*8872*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8874*/ uint16(x86_xReadIw),
	/*8875*/ uint16(x86_xArgImm16),
	/*8876*/ uint16(x86_xMatch),
	/*8877*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8879*/ uint16(x86_xReadId),
	/*8880*/ uint16(x86_xArgImm32),
	/*8881*/ uint16(x86_xMatch),
	/*8882*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8884*/ uint16(x86_xReadId),
	/*8885*/ uint16(x86_xArgImm32),
	/*8886*/ uint16(x86_xMatch),
	/*8887*/ uint16(x86_xCondIs64), 8890, 8910,
	/*8890*/ uint16(x86_xCondDataSize), 8894, 8902, 0,
	/*8894*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8896*/ uint16(x86_xReadSlashR),
	/*8897*/ uint16(x86_xReadIw),
	/*8898*/ uint16(x86_xArgR16),
	/*8899*/ uint16(x86_xArgRM16),
	/*8900*/ uint16(x86_xArgImm16),
	/*8901*/ uint16(x86_xMatch),
	/*8902*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8904*/ uint16(x86_xReadSlashR),
	/*8905*/ uint16(x86_xReadId),
	/*8906*/ uint16(x86_xArgR32),
	/*8907*/ uint16(x86_xArgRM32),
	/*8908*/ uint16(x86_xArgImm32),
	/*8909*/ uint16(x86_xMatch),
	/*8910*/ uint16(x86_xCondDataSize), 8894, 8902, 8914,
	/*8914*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8916*/ uint16(x86_xReadSlashR),
	/*8917*/ uint16(x86_xReadId),
	/*8918*/ uint16(x86_xArgR64),
	/*8919*/ uint16(x86_xArgRM64),
	/*8920*/ uint16(x86_xArgImm32),
	/*8921*/ uint16(x86_xMatch),
	/*8922*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*8924*/ uint16(x86_xReadIb),
	/*8925*/ uint16(x86_xArgImm8),
	/*8926*/ uint16(x86_xMatch),
	/*8927*/ uint16(x86_xCondIs64), 8930, 8950,
	/*8930*/ uint16(x86_xCondDataSize), 8934, 8942, 0,
	/*8934*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8936*/ uint16(x86_xReadSlashR),
	/*8937*/ uint16(x86_xReadIb),
	/*8938*/ uint16(x86_xArgR16),
	/*8939*/ uint16(x86_xArgRM16),
	/*8940*/ uint16(x86_xArgImm8),
	/*8941*/ uint16(x86_xMatch),
	/*8942*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8944*/ uint16(x86_xReadSlashR),
	/*8945*/ uint16(x86_xReadIb),
	/*8946*/ uint16(x86_xArgR32),
	/*8947*/ uint16(x86_xArgRM32),
	/*8948*/ uint16(x86_xArgImm8),
	/*8949*/ uint16(x86_xMatch),
	/*8950*/ uint16(x86_xCondDataSize), 8934, 8942, 8954,
	/*8954*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*8956*/ uint16(x86_xReadSlashR),
	/*8957*/ uint16(x86_xReadIb),
	/*8958*/ uint16(x86_xArgR64),
	/*8959*/ uint16(x86_xArgRM64),
	/*8960*/ uint16(x86_xArgImm8),
	/*8961*/ uint16(x86_xMatch),
	/*8962*/ uint16(x86_xSetOp), uint16(x86_INSB),
	/*8964*/ uint16(x86_xMatch),
	/*8965*/ uint16(x86_xCondDataSize), 8969, 8972, 8975,
	/*8969*/ uint16(x86_xSetOp), uint16(x86_INSW),
	/*8971*/ uint16(x86_xMatch),
	/*8972*/ uint16(x86_xSetOp), uint16(x86_INSD),
	/*8974*/ uint16(x86_xMatch),
	/*8975*/ uint16(x86_xSetOp), uint16(x86_INSD),
	/*8977*/ uint16(x86_xMatch),
	/*8978*/ uint16(x86_xSetOp), uint16(x86_OUTSB),
	/*8980*/ uint16(x86_xMatch),
	/*8981*/ uint16(x86_xCondDataSize), 8985, 8988, 8991,
	/*8985*/ uint16(x86_xSetOp), uint16(x86_OUTSW),
	/*8987*/ uint16(x86_xMatch),
	/*8988*/ uint16(x86_xSetOp), uint16(x86_OUTSD),
	/*8990*/ uint16(x86_xMatch),
	/*8991*/ uint16(x86_xSetOp), uint16(x86_OUTSD),
	/*8993*/ uint16(x86_xMatch),
	/*8994*/ uint16(x86_xSetOp), uint16(x86_JO),
	/*8996*/ uint16(x86_xReadCb),
	/*8997*/ uint16(x86_xArgRel8),
	/*8998*/ uint16(x86_xMatch),
	/*8999*/ uint16(x86_xSetOp), uint16(x86_JNO),
	/*9001*/ uint16(x86_xReadCb),
	/*9002*/ uint16(x86_xArgRel8),
	/*9003*/ uint16(x86_xMatch),
	/*9004*/ uint16(x86_xSetOp), uint16(x86_JB),
	/*9006*/ uint16(x86_xReadCb),
	/*9007*/ uint16(x86_xArgRel8),
	/*9008*/ uint16(x86_xMatch),
	/*9009*/ uint16(x86_xSetOp), uint16(x86_JAE),
	/*9011*/ uint16(x86_xReadCb),
	/*9012*/ uint16(x86_xArgRel8),
	/*9013*/ uint16(x86_xMatch),
	/*9014*/ uint16(x86_xSetOp), uint16(x86_JE),
	/*9016*/ uint16(x86_xReadCb),
	/*9017*/ uint16(x86_xArgRel8),
	/*9018*/ uint16(x86_xMatch),
	/*9019*/ uint16(x86_xSetOp), uint16(x86_JNE),
	/*9021*/ uint16(x86_xReadCb),
	/*9022*/ uint16(x86_xArgRel8),
	/*9023*/ uint16(x86_xMatch),
	/*9024*/ uint16(x86_xSetOp), uint16(x86_JBE),
	/*9026*/ uint16(x86_xReadCb),
	/*9027*/ uint16(x86_xArgRel8),
	/*9028*/ uint16(x86_xMatch),
	/*9029*/ uint16(x86_xSetOp), uint16(x86_JA),
	/*9031*/ uint16(x86_xReadCb),
	/*9032*/ uint16(x86_xArgRel8),
	/*9033*/ uint16(x86_xMatch),
	/*9034*/ uint16(x86_xSetOp), uint16(x86_JS),
	/*9036*/ uint16(x86_xReadCb),
	/*9037*/ uint16(x86_xArgRel8),
	/*9038*/ uint16(x86_xMatch),
	/*9039*/ uint16(x86_xSetOp), uint16(x86_JNS),
	/*9041*/ uint16(x86_xReadCb),
	/*9042*/ uint16(x86_xArgRel8),
	/*9043*/ uint16(x86_xMatch),
	/*9044*/ uint16(x86_xSetOp), uint16(x86_JP),
	/*9046*/ uint16(x86_xReadCb),
	/*9047*/ uint16(x86_xArgRel8),
	/*9048*/ uint16(x86_xMatch),
	/*9049*/ uint16(x86_xSetOp), uint16(x86_JNP),
	/*9051*/ uint16(x86_xReadCb),
	/*9052*/ uint16(x86_xArgRel8),
	/*9053*/ uint16(x86_xMatch),
	/*9054*/ uint16(x86_xSetOp), uint16(x86_JL),
	/*9056*/ uint16(x86_xReadCb),
	/*9057*/ uint16(x86_xArgRel8),
	/*9058*/ uint16(x86_xMatch),
	/*9059*/ uint16(x86_xSetOp), uint16(x86_JGE),
	/*9061*/ uint16(x86_xReadCb),
	/*9062*/ uint16(x86_xArgRel8),
	/*9063*/ uint16(x86_xMatch),
	/*9064*/ uint16(x86_xSetOp), uint16(x86_JLE),
	/*9066*/ uint16(x86_xReadCb),
	/*9067*/ uint16(x86_xArgRel8),
	/*9068*/ uint16(x86_xMatch),
	/*9069*/ uint16(x86_xSetOp), uint16(x86_JG),
	/*9071*/ uint16(x86_xReadCb),
	/*9072*/ uint16(x86_xArgRel8),
	/*9073*/ uint16(x86_xMatch),
	/*9074*/ uint16(x86_xCondSlashR),
	9083, // 0
	9089, // 1
	9095, // 2
	9101, // 3
	9107, // 4
	9113, // 5
	9119, // 6
	9125, // 7
	/*9083*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9085*/ uint16(x86_xReadIb),
	/*9086*/ uint16(x86_xArgRM8),
	/*9087*/ uint16(x86_xArgImm8u),
	/*9088*/ uint16(x86_xMatch),
	/*9089*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9091*/ uint16(x86_xReadIb),
	/*9092*/ uint16(x86_xArgRM8),
	/*9093*/ uint16(x86_xArgImm8u),
	/*9094*/ uint16(x86_xMatch),
	/*9095*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9097*/ uint16(x86_xReadIb),
	/*9098*/ uint16(x86_xArgRM8),
	/*9099*/ uint16(x86_xArgImm8u),
	/*9100*/ uint16(x86_xMatch),
	/*9101*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9103*/ uint16(x86_xReadIb),
	/*9104*/ uint16(x86_xArgRM8),
	/*9105*/ uint16(x86_xArgImm8u),
	/*9106*/ uint16(x86_xMatch),
	/*9107*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9109*/ uint16(x86_xReadIb),
	/*9110*/ uint16(x86_xArgRM8),
	/*9111*/ uint16(x86_xArgImm8u),
	/*9112*/ uint16(x86_xMatch),
	/*9113*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9115*/ uint16(x86_xReadIb),
	/*9116*/ uint16(x86_xArgRM8),
	/*9117*/ uint16(x86_xArgImm8u),
	/*9118*/ uint16(x86_xMatch),
	/*9119*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9121*/ uint16(x86_xReadIb),
	/*9122*/ uint16(x86_xArgRM8),
	/*9123*/ uint16(x86_xArgImm8u),
	/*9124*/ uint16(x86_xMatch),
	/*9125*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9127*/ uint16(x86_xReadIb),
	/*9128*/ uint16(x86_xArgRM8),
	/*9129*/ uint16(x86_xArgImm8u),
	/*9130*/ uint16(x86_xMatch),
	/*9131*/ uint16(x86_xCondSlashR),
	9140, // 0
	9169, // 1
	9198, // 2
	9227, // 3
	9256, // 4
	9285, // 5
	9314, // 6
	9343, // 7
	/*9140*/ uint16(x86_xCondIs64), 9143, 9159,
	/*9143*/ uint16(x86_xCondDataSize), 9147, 9153, 0,
	/*9147*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9149*/ uint16(x86_xReadIw),
	/*9150*/ uint16(x86_xArgRM16),
	/*9151*/ uint16(x86_xArgImm16),
	/*9152*/ uint16(x86_xMatch),
	/*9153*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9155*/ uint16(x86_xReadId),
	/*9156*/ uint16(x86_xArgRM32),
	/*9157*/ uint16(x86_xArgImm32),
	/*9158*/ uint16(x86_xMatch),
	/*9159*/ uint16(x86_xCondDataSize), 9147, 9153, 9163,
	/*9163*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9165*/ uint16(x86_xReadId),
	/*9166*/ uint16(x86_xArgRM64),
	/*9167*/ uint16(x86_xArgImm32),
	/*9168*/ uint16(x86_xMatch),
	/*9169*/ uint16(x86_xCondIs64), 9172, 9188,
	/*9172*/ uint16(x86_xCondDataSize), 9176, 9182, 0,
	/*9176*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9178*/ uint16(x86_xReadIw),
	/*9179*/ uint16(x86_xArgRM16),
	/*9180*/ uint16(x86_xArgImm16),
	/*9181*/ uint16(x86_xMatch),
	/*9182*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9184*/ uint16(x86_xReadId),
	/*9185*/ uint16(x86_xArgRM32),
	/*9186*/ uint16(x86_xArgImm32),
	/*9187*/ uint16(x86_xMatch),
	/*9188*/ uint16(x86_xCondDataSize), 9176, 9182, 9192,
	/*9192*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9194*/ uint16(x86_xReadId),
	/*9195*/ uint16(x86_xArgRM64),
	/*9196*/ uint16(x86_xArgImm32),
	/*9197*/ uint16(x86_xMatch),
	/*9198*/ uint16(x86_xCondIs64), 9201, 9217,
	/*9201*/ uint16(x86_xCondDataSize), 9205, 9211, 0,
	/*9205*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9207*/ uint16(x86_xReadIw),
	/*9208*/ uint16(x86_xArgRM16),
	/*9209*/ uint16(x86_xArgImm16),
	/*9210*/ uint16(x86_xMatch),
	/*9211*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9213*/ uint16(x86_xReadId),
	/*9214*/ uint16(x86_xArgRM32),
	/*9215*/ uint16(x86_xArgImm32),
	/*9216*/ uint16(x86_xMatch),
	/*9217*/ uint16(x86_xCondDataSize), 9205, 9211, 9221,
	/*9221*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9223*/ uint16(x86_xReadId),
	/*9224*/ uint16(x86_xArgRM64),
	/*9225*/ uint16(x86_xArgImm32),
	/*9226*/ uint16(x86_xMatch),
	/*9227*/ uint16(x86_xCondIs64), 9230, 9246,
	/*9230*/ uint16(x86_xCondDataSize), 9234, 9240, 0,
	/*9234*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9236*/ uint16(x86_xReadIw),
	/*9237*/ uint16(x86_xArgRM16),
	/*9238*/ uint16(x86_xArgImm16),
	/*9239*/ uint16(x86_xMatch),
	/*9240*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9242*/ uint16(x86_xReadId),
	/*9243*/ uint16(x86_xArgRM32),
	/*9244*/ uint16(x86_xArgImm32),
	/*9245*/ uint16(x86_xMatch),
	/*9246*/ uint16(x86_xCondDataSize), 9234, 9240, 9250,
	/*9250*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9252*/ uint16(x86_xReadId),
	/*9253*/ uint16(x86_xArgRM64),
	/*9254*/ uint16(x86_xArgImm32),
	/*9255*/ uint16(x86_xMatch),
	/*9256*/ uint16(x86_xCondIs64), 9259, 9275,
	/*9259*/ uint16(x86_xCondDataSize), 9263, 9269, 0,
	/*9263*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9265*/ uint16(x86_xReadIw),
	/*9266*/ uint16(x86_xArgRM16),
	/*9267*/ uint16(x86_xArgImm16),
	/*9268*/ uint16(x86_xMatch),
	/*9269*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9271*/ uint16(x86_xReadId),
	/*9272*/ uint16(x86_xArgRM32),
	/*9273*/ uint16(x86_xArgImm32),
	/*9274*/ uint16(x86_xMatch),
	/*9275*/ uint16(x86_xCondDataSize), 9263, 9269, 9279,
	/*9279*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9281*/ uint16(x86_xReadId),
	/*9282*/ uint16(x86_xArgRM64),
	/*9283*/ uint16(x86_xArgImm32),
	/*9284*/ uint16(x86_xMatch),
	/*9285*/ uint16(x86_xCondIs64), 9288, 9304,
	/*9288*/ uint16(x86_xCondDataSize), 9292, 9298, 0,
	/*9292*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9294*/ uint16(x86_xReadIw),
	/*9295*/ uint16(x86_xArgRM16),
	/*9296*/ uint16(x86_xArgImm16),
	/*9297*/ uint16(x86_xMatch),
	/*9298*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9300*/ uint16(x86_xReadId),
	/*9301*/ uint16(x86_xArgRM32),
	/*9302*/ uint16(x86_xArgImm32),
	/*9303*/ uint16(x86_xMatch),
	/*9304*/ uint16(x86_xCondDataSize), 9292, 9298, 9308,
	/*9308*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9310*/ uint16(x86_xReadId),
	/*9311*/ uint16(x86_xArgRM64),
	/*9312*/ uint16(x86_xArgImm32),
	/*9313*/ uint16(x86_xMatch),
	/*9314*/ uint16(x86_xCondIs64), 9317, 9333,
	/*9317*/ uint16(x86_xCondDataSize), 9321, 9327, 0,
	/*9321*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9323*/ uint16(x86_xReadIw),
	/*9324*/ uint16(x86_xArgRM16),
	/*9325*/ uint16(x86_xArgImm16),
	/*9326*/ uint16(x86_xMatch),
	/*9327*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9329*/ uint16(x86_xReadId),
	/*9330*/ uint16(x86_xArgRM32),
	/*9331*/ uint16(x86_xArgImm32),
	/*9332*/ uint16(x86_xMatch),
	/*9333*/ uint16(x86_xCondDataSize), 9321, 9327, 9337,
	/*9337*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9339*/ uint16(x86_xReadId),
	/*9340*/ uint16(x86_xArgRM64),
	/*9341*/ uint16(x86_xArgImm32),
	/*9342*/ uint16(x86_xMatch),
	/*9343*/ uint16(x86_xCondIs64), 9346, 9362,
	/*9346*/ uint16(x86_xCondDataSize), 9350, 9356, 0,
	/*9350*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9352*/ uint16(x86_xReadIw),
	/*9353*/ uint16(x86_xArgRM16),
	/*9354*/ uint16(x86_xArgImm16),
	/*9355*/ uint16(x86_xMatch),
	/*9356*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9358*/ uint16(x86_xReadId),
	/*9359*/ uint16(x86_xArgRM32),
	/*9360*/ uint16(x86_xArgImm32),
	/*9361*/ uint16(x86_xMatch),
	/*9362*/ uint16(x86_xCondDataSize), 9350, 9356, 9366,
	/*9366*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9368*/ uint16(x86_xReadId),
	/*9369*/ uint16(x86_xArgRM64),
	/*9370*/ uint16(x86_xArgImm32),
	/*9371*/ uint16(x86_xMatch),
	/*9372*/ uint16(x86_xCondSlashR),
	9381, // 0
	9410, // 1
	9439, // 2
	9468, // 3
	9497, // 4
	9526, // 5
	9555, // 6
	9584, // 7
	/*9381*/ uint16(x86_xCondIs64), 9384, 9400,
	/*9384*/ uint16(x86_xCondDataSize), 9388, 9394, 0,
	/*9388*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9390*/ uint16(x86_xReadIb),
	/*9391*/ uint16(x86_xArgRM16),
	/*9392*/ uint16(x86_xArgImm8),
	/*9393*/ uint16(x86_xMatch),
	/*9394*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9396*/ uint16(x86_xReadIb),
	/*9397*/ uint16(x86_xArgRM32),
	/*9398*/ uint16(x86_xArgImm8),
	/*9399*/ uint16(x86_xMatch),
	/*9400*/ uint16(x86_xCondDataSize), 9388, 9394, 9404,
	/*9404*/ uint16(x86_xSetOp), uint16(x86_ADD),
	/*9406*/ uint16(x86_xReadIb),
	/*9407*/ uint16(x86_xArgRM64),
	/*9408*/ uint16(x86_xArgImm8),
	/*9409*/ uint16(x86_xMatch),
	/*9410*/ uint16(x86_xCondIs64), 9413, 9429,
	/*9413*/ uint16(x86_xCondDataSize), 9417, 9423, 0,
	/*9417*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9419*/ uint16(x86_xReadIb),
	/*9420*/ uint16(x86_xArgRM16),
	/*9421*/ uint16(x86_xArgImm8),
	/*9422*/ uint16(x86_xMatch),
	/*9423*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9425*/ uint16(x86_xReadIb),
	/*9426*/ uint16(x86_xArgRM32),
	/*9427*/ uint16(x86_xArgImm8),
	/*9428*/ uint16(x86_xMatch),
	/*9429*/ uint16(x86_xCondDataSize), 9417, 9423, 9433,
	/*9433*/ uint16(x86_xSetOp), uint16(x86_OR),
	/*9435*/ uint16(x86_xReadIb),
	/*9436*/ uint16(x86_xArgRM64),
	/*9437*/ uint16(x86_xArgImm8),
	/*9438*/ uint16(x86_xMatch),
	/*9439*/ uint16(x86_xCondIs64), 9442, 9458,
	/*9442*/ uint16(x86_xCondDataSize), 9446, 9452, 0,
	/*9446*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9448*/ uint16(x86_xReadIb),
	/*9449*/ uint16(x86_xArgRM16),
	/*9450*/ uint16(x86_xArgImm8),
	/*9451*/ uint16(x86_xMatch),
	/*9452*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9454*/ uint16(x86_xReadIb),
	/*9455*/ uint16(x86_xArgRM32),
	/*9456*/ uint16(x86_xArgImm8),
	/*9457*/ uint16(x86_xMatch),
	/*9458*/ uint16(x86_xCondDataSize), 9446, 9452, 9462,
	/*9462*/ uint16(x86_xSetOp), uint16(x86_ADC),
	/*9464*/ uint16(x86_xReadIb),
	/*9465*/ uint16(x86_xArgRM64),
	/*9466*/ uint16(x86_xArgImm8),
	/*9467*/ uint16(x86_xMatch),
	/*9468*/ uint16(x86_xCondIs64), 9471, 9487,
	/*9471*/ uint16(x86_xCondDataSize), 9475, 9481, 0,
	/*9475*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9477*/ uint16(x86_xReadIb),
	/*9478*/ uint16(x86_xArgRM16),
	/*9479*/ uint16(x86_xArgImm8),
	/*9480*/ uint16(x86_xMatch),
	/*9481*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9483*/ uint16(x86_xReadIb),
	/*9484*/ uint16(x86_xArgRM32),
	/*9485*/ uint16(x86_xArgImm8),
	/*9486*/ uint16(x86_xMatch),
	/*9487*/ uint16(x86_xCondDataSize), 9475, 9481, 9491,
	/*9491*/ uint16(x86_xSetOp), uint16(x86_SBB),
	/*9493*/ uint16(x86_xReadIb),
	/*9494*/ uint16(x86_xArgRM64),
	/*9495*/ uint16(x86_xArgImm8),
	/*9496*/ uint16(x86_xMatch),
	/*9497*/ uint16(x86_xCondIs64), 9500, 9516,
	/*9500*/ uint16(x86_xCondDataSize), 9504, 9510, 0,
	/*9504*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9506*/ uint16(x86_xReadIb),
	/*9507*/ uint16(x86_xArgRM16),
	/*9508*/ uint16(x86_xArgImm8),
	/*9509*/ uint16(x86_xMatch),
	/*9510*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9512*/ uint16(x86_xReadIb),
	/*9513*/ uint16(x86_xArgRM32),
	/*9514*/ uint16(x86_xArgImm8),
	/*9515*/ uint16(x86_xMatch),
	/*9516*/ uint16(x86_xCondDataSize), 9504, 9510, 9520,
	/*9520*/ uint16(x86_xSetOp), uint16(x86_AND),
	/*9522*/ uint16(x86_xReadIb),
	/*9523*/ uint16(x86_xArgRM64),
	/*9524*/ uint16(x86_xArgImm8),
	/*9525*/ uint16(x86_xMatch),
	/*9526*/ uint16(x86_xCondIs64), 9529, 9545,
	/*9529*/ uint16(x86_xCondDataSize), 9533, 9539, 0,
	/*9533*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9535*/ uint16(x86_xReadIb),
	/*9536*/ uint16(x86_xArgRM16),
	/*9537*/ uint16(x86_xArgImm8),
	/*9538*/ uint16(x86_xMatch),
	/*9539*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9541*/ uint16(x86_xReadIb),
	/*9542*/ uint16(x86_xArgRM32),
	/*9543*/ uint16(x86_xArgImm8),
	/*9544*/ uint16(x86_xMatch),
	/*9545*/ uint16(x86_xCondDataSize), 9533, 9539, 9549,
	/*9549*/ uint16(x86_xSetOp), uint16(x86_SUB),
	/*9551*/ uint16(x86_xReadIb),
	/*9552*/ uint16(x86_xArgRM64),
	/*9553*/ uint16(x86_xArgImm8),
	/*9554*/ uint16(x86_xMatch),
	/*9555*/ uint16(x86_xCondIs64), 9558, 9574,
	/*9558*/ uint16(x86_xCondDataSize), 9562, 9568, 0,
	/*9562*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9564*/ uint16(x86_xReadIb),
	/*9565*/ uint16(x86_xArgRM16),
	/*9566*/ uint16(x86_xArgImm8),
	/*9567*/ uint16(x86_xMatch),
	/*9568*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9570*/ uint16(x86_xReadIb),
	/*9571*/ uint16(x86_xArgRM32),
	/*9572*/ uint16(x86_xArgImm8),
	/*9573*/ uint16(x86_xMatch),
	/*9574*/ uint16(x86_xCondDataSize), 9562, 9568, 9578,
	/*9578*/ uint16(x86_xSetOp), uint16(x86_XOR),
	/*9580*/ uint16(x86_xReadIb),
	/*9581*/ uint16(x86_xArgRM64),
	/*9582*/ uint16(x86_xArgImm8),
	/*9583*/ uint16(x86_xMatch),
	/*9584*/ uint16(x86_xCondIs64), 9587, 9603,
	/*9587*/ uint16(x86_xCondDataSize), 9591, 9597, 0,
	/*9591*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9593*/ uint16(x86_xReadIb),
	/*9594*/ uint16(x86_xArgRM16),
	/*9595*/ uint16(x86_xArgImm8),
	/*9596*/ uint16(x86_xMatch),
	/*9597*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9599*/ uint16(x86_xReadIb),
	/*9600*/ uint16(x86_xArgRM32),
	/*9601*/ uint16(x86_xArgImm8),
	/*9602*/ uint16(x86_xMatch),
	/*9603*/ uint16(x86_xCondDataSize), 9591, 9597, 9607,
	/*9607*/ uint16(x86_xSetOp), uint16(x86_CMP),
	/*9609*/ uint16(x86_xReadIb),
	/*9610*/ uint16(x86_xArgRM64),
	/*9611*/ uint16(x86_xArgImm8),
	/*9612*/ uint16(x86_xMatch),
	/*9613*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*9615*/ uint16(x86_xReadSlashR),
	/*9616*/ uint16(x86_xArgRM8),
	/*9617*/ uint16(x86_xArgR8),
	/*9618*/ uint16(x86_xMatch),
	/*9619*/ uint16(x86_xCondIs64), 9622, 9638,
	/*9622*/ uint16(x86_xCondDataSize), 9626, 9632, 0,
	/*9626*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*9628*/ uint16(x86_xReadSlashR),
	/*9629*/ uint16(x86_xArgRM16),
	/*9630*/ uint16(x86_xArgR16),
	/*9631*/ uint16(x86_xMatch),
	/*9632*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*9634*/ uint16(x86_xReadSlashR),
	/*9635*/ uint16(x86_xArgRM32),
	/*9636*/ uint16(x86_xArgR32),
	/*9637*/ uint16(x86_xMatch),
	/*9638*/ uint16(x86_xCondDataSize), 9626, 9632, 9642,
	/*9642*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*9644*/ uint16(x86_xReadSlashR),
	/*9645*/ uint16(x86_xArgRM64),
	/*9646*/ uint16(x86_xArgR64),
	/*9647*/ uint16(x86_xMatch),
	/*9648*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9650*/ uint16(x86_xReadSlashR),
	/*9651*/ uint16(x86_xArgRM8),
	/*9652*/ uint16(x86_xArgR8),
	/*9653*/ uint16(x86_xMatch),
	/*9654*/ uint16(x86_xCondIs64), 9657, 9673,
	/*9657*/ uint16(x86_xCondDataSize), 9661, 9667, 0,
	/*9661*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9663*/ uint16(x86_xReadSlashR),
	/*9664*/ uint16(x86_xArgRM16),
	/*9665*/ uint16(x86_xArgR16),
	/*9666*/ uint16(x86_xMatch),
	/*9667*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9669*/ uint16(x86_xReadSlashR),
	/*9670*/ uint16(x86_xArgRM32),
	/*9671*/ uint16(x86_xArgR32),
	/*9672*/ uint16(x86_xMatch),
	/*9673*/ uint16(x86_xCondDataSize), 9661, 9667, 9677,
	/*9677*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9679*/ uint16(x86_xReadSlashR),
	/*9680*/ uint16(x86_xArgRM64),
	/*9681*/ uint16(x86_xArgR64),
	/*9682*/ uint16(x86_xMatch),
	/*9683*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9685*/ uint16(x86_xReadSlashR),
	/*9686*/ uint16(x86_xArgRM8),
	/*9687*/ uint16(x86_xArgR8),
	/*9688*/ uint16(x86_xMatch),
	/*9689*/ uint16(x86_xCondDataSize), 9693, 9699, 9705,
	/*9693*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9695*/ uint16(x86_xReadSlashR),
	/*9696*/ uint16(x86_xArgRM16),
	/*9697*/ uint16(x86_xArgR16),
	/*9698*/ uint16(x86_xMatch),
	/*9699*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9701*/ uint16(x86_xReadSlashR),
	/*9702*/ uint16(x86_xArgRM32),
	/*9703*/ uint16(x86_xArgR32),
	/*9704*/ uint16(x86_xMatch),
	/*9705*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9707*/ uint16(x86_xReadSlashR),
	/*9708*/ uint16(x86_xArgRM64),
	/*9709*/ uint16(x86_xArgR64),
	/*9710*/ uint16(x86_xMatch),
	/*9711*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9713*/ uint16(x86_xReadSlashR),
	/*9714*/ uint16(x86_xArgR8),
	/*9715*/ uint16(x86_xArgRM8),
	/*9716*/ uint16(x86_xMatch),
	/*9717*/ uint16(x86_xCondDataSize), 9721, 9727, 9733,
	/*9721*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9723*/ uint16(x86_xReadSlashR),
	/*9724*/ uint16(x86_xArgR16),
	/*9725*/ uint16(x86_xArgRM16),
	/*9726*/ uint16(x86_xMatch),
	/*9727*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9729*/ uint16(x86_xReadSlashR),
	/*9730*/ uint16(x86_xArgR32),
	/*9731*/ uint16(x86_xArgRM32),
	/*9732*/ uint16(x86_xMatch),
	/*9733*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9735*/ uint16(x86_xReadSlashR),
	/*9736*/ uint16(x86_xArgR64),
	/*9737*/ uint16(x86_xArgRM64),
	/*9738*/ uint16(x86_xMatch),
	/*9739*/ uint16(x86_xCondIs64), 9742, 9758,
	/*9742*/ uint16(x86_xCondDataSize), 9746, 9752, 0,
	/*9746*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9748*/ uint16(x86_xReadSlashR),
	/*9749*/ uint16(x86_xArgRM16),
	/*9750*/ uint16(x86_xArgSreg),
	/*9751*/ uint16(x86_xMatch),
	/*9752*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9754*/ uint16(x86_xReadSlashR),
	/*9755*/ uint16(x86_xArgR32M16),
	/*9756*/ uint16(x86_xArgSreg),
	/*9757*/ uint16(x86_xMatch),
	/*9758*/ uint16(x86_xCondDataSize), 9746, 9752, 9762,
	/*9762*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9764*/ uint16(x86_xReadSlashR),
	/*9765*/ uint16(x86_xArgR64M16),
	/*9766*/ uint16(x86_xArgSreg),
	/*9767*/ uint16(x86_xMatch),
	/*9768*/ uint16(x86_xCondIs64), 9771, 9787,
	/*9771*/ uint16(x86_xCondDataSize), 9775, 9781, 0,
	/*9775*/ uint16(x86_xSetOp), uint16(x86_LEA),
	/*9777*/ uint16(x86_xReadSlashR),
	/*9778*/ uint16(x86_xArgR16),
	/*9779*/ uint16(x86_xArgM),
	/*9780*/ uint16(x86_xMatch),
	/*9781*/ uint16(x86_xSetOp), uint16(x86_LEA),
	/*9783*/ uint16(x86_xReadSlashR),
	/*9784*/ uint16(x86_xArgR32),
	/*9785*/ uint16(x86_xArgM),
	/*9786*/ uint16(x86_xMatch),
	/*9787*/ uint16(x86_xCondDataSize), 9775, 9781, 9791,
	/*9791*/ uint16(x86_xSetOp), uint16(x86_LEA),
	/*9793*/ uint16(x86_xReadSlashR),
	/*9794*/ uint16(x86_xArgR64),
	/*9795*/ uint16(x86_xArgM),
	/*9796*/ uint16(x86_xMatch),
	/*9797*/ uint16(x86_xCondIs64), 9800, 9816,
	/*9800*/ uint16(x86_xCondDataSize), 9804, 9810, 0,
	/*9804*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9806*/ uint16(x86_xReadSlashR),
	/*9807*/ uint16(x86_xArgSreg),
	/*9808*/ uint16(x86_xArgRM16),
	/*9809*/ uint16(x86_xMatch),
	/*9810*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9812*/ uint16(x86_xReadSlashR),
	/*9813*/ uint16(x86_xArgSreg),
	/*9814*/ uint16(x86_xArgR32M16),
	/*9815*/ uint16(x86_xMatch),
	/*9816*/ uint16(x86_xCondDataSize), 9804, 9810, 9820,
	/*9820*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*9822*/ uint16(x86_xReadSlashR),
	/*9823*/ uint16(x86_xArgSreg),
	/*9824*/ uint16(x86_xArgR64M16),
	/*9825*/ uint16(x86_xMatch),
	/*9826*/ uint16(x86_xCondSlashR),
	9835, // 0
	0,    // 1
	0,    // 2
	0,    // 3
	0,    // 4
	0,    // 5
	0,    // 6
	0,    // 7
	/*9835*/ uint16(x86_xCondIs64), 9838, 9850,
	/*9838*/ uint16(x86_xCondDataSize), 9842, 9846, 0,
	/*9842*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*9844*/ uint16(x86_xArgRM16),
	/*9845*/ uint16(x86_xMatch),
	/*9846*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*9848*/ uint16(x86_xArgRM32),
	/*9849*/ uint16(x86_xMatch),
	/*9850*/ uint16(x86_xCondDataSize), 9842, 9854, 9858,
	/*9854*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*9856*/ uint16(x86_xArgRM64),
	/*9857*/ uint16(x86_xMatch),
	/*9858*/ uint16(x86_xSetOp), uint16(x86_POP),
	/*9860*/ uint16(x86_xArgRM64),
	/*9861*/ uint16(x86_xMatch),
	/*9862*/ uint16(x86_xCondIs64), 9865, 9879,
	/*9865*/ uint16(x86_xCondDataSize), 9869, 9874, 0,
	/*9869*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9871*/ uint16(x86_xArgR16op),
	/*9872*/ uint16(x86_xArgAX),
	/*9873*/ uint16(x86_xMatch),
	/*9874*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9876*/ uint16(x86_xArgR32op),
	/*9877*/ uint16(x86_xArgEAX),
	/*9878*/ uint16(x86_xMatch),
	/*9879*/ uint16(x86_xCondDataSize), 9869, 9874, 9883,
	/*9883*/ uint16(x86_xSetOp), uint16(x86_XCHG),
	/*9885*/ uint16(x86_xArgR64op),
	/*9886*/ uint16(x86_xArgRAX),
	/*9887*/ uint16(x86_xMatch),
	/*9888*/ uint16(x86_xCondIs64), 9891, 9901,
	/*9891*/ uint16(x86_xCondDataSize), 9895, 9898, 0,
	/*9895*/ uint16(x86_xSetOp), uint16(x86_CBW),
	/*9897*/ uint16(x86_xMatch),
	/*9898*/ uint16(x86_xSetOp), uint16(x86_CWDE),
	/*9900*/ uint16(x86_xMatch),
	/*9901*/ uint16(x86_xCondDataSize), 9895, 9898, 9905,
	/*9905*/ uint16(x86_xSetOp), uint16(x86_CDQE),
	/*9907*/ uint16(x86_xMatch),
	/*9908*/ uint16(x86_xCondIs64), 9911, 9921,
	/*9911*/ uint16(x86_xCondDataSize), 9915, 9918, 0,
	/*9915*/ uint16(x86_xSetOp), uint16(x86_CWD),
	/*9917*/ uint16(x86_xMatch),
	/*9918*/ uint16(x86_xSetOp), uint16(x86_CDQ),
	/*9920*/ uint16(x86_xMatch),
	/*9921*/ uint16(x86_xCondDataSize), 9915, 9918, 9925,
	/*9925*/ uint16(x86_xSetOp), uint16(x86_CQO),
	/*9927*/ uint16(x86_xMatch),
	/*9928*/ uint16(x86_xCondIs64), 9931, 0,
	/*9931*/ uint16(x86_xCondDataSize), 9935, 9940, 0,
	/*9935*/ uint16(x86_xSetOp), uint16(x86_LCALL),
	/*9937*/ uint16(x86_xReadCd),
	/*9938*/ uint16(x86_xArgPtr16colon16),
	/*9939*/ uint16(x86_xMatch),
	/*9940*/ uint16(x86_xSetOp), uint16(x86_LCALL),
	/*9942*/ uint16(x86_xReadCp),
	/*9943*/ uint16(x86_xArgPtr16colon32),
	/*9944*/ uint16(x86_xMatch),
	/*9945*/ uint16(x86_xSetOp), uint16(x86_FWAIT),
	/*9947*/ uint16(x86_xMatch),
	/*9948*/ uint16(x86_xCondIs64), 9951, 9961,
	/*9951*/ uint16(x86_xCondDataSize), 9955, 9958, 0,
	/*9955*/ uint16(x86_xSetOp), uint16(x86_PUSHF),
	/*9957*/ uint16(x86_xMatch),
	/*9958*/ uint16(x86_xSetOp), uint16(x86_PUSHFD),
	/*9960*/ uint16(x86_xMatch),
	/*9961*/ uint16(x86_xCondDataSize), 9955, 9965, 9968,
	/*9965*/ uint16(x86_xSetOp), uint16(x86_PUSHFQ),
	/*9967*/ uint16(x86_xMatch),
	/*9968*/ uint16(x86_xSetOp), uint16(x86_PUSHFQ),
	/*9970*/ uint16(x86_xMatch),
	/*9971*/ uint16(x86_xCondIs64), 9974, 9984,
	/*9974*/ uint16(x86_xCondDataSize), 9978, 9981, 0,
	/*9978*/ uint16(x86_xSetOp), uint16(x86_POPF),
	/*9980*/ uint16(x86_xMatch),
	/*9981*/ uint16(x86_xSetOp), uint16(x86_POPFD),
	/*9983*/ uint16(x86_xMatch),
	/*9984*/ uint16(x86_xCondDataSize), 9978, 9988, 9991,
	/*9988*/ uint16(x86_xSetOp), uint16(x86_POPFQ),
	/*9990*/ uint16(x86_xMatch),
	/*9991*/ uint16(x86_xSetOp), uint16(x86_POPFQ),
	/*9993*/ uint16(x86_xMatch),
	/*9994*/ uint16(x86_xSetOp), uint16(x86_SAHF),
	/*9996*/ uint16(x86_xMatch),
	/*9997*/ uint16(x86_xSetOp), uint16(x86_LAHF),
	/*9999*/ uint16(x86_xMatch),
	/*10000*/ uint16(x86_xCondIs64), 10003, 10009,
	/*10003*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10005*/ uint16(x86_xReadCm),
	/*10006*/ uint16(x86_xArgAL),
	/*10007*/ uint16(x86_xArgMoffs8),
	/*10008*/ uint16(x86_xMatch),
	/*10009*/ uint16(x86_xCondDataSize), 10003, 10003, 10013,
	/*10013*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10015*/ uint16(x86_xReadCm),
	/*10016*/ uint16(x86_xArgAL),
	/*10017*/ uint16(x86_xArgMoffs8),
	/*10018*/ uint16(x86_xMatch),
	/*10019*/ uint16(x86_xCondDataSize), 10023, 10029, 10035,
	/*10023*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10025*/ uint16(x86_xReadCm),
	/*10026*/ uint16(x86_xArgAX),
	/*10027*/ uint16(x86_xArgMoffs16),
	/*10028*/ uint16(x86_xMatch),
	/*10029*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10031*/ uint16(x86_xReadCm),
	/*10032*/ uint16(x86_xArgEAX),
	/*10033*/ uint16(x86_xArgMoffs32),
	/*10034*/ uint16(x86_xMatch),
	/*10035*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10037*/ uint16(x86_xReadCm),
	/*10038*/ uint16(x86_xArgRAX),
	/*10039*/ uint16(x86_xArgMoffs64),
	/*10040*/ uint16(x86_xMatch),
	/*10041*/ uint16(x86_xCondIs64), 10044, 10050,
	/*10044*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10046*/ uint16(x86_xReadCm),
	/*10047*/ uint16(x86_xArgMoffs8),
	/*10048*/ uint16(x86_xArgAL),
	/*10049*/ uint16(x86_xMatch),
	/*10050*/ uint16(x86_xCondDataSize), 10044, 10044, 10054,
	/*10054*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10056*/ uint16(x86_xReadCm),
	/*10057*/ uint16(x86_xArgMoffs8),
	/*10058*/ uint16(x86_xArgAL),
	/*10059*/ uint16(x86_xMatch),
	/*10060*/ uint16(x86_xCondDataSize), 10064, 10070, 10076,
	/*10064*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10066*/ uint16(x86_xReadCm),
	/*10067*/ uint16(x86_xArgMoffs16),
	/*10068*/ uint16(x86_xArgAX),
	/*10069*/ uint16(x86_xMatch),
	/*10070*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10072*/ uint16(x86_xReadCm),
	/*10073*/ uint16(x86_xArgMoffs32),
	/*10074*/ uint16(x86_xArgEAX),
	/*10075*/ uint16(x86_xMatch),
	/*10076*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10078*/ uint16(x86_xReadCm),
	/*10079*/ uint16(x86_xArgMoffs64),
	/*10080*/ uint16(x86_xArgRAX),
	/*10081*/ uint16(x86_xMatch),
	/*10082*/ uint16(x86_xSetOp), uint16(x86_MOVSB),
	/*10084*/ uint16(x86_xMatch),
	/*10085*/ uint16(x86_xCondIs64), 10088, 10098,
	/*10088*/ uint16(x86_xCondDataSize), 10092, 10095, 0,
	/*10092*/ uint16(x86_xSetOp), uint16(x86_MOVSW),
	/*10094*/ uint16(x86_xMatch),
	/*10095*/ uint16(x86_xSetOp), uint16(x86_MOVSD),
	/*10097*/ uint16(x86_xMatch),
	/*10098*/ uint16(x86_xCondDataSize), 10092, 10095, 10102,
	/*10102*/ uint16(x86_xSetOp), uint16(x86_MOVSQ),
	/*10104*/ uint16(x86_xMatch),
	/*10105*/ uint16(x86_xSetOp), uint16(x86_CMPSB),
	/*10107*/ uint16(x86_xMatch),
	/*10108*/ uint16(x86_xCondIs64), 10111, 10121,
	/*10111*/ uint16(x86_xCondDataSize), 10115, 10118, 0,
	/*10115*/ uint16(x86_xSetOp), uint16(x86_CMPSW),
	/*10117*/ uint16(x86_xMatch),
	/*10118*/ uint16(x86_xSetOp), uint16(x86_CMPSD),
	/*10120*/ uint16(x86_xMatch),
	/*10121*/ uint16(x86_xCondDataSize), 10115, 10118, 10125,
	/*10125*/ uint16(x86_xSetOp), uint16(x86_CMPSQ),
	/*10127*/ uint16(x86_xMatch),
	/*10128*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*10130*/ uint16(x86_xReadIb),
	/*10131*/ uint16(x86_xArgAL),
	/*10132*/ uint16(x86_xArgImm8u),
	/*10133*/ uint16(x86_xMatch),
	/*10134*/ uint16(x86_xCondIs64), 10137, 10153,
	/*10137*/ uint16(x86_xCondDataSize), 10141, 10147, 0,
	/*10141*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*10143*/ uint16(x86_xReadIw),
	/*10144*/ uint16(x86_xArgAX),
	/*10145*/ uint16(x86_xArgImm16),
	/*10146*/ uint16(x86_xMatch),
	/*10147*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*10149*/ uint16(x86_xReadId),
	/*10150*/ uint16(x86_xArgEAX),
	/*10151*/ uint16(x86_xArgImm32),
	/*10152*/ uint16(x86_xMatch),
	/*10153*/ uint16(x86_xCondDataSize), 10141, 10147, 10157,
	/*10157*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*10159*/ uint16(x86_xReadId),
	/*10160*/ uint16(x86_xArgRAX),
	/*10161*/ uint16(x86_xArgImm32),
	/*10162*/ uint16(x86_xMatch),
	/*10163*/ uint16(x86_xSetOp), uint16(x86_STOSB),
	/*10165*/ uint16(x86_xMatch),
	/*10166*/ uint16(x86_xCondIs64), 10169, 10179,
	/*10169*/ uint16(x86_xCondDataSize), 10173, 10176, 0,
	/*10173*/ uint16(x86_xSetOp), uint16(x86_STOSW),
	/*10175*/ uint16(x86_xMatch),
	/*10176*/ uint16(x86_xSetOp), uint16(x86_STOSD),
	/*10178*/ uint16(x86_xMatch),
	/*10179*/ uint16(x86_xCondDataSize), 10173, 10176, 10183,
	/*10183*/ uint16(x86_xSetOp), uint16(x86_STOSQ),
	/*10185*/ uint16(x86_xMatch),
	/*10186*/ uint16(x86_xSetOp), uint16(x86_LODSB),
	/*10188*/ uint16(x86_xMatch),
	/*10189*/ uint16(x86_xCondIs64), 10192, 10202,
	/*10192*/ uint16(x86_xCondDataSize), 10196, 10199, 0,
	/*10196*/ uint16(x86_xSetOp), uint16(x86_LODSW),
	/*10198*/ uint16(x86_xMatch),
	/*10199*/ uint16(x86_xSetOp), uint16(x86_LODSD),
	/*10201*/ uint16(x86_xMatch),
	/*10202*/ uint16(x86_xCondDataSize), 10196, 10199, 10206,
	/*10206*/ uint16(x86_xSetOp), uint16(x86_LODSQ),
	/*10208*/ uint16(x86_xMatch),
	/*10209*/ uint16(x86_xSetOp), uint16(x86_SCASB),
	/*10211*/ uint16(x86_xMatch),
	/*10212*/ uint16(x86_xCondIs64), 10215, 10225,
	/*10215*/ uint16(x86_xCondDataSize), 10219, 10222, 0,
	/*10219*/ uint16(x86_xSetOp), uint16(x86_SCASW),
	/*10221*/ uint16(x86_xMatch),
	/*10222*/ uint16(x86_xSetOp), uint16(x86_SCASD),
	/*10224*/ uint16(x86_xMatch),
	/*10225*/ uint16(x86_xCondDataSize), 10219, 10222, 10229,
	/*10229*/ uint16(x86_xSetOp), uint16(x86_SCASQ),
	/*10231*/ uint16(x86_xMatch),
	/*10232*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10234*/ uint16(x86_xReadIb),
	/*10235*/ uint16(x86_xArgR8op),
	/*10236*/ uint16(x86_xArgImm8u),
	/*10237*/ uint16(x86_xMatch),
	/*10238*/ uint16(x86_xCondIs64), 10241, 10257,
	/*10241*/ uint16(x86_xCondDataSize), 10245, 10251, 0,
	/*10245*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10247*/ uint16(x86_xReadIw),
	/*10248*/ uint16(x86_xArgR16op),
	/*10249*/ uint16(x86_xArgImm16),
	/*10250*/ uint16(x86_xMatch),
	/*10251*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10253*/ uint16(x86_xReadId),
	/*10254*/ uint16(x86_xArgR32op),
	/*10255*/ uint16(x86_xArgImm32),
	/*10256*/ uint16(x86_xMatch),
	/*10257*/ uint16(x86_xCondDataSize), 10245, 10251, 10261,
	/*10261*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10263*/ uint16(x86_xReadIo),
	/*10264*/ uint16(x86_xArgR64op),
	/*10265*/ uint16(x86_xArgImm64),
	/*10266*/ uint16(x86_xMatch),
	/*10267*/ uint16(x86_xCondSlashR),
	10276, // 0
	10282, // 1
	10288, // 2
	10294, // 3
	10300, // 4
	10306, // 5
	0,     // 6
	10312, // 7
	/*10276*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10278*/ uint16(x86_xReadIb),
	/*10279*/ uint16(x86_xArgRM8),
	/*10280*/ uint16(x86_xArgImm8u),
	/*10281*/ uint16(x86_xMatch),
	/*10282*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10284*/ uint16(x86_xReadIb),
	/*10285*/ uint16(x86_xArgRM8),
	/*10286*/ uint16(x86_xArgImm8u),
	/*10287*/ uint16(x86_xMatch),
	/*10288*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10290*/ uint16(x86_xReadIb),
	/*10291*/ uint16(x86_xArgRM8),
	/*10292*/ uint16(x86_xArgImm8u),
	/*10293*/ uint16(x86_xMatch),
	/*10294*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10296*/ uint16(x86_xReadIb),
	/*10297*/ uint16(x86_xArgRM8),
	/*10298*/ uint16(x86_xArgImm8u),
	/*10299*/ uint16(x86_xMatch),
	/*10300*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10302*/ uint16(x86_xReadIb),
	/*10303*/ uint16(x86_xArgRM8),
	/*10304*/ uint16(x86_xArgImm8u),
	/*10305*/ uint16(x86_xMatch),
	/*10306*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10308*/ uint16(x86_xReadIb),
	/*10309*/ uint16(x86_xArgRM8),
	/*10310*/ uint16(x86_xArgImm8u),
	/*10311*/ uint16(x86_xMatch),
	/*10312*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10314*/ uint16(x86_xReadIb),
	/*10315*/ uint16(x86_xArgRM8),
	/*10316*/ uint16(x86_xArgImm8u),
	/*10317*/ uint16(x86_xMatch),
	/*10318*/ uint16(x86_xCondSlashR),
	10327, // 0
	10349, // 1
	10371, // 2
	10400, // 3
	10429, // 4
	10458, // 5
	0,     // 6
	10487, // 7
	/*10327*/ uint16(x86_xCondDataSize), 10331, 10337, 10343,
	/*10331*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10333*/ uint16(x86_xReadIb),
	/*10334*/ uint16(x86_xArgRM16),
	/*10335*/ uint16(x86_xArgImm8u),
	/*10336*/ uint16(x86_xMatch),
	/*10337*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10339*/ uint16(x86_xReadIb),
	/*10340*/ uint16(x86_xArgRM32),
	/*10341*/ uint16(x86_xArgImm8u),
	/*10342*/ uint16(x86_xMatch),
	/*10343*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10345*/ uint16(x86_xReadIb),
	/*10346*/ uint16(x86_xArgRM64),
	/*10347*/ uint16(x86_xArgImm8u),
	/*10348*/ uint16(x86_xMatch),
	/*10349*/ uint16(x86_xCondDataSize), 10353, 10359, 10365,
	/*10353*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10355*/ uint16(x86_xReadIb),
	/*10356*/ uint16(x86_xArgRM16),
	/*10357*/ uint16(x86_xArgImm8u),
	/*10358*/ uint16(x86_xMatch),
	/*10359*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10361*/ uint16(x86_xReadIb),
	/*10362*/ uint16(x86_xArgRM32),
	/*10363*/ uint16(x86_xArgImm8u),
	/*10364*/ uint16(x86_xMatch),
	/*10365*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10367*/ uint16(x86_xReadIb),
	/*10368*/ uint16(x86_xArgRM64),
	/*10369*/ uint16(x86_xArgImm8u),
	/*10370*/ uint16(x86_xMatch),
	/*10371*/ uint16(x86_xCondIs64), 10374, 10390,
	/*10374*/ uint16(x86_xCondDataSize), 10378, 10384, 0,
	/*10378*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10380*/ uint16(x86_xReadIb),
	/*10381*/ uint16(x86_xArgRM16),
	/*10382*/ uint16(x86_xArgImm8u),
	/*10383*/ uint16(x86_xMatch),
	/*10384*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10386*/ uint16(x86_xReadIb),
	/*10387*/ uint16(x86_xArgRM32),
	/*10388*/ uint16(x86_xArgImm8u),
	/*10389*/ uint16(x86_xMatch),
	/*10390*/ uint16(x86_xCondDataSize), 10378, 10384, 10394,
	/*10394*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10396*/ uint16(x86_xReadIb),
	/*10397*/ uint16(x86_xArgRM64),
	/*10398*/ uint16(x86_xArgImm8u),
	/*10399*/ uint16(x86_xMatch),
	/*10400*/ uint16(x86_xCondIs64), 10403, 10419,
	/*10403*/ uint16(x86_xCondDataSize), 10407, 10413, 0,
	/*10407*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10409*/ uint16(x86_xReadIb),
	/*10410*/ uint16(x86_xArgRM16),
	/*10411*/ uint16(x86_xArgImm8u),
	/*10412*/ uint16(x86_xMatch),
	/*10413*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10415*/ uint16(x86_xReadIb),
	/*10416*/ uint16(x86_xArgRM32),
	/*10417*/ uint16(x86_xArgImm8u),
	/*10418*/ uint16(x86_xMatch),
	/*10419*/ uint16(x86_xCondDataSize), 10407, 10413, 10423,
	/*10423*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10425*/ uint16(x86_xReadIb),
	/*10426*/ uint16(x86_xArgRM64),
	/*10427*/ uint16(x86_xArgImm8u),
	/*10428*/ uint16(x86_xMatch),
	/*10429*/ uint16(x86_xCondIs64), 10432, 10448,
	/*10432*/ uint16(x86_xCondDataSize), 10436, 10442, 0,
	/*10436*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10438*/ uint16(x86_xReadIb),
	/*10439*/ uint16(x86_xArgRM16),
	/*10440*/ uint16(x86_xArgImm8u),
	/*10441*/ uint16(x86_xMatch),
	/*10442*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10444*/ uint16(x86_xReadIb),
	/*10445*/ uint16(x86_xArgRM32),
	/*10446*/ uint16(x86_xArgImm8u),
	/*10447*/ uint16(x86_xMatch),
	/*10448*/ uint16(x86_xCondDataSize), 10436, 10442, 10452,
	/*10452*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10454*/ uint16(x86_xReadIb),
	/*10455*/ uint16(x86_xArgRM64),
	/*10456*/ uint16(x86_xArgImm8u),
	/*10457*/ uint16(x86_xMatch),
	/*10458*/ uint16(x86_xCondIs64), 10461, 10477,
	/*10461*/ uint16(x86_xCondDataSize), 10465, 10471, 0,
	/*10465*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10467*/ uint16(x86_xReadIb),
	/*10468*/ uint16(x86_xArgRM16),
	/*10469*/ uint16(x86_xArgImm8u),
	/*10470*/ uint16(x86_xMatch),
	/*10471*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10473*/ uint16(x86_xReadIb),
	/*10474*/ uint16(x86_xArgRM32),
	/*10475*/ uint16(x86_xArgImm8u),
	/*10476*/ uint16(x86_xMatch),
	/*10477*/ uint16(x86_xCondDataSize), 10465, 10471, 10481,
	/*10481*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10483*/ uint16(x86_xReadIb),
	/*10484*/ uint16(x86_xArgRM64),
	/*10485*/ uint16(x86_xArgImm8u),
	/*10486*/ uint16(x86_xMatch),
	/*10487*/ uint16(x86_xCondIs64), 10490, 10506,
	/*10490*/ uint16(x86_xCondDataSize), 10494, 10500, 0,
	/*10494*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10496*/ uint16(x86_xReadIb),
	/*10497*/ uint16(x86_xArgRM16),
	/*10498*/ uint16(x86_xArgImm8u),
	/*10499*/ uint16(x86_xMatch),
	/*10500*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10502*/ uint16(x86_xReadIb),
	/*10503*/ uint16(x86_xArgRM32),
	/*10504*/ uint16(x86_xArgImm8u),
	/*10505*/ uint16(x86_xMatch),
	/*10506*/ uint16(x86_xCondDataSize), 10494, 10500, 10510,
	/*10510*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10512*/ uint16(x86_xReadIb),
	/*10513*/ uint16(x86_xArgRM64),
	/*10514*/ uint16(x86_xArgImm8u),
	/*10515*/ uint16(x86_xMatch),
	/*10516*/ uint16(x86_xSetOp), uint16(x86_RET),
	/*10518*/ uint16(x86_xReadIw),
	/*10519*/ uint16(x86_xArgImm16u),
	/*10520*/ uint16(x86_xMatch),
	/*10521*/ uint16(x86_xSetOp), uint16(x86_RET),
	/*10523*/ uint16(x86_xMatch),
	/*10524*/ uint16(x86_xCondIs64), 10527, 0,
	/*10527*/ uint16(x86_xCondDataSize), 10531, 10537, 0,
	/*10531*/ uint16(x86_xSetOp), uint16(x86_LES),
	/*10533*/ uint16(x86_xReadSlashR),
	/*10534*/ uint16(x86_xArgR16),
	/*10535*/ uint16(x86_xArgM16colon16),
	/*10536*/ uint16(x86_xMatch),
	/*10537*/ uint16(x86_xSetOp), uint16(x86_LES),
	/*10539*/ uint16(x86_xReadSlashR),
	/*10540*/ uint16(x86_xArgR32),
	/*10541*/ uint16(x86_xArgM16colon32),
	/*10542*/ uint16(x86_xMatch),
	/*10543*/ uint16(x86_xCondIs64), 10546, 0,
	/*10546*/ uint16(x86_xCondDataSize), 10550, 10556, 0,
	/*10550*/ uint16(x86_xSetOp), uint16(x86_LDS),
	/*10552*/ uint16(x86_xReadSlashR),
	/*10553*/ uint16(x86_xArgR16),
	/*10554*/ uint16(x86_xArgM16colon16),
	/*10555*/ uint16(x86_xMatch),
	/*10556*/ uint16(x86_xSetOp), uint16(x86_LDS),
	/*10558*/ uint16(x86_xReadSlashR),
	/*10559*/ uint16(x86_xArgR32),
	/*10560*/ uint16(x86_xArgM16colon32),
	/*10561*/ uint16(x86_xMatch),
	/*10562*/ uint16(x86_xCondByte), 1,
	0xF8, 10581,
	/*10566*/ uint16(x86_xCondSlashR),
	10575, // 0
	0,     // 1
	0,     // 2
	0,     // 3
	0,     // 4
	0,     // 5
	0,     // 6
	0,     // 7
	/*10575*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10577*/ uint16(x86_xReadIb),
	/*10578*/ uint16(x86_xArgRM8),
	/*10579*/ uint16(x86_xArgImm8u),
	/*10580*/ uint16(x86_xMatch),
	/*10581*/ uint16(x86_xSetOp), uint16(x86_XABORT),
	/*10583*/ uint16(x86_xReadIb),
	/*10584*/ uint16(x86_xArgImm8u),
	/*10585*/ uint16(x86_xMatch),
	/*10586*/ uint16(x86_xCondByte), 1,
	0xF8, 10628,
	/*10590*/ uint16(x86_xCondSlashR),
	10599, // 0
	0,     // 1
	0,     // 2
	0,     // 3
	0,     // 4
	0,     // 5
	0,     // 6
	0,     // 7
	/*10599*/ uint16(x86_xCondIs64), 10602, 10618,
	/*10602*/ uint16(x86_xCondDataSize), 10606, 10612, 0,
	/*10606*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10608*/ uint16(x86_xReadIw),
	/*10609*/ uint16(x86_xArgRM16),
	/*10610*/ uint16(x86_xArgImm16),
	/*10611*/ uint16(x86_xMatch),
	/*10612*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10614*/ uint16(x86_xReadId),
	/*10615*/ uint16(x86_xArgRM32),
	/*10616*/ uint16(x86_xArgImm32),
	/*10617*/ uint16(x86_xMatch),
	/*10618*/ uint16(x86_xCondDataSize), 10606, 10612, 10622,
	/*10622*/ uint16(x86_xSetOp), uint16(x86_MOV),
	/*10624*/ uint16(x86_xReadId),
	/*10625*/ uint16(x86_xArgRM64),
	/*10626*/ uint16(x86_xArgImm32),
	/*10627*/ uint16(x86_xMatch),
	/*10628*/ uint16(x86_xCondDataSize), 10632, 10637, 10642,
	/*10632*/ uint16(x86_xSetOp), uint16(x86_XBEGIN),
	/*10634*/ uint16(x86_xReadCw),
	/*10635*/ uint16(x86_xArgRel16),
	/*10636*/ uint16(x86_xMatch),
	/*10637*/ uint16(x86_xSetOp), uint16(x86_XBEGIN),
	/*10639*/ uint16(x86_xReadCd),
	/*10640*/ uint16(x86_xArgRel32),
	/*10641*/ uint16(x86_xMatch),
	/*10642*/ uint16(x86_xSetOp), uint16(x86_XBEGIN),
	/*10644*/ uint16(x86_xReadCd),
	/*10645*/ uint16(x86_xArgRel32),
	/*10646*/ uint16(x86_xMatch),
	/*10647*/ uint16(x86_xSetOp), uint16(x86_ENTER),
	/*10649*/ uint16(x86_xReadIw),
	/*10650*/ uint16(x86_xReadIb),
	/*10651*/ uint16(x86_xArgImm16u),
	/*10652*/ uint16(x86_xArgImm8u),
	/*10653*/ uint16(x86_xMatch),
	/*10654*/ uint16(x86_xCondIs64), 10657, 10667,
	/*10657*/ uint16(x86_xCondDataSize), 10661, 10664, 0,
	/*10661*/ uint16(x86_xSetOp), uint16(x86_LEAVE),
	/*10663*/ uint16(x86_xMatch),
	/*10664*/ uint16(x86_xSetOp), uint16(x86_LEAVE),
	/*10666*/ uint16(x86_xMatch),
	/*10667*/ uint16(x86_xCondDataSize), 10661, 10671, 10674,
	/*10671*/ uint16(x86_xSetOp), uint16(x86_LEAVE),
	/*10673*/ uint16(x86_xMatch),
	/*10674*/ uint16(x86_xSetOp), uint16(x86_LEAVE),
	/*10676*/ uint16(x86_xMatch),
	/*10677*/ uint16(x86_xSetOp), uint16(x86_LRET),
	/*10679*/ uint16(x86_xReadIw),
	/*10680*/ uint16(x86_xArgImm16u),
	/*10681*/ uint16(x86_xMatch),
	/*10682*/ uint16(x86_xSetOp), uint16(x86_LRET),
	/*10684*/ uint16(x86_xMatch),
	/*10685*/ uint16(x86_xSetOp), uint16(x86_INT),
	/*10687*/ uint16(x86_xArg3),
	/*10688*/ uint16(x86_xMatch),
	/*10689*/ uint16(x86_xSetOp), uint16(x86_INT),
	/*10691*/ uint16(x86_xReadIb),
	/*10692*/ uint16(x86_xArgImm8u),
	/*10693*/ uint16(x86_xMatch),
	/*10694*/ uint16(x86_xCondIs64), 10697, 0,
	/*10697*/ uint16(x86_xSetOp), uint16(x86_INTO),
	/*10699*/ uint16(x86_xMatch),
	/*10700*/ uint16(x86_xCondIs64), 10703, 10713,
	/*10703*/ uint16(x86_xCondDataSize), 10707, 10710, 0,
	/*10707*/ uint16(x86_xSetOp), uint16(x86_IRET),
	/*10709*/ uint16(x86_xMatch),
	/*10710*/ uint16(x86_xSetOp), uint16(x86_IRETD),
	/*10712*/ uint16(x86_xMatch),
	/*10713*/ uint16(x86_xCondDataSize), 10707, 10710, 10717,
	/*10717*/ uint16(x86_xSetOp), uint16(x86_IRETQ),
	/*10719*/ uint16(x86_xMatch),
	/*10720*/ uint16(x86_xCondSlashR),
	10729, // 0
	10734, // 1
	10739, // 2
	10744, // 3
	10749, // 4
	10754, // 5
	0,     // 6
	10759, // 7
	/*10729*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10731*/ uint16(x86_xArgRM8),
	/*10732*/ uint16(x86_xArg1),
	/*10733*/ uint16(x86_xMatch),
	/*10734*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10736*/ uint16(x86_xArgRM8),
	/*10737*/ uint16(x86_xArg1),
	/*10738*/ uint16(x86_xMatch),
	/*10739*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10741*/ uint16(x86_xArgRM8),
	/*10742*/ uint16(x86_xArg1),
	/*10743*/ uint16(x86_xMatch),
	/*10744*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10746*/ uint16(x86_xArgRM8),
	/*10747*/ uint16(x86_xArg1),
	/*10748*/ uint16(x86_xMatch),
	/*10749*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10751*/ uint16(x86_xArgRM8),
	/*10752*/ uint16(x86_xArg1),
	/*10753*/ uint16(x86_xMatch),
	/*10754*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10756*/ uint16(x86_xArgRM8),
	/*10757*/ uint16(x86_xArg1),
	/*10758*/ uint16(x86_xMatch),
	/*10759*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10761*/ uint16(x86_xArgRM8),
	/*10762*/ uint16(x86_xArg1),
	/*10763*/ uint16(x86_xMatch),
	/*10764*/ uint16(x86_xCondSlashR),
	10773, // 0
	10799, // 1
	10825, // 2
	10851, // 3
	10877, // 4
	10903, // 5
	0,     // 6
	10929, // 7
	/*10773*/ uint16(x86_xCondIs64), 10776, 10790,
	/*10776*/ uint16(x86_xCondDataSize), 10780, 10785, 0,
	/*10780*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10782*/ uint16(x86_xArgRM16),
	/*10783*/ uint16(x86_xArg1),
	/*10784*/ uint16(x86_xMatch),
	/*10785*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10787*/ uint16(x86_xArgRM32),
	/*10788*/ uint16(x86_xArg1),
	/*10789*/ uint16(x86_xMatch),
	/*10790*/ uint16(x86_xCondDataSize), 10780, 10785, 10794,
	/*10794*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10796*/ uint16(x86_xArgRM64),
	/*10797*/ uint16(x86_xArg1),
	/*10798*/ uint16(x86_xMatch),
	/*10799*/ uint16(x86_xCondIs64), 10802, 10816,
	/*10802*/ uint16(x86_xCondDataSize), 10806, 10811, 0,
	/*10806*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10808*/ uint16(x86_xArgRM16),
	/*10809*/ uint16(x86_xArg1),
	/*10810*/ uint16(x86_xMatch),
	/*10811*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10813*/ uint16(x86_xArgRM32),
	/*10814*/ uint16(x86_xArg1),
	/*10815*/ uint16(x86_xMatch),
	/*10816*/ uint16(x86_xCondDataSize), 10806, 10811, 10820,
	/*10820*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10822*/ uint16(x86_xArgRM64),
	/*10823*/ uint16(x86_xArg1),
	/*10824*/ uint16(x86_xMatch),
	/*10825*/ uint16(x86_xCondIs64), 10828, 10842,
	/*10828*/ uint16(x86_xCondDataSize), 10832, 10837, 0,
	/*10832*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10834*/ uint16(x86_xArgRM16),
	/*10835*/ uint16(x86_xArg1),
	/*10836*/ uint16(x86_xMatch),
	/*10837*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10839*/ uint16(x86_xArgRM32),
	/*10840*/ uint16(x86_xArg1),
	/*10841*/ uint16(x86_xMatch),
	/*10842*/ uint16(x86_xCondDataSize), 10832, 10837, 10846,
	/*10846*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10848*/ uint16(x86_xArgRM64),
	/*10849*/ uint16(x86_xArg1),
	/*10850*/ uint16(x86_xMatch),
	/*10851*/ uint16(x86_xCondIs64), 10854, 10868,
	/*10854*/ uint16(x86_xCondDataSize), 10858, 10863, 0,
	/*10858*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10860*/ uint16(x86_xArgRM16),
	/*10861*/ uint16(x86_xArg1),
	/*10862*/ uint16(x86_xMatch),
	/*10863*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10865*/ uint16(x86_xArgRM32),
	/*10866*/ uint16(x86_xArg1),
	/*10867*/ uint16(x86_xMatch),
	/*10868*/ uint16(x86_xCondDataSize), 10858, 10863, 10872,
	/*10872*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10874*/ uint16(x86_xArgRM64),
	/*10875*/ uint16(x86_xArg1),
	/*10876*/ uint16(x86_xMatch),
	/*10877*/ uint16(x86_xCondIs64), 10880, 10894,
	/*10880*/ uint16(x86_xCondDataSize), 10884, 10889, 0,
	/*10884*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10886*/ uint16(x86_xArgRM16),
	/*10887*/ uint16(x86_xArg1),
	/*10888*/ uint16(x86_xMatch),
	/*10889*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10891*/ uint16(x86_xArgRM32),
	/*10892*/ uint16(x86_xArg1),
	/*10893*/ uint16(x86_xMatch),
	/*10894*/ uint16(x86_xCondDataSize), 10884, 10889, 10898,
	/*10898*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10900*/ uint16(x86_xArgRM64),
	/*10901*/ uint16(x86_xArg1),
	/*10902*/ uint16(x86_xMatch),
	/*10903*/ uint16(x86_xCondIs64), 10906, 10920,
	/*10906*/ uint16(x86_xCondDataSize), 10910, 10915, 0,
	/*10910*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10912*/ uint16(x86_xArgRM16),
	/*10913*/ uint16(x86_xArg1),
	/*10914*/ uint16(x86_xMatch),
	/*10915*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10917*/ uint16(x86_xArgRM32),
	/*10918*/ uint16(x86_xArg1),
	/*10919*/ uint16(x86_xMatch),
	/*10920*/ uint16(x86_xCondDataSize), 10910, 10915, 10924,
	/*10924*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10926*/ uint16(x86_xArgRM64),
	/*10927*/ uint16(x86_xArg1),
	/*10928*/ uint16(x86_xMatch),
	/*10929*/ uint16(x86_xCondIs64), 10932, 10946,
	/*10932*/ uint16(x86_xCondDataSize), 10936, 10941, 0,
	/*10936*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10938*/ uint16(x86_xArgRM16),
	/*10939*/ uint16(x86_xArg1),
	/*10940*/ uint16(x86_xMatch),
	/*10941*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10943*/ uint16(x86_xArgRM32),
	/*10944*/ uint16(x86_xArg1),
	/*10945*/ uint16(x86_xMatch),
	/*10946*/ uint16(x86_xCondDataSize), 10936, 10941, 10950,
	/*10950*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10952*/ uint16(x86_xArgRM64),
	/*10953*/ uint16(x86_xArg1),
	/*10954*/ uint16(x86_xMatch),
	/*10955*/ uint16(x86_xCondSlashR),
	10964, // 0
	10969, // 1
	10974, // 2
	10979, // 3
	10984, // 4
	10989, // 5
	0,     // 6
	10994, // 7
	/*10964*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*10966*/ uint16(x86_xArgRM8),
	/*10967*/ uint16(x86_xArgCL),
	/*10968*/ uint16(x86_xMatch),
	/*10969*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*10971*/ uint16(x86_xArgRM8),
	/*10972*/ uint16(x86_xArgCL),
	/*10973*/ uint16(x86_xMatch),
	/*10974*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*10976*/ uint16(x86_xArgRM8),
	/*10977*/ uint16(x86_xArgCL),
	/*10978*/ uint16(x86_xMatch),
	/*10979*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*10981*/ uint16(x86_xArgRM8),
	/*10982*/ uint16(x86_xArgCL),
	/*10983*/ uint16(x86_xMatch),
	/*10984*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*10986*/ uint16(x86_xArgRM8),
	/*10987*/ uint16(x86_xArgCL),
	/*10988*/ uint16(x86_xMatch),
	/*10989*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*10991*/ uint16(x86_xArgRM8),
	/*10992*/ uint16(x86_xArgCL),
	/*10993*/ uint16(x86_xMatch),
	/*10994*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*10996*/ uint16(x86_xArgRM8),
	/*10997*/ uint16(x86_xArgCL),
	/*10998*/ uint16(x86_xMatch),
	/*10999*/ uint16(x86_xCondSlashR),
	11008, // 0
	11034, // 1
	11060, // 2
	11086, // 3
	11112, // 4
	11138, // 5
	0,     // 6
	11164, // 7
	/*11008*/ uint16(x86_xCondIs64), 11011, 11025,
	/*11011*/ uint16(x86_xCondDataSize), 11015, 11020, 0,
	/*11015*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*11017*/ uint16(x86_xArgRM16),
	/*11018*/ uint16(x86_xArgCL),
	/*11019*/ uint16(x86_xMatch),
	/*11020*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*11022*/ uint16(x86_xArgRM32),
	/*11023*/ uint16(x86_xArgCL),
	/*11024*/ uint16(x86_xMatch),
	/*11025*/ uint16(x86_xCondDataSize), 11015, 11020, 11029,
	/*11029*/ uint16(x86_xSetOp), uint16(x86_ROL),
	/*11031*/ uint16(x86_xArgRM64),
	/*11032*/ uint16(x86_xArgCL),
	/*11033*/ uint16(x86_xMatch),
	/*11034*/ uint16(x86_xCondIs64), 11037, 11051,
	/*11037*/ uint16(x86_xCondDataSize), 11041, 11046, 0,
	/*11041*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*11043*/ uint16(x86_xArgRM16),
	/*11044*/ uint16(x86_xArgCL),
	/*11045*/ uint16(x86_xMatch),
	/*11046*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*11048*/ uint16(x86_xArgRM32),
	/*11049*/ uint16(x86_xArgCL),
	/*11050*/ uint16(x86_xMatch),
	/*11051*/ uint16(x86_xCondDataSize), 11041, 11046, 11055,
	/*11055*/ uint16(x86_xSetOp), uint16(x86_ROR),
	/*11057*/ uint16(x86_xArgRM64),
	/*11058*/ uint16(x86_xArgCL),
	/*11059*/ uint16(x86_xMatch),
	/*11060*/ uint16(x86_xCondIs64), 11063, 11077,
	/*11063*/ uint16(x86_xCondDataSize), 11067, 11072, 0,
	/*11067*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*11069*/ uint16(x86_xArgRM16),
	/*11070*/ uint16(x86_xArgCL),
	/*11071*/ uint16(x86_xMatch),
	/*11072*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*11074*/ uint16(x86_xArgRM32),
	/*11075*/ uint16(x86_xArgCL),
	/*11076*/ uint16(x86_xMatch),
	/*11077*/ uint16(x86_xCondDataSize), 11067, 11072, 11081,
	/*11081*/ uint16(x86_xSetOp), uint16(x86_RCL),
	/*11083*/ uint16(x86_xArgRM64),
	/*11084*/ uint16(x86_xArgCL),
	/*11085*/ uint16(x86_xMatch),
	/*11086*/ uint16(x86_xCondIs64), 11089, 11103,
	/*11089*/ uint16(x86_xCondDataSize), 11093, 11098, 0,
	/*11093*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*11095*/ uint16(x86_xArgRM16),
	/*11096*/ uint16(x86_xArgCL),
	/*11097*/ uint16(x86_xMatch),
	/*11098*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*11100*/ uint16(x86_xArgRM32),
	/*11101*/ uint16(x86_xArgCL),
	/*11102*/ uint16(x86_xMatch),
	/*11103*/ uint16(x86_xCondDataSize), 11093, 11098, 11107,
	/*11107*/ uint16(x86_xSetOp), uint16(x86_RCR),
	/*11109*/ uint16(x86_xArgRM64),
	/*11110*/ uint16(x86_xArgCL),
	/*11111*/ uint16(x86_xMatch),
	/*11112*/ uint16(x86_xCondIs64), 11115, 11129,
	/*11115*/ uint16(x86_xCondDataSize), 11119, 11124, 0,
	/*11119*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*11121*/ uint16(x86_xArgRM16),
	/*11122*/ uint16(x86_xArgCL),
	/*11123*/ uint16(x86_xMatch),
	/*11124*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*11126*/ uint16(x86_xArgRM32),
	/*11127*/ uint16(x86_xArgCL),
	/*11128*/ uint16(x86_xMatch),
	/*11129*/ uint16(x86_xCondDataSize), 11119, 11124, 11133,
	/*11133*/ uint16(x86_xSetOp), uint16(x86_SHL),
	/*11135*/ uint16(x86_xArgRM64),
	/*11136*/ uint16(x86_xArgCL),
	/*11137*/ uint16(x86_xMatch),
	/*11138*/ uint16(x86_xCondIs64), 11141, 11155,
	/*11141*/ uint16(x86_xCondDataSize), 11145, 11150, 0,
	/*11145*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*11147*/ uint16(x86_xArgRM16),
	/*11148*/ uint16(x86_xArgCL),
	/*11149*/ uint16(x86_xMatch),
	/*11150*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*11152*/ uint16(x86_xArgRM32),
	/*11153*/ uint16(x86_xArgCL),
	/*11154*/ uint16(x86_xMatch),
	/*11155*/ uint16(x86_xCondDataSize), 11145, 11150, 11159,
	/*11159*/ uint16(x86_xSetOp), uint16(x86_SHR),
	/*11161*/ uint16(x86_xArgRM64),
	/*11162*/ uint16(x86_xArgCL),
	/*11163*/ uint16(x86_xMatch),
	/*11164*/ uint16(x86_xCondIs64), 11167, 11181,
	/*11167*/ uint16(x86_xCondDataSize), 11171, 11176, 0,
	/*11171*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*11173*/ uint16(x86_xArgRM16),
	/*11174*/ uint16(x86_xArgCL),
	/*11175*/ uint16(x86_xMatch),
	/*11176*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*11178*/ uint16(x86_xArgRM32),
	/*11179*/ uint16(x86_xArgCL),
	/*11180*/ uint16(x86_xMatch),
	/*11181*/ uint16(x86_xCondDataSize), 11171, 11176, 11185,
	/*11185*/ uint16(x86_xSetOp), uint16(x86_SAR),
	/*11187*/ uint16(x86_xArgRM64),
	/*11188*/ uint16(x86_xArgCL),
	/*11189*/ uint16(x86_xMatch),
	/*11190*/ uint16(x86_xCondIs64), 11193, 0,
	/*11193*/ uint16(x86_xSetOp), uint16(x86_AAM),
	/*11195*/ uint16(x86_xReadIb),
	/*11196*/ uint16(x86_xArgImm8u),
	/*11197*/ uint16(x86_xMatch),
	/*11198*/ uint16(x86_xCondIs64), 11201, 0,
	/*11201*/ uint16(x86_xSetOp), uint16(x86_AAD),
	/*11203*/ uint16(x86_xReadIb),
	/*11204*/ uint16(x86_xArgImm8u),
	/*11205*/ uint16(x86_xMatch),
	/*11206*/ uint16(x86_xCondIs64), 11209, 11212,
	/*11209*/ uint16(x86_xSetOp), uint16(x86_XLATB),
	/*11211*/ uint16(x86_xMatch),
	/*11212*/ uint16(x86_xCondDataSize), 11209, 11209, 11216,
	/*11216*/ uint16(x86_xSetOp), uint16(x86_XLATB),
	/*11218*/ uint16(x86_xMatch),
	/*11219*/ uint16(x86_xCondByte), 64,
	0xc0, 11390,
	0xc1, 11390,
	0xc2, 11390,
	0xc3, 11390,
	0xc4, 11390,
	0xc5, 11390,
	0xc6, 11390,
	0xc7, 11390,
	0xc8, 11395,
	0xc9, 11395,
	0xca, 11395,
	0xcb, 11395,
	0xcc, 11395,
	0xcd, 11395,
	0xce, 11395,
	0xcf, 11395,
	0xd0, 11400,
	0xd1, 11400,
	0xd2, 11400,
	0xd3, 11400,
	0xd4, 11400,
	0xd5, 11400,
	0xd6, 11400,
	0xd7, 11400,
	0xd8, 11404,
	0xd9, 11404,
	0xda, 11404,
	0xdb, 11404,
	0xdc, 11404,
	0xdd, 11404,
	0xde, 11404,
	0xdf, 11404,
	0xe0, 11408,
	0xe1, 11408,
	0xe2, 11408,
	0xe3, 11408,
	0xe4, 11408,
	0xe5, 11408,
	0xe6, 11408,
	0xe7, 11408,
	0xe8, 11413,
	0xe9, 11413,
	0xea, 11413,
	0xeb, 11413,
	0xec, 11413,
	0xed, 11413,
	0xee, 11413,
	0xef, 11413,
	0xf0, 11418,
	0xf1, 11418,
	0xf2, 11418,
	0xf3, 11418,
	0xf4, 11418,
	0xf5, 11418,
	0xf6, 11418,
	0xf7, 11418,
	0xf8, 11423,
	0xf9, 11423,
	0xfa, 11423,
	0xfb, 11423,
	0xfc, 11423,
	0xfd, 11423,
	0xfe, 11423,
	0xff, 11423,
	/*11349*/ uint16(x86_xCondSlashR),
	11358, // 0
	11362, // 1
	11366, // 2
	11370, // 3
	11374, // 4
	11378, // 5
	11382, // 6
	11386, // 7
	/*11358*/ uint16(x86_xSetOp), uint16(x86_FADD),
	/*11360*/ uint16(x86_xArgM32fp),
	/*11361*/ uint16(x86_xMatch),
	/*11362*/ uint16(x86_xSetOp), uint16(x86_FMUL),
	/*11364*/ uint16(x86_xArgM32fp),
	/*11365*/ uint16(x86_xMatch),
	/*11366*/ uint16(x86_xSetOp), uint16(x86_FCOM),
	/*11368*/ uint16(x86_xArgM32fp),
	/*11369*/ uint16(x86_xMatch),
	/*11370*/ uint16(x86_xSetOp), uint16(x86_FCOMP),
	/*11372*/ uint16(x86_xArgM32fp),
	/*11373*/ uint16(x86_xMatch),
	/*11374*/ uint16(x86_xSetOp), uint16(x86_FSUB),
	/*11376*/ uint16(x86_xArgM32fp),
	/*11377*/ uint16(x86_xMatch),
	/*11378*/ uint16(x86_xSetOp), uint16(x86_FSUBR),
	/*11380*/ uint16(x86_xArgM32fp),
	/*11381*/ uint16(x86_xMatch),
	/*11382*/ uint16(x86_xSetOp), uint16(x86_FDIV),
	/*11384*/ uint16(x86_xArgM32fp),
	/*11385*/ uint16(x86_xMatch),
	/*11386*/ uint16(x86_xSetOp), uint16(x86_FDIVR),
	/*11388*/ uint16(x86_xArgM32fp),
	/*11389*/ uint16(x86_xMatch),
	/*11390*/ uint16(x86_xSetOp), uint16(x86_FADD),
	/*11392*/ uint16(x86_xArgST),
	/*11393*/ uint16(x86_xArgSTi),
	/*11394*/ uint16(x86_xMatch),
	/*11395*/ uint16(x86_xSetOp), uint16(x86_FMUL),
	/*11397*/ uint16(x86_xArgST),
	/*11398*/ uint16(x86_xArgSTi),
	/*11399*/ uint16(x86_xMatch),
	/*11400*/ uint16(x86_xSetOp), uint16(x86_FCOM),
	/*11402*/ uint16(x86_xArgSTi),
	/*11403*/ uint16(x86_xMatch),
	/*11404*/ uint16(x86_xSetOp), uint16(x86_FCOMP),
	/*11406*/ uint16(x86_xArgSTi),
	/*11407*/ uint16(x86_xMatch),
	/*11408*/ uint16(x86_xSetOp), uint16(x86_FSUB),
	/*11410*/ uint16(x86_xArgST),
	/*11411*/ uint16(x86_xArgSTi),
	/*11412*/ uint16(x86_xMatch),
	/*11413*/ uint16(x86_xSetOp), uint16(x86_FSUBR),
	/*11415*/ uint16(x86_xArgST),
	/*11416*/ uint16(x86_xArgSTi),
	/*11417*/ uint16(x86_xMatch),
	/*11418*/ uint16(x86_xSetOp), uint16(x86_FDIV),
	/*11420*/ uint16(x86_xArgST),
	/*11421*/ uint16(x86_xArgSTi),
	/*11422*/ uint16(x86_xMatch),
	/*11423*/ uint16(x86_xSetOp), uint16(x86_FDIVR),
	/*11425*/ uint16(x86_xArgST),
	/*11426*/ uint16(x86_xArgSTi),
	/*11427*/ uint16(x86_xMatch),
	/*11428*/ uint16(x86_xCondByte), 42,
	0xc0, 11551,
	0xc1, 11551,
	0xc2, 11551,
	0xc3, 11551,
	0xc4, 11551,
	0xc5, 11551,
	0xc6, 11551,
	0xc7, 11551,
	0xc8, 11555,
	0xc9, 11555,
	0xca, 11555,
	0xcb, 11555,
	0xcc, 11555,
	0xcd, 11555,
	0xce, 11555,
	0xcf, 11555,
	0xD0, 11559,
	0xE0, 11562,
	0xE1, 11565,
	0xE4, 11568,
	0xE5, 11571,
	0xE8, 11574,
	0xE9, 11577,
	0xEA, 11580,
	0xEB, 11583,
	0xEC, 11586,
	0xF0, 11589,
	0xF1, 11592,
	0xF2, 11595,
	0xF3, 11598,
	0xF4, 11601,
	0xF5, 11604,
	0xF6, 11607,
	0xF7, 11610,
	0xF8, 11613,
	0xF9, 11616,
	0xFA, 11619,
	0xFB, 11622,
	0xFC, 11625,
	0xFD, 11628,
	0xFE, 11631,
	0xFF, 11634,
	/*11514*/ uint16(x86_xCondSlashR),
	11523, // 0
	0,     // 1
	11527, // 2
	11531, // 3
	11535, // 4
	11539, // 5
	11543, // 6
	11547, // 7
	/*11523*/ uint16(x86_xSetOp), uint16(x86_FLD),
	/*11525*/ uint16(x86_xArgM32fp),
	/*11526*/ uint16(x86_xMatch),
	/*11527*/ uint16(x86_xSetOp), uint16(x86_FST),
	/*11529*/ uint16(x86_xArgM32fp),
	/*11530*/ uint16(x86_xMatch),
	/*11531*/ uint16(x86_xSetOp), uint16(x86_FSTP),
	/*11533*/ uint16(x86_xArgM32fp),
	/*11534*/ uint16(x86_xMatch),
	/*11535*/ uint16(x86_xSetOp), uint16(x86_FLDENV),
	/*11537*/ uint16(x86_xArgM1428byte),
	/*11538*/ uint16(x86_xMatch),
	/*11539*/ uint16(x86_xSetOp), uint16(x86_FLDCW),
	/*11541*/ uint16(x86_xArgM2byte),
	/*11542*/ uint16(x86_xMatch),
	/*11543*/ uint16(x86_xSetOp), uint16(x86_FNSTENV),
	/*11545*/ uint16(x86_xArgM1428byte),
	/*11546*/ uint16(x86_xMatch),
	/*11547*/ uint16(x86_xSetOp), uint16(x86_FNSTCW),
	/*11549*/ uint16(x86_xArgM2byte),
	/*11550*/ uint16(x86_xMatch),
	/*11551*/ uint16(x86_xSetOp), uint16(x86_FLD),
	/*11553*/ uint16(x86_xArgSTi),
	/*11554*/ uint16(x86_xMatch),
	/*11555*/ uint16(x86_xSetOp), uint16(x86_FXCH),
	/*11557*/ uint16(x86_xArgSTi),
	/*11558*/ uint16(x86_xMatch),
	/*11559*/ uint16(x86_xSetOp), uint16(x86_FNOP),
	/*11561*/ uint16(x86_xMatch),
	/*11562*/ uint16(x86_xSetOp), uint16(x86_FCHS),
	/*11564*/ uint16(x86_xMatch),
	/*11565*/ uint16(x86_xSetOp), uint16(x86_FABS),
	/*11567*/ uint16(x86_xMatch),
	/*11568*/ uint16(x86_xSetOp), uint16(x86_FTST),
	/*11570*/ uint16(x86_xMatch),
	/*11571*/ uint16(x86_xSetOp), uint16(x86_FXAM),
	/*11573*/ uint16(x86_xMatch),
	/*11574*/ uint16(x86_xSetOp), uint16(x86_FLD1),
	/*11576*/ uint16(x86_xMatch),
	/*11577*/ uint16(x86_xSetOp), uint16(x86_FLDL2T),
	/*11579*/ uint16(x86_xMatch),
	/*11580*/ uint16(x86_xSetOp), uint16(x86_FLDL2E),
	/*11582*/ uint16(x86_xMatch),
	/*11583*/ uint16(x86_xSetOp), uint16(x86_FLDPI),
	/*11585*/ uint16(x86_xMatch),
	/*11586*/ uint16(x86_xSetOp), uint16(x86_FLDLG2),
	/*11588*/ uint16(x86_xMatch),
	/*11589*/ uint16(x86_xSetOp), uint16(x86_F2XM1),
	/*11591*/ uint16(x86_xMatch),
	/*11592*/ uint16(x86_xSetOp), uint16(x86_FYL2X),
	/*11594*/ uint16(x86_xMatch),
	/*11595*/ uint16(x86_xSetOp), uint16(x86_FPTAN),
	/*11597*/ uint16(x86_xMatch),
	/*11598*/ uint16(x86_xSetOp), uint16(x86_FPATAN),
	/*11600*/ uint16(x86_xMatch),
	/*11601*/ uint16(x86_xSetOp), uint16(x86_FXTRACT),
	/*11603*/ uint16(x86_xMatch),
	/*11604*/ uint16(x86_xSetOp), uint16(x86_FPREM1),
	/*11606*/ uint16(x86_xMatch),
	/*11607*/ uint16(x86_xSetOp), uint16(x86_FDECSTP),
	/*11609*/ uint16(x86_xMatch),
	/*11610*/ uint16(x86_xSetOp), uint16(x86_FINCSTP),
	/*11612*/ uint16(x86_xMatch),
	/*11613*/ uint16(x86_xSetOp), uint16(x86_FPREM),
	/*11615*/ uint16(x86_xMatch),
	/*11616*/ uint16(x86_xSetOp), uint16(x86_FYL2XP1),
	/*11618*/ uint16(x86_xMatch),
	/*11619*/ uint16(x86_xSetOp), uint16(x86_FSQRT),
	/*11621*/ uint16(x86_xMatch),
	/*11622*/ uint16(x86_xSetOp), uint16(x86_FSINCOS),
	/*11624*/ uint16(x86_xMatch),
	/*11625*/ uint16(x86_xSetOp), uint16(x86_FRNDINT),
	/*11627*/ uint16(x86_xMatch),
	/*11628*/ uint16(x86_xSetOp), uint16(x86_FSCALE),
	/*11630*/ uint16(x86_xMatch),
	/*11631*/ uint16(x86_xSetOp), uint16(x86_FSIN),
	/*11633*/ uint16(x86_xMatch),
	/*11634*/ uint16(x86_xSetOp), uint16(x86_FCOS),
	/*11636*/ uint16(x86_xMatch),
	/*11637*/ uint16(x86_xCondByte), 33,
	0xc0, 11746,
	0xc1, 11746,
	0xc2, 11746,
	0xc3, 11746,
	0xc4, 11746,
	0xc5, 11746,
	0xc6, 11746,
	0xc7, 11746,
	0xc8, 11751,
	0xc9, 11751,
	0xca, 11751,
	0xcb, 11751,
	0xcc, 11751,
	0xcd, 11751,
	0xce, 11751,
	0xcf, 11751,
	0xd0, 11756,
	0xd1, 11756,
	0xd2, 11756,
	0xd3, 11756,
	0xd4, 11756,
	0xd5, 11756,
	0xd6, 11756,
	0xd7, 11756,
	0xd8, 11761,
	0xd9, 11761,
	0xda, 11761,
	0xdb, 11761,
	0xdc, 11761,
	0xdd, 11761,
	0xde, 11761,
	0xdf, 11761,
	0xE9, 11766,
	/*11705*/ uint16(x86_xCondSlashR),
	11714, // 0
	11718, // 1
	11722, // 2
	11726, // 3
	11730, // 4
	11734, // 5
	11738, // 6
	11742, // 7
	/*11714*/ uint16(x86_xSetOp), uint16(x86_FIADD),
	/*11716*/ uint16(x86_xArgM32int),
	/*11717*/ uint16(x86_xMatch),
	/*11718*/ uint16(x86_xSetOp), uint16(x86_FIMUL),
	/*11720*/ uint16(x86_xArgM32int),
	/*11721*/ uint16(x86_xMatch),
	/*11722*/ uint16(x86_xSetOp), uint16(x86_FICOM),
	/*11724*/ uint16(x86_xArgM32int),
	/*11725*/ uint16(x86_xMatch),
	/*11726*/ uint16(x86_xSetOp), uint16(x86_FICOMP),
	/*11728*/ uint16(x86_xArgM32int),
	/*11729*/ uint16(x86_xMatch),
	/*11730*/ uint16(x86_xSetOp), uint16(x86_FISUB),
	/*11732*/ uint16(x86_xArgM32int),
	/*11733*/ uint16(x86_xMatch),
	/*11734*/ uint16(x86_xSetOp), uint16(x86_FISUBR),
	/*11736*/ uint16(x86_xArgM32int),
	/*11737*/ uint16(x86_xMatch),
	/*11738*/ uint16(x86_xSetOp), uint16(x86_FIDIV),
	/*11740*/ uint16(x86_xArgM32int),
	/*11741*/ uint16(x86_xMatch),
	/*11742*/ uint16(x86_xSetOp), uint16(x86_FIDIVR),
	/*11744*/ uint16(x86_xArgM32int),
	/*11745*/ uint16(x86_xMatch),
	/*11746*/ uint16(x86_xSetOp), uint16(x86_FCMOVB),
	/*11748*/ uint16(x86_xArgST),
	/*11749*/ uint16(x86_xArgSTi),
	/*11750*/ uint16(x86_xMatch),
	/*11751*/ uint16(x86_xSetOp), uint16(x86_FCMOVE),
	/*11753*/ uint16(x86_xArgST),
	/*11754*/ uint16(x86_xArgSTi),
	/*11755*/ uint16(x86_xMatch),
	/*11756*/ uint16(x86_xSetOp), uint16(x86_FCMOVBE),
	/*11758*/ uint16(x86_xArgST),
	/*11759*/ uint16(x86_xArgSTi),
	/*11760*/ uint16(x86_xMatch),
	/*11761*/ uint16(x86_xSetOp), uint16(x86_FCMOVU),
	/*11763*/ uint16(x86_xArgST),
	/*11764*/ uint16(x86_xArgSTi),
	/*11765*/ uint16(x86_xMatch),
	/*11766*/ uint16(x86_xSetOp), uint16(x86_FUCOMPP),
	/*11768*/ uint16(x86_xMatch),
	/*11769*/ uint16(x86_xCondByte), 50,
	0xc0, 11904,
	0xc1, 11904,
	0xc2, 11904,
	0xc3, 11904,
	0xc4, 11904,
	0xc5, 11904,
	0xc6, 11904,
	0xc7, 11904,
	0xc8, 11909,
	0xc9, 11909,
	0xca, 11909,
	0xcb, 11909,
	0xcc, 11909,
	0xcd, 11909,
	0xce, 11909,
	0xcf, 11909,
	0xd0, 11914,
	0xd1, 11914,
	0xd2, 11914,
	0xd3, 11914,
	0xd4, 11914,
	0xd5, 11914,
	0xd6, 11914,
	0xd7, 11914,
	0xd8, 11919,
	0xd9, 11919,
	0xda, 11919,
	0xdb, 11919,
	0xdc, 11919,
	0xdd, 11919,
	0xde, 11919,
	0xdf, 11919,
	0xE2, 11924,
	0xE3, 11927,
	0xe8, 11930,
	0xe9, 11930,
	0xea, 11930,
	0xeb, 11930,
	0xec, 11930,
	0xed, 11930,
	0xee, 11930,
	0xef, 11930,
	0xf0, 11935,
	0xf1, 11935,
	0xf2, 11935,
	0xf3, 11935,
	0xf4, 11935,
	0xf5, 11935,
	0xf6, 11935,
	0xf7, 11935,
	/*11871*/ uint16(x86_xCondSlashR),
	11880, // 0
	11884, // 1
	11888, // 2
	11892, // 3
	0,     // 4
	11896, // 5
	0,     // 6
	11900, // 7
	/*11880*/ uint16(x86_xSetOp), uint16(x86_FILD),
	/*11882*/ uint16(x86_xArgM32int),
	/*11883*/ uint16(x86_xMatch),
	/*11884*/ uint16(x86_xSetOp), uint16(x86_FISTTP),
	/*11886*/ uint16(x86_xArgM32int),
	/*11887*/ uint16(x86_xMatch),
	/*11888*/ uint16(x86_xSetOp), uint16(x86_FIST),
	/*11890*/ uint16(x86_xArgM32int),
	/*11891*/ uint16(x86_xMatch),
	/*11892*/ uint16(x86_xSetOp), uint16(x86_FISTP),
	/*11894*/ uint16(x86_xArgM32int),
	/*11895*/ uint16(x86_xMatch),
	/*11896*/ uint16(x86_xSetOp), uint16(x86_FLD),
	/*11898*/ uint16(x86_xArgM80fp),
	/*11899*/ uint16(x86_xMatch),
	/*11900*/ uint16(x86_xSetOp), uint16(x86_FSTP),
	/*11902*/ uint16(x86_xArgM80fp),
	/*11903*/ uint16(x86_xMatch),
	/*11904*/ uint16(x86_xSetOp), uint16(x86_FCMOVNB),
	/*11906*/ uint16(x86_xArgST),
	/*11907*/ uint16(x86_xArgSTi),
	/*11908*/ uint16(x86_xMatch),
	/*11909*/ uint16(x86_xSetOp), uint16(x86_FCMOVNE),
	/*11911*/ uint16(x86_xArgST),
	/*11912*/ uint16(x86_xArgSTi),
	/*11913*/ uint16(x86_xMatch),
	/*11914*/ uint16(x86_xSetOp), uint16(x86_FCMOVNBE),
	/*11916*/ uint16(x86_xArgST),
	/*11917*/ uint16(x86_xArgSTi),
	/*11918*/ uint16(x86_xMatch),
	/*11919*/ uint16(x86_xSetOp), uint16(x86_FCMOVNU),
	/*11921*/ uint16(x86_xArgST),
	/*11922*/ uint16(x86_xArgSTi),
	/*11923*/ uint16(x86_xMatch),
	/*11924*/ uint16(x86_xSetOp), uint16(x86_FNCLEX),
	/*11926*/ uint16(x86_xMatch),
	/*11927*/ uint16(x86_xSetOp), uint16(x86_FNINIT),
	/*11929*/ uint16(x86_xMatch),
	/*11930*/ uint16(x86_xSetOp), uint16(x86_FUCOMI),
	/*11932*/ uint16(x86_xArgST),
	/*11933*/ uint16(x86_xArgSTi),
	/*11934*/ uint16(x86_xMatch),
	/*11935*/ uint16(x86_xSetOp), uint16(x86_FCOMI),
	/*11937*/ uint16(x86_xArgST),
	/*11938*/ uint16(x86_xArgSTi),
	/*11939*/ uint16(x86_xMatch),
	/*11940*/ uint16(x86_xCondByte), 48,
	0xc0, 12079,
	0xc1, 12079,
	0xc2, 12079,
	0xc3, 12079,
	0xc4, 12079,
	0xc5, 12079,
	0xc6, 12079,
	0xc7, 12079,
	0xc8, 12084,
	0xc9, 12084,
	0xca, 12084,
	0xcb, 12084,
	0xcc, 12084,
	0xcd, 12084,
	0xce, 12084,
	0xcf, 12084,
	0xe0, 12089,
	0xe1, 12089,
	0xe2, 12089,
	0xe3, 12089,
	0xe4, 12089,
	0xe5, 12089,
	0xe6, 12089,
	0xe7, 12089,
	0xe8, 12094,
	0xe9, 12094,
	0xea, 12094,
	0xeb, 12094,
	0xec, 12094,
	0xed, 12094,
	0xee, 12094,
	0xef, 12094,
	0xf0, 12099,
	0xf1, 12099,
	0xf2, 12099,
	0xf3, 12099,
	0xf4, 12099,
	0xf5, 12099,
	0xf6, 12099,
	0xf7, 12099,
	0xf8, 12104,
	0xf9, 12104,
	0xfa, 12104,
	0xfb, 12104,
	0xfc, 12104,
	0xfd, 12104,
	0xfe, 12104,
	0xff, 12104,
	/*12038*/ uint16(x86_xCondSlashR),
	12047, // 0
	12051, // 1
	12055, // 2
	12059, // 3
	12063, // 4
	12067, // 5
	12071, // 6
	12075, // 7
	/*12047*/ uint16(x86_xSetOp), uint16(x86_FADD),
	/*12049*/ uint16(x86_xArgM64fp),
	/*12050*/ uint16(x86_xMatch),
	/*12051*/ uint16(x86_xSetOp), uint16(x86_FMUL),
	/*12053*/ uint16(x86_xArgM64fp),
	/*12054*/ uint16(x86_xMatch),
	/*12055*/ uint16(x86_xSetOp), uint16(x86_FCOM),
	/*12057*/ uint16(x86_xArgM64fp),
	/*12058*/ uint16(x86_xMatch),
	/*12059*/ uint16(x86_xSetOp), uint16(x86_FCOMP),
	/*12061*/ uint16(x86_xArgM64fp),
	/*12062*/ uint16(x86_xMatch),
	/*12063*/ uint16(x86_xSetOp), uint16(x86_FSUB),
	/*12065*/ uint16(x86_xArgM64fp),
	/*12066*/ uint16(x86_xMatch),
	/*12067*/ uint16(x86_xSetOp), uint16(x86_FSUBR),
	/*12069*/ uint16(x86_xArgM64fp),
	/*12070*/ uint16(x86_xMatch),
	/*12071*/ uint16(x86_xSetOp), uint16(x86_FDIV),
	/*12073*/ uint16(x86_xArgM64fp),
	/*12074*/ uint16(x86_xMatch),
	/*12075*/ uint16(x86_xSetOp), uint16(x86_FDIVR),
	/*12077*/ uint16(x86_xArgM64fp),
	/*12078*/ uint16(x86_xMatch),
	/*12079*/ uint16(x86_xSetOp), uint16(x86_FADD),
	/*12081*/ uint16(x86_xArgSTi),
	/*12082*/ uint16(x86_xArgST),
	/*12083*/ uint16(x86_xMatch),
	/*12084*/ uint16(x86_xSetOp), uint16(x86_FMUL),
	/*12086*/ uint16(x86_xArgSTi),
	/*12087*/ uint16(x86_xArgST),
	/*12088*/ uint16(x86_xMatch),
	/*12089*/ uint16(x86_xSetOp), uint16(x86_FSUBR),
	/*12091*/ uint16(x86_xArgSTi),
	/*12092*/ uint16(x86_xArgST),
	/*12093*/ uint16(x86_xMatch),
	/*12094*/ uint16(x86_xSetOp), uint16(x86_FSUB),
	/*12096*/ uint16(x86_xArgSTi),
	/*12097*/ uint16(x86_xArgST),
	/*12098*/ uint16(x86_xMatch),
	/*12099*/ uint16(x86_xSetOp), uint16(x86_FDIVR),
	/*12101*/ uint16(x86_xArgSTi),
	/*12102*/ uint16(x86_xArgST),
	/*12103*/ uint16(x86_xMatch),
	/*12104*/ uint16(x86_xSetOp), uint16(x86_FDIV),
	/*12106*/ uint16(x86_xArgSTi),
	/*12107*/ uint16(x86_xArgST),
	/*12108*/ uint16(x86_xMatch),
	/*12109*/ uint16(x86_xCondByte), 40,
	0xc0, 12228,
	0xc1, 12228,
	0xc2, 12228,
	0xc3, 12228,
	0xc4, 12228,
	0xc5, 12228,
	0xc6, 12228,
	0xc7, 12228,
	0xd0, 12232,
	0xd1, 12232,
	0xd2, 12232,
	0xd3, 12232,
	0xd4, 12232,
	0xd5, 12232,
	0xd6, 12232,
	0xd7, 12232,
	0xd8, 12236,
	0xd9, 12236,
	0xda, 12236,
	0xdb, 12236,
	0xdc, 12236,
	0xdd, 12236,
	0xde, 12236,
	0xdf, 12236,
	0xe0, 12240,
	0xe1, 12240,
	0xe2, 12240,
	0xe3, 12240,
	0xe4, 12240,
	0xe5, 12240,
	0xe6, 12240,
	0xe7, 12240,
	0xe8, 12244,
	0xe9, 12244,
	0xea, 12244,
	0xeb, 12244,
	0xec, 12244,
	0xed, 12244,
	0xee, 12244,
	0xef, 12244,
	/*12191*/ uint16(x86_xCondSlashR),
	12200, // 0
	12204, // 1
	12208, // 2
	12212, // 3
	12216, // 4
	0,     // 5
	12220, // 6
	12224, // 7
	/*12200*/ uint16(x86_xSetOp), uint16(x86_FLD),
	/*12202*/ uint16(x86_xArgM64fp),
	/*12203*/ uint16(x86_xMatch),
	/*12204*/ uint16(x86_xSetOp), uint16(x86_FISTTP),
	/*12206*/ uint16(x86_xArgM64int),
	/*12207*/ uint16(x86_xMatch),
	/*12208*/ uint16(x86_xSetOp), uint16(x86_FST),
	/*12210*/ uint16(x86_xArgM64fp),
	/*12211*/ uint16(x86_xMatch),
	/*12212*/ uint16(x86_xSetOp), uint16(x86_FSTP),
	/*12214*/ uint16(x86_xArgM64fp),
	/*12215*/ uint16(x86_xMatch),
	/*12216*/ uint16(x86_xSetOp), uint16(x86_FRSTOR),
	/*12218*/ uint16(x86_xArgM94108byte),
	/*12219*/ uint16(x86_xMatch),
	/*12220*/ uint16(x86_xSetOp), uint16(x86_FNSAVE),
	/*12222*/ uint16(x86_xArgM94108byte),
	/*12223*/ uint16(x86_xMatch),
	/*12224*/ uint16(x86_xSetOp), uint16(x86_FNSTSW),
	/*12226*/ uint16(x86_xArgM2byte),
	/*12227*/ uint16(x86_xMatch),
	/*12228*/ uint16(x86_xSetOp), uint16(x86_FFREE),
	/*12230*/ uint16(x86_xArgSTi),
	/*12231*/ uint16(x86_xMatch),
	/*12232*/ uint16(x86_xSetOp), uint16(x86_FST),
	/*12234*/ uint16(x86_xArgSTi),
	/*12235*/ uint16(x86_xMatch),
	/*12236*/ uint16(x86_xSetOp), uint16(x86_FSTP),
	/*12238*/ uint16(x86_xArgSTi),
	/*12239*/ uint16(x86_xMatch),
	/*12240*/ uint16(x86_xSetOp), uint16(x86_FUCOM),
	/*12242*/ uint16(x86_xArgSTi),
	/*12243*/ uint16(x86_xMatch),
	/*12244*/ uint16(x86_xSetOp), uint16(x86_FUCOMP),
	/*12246*/ uint16(x86_xArgSTi),
	/*12247*/ uint16(x86_xMatch),
	/*12248*/ uint16(x86_xCondByte), 49,
	0xc0, 12389,
	0xc1, 12389,
	0xc2, 12389,
	0xc3, 12389,
	0xc4, 12389,
	0xc5, 12389,
	0xc6, 12389,
	0xc7, 12389,
	0xc8, 12394,
	0xc9, 12394,
	0xca, 12394,
	0xcb, 12394,
	0xcc, 12394,
	0xcd, 12394,
	0xce, 12394,
	0xcf, 12394,
	0xD9, 12399,
	0xe0, 12402,
	0xe1, 12402,
	0xe2, 12402,
	0xe3, 12402,
	0xe4, 12402,
	0xe5, 12402,
	0xe6, 12402,
	0xe7, 12402,
	0xe8, 12407,
	0xe9, 12407,
	0xea, 12407,
	0xeb, 12407,
	0xec, 12407,
	0xed, 12407,
	0xee, 12407,
	0xef, 12407,
	0xf0, 12412,
	0xf1, 12412,
	0xf2, 12412,
	0xf3, 12412,
	0xf4, 12412,
	0xf5, 12412,
	0xf6, 12412,
	0xf7, 12412,
	0xf8, 12417,
	0xf9, 12417,
	0xfa, 12417,
	0xfb, 12417,
	0xfc, 12417,
	0xfd, 12417,
	0xfe, 12417,
	0xff, 12417,
	/*12348*/ uint16(x86_xCondSlashR),
	12357, // 0
	12361, // 1
	12365, // 2
	12369, // 3
	12373, // 4
	12377, // 5
	12381, // 6
	12385, // 7
	/*12357*/ uint16(x86_xSetOp), uint16(x86_FIADD),
	/*12359*/ uint16(x86_xArgM16int),
	/*12360*/ uint16(x86_xMatch),
	/*12361*/ uint16(x86_xSetOp), uint16(x86_FIMUL),
	/*12363*/ uint16(x86_xArgM16int),
	/*12364*/ uint16(x86_xMatch),
	/*12365*/ uint16(x86_xSetOp), uint16(x86_FICOM),
	/*12367*/ uint16(x86_xArgM16int),
	/*12368*/ uint16(x86_xMatch),
	/*12369*/ uint16(x86_xSetOp), uint16(x86_FICOMP),
	/*12371*/ uint16(x86_xArgM16int),
	/*12372*/ uint16(x86_xMatch),
	/*12373*/ uint16(x86_xSetOp), uint16(x86_FISUB),
	/*12375*/ uint16(x86_xArgM16int),
	/*12376*/ uint16(x86_xMatch),
	/*12377*/ uint16(x86_xSetOp), uint16(x86_FISUBR),
	/*12379*/ uint16(x86_xArgM16int),
	/*12380*/ uint16(x86_xMatch),
	/*12381*/ uint16(x86_xSetOp), uint16(x86_FIDIV),
	/*12383*/ uint16(x86_xArgM16int),
	/*12384*/ uint16(x86_xMatch),
	/*12385*/ uint16(x86_xSetOp), uint16(x86_FIDIVR),
	/*12387*/ uint16(x86_xArgM16int),
	/*12388*/ uint16(x86_xMatch),
	/*12389*/ uint16(x86_xSetOp), uint16(x86_FADDP),
	/*12391*/ uint16(x86_xArgSTi),
	/*12392*/ uint16(x86_xArgST),
	/*12393*/ uint16(x86_xMatch),
	/*12394*/ uint16(x86_xSetOp), uint16(x86_FMULP),
	/*12396*/ uint16(x86_xArgSTi),
	/*12397*/ uint16(x86_xArgST),
	/*12398*/ uint16(x86_xMatch),
	/*12399*/ uint16(x86_xSetOp), uint16(x86_FCOMPP),
	/*12401*/ uint16(x86_xMatch),
	/*12402*/ uint16(x86_xSetOp), uint16(x86_FSUBRP),
	/*12404*/ uint16(x86_xArgSTi),
	/*12405*/ uint16(x86_xArgST),
	/*12406*/ uint16(x86_xMatch),
	/*12407*/ uint16(x86_xSetOp), uint16(x86_FSUBP),
	/*12409*/ uint16(x86_xArgSTi),
	/*12410*/ uint16(x86_xArgST),
	/*12411*/ uint16(x86_xMatch),
	/*12412*/ uint16(x86_xSetOp), uint16(x86_FDIVRP),
	/*12414*/ uint16(x86_xArgSTi),
	/*12415*/ uint16(x86_xArgST),
	/*12416*/ uint16(x86_xMatch),
	/*12417*/ uint16(x86_xSetOp), uint16(x86_FDIVP),
	/*12419*/ uint16(x86_xArgSTi),
	/*12420*/ uint16(x86_xArgST),
	/*12421*/ uint16(x86_xMatch),
	/*12422*/ uint16(x86_xCondByte), 25,
	0xc0, 12515,
	0xc1, 12515,
	0xc2, 12515,
	0xc3, 12515,
	0xc4, 12515,
	0xc5, 12515,
	0xc6, 12515,
	0xc7, 12515,
	0xE0, 12519,
	0xe8, 12523,
	0xe9, 12523,
	0xea, 12523,
	0xeb, 12523,
	0xec, 12523,
	0xed, 12523,
	0xee, 12523,
	0xef, 12523,
	0xf0, 12528,
	0xf1, 12528,
	0xf2, 12528,
	0xf3, 12528,
	0xf4, 12528,
	0xf5, 12528,
	0xf6, 12528,
	0xf7, 12528,
	/*12474*/ uint16(x86_xCondSlashR),
	12483, // 0
	12487, // 1
	12491, // 2
	12495, // 3
	12499, // 4
	12503, // 5
	12507, // 6
	12511, // 7
	/*12483*/ uint16(x86_xSetOp), uint16(x86_FILD),
	/*12485*/ uint16(x86_xArgM16int),
	/*12486*/ uint16(x86_xMatch),
	/*12487*/ uint16(x86_xSetOp), uint16(x86_FISTTP),
	/*12489*/ uint16(x86_xArgM16int),
	/*12490*/ uint16(x86_xMatch),
	/*12491*/ uint16(x86_xSetOp), uint16(x86_FIST),
	/*12493*/ uint16(x86_xArgM16int),
	/*12494*/ uint16(x86_xMatch),
	/*12495*/ uint16(x86_xSetOp), uint16(x86_FISTP),
	/*12497*/ uint16(x86_xArgM16int),
	/*12498*/ uint16(x86_xMatch),
	/*12499*/ uint16(x86_xSetOp), uint16(x86_FBLD),
	/*12501*/ uint16(x86_xArgM80dec),
	/*12502*/ uint16(x86_xMatch),
	/*12503*/ uint16(x86_xSetOp), uint16(x86_FILD),
	/*12505*/ uint16(x86_xArgM64int),
	/*12506*/ uint16(x86_xMatch),
	/*12507*/ uint16(x86_xSetOp), uint16(x86_FBSTP),
	/*12509*/ uint16(x86_xArgM80bcd),
	/*12510*/ uint16(x86_xMatch),
	/*12511*/ uint16(x86_xSetOp), uint16(x86_FISTP),
	/*12513*/ uint16(x86_xArgM64int),
	/*12514*/ uint16(x86_xMatch),
	/*12515*/ uint16(x86_xSetOp), uint16(x86_FFREEP),
	/*12517*/ uint16(x86_xArgSTi),
	/*12518*/ uint16(x86_xMatch),
	/*12519*/ uint16(x86_xSetOp), uint16(x86_FNSTSW),
	/*12521*/ uint16(x86_xArgAX),
	/*12522*/ uint16(x86_xMatch),
	/*12523*/ uint16(x86_xSetOp), uint16(x86_FUCOMIP),
	/*12525*/ uint16(x86_xArgST),
	/*12526*/ uint16(x86_xArgSTi),
	/*12527*/ uint16(x86_xMatch),
	/*12528*/ uint16(x86_xSetOp), uint16(x86_FCOMIP),
	/*12530*/ uint16(x86_xArgST),
	/*12531*/ uint16(x86_xArgSTi),
	/*12532*/ uint16(x86_xMatch),
	/*12533*/ uint16(x86_xSetOp), uint16(x86_LOOPNE),
	/*12535*/ uint16(x86_xReadCb),
	/*12536*/ uint16(x86_xArgRel8),
	/*12537*/ uint16(x86_xMatch),
	/*12538*/ uint16(x86_xSetOp), uint16(x86_LOOPE),
	/*12540*/ uint16(x86_xReadCb),
	/*12541*/ uint16(x86_xArgRel8),
	/*12542*/ uint16(x86_xMatch),
	/*12543*/ uint16(x86_xSetOp), uint16(x86_LOOP),
	/*12545*/ uint16(x86_xReadCb),
	/*12546*/ uint16(x86_xArgRel8),
	/*12547*/ uint16(x86_xMatch),
	/*12548*/ uint16(x86_xCondIs64), 12551, 12565,
	/*12551*/ uint16(x86_xCondAddrSize), 12555, 12560, 0,
	/*12555*/ uint16(x86_xSetOp), uint16(x86_JCXZ),
	/*12557*/ uint16(x86_xReadCb),
	/*12558*/ uint16(x86_xArgRel8),
	/*12559*/ uint16(x86_xMatch),
	/*12560*/ uint16(x86_xSetOp), uint16(x86_JECXZ),
	/*12562*/ uint16(x86_xReadCb),
	/*12563*/ uint16(x86_xArgRel8),
	/*12564*/ uint16(x86_xMatch),
	/*12565*/ uint16(x86_xCondAddrSize), 0, 12560, 12569,
	/*12569*/ uint16(x86_xSetOp), uint16(x86_JRCXZ),
	/*12571*/ uint16(x86_xReadCb),
	/*12572*/ uint16(x86_xArgRel8),
	/*12573*/ uint16(x86_xMatch),
	/*12574*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12576*/ uint16(x86_xReadIb),
	/*12577*/ uint16(x86_xArgAL),
	/*12578*/ uint16(x86_xArgImm8u),
	/*12579*/ uint16(x86_xMatch),
	/*12580*/ uint16(x86_xCondDataSize), 12584, 12590, 12596,
	/*12584*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12586*/ uint16(x86_xReadIb),
	/*12587*/ uint16(x86_xArgAX),
	/*12588*/ uint16(x86_xArgImm8u),
	/*12589*/ uint16(x86_xMatch),
	/*12590*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12592*/ uint16(x86_xReadIb),
	/*12593*/ uint16(x86_xArgEAX),
	/*12594*/ uint16(x86_xArgImm8u),
	/*12595*/ uint16(x86_xMatch),
	/*12596*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12598*/ uint16(x86_xReadIb),
	/*12599*/ uint16(x86_xArgEAX),
	/*12600*/ uint16(x86_xArgImm8u),
	/*12601*/ uint16(x86_xMatch),
	/*12602*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12604*/ uint16(x86_xReadIb),
	/*12605*/ uint16(x86_xArgImm8u),
	/*12606*/ uint16(x86_xArgAL),
	/*12607*/ uint16(x86_xMatch),
	/*12608*/ uint16(x86_xCondDataSize), 12612, 12618, 12624,
	/*12612*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12614*/ uint16(x86_xReadIb),
	/*12615*/ uint16(x86_xArgImm8u),
	/*12616*/ uint16(x86_xArgAX),
	/*12617*/ uint16(x86_xMatch),
	/*12618*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12620*/ uint16(x86_xReadIb),
	/*12621*/ uint16(x86_xArgImm8u),
	/*12622*/ uint16(x86_xArgEAX),
	/*12623*/ uint16(x86_xMatch),
	/*12624*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12626*/ uint16(x86_xReadIb),
	/*12627*/ uint16(x86_xArgImm8u),
	/*12628*/ uint16(x86_xArgEAX),
	/*12629*/ uint16(x86_xMatch),
	/*12630*/ uint16(x86_xCondIs64), 12633, 12647,
	/*12633*/ uint16(x86_xCondDataSize), 12637, 12642, 0,
	/*12637*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*12639*/ uint16(x86_xReadCw),
	/*12640*/ uint16(x86_xArgRel16),
	/*12641*/ uint16(x86_xMatch),
	/*12642*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*12644*/ uint16(x86_xReadCd),
	/*12645*/ uint16(x86_xArgRel32),
	/*12646*/ uint16(x86_xMatch),
	/*12647*/ uint16(x86_xCondDataSize), 12651, 12642, 12656,
	/*12651*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*12653*/ uint16(x86_xReadCd),
	/*12654*/ uint16(x86_xArgRel32),
	/*12655*/ uint16(x86_xMatch),
	/*12656*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*12658*/ uint16(x86_xReadCd),
	/*12659*/ uint16(x86_xArgRel32),
	/*12660*/ uint16(x86_xMatch),
	/*12661*/ uint16(x86_xCondIs64), 12664, 12678,
	/*12664*/ uint16(x86_xCondDataSize), 12668, 12673, 0,
	/*12668*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*12670*/ uint16(x86_xReadCw),
	/*12671*/ uint16(x86_xArgRel16),
	/*12672*/ uint16(x86_xMatch),
	/*12673*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*12675*/ uint16(x86_xReadCd),
	/*12676*/ uint16(x86_xArgRel32),
	/*12677*/ uint16(x86_xMatch),
	/*12678*/ uint16(x86_xCondDataSize), 12682, 12673, 12687,
	/*12682*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*12684*/ uint16(x86_xReadCd),
	/*12685*/ uint16(x86_xArgRel32),
	/*12686*/ uint16(x86_xMatch),
	/*12687*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*12689*/ uint16(x86_xReadCd),
	/*12690*/ uint16(x86_xArgRel32),
	/*12691*/ uint16(x86_xMatch),
	/*12692*/ uint16(x86_xCondIs64), 12695, 0,
	/*12695*/ uint16(x86_xCondDataSize), 12699, 12704, 0,
	/*12699*/ uint16(x86_xSetOp), uint16(x86_LJMP),
	/*12701*/ uint16(x86_xReadCd),
	/*12702*/ uint16(x86_xArgPtr16colon16),
	/*12703*/ uint16(x86_xMatch),
	/*12704*/ uint16(x86_xSetOp), uint16(x86_LJMP),
	/*12706*/ uint16(x86_xReadCp),
	/*12707*/ uint16(x86_xArgPtr16colon32),
	/*12708*/ uint16(x86_xMatch),
	/*12709*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*12711*/ uint16(x86_xReadCb),
	/*12712*/ uint16(x86_xArgRel8),
	/*12713*/ uint16(x86_xMatch),
	/*12714*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12716*/ uint16(x86_xArgAL),
	/*12717*/ uint16(x86_xArgDX),
	/*12718*/ uint16(x86_xMatch),
	/*12719*/ uint16(x86_xCondDataSize), 12723, 12728, 12733,
	/*12723*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12725*/ uint16(x86_xArgAX),
	/*12726*/ uint16(x86_xArgDX),
	/*12727*/ uint16(x86_xMatch),
	/*12728*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12730*/ uint16(x86_xArgEAX),
	/*12731*/ uint16(x86_xArgDX),
	/*12732*/ uint16(x86_xMatch),
	/*12733*/ uint16(x86_xSetOp), uint16(x86_IN),
	/*12735*/ uint16(x86_xArgEAX),
	/*12736*/ uint16(x86_xArgDX),
	/*12737*/ uint16(x86_xMatch),
	/*12738*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12740*/ uint16(x86_xArgDX),
	/*12741*/ uint16(x86_xArgAL),
	/*12742*/ uint16(x86_xMatch),
	/*12743*/ uint16(x86_xCondDataSize), 12747, 12752, 12757,
	/*12747*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12749*/ uint16(x86_xArgDX),
	/*12750*/ uint16(x86_xArgAX),
	/*12751*/ uint16(x86_xMatch),
	/*12752*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12754*/ uint16(x86_xArgDX),
	/*12755*/ uint16(x86_xArgEAX),
	/*12756*/ uint16(x86_xMatch),
	/*12757*/ uint16(x86_xSetOp), uint16(x86_OUT),
	/*12759*/ uint16(x86_xArgDX),
	/*12760*/ uint16(x86_xArgEAX),
	/*12761*/ uint16(x86_xMatch),
	/*12762*/ uint16(x86_xSetOp), uint16(x86_ICEBP),
	/*12764*/ uint16(x86_xMatch),
	/*12765*/ uint16(x86_xSetOp), uint16(x86_HLT),
	/*12767*/ uint16(x86_xMatch),
	/*12768*/ uint16(x86_xSetOp), uint16(x86_CMC),
	/*12770*/ uint16(x86_xMatch),
	/*12771*/ uint16(x86_xCondSlashR),
	12780, // 0
	0,     // 1
	12786, // 2
	12790, // 3
	12794, // 4
	12798, // 5
	12802, // 6
	12806, // 7
	/*12780*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*12782*/ uint16(x86_xReadIb),
	/*12783*/ uint16(x86_xArgRM8),
	/*12784*/ uint16(x86_xArgImm8u),
	/*12785*/ uint16(x86_xMatch),
	/*12786*/ uint16(x86_xSetOp), uint16(x86_NOT),
	/*12788*/ uint16(x86_xArgRM8),
	/*12789*/ uint16(x86_xMatch),
	/*12790*/ uint16(x86_xSetOp), uint16(x86_NEG),
	/*12792*/ uint16(x86_xArgRM8),
	/*12793*/ uint16(x86_xMatch),
	/*12794*/ uint16(x86_xSetOp), uint16(x86_MUL),
	/*12796*/ uint16(x86_xArgRM8),
	/*12797*/ uint16(x86_xMatch),
	/*12798*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*12800*/ uint16(x86_xArgRM8),
	/*12801*/ uint16(x86_xMatch),
	/*12802*/ uint16(x86_xSetOp), uint16(x86_DIV),
	/*12804*/ uint16(x86_xArgRM8),
	/*12805*/ uint16(x86_xMatch),
	/*12806*/ uint16(x86_xSetOp), uint16(x86_IDIV),
	/*12808*/ uint16(x86_xArgRM8),
	/*12809*/ uint16(x86_xMatch),
	/*12810*/ uint16(x86_xCondSlashR),
	12819, // 0
	0,     // 1
	12848, // 2
	12871, // 3
	12894, // 4
	12917, // 5
	12940, // 6
	12963, // 7
	/*12819*/ uint16(x86_xCondIs64), 12822, 12838,
	/*12822*/ uint16(x86_xCondDataSize), 12826, 12832, 0,
	/*12826*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*12828*/ uint16(x86_xReadIw),
	/*12829*/ uint16(x86_xArgRM16),
	/*12830*/ uint16(x86_xArgImm16),
	/*12831*/ uint16(x86_xMatch),
	/*12832*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*12834*/ uint16(x86_xReadId),
	/*12835*/ uint16(x86_xArgRM32),
	/*12836*/ uint16(x86_xArgImm32),
	/*12837*/ uint16(x86_xMatch),
	/*12838*/ uint16(x86_xCondDataSize), 12826, 12832, 12842,
	/*12842*/ uint16(x86_xSetOp), uint16(x86_TEST),
	/*12844*/ uint16(x86_xReadId),
	/*12845*/ uint16(x86_xArgRM64),
	/*12846*/ uint16(x86_xArgImm32),
	/*12847*/ uint16(x86_xMatch),
	/*12848*/ uint16(x86_xCondIs64), 12851, 12863,
	/*12851*/ uint16(x86_xCondDataSize), 12855, 12859, 0,
	/*12855*/ uint16(x86_xSetOp), uint16(x86_NOT),
	/*12857*/ uint16(x86_xArgRM16),
	/*12858*/ uint16(x86_xMatch),
	/*12859*/ uint16(x86_xSetOp), uint16(x86_NOT),
	/*12861*/ uint16(x86_xArgRM32),
	/*12862*/ uint16(x86_xMatch),
	/*12863*/ uint16(x86_xCondDataSize), 12855, 12859, 12867,
	/*12867*/ uint16(x86_xSetOp), uint16(x86_NOT),
	/*12869*/ uint16(x86_xArgRM64),
	/*12870*/ uint16(x86_xMatch),
	/*12871*/ uint16(x86_xCondIs64), 12874, 12886,
	/*12874*/ uint16(x86_xCondDataSize), 12878, 12882, 0,
	/*12878*/ uint16(x86_xSetOp), uint16(x86_NEG),
	/*12880*/ uint16(x86_xArgRM16),
	/*12881*/ uint16(x86_xMatch),
	/*12882*/ uint16(x86_xSetOp), uint16(x86_NEG),
	/*12884*/ uint16(x86_xArgRM32),
	/*12885*/ uint16(x86_xMatch),
	/*12886*/ uint16(x86_xCondDataSize), 12878, 12882, 12890,
	/*12890*/ uint16(x86_xSetOp), uint16(x86_NEG),
	/*12892*/ uint16(x86_xArgRM64),
	/*12893*/ uint16(x86_xMatch),
	/*12894*/ uint16(x86_xCondIs64), 12897, 12909,
	/*12897*/ uint16(x86_xCondDataSize), 12901, 12905, 0,
	/*12901*/ uint16(x86_xSetOp), uint16(x86_MUL),
	/*12903*/ uint16(x86_xArgRM16),
	/*12904*/ uint16(x86_xMatch),
	/*12905*/ uint16(x86_xSetOp), uint16(x86_MUL),
	/*12907*/ uint16(x86_xArgRM32),
	/*12908*/ uint16(x86_xMatch),
	/*12909*/ uint16(x86_xCondDataSize), 12901, 12905, 12913,
	/*12913*/ uint16(x86_xSetOp), uint16(x86_MUL),
	/*12915*/ uint16(x86_xArgRM64),
	/*12916*/ uint16(x86_xMatch),
	/*12917*/ uint16(x86_xCondIs64), 12920, 12932,
	/*12920*/ uint16(x86_xCondDataSize), 12924, 12928, 0,
	/*12924*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*12926*/ uint16(x86_xArgRM16),
	/*12927*/ uint16(x86_xMatch),
	/*12928*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*12930*/ uint16(x86_xArgRM32),
	/*12931*/ uint16(x86_xMatch),
	/*12932*/ uint16(x86_xCondDataSize), 12924, 12928, 12936,
	/*12936*/ uint16(x86_xSetOp), uint16(x86_IMUL),
	/*12938*/ uint16(x86_xArgRM64),
	/*12939*/ uint16(x86_xMatch),
	/*12940*/ uint16(x86_xCondIs64), 12943, 12955,
	/*12943*/ uint16(x86_xCondDataSize), 12947, 12951, 0,
	/*12947*/ uint16(x86_xSetOp), uint16(x86_DIV),
	/*12949*/ uint16(x86_xArgRM16),
	/*12950*/ uint16(x86_xMatch),
	/*12951*/ uint16(x86_xSetOp), uint16(x86_DIV),
	/*12953*/ uint16(x86_xArgRM32),
	/*12954*/ uint16(x86_xMatch),
	/*12955*/ uint16(x86_xCondDataSize), 12947, 12951, 12959,
	/*12959*/ uint16(x86_xSetOp), uint16(x86_DIV),
	/*12961*/ uint16(x86_xArgRM64),
	/*12962*/ uint16(x86_xMatch),
	/*12963*/ uint16(x86_xCondIs64), 12966, 12978,
	/*12966*/ uint16(x86_xCondDataSize), 12970, 12974, 0,
	/*12970*/ uint16(x86_xSetOp), uint16(x86_IDIV),
	/*12972*/ uint16(x86_xArgRM16),
	/*12973*/ uint16(x86_xMatch),
	/*12974*/ uint16(x86_xSetOp), uint16(x86_IDIV),
	/*12976*/ uint16(x86_xArgRM32),
	/*12977*/ uint16(x86_xMatch),
	/*12978*/ uint16(x86_xCondDataSize), 12970, 12974, 12982,
	/*12982*/ uint16(x86_xSetOp), uint16(x86_IDIV),
	/*12984*/ uint16(x86_xArgRM64),
	/*12985*/ uint16(x86_xMatch),
	/*12986*/ uint16(x86_xSetOp), uint16(x86_CLC),
	/*12988*/ uint16(x86_xMatch),
	/*12989*/ uint16(x86_xSetOp), uint16(x86_STC),
	/*12991*/ uint16(x86_xMatch),
	/*12992*/ uint16(x86_xSetOp), uint16(x86_CLI),
	/*12994*/ uint16(x86_xMatch),
	/*12995*/ uint16(x86_xSetOp), uint16(x86_STI),
	/*12997*/ uint16(x86_xMatch),
	/*12998*/ uint16(x86_xSetOp), uint16(x86_CLD),
	/*13000*/ uint16(x86_xMatch),
	/*13001*/ uint16(x86_xSetOp), uint16(x86_STD),
	/*13003*/ uint16(x86_xMatch),
	/*13004*/ uint16(x86_xCondSlashR),
	13013, // 0
	13017, // 1
	0,     // 2
	0,     // 3
	0,     // 4
	0,     // 5
	0,     // 6
	0,     // 7
	/*13013*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*13015*/ uint16(x86_xArgRM8),
	/*13016*/ uint16(x86_xMatch),
	/*13017*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*13019*/ uint16(x86_xArgRM8),
	/*13020*/ uint16(x86_xMatch),
	/*13021*/ uint16(x86_xCondSlashR),
	13030, // 0
	13053, // 1
	13076, // 2
	13095, // 3
	13118, // 4
	13137, // 5
	13160, // 6
	0,     // 7
	/*13030*/ uint16(x86_xCondIs64), 13033, 13045,
	/*13033*/ uint16(x86_xCondDataSize), 13037, 13041, 0,
	/*13037*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*13039*/ uint16(x86_xArgRM16),
	/*13040*/ uint16(x86_xMatch),
	/*13041*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*13043*/ uint16(x86_xArgRM32),
	/*13044*/ uint16(x86_xMatch),
	/*13045*/ uint16(x86_xCondDataSize), 13037, 13041, 13049,
	/*13049*/ uint16(x86_xSetOp), uint16(x86_INC),
	/*13051*/ uint16(x86_xArgRM64),
	/*13052*/ uint16(x86_xMatch),
	/*13053*/ uint16(x86_xCondIs64), 13056, 13068,
	/*13056*/ uint16(x86_xCondDataSize), 13060, 13064, 0,
	/*13060*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*13062*/ uint16(x86_xArgRM16),
	/*13063*/ uint16(x86_xMatch),
	/*13064*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*13066*/ uint16(x86_xArgRM32),
	/*13067*/ uint16(x86_xMatch),
	/*13068*/ uint16(x86_xCondDataSize), 13060, 13064, 13072,
	/*13072*/ uint16(x86_xSetOp), uint16(x86_DEC),
	/*13074*/ uint16(x86_xArgRM64),
	/*13075*/ uint16(x86_xMatch),
	/*13076*/ uint16(x86_xCondIs64), 13079, 13091,
	/*13079*/ uint16(x86_xCondDataSize), 13083, 13087, 0,
	/*13083*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*13085*/ uint16(x86_xArgRM16),
	/*13086*/ uint16(x86_xMatch),
	/*13087*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*13089*/ uint16(x86_xArgRM32),
	/*13090*/ uint16(x86_xMatch),
	/*13091*/ uint16(x86_xSetOp), uint16(x86_CALL),
	/*13093*/ uint16(x86_xArgRM64),
	/*13094*/ uint16(x86_xMatch),
	/*13095*/ uint16(x86_xCondIs64), 13098, 13110,
	/*13098*/ uint16(x86_xCondDataSize), 13102, 13106, 0,
	/*13102*/ uint16(x86_xSetOp), uint16(x86_LCALL),
	/*13104*/ uint16(x86_xArgM16colon16),
	/*13105*/ uint16(x86_xMatch),
	/*13106*/ uint16(x86_xSetOp), uint16(x86_LCALL),
	/*13108*/ uint16(x86_xArgM16colon32),
	/*13109*/ uint16(x86_xMatch),
	/*13110*/ uint16(x86_xCondDataSize), 13102, 13106, 13114,
	/*13114*/ uint16(x86_xSetOp), uint16(x86_LCALL),
	/*13116*/ uint16(x86_xArgM16colon64),
	/*13117*/ uint16(x86_xMatch),
	/*13118*/ uint16(x86_xCondIs64), 13121, 13133,
	/*13121*/ uint16(x86_xCondDataSize), 13125, 13129, 0,
	/*13125*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*13127*/ uint16(x86_xArgRM16),
	/*13128*/ uint16(x86_xMatch),
	/*13129*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*13131*/ uint16(x86_xArgRM32),
	/*13132*/ uint16(x86_xMatch),
	/*13133*/ uint16(x86_xSetOp), uint16(x86_JMP),
	/*13135*/ uint16(x86_xArgRM64),
	/*13136*/ uint16(x86_xMatch),
	/*13137*/ uint16(x86_xCondIs64), 13140, 13152,
	/*13140*/ uint16(x86_xCondDataSize), 13144, 13148, 0,
	/*13144*/ uint16(x86_xSetOp), uint16(x86_LJMP),
	/*13146*/ uint16(x86_xArgM16colon16),
	/*13147*/ uint16(x86_xMatch),
	/*13148*/ uint16(x86_xSetOp), uint16(x86_LJMP),
	/*13150*/ uint16(x86_xArgM16colon32),
	/*13151*/ uint16(x86_xMatch),
	/*13152*/ uint16(x86_xCondDataSize), 13144, 13148, 13156,
	/*13156*/ uint16(x86_xSetOp), uint16(x86_LJMP),
	/*13158*/ uint16(x86_xArgM16colon64),
	/*13159*/ uint16(x86_xMatch),
	/*13160*/ uint16(x86_xCondIs64), 13163, 13175,
	/*13163*/ uint16(x86_xCondDataSize), 13167, 13171, 0,
	/*13167*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*13169*/ uint16(x86_xArgRM16),
	/*13170*/ uint16(x86_xMatch),
	/*13171*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*13173*/ uint16(x86_xArgRM32),
	/*13174*/ uint16(x86_xMatch),
	/*13175*/ uint16(x86_xCondDataSize), 13167, 13179, 13183,
	/*13179*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*13181*/ uint16(x86_xArgRM64),
	/*13182*/ uint16(x86_xMatch),
	/*13183*/ uint16(x86_xSetOp), uint16(x86_PUSH),
	/*13185*/ uint16(x86_xArgRM64),
	/*13186*/ uint16(x86_xMatch),
}

const (
	_ x86_Op = iota

	x86_AAA
	x86_AAD
	x86_AAM
	x86_AAS
	x86_ADC
	x86_ADD
	x86_ADDPD
	x86_ADDPS
	x86_ADDSD
	x86_ADDSS
	x86_ADDSUBPD
	x86_ADDSUBPS
	x86_AESDEC
	x86_AESDECLAST
	x86_AESENC
	x86_AESENCLAST
	x86_AESIMC
	x86_AESKEYGENASSIST
	x86_AND
	x86_ANDNPD
	x86_ANDNPS
	x86_ANDPD
	x86_ANDPS
	x86_ARPL
	x86_BLENDPD
	x86_BLENDPS
	x86_BLENDVPD
	x86_BLENDVPS
	x86_BOUND
	x86_BSF
	x86_BSR
	x86_BSWAP
	x86_BT
	x86_BTC
	x86_BTR
	x86_BTS
	x86_CALL
	x86_CBW
	x86_CDQ
	x86_CDQE
	x86_CLC
	x86_CLD
	x86_CLFLUSH
	x86_CLI
	x86_CLTS
	x86_CMC
	x86_CMOVA
	x86_CMOVAE
	x86_CMOVB
	x86_CMOVBE
	x86_CMOVE
	x86_CMOVG
	x86_CMOVGE
	x86_CMOVL
	x86_CMOVLE
	x86_CMOVNE
	x86_CMOVNO
	x86_CMOVNP
	x86_CMOVNS
	x86_CMOVO
	x86_CMOVP
	x86_CMOVS
	x86_CMP
	x86_CMPPD
	x86_CMPPS
	x86_CMPSB
	x86_CMPSD
	x86_CMPSD_XMM
	x86_CMPSQ
	x86_CMPSS
	x86_CMPSW
	x86_CMPXCHG
	x86_CMPXCHG16B
	x86_CMPXCHG8B
	x86_COMISD
	x86_COMISS
	x86_CPUID
	x86_CQO
	x86_CRC32
	x86_CVTDQ2PD
	x86_CVTDQ2PS
	x86_CVTPD2DQ
	x86_CVTPD2PI
	x86_CVTPD2PS
	x86_CVTPI2PD
	x86_CVTPI2PS
	x86_CVTPS2DQ
	x86_CVTPS2PD
	x86_CVTPS2PI
	x86_CVTSD2SI
	x86_CVTSD2SS
	x86_CVTSI2SD
	x86_CVTSI2SS
	x86_CVTSS2SD
	x86_CVTSS2SI
	x86_CVTTPD2DQ
	x86_CVTTPD2PI
	x86_CVTTPS2DQ
	x86_CVTTPS2PI
	x86_CVTTSD2SI
	x86_CVTTSS2SI
	x86_CWD
	x86_CWDE
	x86_DAA
	x86_DAS
	x86_DEC
	x86_DIV
	x86_DIVPD
	x86_DIVPS
	x86_DIVSD
	x86_DIVSS
	x86_DPPD
	x86_DPPS
	x86_EMMS
	x86_ENTER
	x86_EXTRACTPS
	x86_F2XM1
	x86_FABS
	x86_FADD
	x86_FADDP
	x86_FBLD
	x86_FBSTP
	x86_FCHS
	x86_FCMOVB
	x86_FCMOVBE
	x86_FCMOVE
	x86_FCMOVNB
	x86_FCMOVNBE
	x86_FCMOVNE
	x86_FCMOVNU
	x86_FCMOVU
	x86_FCOM
	x86_FCOMI
	x86_FCOMIP
	x86_FCOMP
	x86_FCOMPP
	x86_FCOS
	x86_FDECSTP
	x86_FDIV
	x86_FDIVP
	x86_FDIVR
	x86_FDIVRP
	x86_FFREE
	x86_FFREEP
	x86_FIADD
	x86_FICOM
	x86_FICOMP
	x86_FIDIV
	x86_FIDIVR
	x86_FILD
	x86_FIMUL
	x86_FINCSTP
	x86_FIST
	x86_FISTP
	x86_FISTTP
	x86_FISUB
	x86_FISUBR
	x86_FLD
	x86_FLD1
	x86_FLDCW
	x86_FLDENV
	x86_FLDL2E
	x86_FLDL2T
	x86_FLDLG2
	x86_FLDPI
	x86_FMUL
	x86_FMULP
	x86_FNCLEX
	x86_FNINIT
	x86_FNOP
	x86_FNSAVE
	x86_FNSTCW
	x86_FNSTENV
	x86_FNSTSW
	x86_FPATAN
	x86_FPREM
	x86_FPREM1
	x86_FPTAN
	x86_FRNDINT
	x86_FRSTOR
	x86_FSCALE
	x86_FSIN
	x86_FSINCOS
	x86_FSQRT
	x86_FST
	x86_FSTP
	x86_FSUB
	x86_FSUBP
	x86_FSUBR
	x86_FSUBRP
	x86_FTST
	x86_FUCOM
	x86_FUCOMI
	x86_FUCOMIP
	x86_FUCOMP
	x86_FUCOMPP
	x86_FWAIT
	x86_FXAM
	x86_FXCH
	x86_FXRSTOR
	x86_FXRSTOR64
	x86_FXSAVE
	x86_FXSAVE64
	x86_FXTRACT
	x86_FYL2X
	x86_FYL2XP1
	x86_HADDPD
	x86_HADDPS
	x86_HLT
	x86_HSUBPD
	x86_HSUBPS
	x86_ICEBP
	x86_IDIV
	x86_IMUL
	x86_IN
	x86_INC
	x86_INSB
	x86_INSD
	x86_INSERTPS
	x86_INSW
	x86_INT
	x86_INTO
	x86_INVD
	x86_INVLPG
	x86_INVPCID
	x86_IRET
	x86_IRETD
	x86_IRETQ
	x86_JA
	x86_JAE
	x86_JB
	x86_JBE
	x86_JCXZ
	x86_JE
	x86_JECXZ
	x86_JG
	x86_JGE
	x86_JL
	x86_JLE
	x86_JMP
	x86_JNE
	x86_JNO
	x86_JNP
	x86_JNS
	x86_JO
	x86_JP
	x86_JRCXZ
	x86_JS
	x86_LAHF
	x86_LAR
	x86_LCALL
	x86_LDDQU
	x86_LDMXCSR
	x86_LDS
	x86_LEA
	x86_LEAVE
	x86_LES
	x86_LFENCE
	x86_LFS
	x86_LGDT
	x86_LGS
	x86_LIDT
	x86_LJMP
	x86_LLDT
	x86_LMSW
	x86_LODSB
	x86_LODSD
	x86_LODSQ
	x86_LODSW
	x86_LOOP
	x86_LOOPE
	x86_LOOPNE
	x86_LRET
	x86_LSL
	x86_LSS
	x86_LTR
	x86_LZCNT
	x86_MASKMOVDQU
	x86_MASKMOVQ
	x86_MAXPD
	x86_MAXPS
	x86_MAXSD
	x86_MAXSS
	x86_MFENCE
	x86_MINPD
	x86_MINPS
	x86_MINSD
	x86_MINSS
	x86_MONITOR
	x86_MOV
	x86_MOVAPD
	x86_MOVAPS
	x86_MOVBE
	x86_MOVD
	x86_MOVDDUP
	x86_MOVDQ2Q
	x86_MOVDQA
	x86_MOVDQU
	x86_MOVHLPS
	x86_MOVHPD
	x86_MOVHPS
	x86_MOVLHPS
	x86_MOVLPD
	x86_MOVLPS
	x86_MOVMSKPD
	x86_MOVMSKPS
	x86_MOVNTDQ
	x86_MOVNTDQA
	x86_MOVNTI
	x86_MOVNTPD
	x86_MOVNTPS
	x86_MOVNTQ
	x86_MOVNTSD
	x86_MOVNTSS
	x86_MOVQ
	x86_MOVQ2DQ
	x86_MOVSB
	x86_MOVSD
	x86_MOVSD_XMM
	x86_MOVSHDUP
	x86_MOVSLDUP
	x86_MOVSQ
	x86_MOVSS
	x86_MOVSW
	x86_MOVSX
	x86_MOVSXD
	x86_MOVUPD
	x86_MOVUPS
	x86_MOVZX
	x86_MPSADBW
	x86_MUL
	x86_MULPD
	x86_MULPS
	x86_MULSD
	x86_MULSS
	x86_MWAIT
	x86_NEG
	x86_NOP
	x86_NOT
	x86_OR
	x86_ORPD
	x86_ORPS
	x86_OUT
	x86_OUTSB
	x86_OUTSD
	x86_OUTSW
	x86_PABSB
	x86_PABSD
	x86_PABSW
	x86_PACKSSDW
	x86_PACKSSWB
	x86_PACKUSDW
	x86_PACKUSWB
	x86_PADDB
	x86_PADDD
	x86_PADDQ
	x86_PADDSB
	x86_PADDSW
	x86_PADDUSB
	x86_PADDUSW
	x86_PADDW
	x86_PALIGNR
	x86_PAND
	x86_PANDN
	x86_PAUSE
	x86_PAVGB
	x86_PAVGW
	x86_PBLENDVB
	x86_PBLENDW
	x86_PCLMULQDQ
	x86_PCMPEQB
	x86_PCMPEQD
	x86_PCMPEQQ
	x86_PCMPEQW
	x86_PCMPESTRI
	x86_PCMPESTRM
	x86_PCMPGTB
	x86_PCMPGTD
	x86_PCMPGTQ
	x86_PCMPGTW
	x86_PCMPISTRI
	x86_PCMPISTRM
	x86_PEXTRB
	x86_PEXTRD
	x86_PEXTRQ
	x86_PEXTRW
	x86_PHADDD
	x86_PHADDSW
	x86_PHADDW
	x86_PHMINPOSUW
	x86_PHSUBD
	x86_PHSUBSW
	x86_PHSUBW
	x86_PINSRB
	x86_PINSRD
	x86_PINSRQ
	x86_PINSRW
	x86_PMADDUBSW
	x86_PMADDWD
	x86_PMAXSB
	x86_PMAXSD
	x86_PMAXSW
	x86_PMAXUB
	x86_PMAXUD
	x86_PMAXUW
	x86_PMINSB
	x86_PMINSD
	x86_PMINSW
	x86_PMINUB
	x86_PMINUD
	x86_PMINUW
	x86_PMOVMSKB
	x86_PMOVSXBD
	x86_PMOVSXBQ
	x86_PMOVSXBW
	x86_PMOVSXDQ
	x86_PMOVSXWD
	x86_PMOVSXWQ
	x86_PMOVZXBD
	x86_PMOVZXBQ
	x86_PMOVZXBW
	x86_PMOVZXDQ
	x86_PMOVZXWD
	x86_PMOVZXWQ
	x86_PMULDQ
	x86_PMULHRSW
	x86_PMULHUW
	x86_PMULHW
	x86_PMULLD
	x86_PMULLW
	x86_PMULUDQ
	x86_POP
	x86_POPA
	x86_POPAD
	x86_POPCNT
	x86_POPF
	x86_POPFD
	x86_POPFQ
	x86_POR
	x86_PREFETCHNTA
	x86_PREFETCHT0
	x86_PREFETCHT1
	x86_PREFETCHT2
	x86_PREFETCHW
	x86_PSADBW
	x86_PSHUFB
	x86_PSHUFD
	x86_PSHUFHW
	x86_PSHUFLW
	x86_PSHUFW
	x86_PSIGNB
	x86_PSIGND
	x86_PSIGNW
	x86_PSLLD
	x86_PSLLDQ
	x86_PSLLQ
	x86_PSLLW
	x86_PSRAD
	x86_PSRAW
	x86_PSRLD
	x86_PSRLDQ
	x86_PSRLQ
	x86_PSRLW
	x86_PSUBB
	x86_PSUBD
	x86_PSUBQ
	x86_PSUBSB
	x86_PSUBSW
	x86_PSUBUSB
	x86_PSUBUSW
	x86_PSUBW
	x86_PTEST
	x86_PUNPCKHBW
	x86_PUNPCKHDQ
	x86_PUNPCKHQDQ
	x86_PUNPCKHWD
	x86_PUNPCKLBW
	x86_PUNPCKLDQ
	x86_PUNPCKLQDQ
	x86_PUNPCKLWD
	x86_PUSH
	x86_PUSHA
	x86_PUSHAD
	x86_PUSHF
	x86_PUSHFD
	x86_PUSHFQ
	x86_PXOR
	x86_RCL
	x86_RCPPS
	x86_RCPSS
	x86_RCR
	x86_RDFSBASE
	x86_RDGSBASE
	x86_RDMSR
	x86_RDPMC
	x86_RDRAND
	x86_RDTSC
	x86_RDTSCP
	x86_RET
	x86_ROL
	x86_ROR
	x86_ROUNDPD
	x86_ROUNDPS
	x86_ROUNDSD
	x86_ROUNDSS
	x86_RSM
	x86_RSQRTPS
	x86_RSQRTSS
	x86_SAHF
	x86_SAR
	x86_SBB
	x86_SCASB
	x86_SCASD
	x86_SCASQ
	x86_SCASW
	x86_SETA
	x86_SETAE
	x86_SETB
	x86_SETBE
	x86_SETE
	x86_SETG
	x86_SETGE
	x86_SETL
	x86_SETLE
	x86_SETNE
	x86_SETNO
	x86_SETNP
	x86_SETNS
	x86_SETO
	x86_SETP
	x86_SETS
	x86_SFENCE
	x86_SGDT
	x86_SHL
	x86_SHLD
	x86_SHR
	x86_SHRD
	x86_SHUFPD
	x86_SHUFPS
	x86_SIDT
	x86_SLDT
	x86_SMSW
	x86_SQRTPD
	x86_SQRTPS
	x86_SQRTSD
	x86_SQRTSS
	x86_STC
	x86_STD
	x86_STI
	x86_STMXCSR
	x86_STOSB
	x86_STOSD
	x86_STOSQ
	x86_STOSW
	x86_STR
	x86_SUB
	x86_SUBPD
	x86_SUBPS
	x86_SUBSD
	x86_SUBSS
	x86_SWAPGS
	x86_SYSCALL
	x86_SYSENTER
	x86_SYSEXIT
	x86_SYSRET
	x86_TEST
	x86_TZCNT
	x86_UCOMISD
	x86_UCOMISS
	x86_UD1
	x86_UD2
	x86_UNPCKHPD
	x86_UNPCKHPS
	x86_UNPCKLPD
	x86_UNPCKLPS
	x86_VERR
	x86_VERW
	x86_WBINVD
	x86_WRFSBASE
	x86_WRGSBASE
	x86_WRMSR
	x86_XABORT
	x86_XADD
	x86_XBEGIN
	x86_XCHG
	x86_XEND
	x86_XGETBV
	x86_XLATB
	x86_XOR
	x86_XORPD
	x86_XORPS
	x86_XRSTOR
	x86_XRSTOR64
	x86_XRSTORS
	x86_XRSTORS64
	x86_XSAVE
	x86_XSAVE64
	x86_XSAVEC
	x86_XSAVEC64
	x86_XSAVEOPT
	x86_XSAVEOPT64
	x86_XSAVES
	x86_XSAVES64
	x86_XSETBV
	x86_XTEST
)

const x86_maxOp = x86_XTEST

var x86_opNames = [...]string{
	x86_AAA:             "AAA",
	x86_AAD:             "AAD",
	x86_AAM:             "AAM",
	x86_AAS:             "AAS",
	x86_ADC:             "ADC",
	x86_ADD:             "ADD",
	x86_ADDPD:           "ADDPD",
	x86_ADDPS:           "ADDPS",
	x86_ADDSD:           "ADDSD",
	x86_ADDSS:           "ADDSS",
	x86_ADDSUBPD:        "ADDSUBPD",
	x86_ADDSUBPS:        "ADDSUBPS",
	x86_AESDEC:          "AESDEC",
	x86_AESDECLAST:      "AESDECLAST",
	x86_AESENC:          "AESENC",
	x86_AESENCLAST:      "AESENCLAST",
	x86_AESIMC:          "AESIMC",
	x86_AESKEYGENASSIST: "AESKEYGENASSIST",
	x86_AND:             "AND",
	x86_ANDNPD:          "ANDNPD",
	x86_ANDNPS:          "ANDNPS",
	x86_ANDPD:           "ANDPD",
	x86_ANDPS:           "ANDPS",
	x86_ARPL:            "ARPL",
	x86_BLENDPD:         "BLENDPD",
	x86_BLENDPS:         "BLENDPS",
	x86_BLENDVPD:        "BLENDVPD",
	x86_BLENDVPS:        "BLENDVPS",
	x86_BOUND:           "BOUND",
	x86_BSF:             "BSF",
	x86_BSR:             "BSR",
	x86_BSWAP:           "BSWAP",
	x86_BT:              "BT",
	x86_BTC:             "BTC",
	x86_BTR:             "BTR",
	x86_BTS:             "BTS",
	x86_CALL:            "CALL",
	x86_CBW:             "CBW",
	x86_CDQ:             "CDQ",
	x86_CDQE:            "CDQE",
	x86_CLC:             "CLC",
	x86_CLD:             "CLD",
	x86_CLFLUSH:         "CLFLUSH",
	x86_CLI:             "CLI",
	x86_CLTS:            "CLTS",
	x86_CMC:             "CMC",
	x86_CMOVA:           "CMOVA",
	x86_CMOVAE:          "CMOVAE",
	x86_CMOVB:           "CMOVB",
	x86_CMOVBE:          "CMOVBE",
	x86_CMOVE:           "CMOVE",
	x86_CMOVG:           "CMOVG",
	x86_CMOVGE:          "CMOVGE",
	x86_CMOVL:           "CMOVL",
	x86_CMOVLE:          "CMOVLE",
	x86_CMOVNE:          "CMOVNE",
	x86_CMOVNO:          "CMOVNO",
	x86_CMOVNP:          "CMOVNP",
	x86_CMOVNS:          "CMOVNS",
	x86_CMOVO:           "CMOVO",
	x86_CMOVP:           "CMOVP",
	x86_CMOVS:           "CMOVS",
	x86_CMP:             "CMP",
	x86_CMPPD:           "CMPPD",
	x86_CMPPS:           "CMPPS",
	x86_CMPSB:           "CMPSB",
	x86_CMPSD:           "CMPSD",
	x86_CMPSD_XMM:       "CMPSD_XMM",
	x86_CMPSQ:           "CMPSQ",
	x86_CMPSS:           "CMPSS",
	x86_CMPSW:           "CMPSW",
	x86_CMPXCHG:         "CMPXCHG",
	x86_CMPXCHG16B:      "CMPXCHG16B",
	x86_CMPXCHG8B:       "CMPXCHG8B",
	x86_COMISD:          "COMISD",
	x86_COMISS:          "COMISS",
	x86_CPUID:           "CPUID",
	x86_CQO:             "CQO",
	x86_CRC32:           "CRC32",
	x86_CVTDQ2PD:        "CVTDQ2PD",
	x86_CVTDQ2PS:        "CVTDQ2PS",
	x86_CVTPD2DQ:        "CVTPD2DQ",
	x86_CVTPD2PI:        "CVTPD2PI",
	x86_CVTPD2PS:        "CVTPD2PS",
	x86_CVTPI2PD:        "CVTPI2PD",
	x86_CVTPI2PS:        "CVTPI2PS",
	x86_CVTPS2DQ:        "CVTPS2DQ",
	x86_CVTPS2PD:        "CVTPS2PD",
	x86_CVTPS2PI:        "CVTPS2PI",
	x86_CVTSD2SI:        "CVTSD2SI",
	x86_CVTSD2SS:        "CVTSD2SS",
	x86_CVTSI2SD:        "CVTSI2SD",
	x86_CVTSI2SS:        "CVTSI2SS",
	x86_CVTSS2SD:        "CVTSS2SD",
	x86_CVTSS2SI:        "CVTSS2SI",
	x86_CVTTPD2DQ:       "CVTTPD2DQ",
	x86_CVTTPD2PI:       "CVTTPD2PI",
	x86_CVTTPS2DQ:       "CVTTPS2DQ",
	x86_CVTTPS2PI:       "CVTTPS2PI",
	x86_CVTTSD2SI:       "CVTTSD2SI",
	x86_CVTTSS2SI:       "CVTTSS2SI",
	x86_CWD:             "CWD",
	x86_CWDE:            "CWDE",
	x86_DAA:             "DAA",
	x86_DAS:             "DAS",
	x86_DEC:             "DEC",
	x86_DIV:             "DIV",
	x86_DIVPD:           "DIVPD",
	x86_DIVPS:           "DIVPS",
	x86_DIVSD:           "DIVSD",
	x86_DIVSS:           "DIVSS",
	x86_DPPD:            "DPPD",
	x86_DPPS:            "DPPS",
	x86_EMMS:            "EMMS",
	x86_ENTER:           "ENTER",
	x86_EXTRACTPS:       "EXTRACTPS",
	x86_F2XM1:           "F2XM1",
	x86_FABS:            "FABS",
	x86_FADD:            "FADD",
	x86_FADDP:           "FADDP",
	x86_FBLD:            "FBLD",
	x86_FBSTP:           "FBSTP",
	x86_FCHS:            "FCHS",
	x86_FCMOVB:          "FCMOVB",
	x86_FCMOVBE:         "FCMOVBE",
	x86_FCMOVE:          "FCMOVE",
	x86_FCMOVNB:         "FCMOVNB",
	x86_FCMOVNBE:        "FCMOVNBE",
	x86_FCMOVNE:         "FCMOVNE",
	x86_FCMOVNU:         "FCMOVNU",
	x86_FCMOVU:          "FCMOVU",
	x86_FCOM:            "FCOM",
	x86_FCOMI:           "FCOMI",
	x86_FCOMIP:          "FCOMIP",
	x86_FCOMP:           "FCOMP",
	x86_FCOMPP:          "FCOMPP",
	x86_FCOS:            "FCOS",
	x86_FDECSTP:         "FDECSTP",
	x86_FDIV:            "FDIV",
	x86_FDIVP:           "FDIVP",
	x86_FDIVR:           "FDIVR",
	x86_FDIVRP:          "FDIVRP",
	x86_FFREE:           "FFREE",
	x86_FFREEP:          "FFREEP",
	x86_FIADD:           "FIADD",
	x86_FICOM:           "FICOM",
	x86_FICOMP:          "FICOMP",
	x86_FIDIV:           "FIDIV",
	x86_FIDIVR:          "FIDIVR",
	x86_FILD:            "FILD",
	x86_FIMUL:           "FIMUL",
	x86_FINCSTP:         "FINCSTP",
	x86_FIST:            "FIST",
	x86_FISTP:           "FISTP",
	x86_FISTTP:          "FISTTP",
	x86_FISUB:           "FISUB",
	x86_FISUBR:          "FISUBR",
	x86_FLD:             "FLD",
	x86_FLD1:            "FLD1",
	x86_FLDCW:           "FLDCW",
	x86_FLDENV:          "FLDENV",
	x86_FLDL2E:          "FLDL2E",
	x86_FLDL2T:          "FLDL2T",
	x86_FLDLG2:          "FLDLG2",
	x86_FLDPI:           "FLDPI",
	x86_FMUL:            "FMUL",
	x86_FMULP:           "FMULP",
	x86_FNCLEX:          "FNCLEX",
	x86_FNINIT:          "FNINIT",
	x86_FNOP:            "FNOP",
	x86_FNSAVE:          "FNSAVE",
	x86_FNSTCW:          "FNSTCW",
	x86_FNSTENV:         "FNSTENV",
	x86_FNSTSW:          "FNSTSW",
	x86_FPATAN:          "FPATAN",
	x86_FPREM:           "FPREM",
	x86_FPREM1:          "FPREM1",
	x86_FPTAN:           "FPTAN",
	x86_FRNDINT:         "FRNDINT",
	x86_FRSTOR:          "FRSTOR",
	x86_FSCALE:          "FSCALE",
	x86_FSIN:            "FSIN",
	x86_FSINCOS:         "FSINCOS",
	x86_FSQRT:           "FSQRT",
	x86_FST:             "FST",
	x86_FSTP:            "FSTP",
	x86_FSUB:            "FSUB",
	x86_FSUBP:           "FSUBP",
	x86_FSUBR:           "FSUBR",
	x86_FSUBRP:          "FSUBRP",
	x86_FTST:            "FTST",
	x86_FUCOM:           "FUCOM",
	x86_FUCOMI:          "FUCOMI",
	x86_FUCOMIP:         "FUCOMIP",
	x86_FUCOMP:          "FUCOMP",
	x86_FUCOMPP:         "FUCOMPP",
	x86_FWAIT:           "FWAIT",
	x86_FXAM:            "FXAM",
	x86_FXCH:            "FXCH",
	x86_FXRSTOR:         "FXRSTOR",
	x86_FXRSTOR64:       "FXRSTOR64",
	x86_FXSAVE:          "FXSAVE",
	x86_FXSAVE64:        "FXSAVE64",
	x86_FXTRACT:         "FXTRACT",
	x86_FYL2X:           "FYL2X",
	x86_FYL2XP1:         "FYL2XP1",
	x86_HADDPD:          "HADDPD",
	x86_HADDPS:          "HADDPS",
	x86_HLT:             "HLT",
	x86_HSUBPD:          "HSUBPD",
	x86_HSUBPS:          "HSUBPS",
	x86_ICEBP:           "ICEBP",
	x86_IDIV:            "IDIV",
	x86_IMUL:            "IMUL",
	x86_IN:              "IN",
	x86_INC:             "INC",
	x86_INSB:            "INSB",
	x86_INSD:            "INSD",
	x86_INSERTPS:        "INSERTPS",
	x86_INSW:            "INSW",
	x86_INT:             "INT",
	x86_INTO:            "INTO",
	x86_INVD:            "INVD",
	x86_INVLPG:          "INVLPG",
	x86_INVPCID:         "INVPCID",
	x86_IRET:            "IRET",
	x86_IRETD:           "IRETD",
	x86_IRETQ:           "IRETQ",
	x86_JA:              "JA",
	x86_JAE:             "JAE",
	x86_JB:              "JB",
	x86_JBE:             "JBE",
	x86_JCXZ:            "JCXZ",
	x86_JE:              "JE",
	x86_JECXZ:           "JECXZ",
	x86_JG:              "JG",
	x86_JGE:             "JGE",
	x86_JL:              "JL",
	x86_JLE:             "JLE",
	x86_JMP:             "JMP",
	x86_JNE:             "JNE",
	x86_JNO:             "JNO",
	x86_JNP:             "JNP",
	x86_JNS:             "JNS",
	x86_JO:              "JO",
	x86_JP:              "JP",
	x86_JRCXZ:           "JRCXZ",
	x86_JS:              "JS",
	x86_LAHF:            "LAHF",
	x86_LAR:             "LAR",
	x86_LCALL:           "LCALL",
	x86_LDDQU:           "LDDQU",
	x86_LDMXCSR:         "LDMXCSR",
	x86_LDS:             "LDS",
	x86_LEA:             "LEA",
	x86_LEAVE:           "LEAVE",
	x86_LES:             "LES",
	x86_LFENCE:          "LFENCE",
	x86_LFS:             "LFS",
	x86_LGDT:            "LGDT",
	x86_LGS:             "LGS",
	x86_LIDT:            "LIDT",
	x86_LJMP:            "LJMP",
	x86_LLDT:            "LLDT",
	x86_LMSW:            "LMSW",
	x86_LODSB:           "LODSB",
	x86_LODSD:           "LODSD",
	x86_LODSQ:           "LODSQ",
	x86_LODSW:           "LODSW",
	x86_LOOP:            "LOOP",
	x86_LOOPE:           "LOOPE",
	x86_LOOPNE:          "LOOPNE",
	x86_LRET:            "LRET",
	x86_LSL:             "LSL",
	x86_LSS:             "LSS",
	x86_LTR:             "LTR",
	x86_LZCNT:           "LZCNT",
	x86_MASKMOVDQU:      "MASKMOVDQU",
	x86_MASKMOVQ:        "MASKMOVQ",
	x86_MAXPD:           "MAXPD",
	x86_MAXPS:           "MAXPS",
	x86_MAXSD:           "MAXSD",
	x86_MAXSS:           "MAXSS",
	x86_MFENCE:          "MFENCE",
	x86_MINPD:           "MINPD",
	x86_MINPS:           "MINPS",
	x86_MINSD:           "MINSD",
	x86_MINSS:           "MINSS",
	x86_MONITOR:         "MONITOR",
	x86_MOV:             "MOV",
	x86_MOVAPD:          "MOVAPD",
	x86_MOVAPS:          "MOVAPS",
	x86_MOVBE:           "MOVBE",
	x86_MOVD:            "MOVD",
	x86_MOVDDUP:         "MOVDDUP",
	x86_MOVDQ2Q:         "MOVDQ2Q",
	x86_MOVDQA:          "MOVDQA",
	x86_MOVDQU:          "MOVDQU",
	x86_MOVHLPS:         "MOVHLPS",
	x86_MOVHPD:          "MOVHPD",
	x86_MOVHPS:          "MOVHPS",
	x86_MOVLHPS:         "MOVLHPS",
	x86_MOVLPD:          "MOVLPD",
	x86_MOVLPS:          "MOVLPS",
	x86_MOVMSKPD:        "MOVMSKPD",
	x86_MOVMSKPS:        "MOVMSKPS",
	x86_MOVNTDQ:         "MOVNTDQ",
	x86_MOVNTDQA:        "MOVNTDQA",
	x86_MOVNTI:          "MOVNTI",
	x86_MOVNTPD:         "MOVNTPD",
	x86_MOVNTPS:         "MOVNTPS",
	x86_MOVNTQ:          "MOVNTQ",
	x86_MOVNTSD:         "MOVNTSD",
	x86_MOVNTSS:         "MOVNTSS",
	x86_MOVQ:            "MOVQ",
	x86_MOVQ2DQ:         "MOVQ2DQ",
	x86_MOVSB:           "MOVSB",
	x86_MOVSD:           "MOVSD",
	x86_MOVSD_XMM:       "MOVSD_XMM",
	x86_MOVSHDUP:        "MOVSHDUP",
	x86_MOVSLDUP:        "MOVSLDUP",
	x86_MOVSQ:           "MOVSQ",
	x86_MOVSS:           "MOVSS",
	x86_MOVSW:           "MOVSW",
	x86_MOVSX:           "MOVSX",
	x86_MOVSXD:          "MOVSXD",
	x86_MOVUPD:          "MOVUPD",
	x86_MOVUPS:          "MOVUPS",
	x86_MOVZX:           "MOVZX",
	x86_MPSADBW:         "MPSADBW",
	x86_MUL:             "MUL",
	x86_MULPD:           "MULPD",
	x86_MULPS:           "MULPS",
	x86_MULSD:           "MULSD",
	x86_MULSS:           "MULSS",
	x86_MWAIT:           "MWAIT",
	x86_NEG:             "NEG",
	x86_NOP:             "NOP",
	x86_NOT:             "NOT",
	x86_OR:              "OR",
	x86_ORPD:            "ORPD",
	x86_ORPS:            "ORPS",
	x86_OUT:             "OUT",
	x86_OUTSB:           "OUTSB",
	x86_OUTSD:           "OUTSD",
	x86_OUTSW:           "OUTSW",
	x86_PABSB:           "PABSB",
	x86_PABSD:           "PABSD",
	x86_PABSW:           "PABSW",
	x86_PACKSSDW:        "PACKSSDW",
	x86_PACKSSWB:        "PACKSSWB",
	x86_PACKUSDW:        "PACKUSDW",
	x86_PACKUSWB:        "PACKUSWB",
	x86_PADDB:           "PADDB",
	x86_PADDD:           "PADDD",
	x86_PADDQ:           "PADDQ",
	x86_PADDSB:          "PADDSB",
	x86_PADDSW:          "PADDSW",
	x86_PADDUSB:         "PADDUSB",
	x86_PADDUSW:         "PADDUSW",
	x86_PADDW:           "PADDW",
	x86_PALIGNR:         "PALIGNR",
	x86_PAND:            "PAND",
	x86_PANDN:           "PANDN",
	x86_PAUSE:           "PAUSE",
	x86_PAVGB:           "PAVGB",
	x86_PAVGW:           "PAVGW",
	x86_PBLENDVB:        "PBLENDVB",
	x86_PBLENDW:         "PBLENDW",
	x86_PCLMULQDQ:       "PCLMULQDQ",
	x86_PCMPEQB:         "PCMPEQB",
	x86_PCMPEQD:         "PCMPEQD",
	x86_PCMPEQQ:         "PCMPEQQ",
	x86_PCMPEQW:         "PCMPEQW",
	x86_PCMPESTRI:       "PCMPESTRI",
	x86_PCMPESTRM:       "PCMPESTRM",
	x86_PCMPGTB:         "PCMPGTB",
	x86_PCMPGTD:         "PCMPGTD",
	x86_PCMPGTQ:         "PCMPGTQ",
	x86_PCMPGTW:         "PCMPGTW",
	x86_PCMPISTRI:       "PCMPISTRI",
	x86_PCMPISTRM:       "PCMPISTRM",
	x86_PEXTRB:          "PEXTRB",
	x86_PEXTRD:          "PEXTRD",
	x86_PEXTRQ:          "PEXTRQ",
	x86_PEXTRW:          "PEXTRW",
	x86_PHADDD:          "PHADDD",
	x86_PHADDSW:         "PHADDSW",
	x86_PHADDW:          "PHADDW",
	x86_PHMINPOSUW:      "PHMINPOSUW",
	x86_PHSUBD:          "PHSUBD",
	x86_PHSUBSW:         "PHSUBSW",
	x86_PHSUBW:          "PHSUBW",
	x86_PINSRB:          "PINSRB",
	x86_PINSRD:          "PINSRD",
	x86_PINSRQ:          "PINSRQ",
	x86_PINSRW:          "PINSRW",
	x86_PMADDUBSW:       "PMADDUBSW",
	x86_PMADDWD:         "PMADDWD",
	x86_PMAXSB:          "PMAXSB",
	x86_PMAXSD:          "PMAXSD",
	x86_PMAXSW:          "PMAXSW",
	x86_PMAXUB:          "PMAXUB",
	x86_PMAXUD:          "PMAXUD",
	x86_PMAXUW:          "PMAXUW",
	x86_PMINSB:          "PMINSB",
	x86_PMINSD:          "PMINSD",
	x86_PMINSW:          "PMINSW",
	x86_PMINUB:          "PMINUB",
	x86_PMINUD:          "PMINUD",
	x86_PMINUW:          "PMINUW",
	x86_PMOVMSKB:        "PMOVMSKB",
	x86_PMOVSXBD:        "PMOVSXBD",
	x86_PMOVSXBQ:        "PMOVSXBQ",
	x86_PMOVSXBW:        "PMOVSXBW",
	x86_PMOVSXDQ:        "PMOVSXDQ",
	x86_PMOVSXWD:        "PMOVSXWD",
	x86_PMOVSXWQ:        "PMOVSXWQ",
	x86_PMOVZXBD:        "PMOVZXBD",
	x86_PMOVZXBQ:        "PMOVZXBQ",
	x86_PMOVZXBW:        "PMOVZXBW",
	x86_PMOVZXDQ:        "PMOVZXDQ",
	x86_PMOVZXWD:        "PMOVZXWD",
	x86_PMOVZXWQ:        "PMOVZXWQ",
	x86_PMULDQ:          "PMULDQ",
	x86_PMULHRSW:        "PMULHRSW",
	x86_PMULHUW:         "PMULHUW",
	x86_PMULHW:          "PMULHW",
	x86_PMULLD:          "PMULLD",
	x86_PMULLW:          "PMULLW",
	x86_PMULUDQ:         "PMULUDQ",
	x86_POP:             "POP",
	x86_POPA:            "POPA",
	x86_POPAD:           "POPAD",
	x86_POPCNT:          "POPCNT",
	x86_POPF:            "POPF",
	x86_POPFD:           "POPFD",
	x86_POPFQ:           "POPFQ",
	x86_POR:             "POR",
	x86_PREFETCHNTA:     "PREFETCHNTA",
	x86_PREFETCHT0:      "PREFETCHT0",
	x86_PREFETCHT1:      "PREFETCHT1",
	x86_PREFETCHT2:      "PREFETCHT2",
	x86_PREFETCHW:       "PREFETCHW",
	x86_PSADBW:          "PSADBW",
	x86_PSHUFB:          "PSHUFB",
	x86_PSHUFD:          "PSHUFD",
	x86_PSHUFHW:         "PSHUFHW",
	x86_PSHUFLW:         "PSHUFLW",
	x86_PSHUFW:          "PSHUFW",
	x86_PSIGNB:          "PSIGNB",
	x86_PSIGND:          "PSIGND",
	x86_PSIGNW:          "PSIGNW",
	x86_PSLLD:           "PSLLD",
	x86_PSLLDQ:          "PSLLDQ",
	x86_PSLLQ:           "PSLLQ",
	x86_PSLLW:           "PSLLW",
	x86_PSRAD:           "PSRAD",
	x86_PSRAW:           "PSRAW",
	x86_PSRLD:           "PSRLD",
	x86_PSRLDQ:          "PSRLDQ",
	x86_PSRLQ:           "PSRLQ",
	x86_PSRLW:           "PSRLW",
	x86_PSUBB:           "PSUBB",
	x86_PSUBD:           "PSUBD",
	x86_PSUBQ:           "PSUBQ",
	x86_PSUBSB:          "PSUBSB",
	x86_PSUBSW:          "PSUBSW",
	x86_PSUBUSB:         "PSUBUSB",
	x86_PSUBUSW:         "PSUBUSW",
	x86_PSUBW:           "PSUBW",
	x86_PTEST:           "PTEST",
	x86_PUNPCKHBW:       "PUNPCKHBW",
	x86_PUNPCKHDQ:       "PUNPCKHDQ",
	x86_PUNPCKHQDQ:      "PUNPCKHQDQ",
	x86_PUNPCKHWD:       "PUNPCKHWD",
	x86_PUNPCKLBW:       "PUNPCKLBW",
	x86_PUNPCKLDQ:       "PUNPCKLDQ",
	x86_PUNPCKLQDQ:      "PUNPCKLQDQ",
	x86_PUNPCKLWD:       "PUNPCKLWD",
	x86_PUSH:            "PUSH",
	x86_PUSHA:           "PUSHA",
	x86_PUSHAD:          "PUSHAD",
	x86_PUSHF:           "PUSHF",
	x86_PUSHFD:          "PUSHFD",
	x86_PUSHFQ:          "PUSHFQ",
	x86_PXOR:            "PXOR",
	x86_RCL:             "RCL",
	x86_RCPPS:           "RCPPS",
	x86_RCPSS:           "RCPSS",
	x86_RCR:             "RCR",
	x86_RDFSBASE:        "RDFSBASE",
	x86_RDGSBASE:        "RDGSBASE",
	x86_RDMSR:           "RDMSR",
	x86_RDPMC:           "RDPMC",
	x86_RDRAND:          "RDRAND",
	x86_RDTSC:           "RDTSC",
	x86_RDTSCP:          "RDTSCP",
	x86_RET:             "RET",
	x86_ROL:             "ROL",
	x86_ROR:             "ROR",
	x86_ROUNDPD:         "ROUNDPD",
	x86_ROUNDPS:         "ROUNDPS",
	x86_ROUNDSD:         "ROUNDSD",
	x86_ROUNDSS:         "ROUNDSS",
	x86_RSM:             "RSM",
	x86_RSQRTPS:         "RSQRTPS",
	x86_RSQRTSS:         "RSQRTSS",
	x86_SAHF:            "SAHF",
	x86_SAR:             "SAR",
	x86_SBB:             "SBB",
	x86_SCASB:           "SCASB",
	x86_SCASD:           "SCASD",
	x86_SCASQ:           "SCASQ",
	x86_SCASW:           "SCASW",
	x86_SETA:            "SETA",
	x86_SETAE:           "SETAE",
	x86_SETB:            "SETB",
	x86_SETBE:           "SETBE",
	x86_SETE:            "SETE",
	x86_SETG:            "SETG",
	x86_SETGE:           "SETGE",
	x86_SETL:            "SETL",
	x86_SETLE:           "SETLE",
	x86_SETNE:           "SETNE",
	x86_SETNO:           "SETNO",
	x86_SETNP:           "SETNP",
	x86_SETNS:           "SETNS",
	x86_SETO:            "SETO",
	x86_SETP:            "SETP",
	x86_SETS:            "SETS",
	x86_SFENCE:          "SFENCE",
	x86_SGDT:            "SGDT",
	x86_SHL:             "SHL",
	x86_SHLD:            "SHLD",
	x86_SHR:             "SHR",
	x86_SHRD:            "SHRD",
	x86_SHUFPD:          "SHUFPD",
	x86_SHUFPS:          "SHUFPS",
	x86_SIDT:            "SIDT",
	x86_SLDT:            "SLDT",
	x86_SMSW:            "SMSW",
	x86_SQRTPD:          "SQRTPD",
	x86_SQRTPS:          "SQRTPS",
	x86_SQRTSD:          "SQRTSD",
	x86_SQRTSS:          "SQRTSS",
	x86_STC:             "STC",
	x86_STD:             "STD",
	x86_STI:             "STI",
	x86_STMXCSR:         "STMXCSR",
	x86_STOSB:           "STOSB",
	x86_STOSD:           "STOSD",
	x86_STOSQ:           "STOSQ",
	x86_STOSW:           "STOSW",
	x86_STR:             "STR",
	x86_SUB:             "SUB",
	x86_SUBPD:           "SUBPD",
	x86_SUBPS:           "SUBPS",
	x86_SUBSD:           "SUBSD",
	x86_SUBSS:           "SUBSS",
	x86_SWAPGS:          "SWAPGS",
	x86_SYSCALL:         "SYSCALL",
	x86_SYSENTER:        "SYSENTER",
	x86_SYSEXIT:         "SYSEXIT",
	x86_SYSRET:          "SYSRET",
	x86_TEST:            "TEST",
	x86_TZCNT:           "TZCNT",
	x86_UCOMISD:         "UCOMISD",
	x86_UCOMISS:         "UCOMISS",
	x86_UD1:             "UD1",
	x86_UD2:             "UD2",
	x86_UNPCKHPD:        "UNPCKHPD",
	x86_UNPCKHPS:        "UNPCKHPS",
	x86_UNPCKLPD:        "UNPCKLPD",
	x86_UNPCKLPS:        "UNPCKLPS",
	x86_VERR:            "VERR",
	x86_VERW:            "VERW",
	x86_WBINVD:          "WBINVD",
	x86_WRFSBASE:        "WRFSBASE",
	x86_WRGSBASE:        "WRGSBASE",
	x86_WRMSR:           "WRMSR",
	x86_XABORT:          "XABORT",
	x86_XADD:            "XADD",
	x86_XBEGIN:          "XBEGIN",
	x86_XCHG:            "XCHG",
	x86_XEND:            "XEND",
	x86_XGETBV:          "XGETBV",
	x86_XLATB:           "XLATB",
	x86_XOR:             "XOR",
	x86_XORPD:           "XORPD",
	x86_XORPS:           "XORPS",
	x86_XRSTOR:          "XRSTOR",
	x86_XRSTOR64:        "XRSTOR64",
	x86_XRSTORS:         "XRSTORS",
	x86_XRSTORS64:       "XRSTORS64",
	x86_XSAVE:           "XSAVE",
	x86_XSAVE64:         "XSAVE64",
	x86_XSAVEC:          "XSAVEC",
	x86_XSAVEC64:        "XSAVEC64",
	x86_XSAVEOPT:        "XSAVEOPT",
	x86_XSAVEOPT64:      "XSAVEOPT64",
	x86_XSAVES:          "XSAVES",
	x86_XSAVES64:        "XSAVES64",
	x86_XSETBV:          "XSETBV",
	x86_XTEST:           "XTEST",
}
